/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/sta/send.c#5 $
 *
 * Copyright Â© 2000-2004 Atheros Communications, Inc., All Rights Reserved
 *
 * Atheros and the Atheros logo and design are trademarks of Atheros
 * Communications, Inc.
 *
 * Sample Code from Microsoft Windows 2000 Driver Development Kit is
 * used under license from Microsoft Corporation and was developed for
 * Microsoft by Intel Corp., Hillsboro, Oregon: Copyright (c) 1994-1997
 * by Intel Corporation.
 *
 * Contains all the routines to queue protocol/mgmt transmits onto the h/w
 * and process their completions for NDIS driver
 */

#include "wlantype.h"
#include "wlandrv.h"
#include "wlanext.h"
#include "wlansmeext.h"
#include "wlanDev.h"
#include "wlanCapDe.h"
#include "wlanframe.h"
#include "wlanSend.h"
#include "wlanDev.h"
#include "stacserv.h"
#include "intercept.h"
#include "wdcApi.h"
#if NDIS_WDM
#include "athusbapi.h"
#endif
#include "usbstub.h"

#ifdef DEBUG
int txDebugLevel;
#endif

/* Forward declarations */
static NDIS_STATUS
SetupNextSend(WLAN_DEV_INFO *pdevInfo, PNDIS_PACKET Packet);

static VOID
queueSendPacket(WLAN_DEV_INFO *pDevInfo, PNDIS_PACKET Packet);

static A_BOOL
wlanSendCheck(WLAN_DEV_INFO *pDev);
/*******************************************************************************
 * setupEncrypt - If this is a WEP frame we need to:
 *  - generate the random IV (incrementing is good enough)
 *  - Add the ICV to the last descriptor's length.
 *  - perform sw encryption if needed
 *  - Adjust the header length to accomodate the PCU's need to include
 *    need to include the 32 bit phys pointer to the descriptor.
 *  - Set the key cache index.
 *  - Set the descriptor key valid flag.
 *
 * returns hw index.
 */

static INLINE A_UINT32
setupEncrypt(WLAN_DEV_INFO *pdevInfo, ATHEROS_DESC *pDesc)
{
    WLAN_FRAME_HEADER *pWlanHdr = pDesc->pBufferVirtPtr.header;
    SIB_ENTRY         *pSib     = pDesc->pDestSibEntry;
    WLAN_PRIV_RECORD  *pKey;
    A_UINT16           hwidx;
    A_BOOL             faulted;

    pDesc->pKey = NULL;

    if (!pWlanHdr->frameControl.wep) {
        if (!pSib) {
            return 0;
        }

        /*
         * A key slot should have been allocated at that point in IBSS mode.
         * It might not be the case in ad hoc.
         */
        if ((pSib->pPriv == NULL) &&
            (pdevInfo->bssDescr) &&
            (pdevInfo->bssDescr->bsstype == INDEPENDENT_BSS)) {

            if (pdevInfo->staConfig.privacyInvoked) {
                WLAN_PRIV_RECORD *pKey;

                pKey = &pdevInfo->keyTable[pdevInfo->staConfig.defaultKey];
                pSib->pPriv = pKey;
                wlanInstallConnectionKey(pdevInfo,
                                      pSib,
                                      &pSib->macAddr,
                                      pKey);
            } else {
                WLAN_PRIV_RECORD key;

                tx2Printf ("wlanMlmeAuthRequest: installing dummy default key\n");
                A_MEM_ZERO(&key, sizeof(key));
                key.keyType = PRIV_KEY_TYPE_NULL;

                wlanInstallConnectionKey(pdevInfo,
                                         pSib,
                                         &pSib->macAddr,
                                         &key);
            }
        }
        return pSib->wdcConnId;
    }

    /* multicast frames do not have a SIB - use shared key. */
    /* XXX Divy. Should change hwidx variable name */
    if (pSib == NULL) {
        hwidx = pdevInfo->staConfig.defaultKey;
        if (hwidx == HWINDEX_INVALID) {
            return HWINDEX_INVALID;
        }
        pKey = &pdevInfo->keyTable[hwidx];

    } else {
        /* Ad Hoc can tx encrypted frame w/o much notice. */
        /* XXX Divy. Need to work on ad hoc */
        if (pSib->pPriv == NULL) {
            wlanKeyEnable(pdevInfo, pSib, pdevInfo->staConfig.defaultKey);
            if (pSib->pPriv == NULL) {
                return A_ERROR;
            }
        }
        pKey = pSib->pPriv;
        hwidx = pSib->wdcConnId;
    }

    ASSERT(pKey);

    pDesc->pKey = pKey;

    frameIVSetup(pdevInfo, pKey, pWlanHdr);

    if (!cipherSupported(pdevInfo, pKey->keyType)) {
        if (swEnCipher(pdevInfo, pKey, pDesc) != A_OK) {
            return HWINDEX_INVALID;
        }

        /* add icv to the last buffer and frame length */
        pDesc->pTxLastDesc->bufferLength += icvLengthGet(pKey);
    }

    pDesc->frameLength += icvLengthGet(pKey);
    pDesc->pKey = pKey;

    cryptoEndianChange(pDesc->pBufferVirtPtr.header, CPU2LE);

    return hwidx;
}

//xxxxPGSxxxx - this should go back to common code (reintroduce wlansend.c)!!!!!!
/*************************************************************************
 *
 * wlanFrameTransmit - entry point for transmit of a single new frame;
 *   called for all types of 802.11 frames after the frame has been
 *   fully assembled - except for fragmentation, and is ready for final
 *   xmit preparation
 *
 * RETURNS: error status
 *   Discards the frame on error
 */

A_STATUS
wlanFrameTransmit(WLAN_DEV_INFO *pdevInfo, QUEUE_DETAILS *pTxQueue, ATHEROS_DESC *pDesc)
{
    SIB_ENTRY    *pSib      = pDesc->pDestSibEntry;
    A_UINT16      frameType = pDesc->pBufferVirtPtr.header->frameControl.fType;
    ATHEROS_DESC *pTxDesc;
    A_BOOL        isFrag;

    if (pdevInfo->NicResetInProgress || (!apDeviceIsUp(pdevInfo))) {
        freeBuffandDescChain(pdevInfo, pDesc);
        return A_ERROR;
    }


    pdevInfo->localSta->stats.TxFrames[frameType]++;
    if (pSib) {
        pSib->stats.TxFrames[frameType]++;
    }

    ASSERT(frameType != FRAME_RESERVED);

    /*
     * track time for when the frame first came in for transmit for
     * subsequent aging / MSDU lifetime type of operations
     */
    pDesc->timeStamp = A_MS_TICKGET();

    /* fragment the frame */
    pDesc = wlanFrag(pdevInfo, pDesc);
    if (!pDesc) {
        return A_ERROR;
    }

    isFrag = pDesc->pBufferVirtPtr.header3->frameControl.moreFrag;

    /* set up each of the fragments for transmit */

    for (pTxDesc = pDesc; pTxDesc; pTxDesc = pTxDesc->pTxLastDesc->pNextVirtPtr) {
        A_TX_DESC_CHAIN_FORMAT(pTxDesc);

        /* Increase the buffer length with the FCS_FIELD_SIZE */
        pTxDesc->frameLength += FCS_FIELD_SIZE;

        /* Handle key cache, update descriptor + IV for WEP frames */
        pTxDesc->hwIndex = setupEncrypt(pdevInfo, pTxDesc);
        if (pTxDesc->hwIndex == HWINDEX_INVALID) {
            /* oversubscribed key cache or swencryption failure */
#if TBD /* xxxxLMWxxxx */
            LOG_DROP_FRAME(pdevInfo, tx);
#endif
            logMsg("setupEncrypt failed: frame dropped for SIB %p\n", pSib);

            freeBuffandDescChain(pdevInfo, pDesc);
            return A_ERROR;
        }
    }


    NdisInterlockedIncrement(&pdevInfo->pOSHandle->txIrpPended);

    if (!isFrag) {

        /* Convert ATHEROS_DESC to WDC_BUFFER_DESC (WDC) */
        A_TX_DESC_CHAIN_TO_WDC(pDesc);

        if (wdcSend(pdevInfo->targetHandle,
                    (HOST_FRAME_HANDLE) pDesc,
                    pTxQueue->targetQueueId,
                    pDesc->pDestSibEntry ?
                    pDesc->pDestSibEntry->wdcConnId : WDC_INVALID_CONN_ID,
                    &pDesc->bufDesc,
                    (A_BOOL)pDesc->bIndicateXmitStatus) == A_ERROR)
        {
            NdisInterlockedDecrement(&pdevInfo->pOSHandle->txIrpPended);
            freeBuffandDescChain(pdevInfo, pDesc);
            return A_ERROR;
        }

        if (pDesc->bIndicateXmitStatus) {
            A_SEM_LOCK(pdevInfo->txPendingQueue.qSem, WAIT_FOREVER);
            descQPushTail(&pdevInfo->txPendingQueue, pDesc, pDesc);
            A_SEM_UNLOCK(pdevInfo->txPendingQueue.qSem);
        }

    } else {
        ATHEROS_DESC *pDescTmp = pDesc;

        while (pDescTmp) {

            A_TX_DESC_CHAIN_TO_WDC(pDescTmp);

            /* bIndicateXmitStatus=1 w/ Fragmentation is untested -
               we should never hit this condition */
            ASSERT(pDescTmp->bIndicateXmitStatus == 0);

            if (wdcSend(pdevInfo->targetHandle,
                        (HOST_FRAME_HANDLE) pDescTmp,
                        pTxQueue->targetQueueId,
                        pDescTmp->pDestSibEntry ?
                        pDescTmp->pDestSibEntry->wdcConnId : WDC_INVALID_CONN_ID,
                        &pDescTmp->bufDesc,
                        (A_BOOL)pDesc->bIndicateXmitStatus) == A_ERROR)
            {
                NdisInterlockedDecrement(&pdevInfo->pOSHandle->txIrpPended);
                freeBuffandDescChain(pdevInfo, pDescTmp);
                return A_ERROR;
            }

            if (pDescTmp->bIndicateXmitStatus) {
                A_SEM_LOCK(pdevInfo->txPendingQueue.qSem, WAIT_FOREVER);
                descQPushTail(&pdevInfo->txPendingQueue, pDescTmp, pDescTmp);
                A_SEM_UNLOCK(pdevInfo->txPendingQueue.qSem);
            }

            if (pDescTmp->more) {
                /* Skip over all continuation descriptors */
                while(pDescTmp && pDescTmp->more)
                    pDescTmp = pDescTmp->pNextVirtPtr;

                /* Skip over tail descriptor */
                if (pDescTmp)
                    pDescTmp = pDescTmp->pNextVirtPtr;
            } else {
                pDescTmp = NULL;
            }
        }
    }

    return A_OK;
}

A_BOOL
wlanSendCheck(WLAN_DEV_INFO *pDev)
{
    if ((pDev->targetHandle == NULL) || 
        (pDev->powerMgmt.hibernate == ENABLE) ||
        (pDev->busStatus == ATHUSB_BUS_STATE_SURPRISE_REMOVED) ||
        (pDev->busStatus == ATHUSB_BUS_STATE_FATAL) ||
        (pDev->busStatus == ATHUSB_BUS_STATE_SUSPEND))
    {
        uiPrintf("Device Not Present\n");
        return FALSE;
    }

    return TRUE;
}


NDIS_STATUS
athSendSinglePacket(WLAN_DEV_INFO *pDevInfo, PNDIS_PACKET Packet)
{
    ATHEROS_DESC *pDesc;
    WLAN_MACADDR *pDestAddr;
    A_STATUS     status;

    /*
     * Convert the given NDIS packet to a chain of Atheros descriptors.
     * Coalesce the buffers if necessary.
     */
    ATH_RELEASE_SPINLOCK(pDevInfo->sendLock);
    pDesc = osPacketToAthDesc(pDevInfo, Packet, &status);
    ATH_ACQUIRE_SPINLOCK(pDevInfo->sendLock);

    if (pDesc == NULL) {
        /*
         * For deserialized driver, don't return NDIS_STATUS_RESOURCES, we'll
         * need to hold on to the packet in a separate place.
         */
        logMsg("SetupNextSend: out of descriptor/buffer/map resources\n");
        if (status == A_OK) {
            return NDIS_STATUS_RESOURCES;
        } else {
            NdisMSendComplete(pDevInfo->pOSHandle->NicAdapterHandle, 
                              Packet, NDIS_STATUS_FAILURE);
            NdisInterlockedDecrement(&pDevInfo->pOSHandle->txFramesPended);
            pDevInfo->localSta->stats.TransmitErrors--;
            return NDIS_STATUS_FAILURE;
        }
    }

    /* station uses the basic virtual device only */
    pDesc->pOpBss = &pDevInfo->baseBss;
    pDestAddr = (pDevInfo->bssDescr->bsstype == INFRASTRUCTURE_BSS)
                ? &pDesc->pOpBss->bssId
                : &pDesc->pBufferVirtPtr.lanHeader->destAddr;
    pDesc->pDestSibEntry = sibEntryFind(pDevInfo, pDestAddr, NULL);

    status = wlanDataFrameTransmit(pDevInfo, pDesc);
    if (status != A_OK) {
        return NDIS_STATUS_FAILURE;
    }

    cservUpdateLastActivityTime(pDevInfo);

    return NDIS_STATUS_SUCCESS;
}

#define MAX_QUEUE_TARGET   384

VOID
queueSendPacket(WLAN_DEV_INFO *pDevInfo, PNDIS_PACKET Packet)
{
    MP_RSRVD_SET_REF_COUNT(Packet, 0);
    InsertTailQueue(&pDevInfo->sendWaitQueue,
                    ATH_GET_PACKET_MR(Packet));
    pDevInfo->nWaitSend ++;
    txPrintf("queueSendPacket: Packet %X, nWaitSend %d\n", Packet, pDevInfo->nWaitSend); 
}

VOID
drainSendWaitQueue(WLAN_DEV_INFO *pDevInfo)
{
    PNDIS_PACKET        packet;
    PPACKET_QUEUE_ENTRY pEntry;
    NDIS_STATUS         status;
    OS_DEV_INFO         *pOsInfo  = pDevInfo->pOSHandle;

    tx2Printf("drainSendWaitQueue\n");

    if (pDevInfo->invalidTxChannel ||
        pDevInfo->NicResetInProgress)
    {
        return;
    }

    ATH_ACQUIRE_SPINLOCK(pDevInfo->lock);
    pDevInfo->progressWaitQueue ++;
    if (pDevInfo->progressWaitQueue > 1) {
        pDevInfo->progressWaitQueue --;
        ATH_RELEASE_SPINLOCK(pDevInfo->lock);
        return;
    }
    ATH_RELEASE_SPINLOCK(pDevInfo->lock);
    
    ATH_ACQUIRE_SPINLOCK(pDevInfo->sendLock);
    ATH_ACQUIRE_SPINLOCK(pDevInfo->lock);

    if (!IsQueueEmpty(&pDevInfo->sendWaitQueue) &&
        (powerSendRequest(pDevInfo) != A_OK))
    {
        pDevInfo->progressWaitQueue --;
        ATH_RELEASE_SPINLOCK(pDevInfo->lock);
        ATH_RELEASE_SPINLOCK(pDevInfo->sendLock);
        return;
    }

    ATH_RELEASE_SPINLOCK(pDevInfo->lock);

    while (!IsQueueEmpty(&pDevInfo->sendWaitQueue) &&
            pDevInfo->pOSHandle->resourceAvail)
    {
        pEntry = RemoveHeadQueue(&pDevInfo->sendWaitQueue);

        ASSERT(pEntry);

        pDevInfo->nWaitSend--;

#ifdef Linux
        packet = (PNDIS_PACKET)pEntry;
#endif
        txPrintf("drainSendWaitQueue: Packet remove from wait queue %x, nWaitSend %d\n", packet, pDevInfo->nWaitSend);

        MP_RSRVD_SET_REF_COUNT(packet, 0);

        if (pDevInfo->nWaitSend > MAX_QUEUE_TARGET) {
            /* If we queue too many packet and we can't drain it 
             * in the reasonable time, we have trouble to pass
             * NDIS test*/
            ATH_RELEASE_SPINLOCK(pDevInfo->sendLock);
            NdisMSendComplete(pOsInfo->NicAdapterHandle,
                              packet,
                              NDIS_STATUS_SUCCESS);
            txPrintf("Complete the buffer since we queue %d packet\n",MAX_QUEUE_TARGET);
            ATH_ACQUIRE_SPINLOCK(pDevInfo->sendLock);
            NdisInterlockedDecrement(&pDevInfo->pOSHandle->txFramesPended);
            pDevInfo->localSta->stats.GoodTransmits ++;
            continue;
        }

        if (!pOsInfo->openForBusiness  ||
            !pOsInfo->connectionStatus ||
            !wlanSendCheck(pDevInfo)  ||
            pDevInfo->NdisHaltInProgress)
        {
            txPrintf("drainSendWaitQueue: Packet completed with failure %x\n", packet);
            ATH_RELEASE_SPINLOCK(pDevInfo->sendLock);
            NdisMSendComplete(pOsInfo->NicAdapterHandle,
                              packet,
                              NDIS_STATUS_FAILURE);
            ATH_ACQUIRE_SPINLOCK(pDevInfo->sendLock);
            NdisInterlockedDecrement(&pDevInfo->pOSHandle->txFramesPended);
        } else {
            if (pDevInfo->busStatus == ATHUSB_BUS_STATE_ERROR) {
                /* 
                 * still keep the packet in the queue before endpoint reset
                 * finished
                 */
                status = NDIS_STATUS_RESOURCES;
            } else {
                status = athSendSinglePacket(pDevInfo,packet);
            }

            if (status == NDIS_STATUS_RESOURCES) {
                txPrintf("drainSendWaitQueue: Out of resources, requeue packet %x\n", packet);
                MP_RSRVD_SET_REF_COUNT(packet, 0);
                InsertHeadQueue(&pDevInfo->sendWaitQueue,
                                ATH_GET_PACKET_MR(packet));
                pDevInfo->nWaitSend ++;
                break;
            }
        }
    }

    ATH_RELEASE_SPINLOCK(pDevInfo->sendLock);

    ATH_ACQUIRE_SPINLOCK(pDevInfo->lock);
    pDevInfo->progressWaitQueue --;
    ATH_RELEASE_SPINLOCK(pDevInfo->lock);
}

VOID
abortSendWaitQueue(WLAN_DEV_INFO *pDevInfo)
{
    PNDIS_PACKET        packet;
    PPACKET_QUEUE_ENTRY pEntry;
    NDIS_STATUS         status;
    OS_DEV_INFO        *pOsInfo  = pDevInfo->pOSHandle;

    txPrintf("abortSendWaitQueue\n");

    ATH_ACQUIRE_SPINLOCK(pDevInfo->sendLock);

    while (!IsQueueEmpty(&pDevInfo->sendWaitQueue)) {
        pEntry = RemoveHeadQueue(&pDevInfo->sendWaitQueue);

        ASSERT(pEntry);

        pDevInfo->nWaitSend--;

#ifdef Linux
        packet = (PNDIS_PACKET)pEntry;
#endif

        ATH_RELEASE_SPINLOCK(pDevInfo->sendLock);

        if (pDevInfo->NdisResetInProgress) {
            status = NDIS_STATUS_REQUEST_ABORTED;
        } else {
            status = NDIS_STATUS_SUCCESS;
        }
        NdisMSendComplete(pOsInfo->NicAdapterHandle,
                          packet,
                          status);

        NdisInterlockedDecrement(&pOsInfo->txFramesPended);
        ATH_ACQUIRE_SPINLOCK(pDevInfo->sendLock);
        pDevInfo->localSta->stats.GoodTransmits ++;
    }

    ATH_RELEASE_SPINLOCK(pDevInfo->sendLock);
}

/*******************************************************************************
 * TransmitCompletion
 *
 * This routine cleans up after a transmitted frame. It updates the transmit
 * statistic counters, frees up descriptors and map regs, and issues a send
 * completion to the OS which will unlock the pages.
 *
 * RETURNS: TRUE   If we completed packets to the OS
 *          FALSE  If they were internally generated frames
 */

void
TransmitCompletion(WLAN_DEV_INFO *pDevInfo, ATHEROS_DESC *pDesc,
                   A_BOOL success, COMPLETION_ENUM completePacket)
{
    ATHEROS_DESC   *pTmpDesc = pDesc;
    A_UINT32        packetRefCount;
    WLAN_MACADDR   *pDstMacAddr;

    /* always free the associated resources such as map registers */
    athTxFrameReleaseBufferMaps(pDevInfo, pDesc);

    if (completePacket == DONT_COMPLETE) {
        /* Do not complete the transmit to NDIS/SME */
        return;
    }

    /*
     * Inform power management of PM status; if internally-generated PM frame
     * no need to process further
     */
    ATH_ACQUIRE_SPINLOCK(pDevInfo->lock);
    if ((!pDesc->isEthFrm) && powerTxIndication(pDevInfo, success, pDesc)) {
        ATH_RELEASE_SPINLOCK(pDevInfo->lock);
        return;
    }
    ATH_RELEASE_SPINLOCK(pDevInfo->lock);

    while (pTmpDesc) {
        PNDIS_PACKET pPacket = pTmpDesc->pOSHandle;
        
        if (pPacket) {
            ATH_OSPKTREF_FREE(pTmpDesc);
        }

        if (pPacket && (MP_RSRVD_GET_REF_COUNT(pPacket) == 0)) {
            
            txPrintf("TransmitCompletion: Packet %X\n", pPacket);

            ATH_RELEASE_SPINLOCK(pDevInfo->sendLock);
            drainSendWaitQueue(pDevInfo);
            ATH_ACQUIRE_SPINLOCK(pDevInfo->sendLock);

            NdisInterlockedDecrement(&pDevInfo->pOSHandle->txFramesPended);
        }

        if (!pTmpDesc->pOrigBufferVirtPtr) {
            ASSERT(pTmpDesc->freeQueue == &pDevInfo->emptyDescQueue);
            pTmpDesc->pBufferVirtPtr.byte = NULL;
        }

        if (pTmpDesc == pDesc->pTxLastDesc) {
            break;
        }
        pTmpDesc = pTmpDesc->pNextVirtPtr;
    }
}

#if NDIS_WDM
WLAN_STATS dummyStats;
#else
extern WLAN_STATS dummyStats;
#endif
/*******************************************************************************
 * txCallback
 *
 * This routine cleans up after a transmitted frame. It updates the transmit
 * statistic counters, frees up descriptors and map regs, and issues a send
 * completion to the OS which will unlock the pages.
 *
 * RETURNS: NONE
 */
void
txHostCallback(

    WLAN_DEV_INFO    *pdevInfo,
    ATHEROS_DESC     *pDesc,
    TRANSMIT_INFO    *pInfo)
{
    WLAN_FRAME_HEADER   *pHdr;
    WLAN_STATS          *pLocalStats;
    SIB_ENTRY           *pSib;
    WLAN_STATS          *pSibStats;
    ATHEROS_DESC        *pLastDesc;

    if (!pdevInfo->pOSHandle->openForBusiness) {
        return;
    }

    pHdr        = pDesc->pBufferVirtPtr.header;
    pLocalStats = &pdevInfo->localSta->stats;
    pSib        = pDesc->pDestSibEntry;
    pSibStats   = pSib ? &pSib->stats : &dummyStats;
    pLastDesc   = pDesc->pTxLastDesc;

    ASSERT(pLastDesc);

    NdisInterlockedDecrement(&pdevInfo->pOSHandle->txIrpPended);

    if (pDesc->bIndicateXmitStatus) {
        ATHEROS_DESC    *pTmpDesc;
        pTmpDesc = getPktFromQueue(&pdevInfo->txPendingQueue,pDesc);
        if (pTmpDesc == NULL) {
            return;
        }
    }

    if (!wlanSendCheck(pdevInfo)) {
        freeBuffandDescChain(pdevInfo, pDesc);
        return;
    }

    pLastDesc->status.tx.status     = (pInfo->status == TX_OK) ? TRANSMIT_OK : EXCESSIVE_RETRIES ;
    pLastDesc->status.tx.timestamp  = (A_UINT16) pInfo->timeStamp;
    pLastDesc->status.tx.seqNum     = 0;    /* xxxxPGSxxxx TODO */
    pLastDesc->status.tx.retryCount = (A_UINT16) pInfo->retryCount;


    if (pHdr->frameControl.fType == FRAME_MGT) {
        if (pDesc->bIndicateXmitStatus) {
            /*
             * Call tx completion on management frames; also do pending
             * descriptor processing for the corresponding station
             */
            ATH_ACQUIRE_SPINLOCK(pdevInfo->lock);
            mlmeMgtTxCompletion(pdevInfo, pDesc);
            ATH_RELEASE_SPINLOCK(pdevInfo->lock);
        }
    } else {
        if ((pInfo->status == TX_ERR) && 
            (pdevInfo->busStatus == ATHUSB_BUS_STATE_ERROR)) 
        {
            PNDIS_PACKET    packet = pDesc->pOSHandle;
            ATHEROS_DESC    *pTmpDesc;

            if (pdevInfo->nWaitSend < MAX_QUEUE_TARGET) {
                /* 
                 * still keep the packet in the queue before endpoint reset
                 * finished
                 */                   

                pTmpDesc = pDesc;
                while (pTmpDesc != NULL) {
                    packet = pTmpDesc->pOSHandle;

                    if (packet) {
                        ATH_OSPKTREF_FREE(pTmpDesc);
    
                        if (packet && (MP_RSRVD_GET_REF_COUNT(packet) == 0)) {
                            InsertHeadQueue(&pdevInfo->sendWaitQueue,
                                ATH_GET_PACKET_MR(packet));
                            pdevInfo->nWaitSend ++;
                        }                        
                        pTmpDesc = pTmpDesc->pNextVirtPtr;
                    }
                }
            }
        }

        /*
         * STA side completion processing - called for non-mgt frames
         * only, and mostly for indication to NDIS for data frames.
         */
        staTxCompletion(pdevInfo, pDesc,
                        pInfo->status == TX_OK, DO_COMPLETE);
    }

    /*
     * if the frame was a successful unicast frame, update the STA
     * flag that keeps track of the AP's status.
     */
    if (pInfo->status == TX_OK && !isGrp(&pHdr->address1)) {
        /*
         * Since directed probe request are sent when scanning hidden ssids,
         * Need to check the bssid before clearing the noApActivity flag.
         */
        if ((pdevInfo->localSta->serviceType == WLAN_STA_SERVICE) &&
            (pdevInfo->bssDescr->bsstype == INFRASTRUCTURE_BSS) &&
            (A_MACADDR_COMP(&pHdr->address1, &pdevInfo->baseBss.bssId) == 0) &&
            pDesc->bIndicateXmitStatus)
        {
#if NDIS_WDM
            /*
            * (LW) Attention !!!
            * This is temp hack, should revisit later
            */
            pdevInfo->noApActivity = FALSE;
#else
            /* xxxxPGSxxxx TODO - noApActivity - is this host side only??? */
            HACK_TO_PARDEV(pdevInfo)->noApActivity = FALSE;
#endif
        }
    }

    /* Free descriptor and buffers */
    pDesc->pTxLastDesc->pNextVirtPtr = NULL;
    freeBuffandDescChain(pdevInfo, pDesc);
}

/* xxxxPGSxxxx - TODO - move wlan send functions back into wlanSend.c */

/*
 * Called when the WDC's data queue becomes empty.
 * We try to pull things from the Fast Frame TX Staging Queues,
 * since it's better to keep the hardware busy transmitting than
 * to wait around for another transmit request that could be
 * aggregated with previous requests.
 */
void noDataEventHandler(
    TARGET_HANDLE targetHandle,
    WDC_EVENT     event,
    void *        param,
    A_UINT32      unused1,
    A_UCHAR      *unused2
)
{
    WLAN_DEV_INFO *pDev = (WLAN_DEV_INFO *)param;

    txFfFlush(pDev);
}

/*******************************************************************************
 * wlanTransmitInit - initialize the transmit related queue structures
 *      (cab/beacon queues are handled separately by the beacon code;
 *      pollQueue is handled sepeartely by the sta side PowerSave
 *      related code or XR AP code; this routine is for data transmit
 *      queue(s) only)
 *
 * RETURNS: success or failure
 */
A_STATUS
wlanTransmitInit(WLAN_DEV_INFO *pdevInfo)
{
    A_STATUS          status;

    A_ATOMIC_SET(&pdevInfo->pOSHandle->txIrpPended, 0);

    /* Host initialization */
    if ((status = descQInit(&pdevInfo->baseBss.txQueue)) != A_OK) {
        return status;
    }

{
#ifdef WME
    int qnum;

    wlanInitStaWmeParams(pdevInfo);
    if (pdevInfo->staConfig.WmeEnabled) {
        for (qnum = 0; qnum < pdevInfo->hwCapInfo.numTxQueues; qnum++) {
            switch (qnum) {
            case TXQ_ID_FOR_AC0:
            case TXQ_ID_FOR_AC1:
            case TXQ_ID_FOR_AC2:
            case TXQ_ID_FOR_AC3:
            case TXQ_ID_FOR_UPSD:
#ifdef GPRS
            case TXQ_ID_FOR_GPRS:
#endif
                break;
            default:
                continue;
            }
            status = descQInit(&pdevInfo->baseBss.wmequeues[qnum]);
            if (status != A_OK) {
                return status;
            }
        }
    } else
#endif
    {
        if ((status = descQInit(&pdevInfo->txStagingQueue)) != A_OK) {
            return status;
        }
        pdevInfo->baseBss.txQueue.pStagingQueue = &pdevInfo->txStagingQueue;
    }
}

    if (pdevInfo->staConfig.abolt & ABOLT_BURST) {
        if ((status = descQInit(&pdevInfo->baseBss.burstQueue)) != A_OK) {
            return status;
        }
        if ((status = descQInit(&pdevInfo->burstStagingQueue)) != A_OK) {
            return status;
        }
        pdevInfo->baseBss.burstQueue.pStagingQueue = &pdevInfo->burstStagingQueue;
    }

    if (isXrAp(pdevInfo)) {
        if ((status = descQInit(&pdevInfo->xrBss.txQueue)) != A_OK) {
            return status;
        }
    }

    wdcRegisterEventHandler(pdevInfo->targetHandle,
                            WDCEVT_NO_TX_DATA,
                            noDataEventHandler,
                            pdevInfo,
                            NULL);


    return A_OK;
}

/*******************************************************************************
 * wlanTransmitStart - set up the hardware queues for the transmit
 *      function;
 *      (cab/beacon queues are handled separately by the beacon code;
 *      pollQueue is handled sepeartely by the sta side PowerSave
 *      related code or XR AP code; this routine is for data transmit
 *      queue(s) only)
 *
 * RETURNS: success or failure
 */
A_STATUS
wlanTransmitStart(WLAN_DEV_INFO *pdevInfo)
{
    A_STATUS            status;
    TARGET_HANDLE       targetHandle = pdevInfo->targetHandle;
    TXQ_ATTRIBUTES      txQueueInfo;

    A_QUEUE_INFO_INIT(&txQueueInfo, &pdevInfo->baseBss.phyChAPs.ac[ACI_BE]);
    txQueueInfo.priority = TXQ_ID_FOR_DATA;

    if (pdevInfo->staConfig.compressionSupport) {
        txQueueInfo.compression = TRUE;
    }

    if (pdevInfo->staConfig.abolt & ABOLT_BURST) {
        txQueueInfo.burstTime   = TXOP_TO_US(pdevInfo->baseBss.phyChAPs.ac[ACI_BE].txOpLimit);
    }

{
#ifdef WME
    int qnum;

    if (pdevInfo->staConfig.WmeEnabled) {

        A_QUEUE_INFO_INIT(&txQueueInfo, &pdevInfo->baseBss.phyChAPs.ac[ACI_BE]);

        for (qnum = 0; qnum < pdevInfo->hwCapInfo.numTxQueues; qnum++) {
            switch (qnum) {
            case TXQ_ID_FOR_AC0: /* ACI_BK */
                txQueueInfo.aifs = pdevInfo->baseBss.phyChAPs.ac[ACI_BK].aifs;
                txQueueInfo.logCwMin = pdevInfo->baseBss.phyChAPs.ac[ACI_BK].logCwMin;
                break;
            case TXQ_ID_FOR_AC1: /* ACI_BE */
                txQueueInfo.aifs  = pdevInfo->baseBss.phyChAPs.ac[ACI_BE].aifs;
                txQueueInfo.logCwMin = pdevInfo->baseBss.phyChAPs.ac[ACI_BE].logCwMin;
                break;
            case TXQ_ID_FOR_AC2: /* ACI_Vi */
                txQueueInfo.aifs  = pdevInfo->baseBss.phyChAPs.ac[ACI_Vi].aifs;
                txQueueInfo.logCwMin = pdevInfo->baseBss.phyChAPs.ac[ACI_Vi].logCwMin;
                txQueueInfo.logCwMax = pdevInfo->baseBss.phyChAPs.ac[ACI_Vi].logCwMax;
                break;
            case TXQ_ID_FOR_AC3: /* ACI_Vo */
                txQueueInfo.aifs  = pdevInfo->baseBss.phyChAPs.ac[ACI_Vo].aifs;
                txQueueInfo.logCwMin = pdevInfo->baseBss.phyChAPs.ac[ACI_Vo].logCwMin;
                txQueueInfo.logCwMax = pdevInfo->baseBss.phyChAPs.ac[ACI_Vo].logCwMax;
                break;
            case TXQ_ID_FOR_UPSD:
                txQueueInfo.aifs  = 0;
                txQueueInfo.logCwMin= 0;
                txQueueInfo.logCwMax= 0;
                break;
#ifdef GPRS
            case TXQ_ID_FOR_GPRS:
                txQueueInfo.aifs = 0;
                txQueueInfo.logCwMin = 0;
                txQueueInfo.logCwMax = 0;

                break;
#endif
            default:
                continue;
            }

            txQueueInfo.priority = qnum;
            ASSERT(A_SEM_VALID(pdevInfo->baseBss.wmequeues[qnum].qSem));

            wdcSetupTxQueue(pTargetDevice, qnum,  &txQueueInfo);
            pdevInfo->baseBss.wmequeues[qnum].pTargetQueueId = qnum;
        }
    } else
#endif
    {
        ASSERT(A_SEM_VALID(pdevInfo->baseBss.txQueue.qSem));

        wdcSetupTxQueue(targetHandle, TXQ_ID_FOR_DATA, &txQueueInfo);
        pdevInfo->baseBss.txQueue.targetQueueId = TXQ_ID_FOR_DATA;
    }
}

    if (pdevInfo->staConfig.abolt & ABOLT_BURST) {
        /* Initialize the Burst Queue parameters*/
        A_QUEUE_INFO_INIT(&txQueueInfo, &pdevInfo->baseBss.phyChAPs.ac[ACI_BE]);
        txQueueInfo.priority  = TXQ_ID_FOR_GBURST;
        txQueueInfo.burstTime = TXOP_TO_US(pdevInfo->baseBss.phyChAPs.ac[ACI_BE].txOpLimit);
        if (pdevInfo->staConfig.compressionSupport) {
            txQueueInfo.compression = TRUE;
        }
        ASSERT(A_SEM_VALID(pdevInfo->baseBss.burstQueue.qSem));

        wdcSetupTxQueue(targetHandle, TXQ_ID_FOR_GBURST, &txQueueInfo);
        pdevInfo->baseBss.burstQueue.targetQueueId = TXQ_ID_FOR_GBURST;
    }

    if (isXrAp(pdevInfo)) {
        /* AP should use base BSS params to access XR channel */
        A_QUEUE_INFO_INIT(&txQueueInfo, &pdevInfo->baseBss.phyChAPs.ac[ACI_BE]);
        txQueueInfo.priority  = TXQ_ID_FOR_XR_DATA;
        txQueueInfo.aifs     += XR_DNLINK_QUEUE_AIFS_SHIFT;
        txQueueInfo.logCwMin += XR_DNLINK_QUEUE_CWMIN_SHIFT;
        ASSERT(A_SEM_VALID(pdevInfo->xrBss.txQueue.qSem));

        wdcSetupTxQueue(targetHandle, TXQ_ID_FOR_XR_DATA, &txQueueInfo);
        pdevInfo->xrBss.txQueue.targetQueueId = TXQ_ID_FOR_XR_DATA;
    }

    return A_OK;
}

/*******************************************************************************
 * wlanTransmitStop - stop all the Tx queues
 *
 * RETURNS: void
 */
void
wlanTransmitStop(WLAN_DEV_INFO *pdevInfo)
{
    /* xxxxPGSxxxx HACK - this needs to be fixed during the tx partitioning
     * need host side transmit stop
     */
}

/*******************************************************************************
 * wlanTransmitRelease - release all the Tx resources
 *
 * RETURNS: void
 */
void
wlanTransmitRelease(WLAN_DEV_INFO *pdevInfo)
{
    /* Cancel pending transmit requests */
    ATHEROS_DESC    *pDesc;
    while (pDesc = getPktFromQueueStart(&pdevInfo->txPendingQueue)) {
        freeBuffandDescChain(pdevInfo, pDesc);
    }

    #define ASSERT_TXQ_EMPTY(_pQueue)   ASSERT((_pQueue)->pDescQueueHead == NULL)

#ifdef WME
    int qnum;

    if (pdevInfo->staConfig.WmeEnabled) {
        for (qnum = 0; qnum < NWME; qnum++) {
            if (A_SEM_VALID(pdevInfo->baseBss.wmequeues[qnum].qSem)) {
                A_SEM_DELETE(pdevInfo->baseBss.wmequeues[qnum].qSem);
            }
        }
    } else
#endif
    {
        ASSERT_TXQ_EMPTY(&pdevInfo->baseBss.txQueue);
        if (A_SEM_VALID(pdevInfo->baseBss.txQueue.qSem)) {
            A_SEM_DELETE(pdevInfo->baseBss.txQueue.qSem);
            ASSERT_TXQ_EMPTY(pdevInfo->baseBss.txQueue.pStagingQueue);
            A_SEM_DELETE(pdevInfo->baseBss.txQueue.pStagingQueue->qSem);
            pdevInfo->baseBss.txQueue.pStagingQueue = NULL;
        }
    }

    if (pdevInfo->staConfig.abolt & ABOLT_BURST) {
        ASSERT_TXQ_EMPTY(&pdevInfo->baseBss.burstQueue);
        if (A_SEM_VALID(pdevInfo->baseBss.burstQueue.qSem)) {
            A_SEM_DELETE(pdevInfo->baseBss.burstQueue.qSem);
            ASSERT_TXQ_EMPTY(pdevInfo->baseBss.burstQueue.pStagingQueue);
            A_SEM_DELETE(pdevInfo->baseBss.burstQueue.pStagingQueue->qSem);
            pdevInfo->baseBss.burstQueue.pStagingQueue = NULL;
        }
    }

    if (isXrAp(pdevInfo)) {
        ASSERT_TXQ_EMPTY(&pdevInfo->xrBss.txQueue);
        if (A_SEM_VALID(pdevInfo->xrBss.txQueue.qSem)) {
            A_SEM_DELETE(pdevInfo->xrBss.txQueue.qSem);
        }
    }
}


