/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/sta/stacserv.c#4 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Connection services routines for station
 */

#include "wlantype.h"
#include "wlanproto.h"
#include "wlansme.h"
#include "wlansmeext.h"
#include "wlanext.h"
#include "display.h"
#include "stacserv.h"
#include "ccx.h"
#include "wlanchannel.h"
#include "usbstub.h"
#ifndef BUILD_AP
#include "linuxext.h"
#else
#include "ui.h"
#endif


#ifdef DEBUG

int cservDebugLevel  = 0;

#endif /* DEBUG */

/* Hysteresis Table entry */
typedef struct CservHysteresis {
    A_UINT32  thresh;
    A_UINT32  hysAdjust;
} CSERV_HYSTERESIS;

/*
 * Hysteresis Table.  Algorith assume threshold
 * is in descending order and thresh should end in 0
 */
static const CSERV_HYSTERESIS cservHysTable[] = {
    {40, 30},
    {30, 18},
    {20, 12},
    {13, 8},
    {00, 6}
};

#define HYS_TABLE_SIZE (sizeof(cservHysTable) / sizeof(CSERV_HYSTERESIS))

/* Margin table entry */
typedef struct CservMargin {
    A_INT32 startRssi;                  /* start of rssi range (inclusive) */
    A_INT32 endRssi;                    /* end of rssi range (inclusive) */
    /* 
     * WIRELESS_MODE defines the order of entries in the following arrays.
     * 11a, 11a Turbo, 11b, 11g, 108g, XR
     */
    A_INT32 margin[WIRELESS_MODE_MAX];  /* rssi margin for each wireless mode */
    A_INT32 pref[WIRELESS_MODE_MAX];    /* order of preference for each wireless mode
                                         * higher number = higher preference
                                         */
} CSERV_MARGIN;

/* Margin Table - used when evaluating APs for roaming */
static const CSERV_MARGIN cservMarginTable[] = {
    {   31, 9999,                       /* rssi range (inclusive) */
        { 20, 20, 40, 25, 25, 0},       /* rssi margin */
        {  5,  6,  2,  3,  4, 1} },     /* order of preference */    

    {   6, 30,                          /* rssi range (inclusive) */
        { 10, 10, 40, 15, 15, 0},       /* rssi margin */
        {  5,  6,  2,  3,  4, 1} },     /* order of preference */    

    {   -9999, 5,                       /* rssi range (inclusive) */
        { 0,   0,  0,  0,  0, 0},       /* rssi margin */
        { 5,   6,  2,  3,  4, 1}    }   /* order of preference */    
};

#define NUM_MARGIN_TABLE (sizeof(cservMarginTable) / sizeof(CSERV_MARGIN))

/*
 * If we are allowed to try open sytem authentication try it first.  If
 * privacy is invoked, a failed open system authentication will be followed
 * by shared key request.
 * If configured for LEAP, only do that.
 */

static INLINE A_UINT16
authAlgNum(WLAN_DEV_INFO *pDevInfo)
{
#ifdef WLAN_CONFIG_LEAP
    if (pDevInfo->staConfig.leapEnabled) {
        return AUTH_LEAP;
    }
#endif

    if (pDevInfo->staConfig.authOpenFirst == FALSE &&
        pDevInfo->staConfig.privacyInvoked)
    {
        return AUTH_SHARED_KEY;
    } else {
        return AUTH_OPEN_SYSTEM;
    }
}

#define CSERV_OP_RETRY_SCALE_NUM     (sizeof(cservOpRetryScale) / sizeof(A_UINT32))

/*
 * Maximum scaling of scan before starting an adhoc network
 * This is used to limit the time needed to start a new adhoc network
 */
#define CSERV_MAX_SCALE_ADHOC        (cservOpRetryScale[CSERV_OP_RETRY_SCALE_NUM - 1])

/* Operation Scaling Macros */
#define RETRY_SCALE_GET(pCsInfo)     (cservOpRetryScale[(pCsInfo)->opRetryScaleIndex])
#define RETRY_SCALE_UP(pCsInfo)      ((pCsInfo)->opRetryScaleIndex =                                   \
                                      ((pCsInfo)->opRetryScaleIndex < (CSERV_OP_RETRY_SCALE_NUM-1)) ?  \
                                      (pCsInfo)->opRetryScaleIndex + 1 : (CSERV_OP_RETRY_SCALE_NUM-1))
#define RETRY_SCALE_RESET(pCsInfo)   ((pCsInfo)->opRetryScaleIndex = 0)
#define IS_MAX_RETRY_SCALE(pCsInfo)  ((pCsInfo)->opRetryScaleIndex >= (CSERV_OP_RETRY_SCALE_NUM - 1))
#define RETRY_SCALE_SET_MAX(pCsInfo) ((pCsInfo)->opRetryScaleIndex = CSERV_OP_RETRY_SCALE_NUM-1)
#define RETRY_MUL(pCsInfo, x)        ((x) * RETRY_SCALE_GET(pCsInfo))

/*
 * Prototypes
 */

static void staSmeAuthCallback(WLAN_DEV_INFO *, WLAN_MACADDR *, A_UINT16, MLME_RESULT);
static void staSmeDeauthCallback(WLAN_DEV_INFO *, WLAN_MACADDR *, MLME_RESULT);
static void staSmeDeauthIndCallback(WLAN_DEV_INFO *, WLAN_MACADDR *, MLME_RESULT);
static void staSmeAssocCallback(WLAN_DEV_INFO *, MLME_RESULT);
static void staSmeReassocCallback(WLAN_DEV_INFO *, MLME_RESULT);
static void staSmeDisassocCallback(WLAN_DEV_INFO *, WLAN_MACADDR *, MLME_RESULT);
static void staSmeDisassocIndCallback(WLAN_DEV_INFO *, WLAN_MACADDR *, MLME_RESULT);
static void staSmeScanCallback(WLAN_DEV_INFO *, BSSDESCR_SET *, MLME_RESULT, A_BOOL, A_BOOL);
static void staSmeJoinCallback(WLAN_DEV_INFO *, MLME_RESULT);
static void staSmeStartCallback(WLAN_DEV_INFO *, MLME_RESULT);
static void staSmePowerCallback(WLAN_DEV_INFO *, A_STATUS, A_BOOL);
static void cservBssSetPurge(WLAN_DEV_INFO *pDevInfo);
static void cservTxRateNotify(WLAN_DEV_INFO *pDev, A_UINT32 rate, A_UINT32 rssi);
static void cservConnectionCompleteNotify(WLAN_DEV_INFO *pDev);

/*
 * Local storage
 */

/* Operation Time Scaling factor */
static A_UINT32 cservOpRetryScale[] = {1, 5};

/* Initialize the structure with our SME info */
SME_INFO_STRUCT staSmeInfo = {
    WLAN_STA_SERVICE,          // Service type default

    staSmeAuthCallback,        // auth confirm
    NULL,                      // auth indication

    staSmeDeauthCallback,      // deauth confirm
    staSmeDeauthIndCallback,   // deauth indication

    staSmeAssocCallback,       // assoc confirm
    NULL,                      // assoc indication

    staSmeReassocCallback,     // reassoc confirm
    NULL,                      // reassociate callback

    staSmeDisassocCallback,    // disassoc confirm
    staSmeDisassocIndCallback, // disassoc indication

    staSmePowerCallback,       // power mgmt callback
    staSmeScanCallback,
    staSmeJoinCallback,
    staSmeStartCallback,
    NULL
};

/*
 * Callback routines
 */

/*
 * Notification mechanism
 *
 * CS routines are asynchronous by nature. They use callbacks for SME.
 * Sometimes the process that called the original CS routine needs to be
 * notified.
 *
 * Most of CS callback routines have a block of code that generates notification.
 */

/*******************************************************************
 * staSmeAuthCallback - Handled callbck of authentication request to SME
 * Called after wlanMlmeAuthRequest()
 * It checks the result of authentication request and fires off association
 * if it was successful. If not, it sends a notification, if any.
 */
void
staSmeAuthCallback(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR *peerAddr,
                   A_UINT16 algNo,
                   MLME_RESULT resultCode)
{
    CSERV_INFO *pCsInfo = pDevInfo->pCsInfo;

    cservPrintf("authCallback: Authentication callback on STA.\n");
    cservPrintf("cservOp %x\n", pCsInfo->cservOp);

    if (resultCode == MLME_SUCCESS) {
        uiPrintf("authCallback: Successfully authenticated with ");
        printMacAddress(*peerAddr);
        uiPrintf(" \"");
        printSsid(&pDevInfo->bssDescr->ssid);
        uiPrintf("\"\n");
    } else {
        BSSDESCR *pBssdescr =
            mlmeBsssetFind(&pCsInfo->scanInfo.bssSet, 
            &pDevInfo->bssDescr->bssId,
            pDevInfo->staConfig.pChannel->channelFlags);

        uiPrintf("authCallback:Authentication failed, code = 0x%x\n",
                 resultCode);
        /*
         * If we haven't indicated MEDIA_DISCONNECT to the OS yet,
         * start that process. This happens if this failed authcallback
         * is while roaming from one AP to other.
         */
        if (GET_CONNECTION_STATUS(pDevInfo) &&
            pCsInfo->disconnectCur == 0)
        {
            cservStartDisconnectTimer(pDevInfo);
        }

        /* We allow two attempts at authentication */
        if (pBssdescr) {
            if (pBssdescr->apState & AP_RETRY) {
                pBssdescr->apState |= BAD_AP;
        } else {
                pBssdescr->apState |= AP_RETRY;
            }
        }
    }

    if (pCsInfo->cservOp) {
        /* We had some CS action in progress */
        pCsInfo->cservOp &= ~CSERV_OP_AUTH;
        if (resultCode == MLME_SUCCESS) {
            /* Take the next action */
            if (pCsInfo->cservOp & (CSERV_OP_ASSOC | CSERV_OP_REASSOC)) {
                A_UINT16 beaconInterval = pDevInfo->bssDescr->beaconInterval;
                A_UINT16 listenInterval = (pDevInfo->staConfig.listenInterval +
                                           beaconInterval - 1) / beaconInterval;

                if (pCsInfo->cservOp & CSERV_OP_ASSOC) {
                    cservPrintf("Calling Associate\n");
                    /* clear auth bit & start association process */
                    /* express listen interval in beacon interval units, rounded up. */
                    resultCode =
                        wlanMlmeAssocRequest(pDevInfo,
                                             &pDevInfo->bssDescr->bssId,
                                             RETRY_MUL(pCsInfo,
                                                       CSERV_TIME_AA_MAX),
                                             &pDevInfo->localSta->capInfo,
                                             listenInterval);
                } else {
                    cservPrintf("Calling ReAssociate\n");
                    /* clear auth bit & start association process */
                    /* express listen interval in beacon interval units, rounded up. */
                    resultCode =
                        wlanMlmeReassocRequest(pDevInfo,
                                               &pDevInfo->bssDescr->bssId,
                                               RETRY_MUL(pCsInfo,
                                                         CSERV_TIME_AA_MAX),
                                               &pDevInfo->localSta->capInfo,
                                               listenInterval);
                }
            }
        } else if (algNo == AUTH_SHARED_KEY &&
                   pDevInfo->staConfig.authOpenFirst == FALSE &&
                   pDevInfo->staConfig.authAllowOpen == TRUE)
        {
            /* retry with open system authentication (Microsoft order) */
            pCsInfo->cservOp |= CSERV_OP_AUTH;
            resultCode =
                wlanMlmeAuthRequest(pDevInfo,
                                    &pDevInfo->bssDescr->bssId,
                                    AUTH_OPEN_SYSTEM,
                                    RETRY_MUL(pCsInfo, CSERV_TIME_AA_MAX));
        } else if ( algNo == AUTH_OPEN_SYSTEM &&
                    pDevInfo->staConfig.authOpenFirst == TRUE &&
                    pDevInfo->staConfig.authAllowShared == TRUE )
        {
            /* retry with shared key authentication */
            pCsInfo->cservOp |= CSERV_OP_AUTH;
            resultCode =
                wlanMlmeAuthRequest(pDevInfo,
                                    &pDevInfo->bssDescr->bssId,
                                    AUTH_SHARED_KEY,
                                    RETRY_MUL(pCsInfo, CSERV_TIME_AA_MAX));
        }
#ifdef WLAN_CONFIG_CCX
        else if (algNo == AUTH_LEAP) {
            ccxLeapAuthCallback(pDevInfo, peerAddr, resultCode);
        }
#endif /* WLAN_CONFIG_CCX */
        if (resultCode != MLME_SUCCESS || pCsInfo->cservOp == 0) {
            /* resultCode may have changed! */
            /* Cancel CS operation */
            pCsInfo->cservOp = 0;
            /*
             *  bump up the channel dwell time to account for 
             *  long beacon periods and busy AP's
             */
            RETRY_SCALE_UP(pCsInfo);
        }
    }
}

/*******************************************************************
 * staSmeDeauthCallback
 * Called after wlanMlmeDeauthRequest()
 */
void
staSmeDeauthCallback(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR *peerAddr,
                     MLME_RESULT resultCode)
{
    if (resultCode == MLME_SUCCESS) {
        uiPrintf("deauthCallback: Successfully deauthenticated from ");
        printMacAddress(*peerAddr);
        uiPrintf("\n");
    } else {
        uiPrintf("deauthCallback: Deauthentication failed, code = 0x%x\n",
                 resultCode);
    }
}

/*******************************************************************
 * staSmeAssocCallback
 * Called after wlanMlmeAssocRequest()
 * This callback is an end of chain actions of CS. It notifies OS on
 * successful association so that OS can link the device into network stack.
 */
void
staSmeAssocCallback(WLAN_DEV_INFO *pDevInfo, MLME_RESULT resultCode)
{
    CSERV_INFO *pCsInfo = pDevInfo->pCsInfo;
    SIB_ENTRY  *pSib;

    pSib = sibEntryFind(pDevInfo, &pDevInfo->bssDescr->bssId, NULL);

    if (resultCode == MLME_SUCCESS) {
#if defined(WLAN_CONFIG_LEAP) && defined(LEAP_IN_DRIVER)
        if (pDevInfo->staConfig.leapInDriver && pSib != NULL) {
            leapAssocCallback(pDevInfo, pSib);
        } else
#endif  /* WLAN_CONFIG_LEAP && LEAP_IN_DRIVER */
        {
            /*
             * Cancel OS disconnect indicate timer if it
             * was started for previous failure to associate
             */
            osIndicateConnectStatus(pDevInfo, TRUE);
            pCsInfo->apConnAttemptCur = 0;
        }

        /* Inform CCX that we're associated */
        ccxAssocCallback(pDevInfo);
        uiPrintf("assocCallback: Successfully associated with AP\n");

        cservConnectionCompleteNotify(pDevInfo);
    } else {
        uiPrintf("assocCallback: Assoc failed, code = 0x%x\n", resultCode);
        /*
         * Bump up the channel dwell time to account for long beacon
         * periods and busy AP's
         */
        RETRY_SCALE_UP(pCsInfo);

#ifdef WLAN_CONFIG_LEAP
        /*
         *  Some Cisco APs accept the LEAP algorithm in the auth, but then
         *  issue an assocResp with the 'unsupported algorithm' status code
         */
        if (pDevInfo->staConfig.leapEnabled) {
            ccxLeapAuthCallback(pDevInfo, &pDevInfo->bssDescr->bssId,
                                resultCode);
        }
#endif /* WLAN_CONFIG_LEAP */

        /*
         * if we haven't indicated MEDIA_DISCONNECT to the OS yet,
         * start that process. This happens if this failed assoccallback
         * is while roaming from one AP to other.
         */
        if (GET_CONNECTION_STATUS(pDevInfo) && pCsInfo->disconnectCur == 0) {
            cservStartDisconnectTimer(pDevInfo);
        }

        if (resultCode != MLME_TIMEOUT || 
            pCsInfo->apConnAttemptCur == pCsInfo->apConnAttemptsMax)
        {
            BSSDESCR *pBssdescr;
            pBssdescr = mlmeBsssetFind(&pCsInfo->scanInfo.bssSet, 
                                       &pDevInfo->bssDescr->bssId,
                                       pDevInfo->staConfig.pChannel->channelFlags);
            /* 
             * AP rejected assoc req for specific reason. Don't try it again.
             * If we have tried max attempts, timeout error is also bad.
             * Mark it as bad AP and give chance to others.
             */
            if (pBssdescr != NULL) {
                pBssdescr->apState |= BAD_AP;
            }
            /* Let cservPollFunction() take care of cleanup */
            pCsInfo->apConnAttemptCur = pCsInfo->apConnAttemptsMax;
        }

    }
    if (pCsInfo->cservOp) {
        /* We had some CS action in progress, this is the last step */
        pCsInfo->cservOp = 0;
    }
}


/*******************************************************************
 * staSmeReassocCallback
 * Called after wlanMlmeReassocRequest()
 * This callback is an end of chain actions of CS. It notifies OS on
 * successful association so that OS can link the device into network stack.
 */
void
staSmeReassocCallback(WLAN_DEV_INFO *pDevInfo, MLME_RESULT resultCode)
{
    CSERV_INFO *pCsInfo = pDevInfo->pCsInfo;
    SIB_ENTRY  *pSib;

    pSib = sibEntryFind(pDevInfo, &pDevInfo->bssDescr->bssId, NULL);

    if (resultCode == MLME_SUCCESS) {
#if defined(WLAN_CONFIG_LEAP) && defined(LEAP_IN_DRIVER)
        if (pDevInfo->staConfig.leapInDriver && pSib != NULL) {
            leapAssocCallback(pDevInfo, pSib);
        } else
#endif /* WLAN_CONFIG_LEAP && LEAP_IN_DRIVER */
        {
            /*
             * Cancel OS disconnect indicate timer if it
             * was started for previous failure to associate
             */
            osIndicateConnectStatus(pDevInfo, TRUE);
            pCsInfo->apConnAttemptCur = 0;
        }

        /* Inform CCX that we're associated */
        ccxAssocCallback(pDevInfo);

        uiPrintf("reAssocCallback: Successfully associated with AP\n");

        cservConnectionCompleteNotify(pDevInfo);
    } else {
        uiPrintf("reAssocCallback: reAssoc failed, code = 0x%x\n", resultCode);
        /*
         * Bump up the channel dwell time to account for long beacon
         * periods and busy AP's
         */
        RETRY_SCALE_UP(pCsInfo);
        /*
         * if we haven't indicated MEDIA_DISCONNECT to the OS yet,
         * start that process. This happens if this failed reassoccallback
         * is while roaming from one AP to other.
         */
        if (GET_CONNECTION_STATUS(pDevInfo) && pCsInfo->disconnectCur == 0) {
            cservStartDisconnectTimer(pDevInfo);
        }

        if (resultCode != MLME_TIMEOUT || 
            pCsInfo->apConnAttemptCur == pCsInfo->apConnAttemptsMax) {
            BSSDESCR *pBssdescr = mlmeBsssetFind(
                &pCsInfo->scanInfo.bssSet, 
                &pDevInfo->bssDescr->bssId,
                pDevInfo->staConfig.pChannel->channelFlags);
            /* 
             * AP rejected assoc req for specific reason. Don't try it again.
             * If we have tried max attempts, timeout error is also bad.
             * Mark it as bad AP and give chance to others.
             */
            pBssdescr->apState |= BAD_AP;
            /* Let cservPollFunction() take care of cleanup */
            pCsInfo->apConnAttemptCur = pCsInfo->apConnAttemptsMax;
        }

    }
    if (pCsInfo->cservOp) {
        /* We had some CS action in progress, this is the last step */
        pCsInfo->cservOp = 0;
    }
}


/*******************************************************************
 * staSmeDisassocCallback
 * Called after wlanMlmeDisassocRequest()
 * It notifies OS of disconnecting from the network.
 */
void
staSmeDisassocCallback(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR *pmacAddr,
                 MLME_RESULT resultCode)
{
    CSERV_INFO *pCsInfo = pDevInfo->pCsInfo;

    ASSERT(pCsInfo);

    if (resultCode == MLME_SUCCESS) {
        cservStartDisconnectTimer(pDevInfo);
        uiPrintf("disassocCallback: Successfully disassociated\n");
    } else {
        uiPrintf("disassocCallback: Disassociation failed, code = 0x%x\n",
                 resultCode);
    }

    pCsInfo->wasDisassociated = TRUE;

    /* Let CCX know that we've disassociated */
    ccxDisassocCallback(pDevInfo);
}

/*******************************************************************
 * staSmeDeauthIndCallback is called when a deauth frame arrives at our doorstep.
 */
void
staSmeDeauthIndCallback(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR *macAddr,
                  MLME_RESULT result)
{
    REMOVE_BSS_ENUM  remove = (result == REASON_AUTH_LEAVING) ?
                                REMOVE_BSS : DONT_REMOVE_BSS;

    /* AP is leaving. Possible radar detected */
    /* Disconnect from our current AP and start looking for another */
    cservStopBss(pDevInfo, DISCONNECT_DELAYED, remove, DONT_SEND_DEAUTH);
    return;
}

/*******************************************************************
 * staSmeDisassocIndCallback is called when a deauth frame arrives at our doorstep.
 */
void
staSmeDisassocIndCallback(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR *macAddr,
                  MLME_RESULT result)
{
    REMOVE_BSS_ENUM  remove = (result == REASON_ASSOC_LEAVING) ?
                                REMOVE_BSS : DONT_REMOVE_BSS;

    /* AP is leaving. Possible radar detected */
    /* Disconnect from our current AP and start looking for another */
    cservStopBss(pDevInfo, DISCONNECT_DELAYED, remove, DONT_SEND_DEAUTH);

    return;
}

/*******************************************************************
 * staSmeScanCallback
 * Called after wlanMlmeScanRequest()
 * This routine does a lot of work for CS. It starts a join on successful scan.
 * It starts an adhoc network too.
 */
void
staSmeScanCallback(WLAN_DEV_INFO *pDevInfo, BSSDESCR_SET *pbssSet,
                   MLME_RESULT resultCode, A_BOOL wasScanInterrupted,
                   A_BOOL isScanDone)
{
    A_INT32     i;
    CSERV_INFO  *pCsInfo = pDevInfo->pCsInfo;
    A_BOOL      isIbss   = (pDevInfo->staConfig.bssType == INDEPENDENT_BSS);
    A_BOOL      isJoined = (pDevInfo->localSta->staState & STATE_JOINED) ? TRUE : FALSE;
    A_INT32     status;

    cservPrintf("scanCallback -- result: %x, BSS: %d found, %d in cache, op: %x, staState: %x\n",
             resultCode, pbssSet ? pbssSet->count: 0,
             pCsInfo->scanInfo.bssSet.count, pCsInfo->cservOp, pDevInfo->localSta->staState);

    if ((!wasScanInterrupted) && (isIbss && isJoined)) {
        /*
         * We are in ad hoc. We may be the station sending the beacons.
         * our bssid should be part of the scan list.
         */
        cservPrintf("Adding my adhoc BSS to scan list -- ");
        cservPrintMacAddress(pDevInfo->bssDescr->bssId);
        cservPrintf("\n");
        mlmeBsssetAdd(pDevInfo, pDevInfo->bssDescr, &pCsInfo->scanInfo.bssSet);
    }

    if (resultCode == MLME_SUCCESS) {
        /* Update the connection services BSS cache. */
        for (i = 0; pbssSet != NULL && i < pbssSet->count ; i++) {
            BSSDESCR *pBss = &pbssSet->bssDescrArray[i];

            mlmeBsssetAdd(pDevInfo, pBss, &pCsInfo->scanInfo.bssSet);
            cservPrintf("[%2d]: ", i);
            cservPrintMacAddress(pBss->bssId);
            cservPrintf(" RSSI:%2d, %4d(%4x), %4s, Rank:%4d ",
                     pBss->rssi, pBss->pChannel->channel, pBss->pChannel->channelFlags,
                     pBss->bsstype == INDEPENDENT_BSS ? "IBSS" : "AP",
                     pBss->rank);
            cservPrintSsid(&pBss->ssid);
            cservPrintf("\n");
        }

        /* Display all of the BSSes in the CSERV BSS cache. */
        uiPrintf("\n%d BSSes in the CS cache...\n",
                 pCsInfo->scanInfo.bssSet.count);
        for (i = 0; i < pCsInfo->scanInfo.bssSet.count ; i++) {
            BSSDESCR    *pBss = &pCsInfo->scanInfo.bssSet.bssDescrArray[i];
            WLAN_CFLAGS cf;

            cservPrintf("[%2d]: ", i);
            cservPrintMacAddress(pBss->bssId);
            cf = pBss->pChannel->channelFlags & CHANNEL_ALL;
            cservPrintf(" RSSI:%2d, %4d(%4x) %1s %s%4s, Rank %4d ", pBss->rssi,
                     pBss->pChannel->channel, cf, CHAN_TO_CHAR(cf),
                     pBss->apState & BAD_AP ? "BAD " : "",
                     pBss->bsstype == INDEPENDENT_BSS ? "IBSS" : "AP",
                     pBss->rank);
            cservPrintSsid(&pBss->ssid);
            cservPrintf("\n");
        }
    }

    /*
     * If the scan is not yet complete, don't do any of the autoconnection
     * stuff below.  This happens when the scan is interrupted by an OS
     * request to associate to a particular SSID, and we want to stop and
     * check if we've seen that SSID yet in our scan.
     */
    if (wasScanInterrupted) {
        return;
    }

    /*
     * A scan cycle has just completed.  Keep track so we can inform upper
     * layers, if needed.
     */
    if (isScanDone) {
        pCsInfo->isScanRunDone = TRUE;
    }
    
    if (pbssSet && pbssSet->count <= 0) {
        resultCode = MLME_TIMEOUT;
        /* don't try to put device in power save mode here, let power mgt poll handle it. */
    }

    /* Record the ending time of this scan */
    pCsInfo->scanInfo.scanEndTime = A_MS_TICKGET();

    if (!pCsInfo->cservOp) {
        /* No operations in progress.  We're done. */
        return;
    }

    /* We had some CS action in progress */
    pCsInfo->cservOp &= ~CSERV_OP_SCAN;
    if (pCsInfo->cservState & CSERV_STATE_BKSCANNING) {
        /* we are done with one channel of background scan. */
        pCsInfo->cservOp = 0;

        /* Stay asleep if we're connected */
        pCsInfo->bkPowerReq = 0;
        if (IS_STATE_CONNECTED(pDevInfo)) {
            wlanMlmePwrmgtRequest(pDevInfo, PS_MODE_POWER_SAVE, FALSE);
            pCsInfo->cservFakeSleep = FALSE;
        } else {
            wlanMlmePwrmgtRequest(pDevInfo, PS_MODE_ACTIVE, FALSE);
        }
        pCsInfo->bkScanCur = pCsInfo->bkScanInterval;

        return;
    }

    /*
     * Search and process any country code element if applicable
     */
    status = cserv11dHandler(pDevInfo, &pCsInfo->scanInfo.bssSet);
    switch (status) {
    case CSERV_11D_NEW_COUNTRY_FOUND:
        cservPrintf("cserv11dHandler return: New country code\n");

        /*
         * We've just rebuilt the channel list.
         * Must clear out the cache count because we've rebuilt
         * the channel list and the BSS cache contains pointers
         * to the old channel list
         */
        cservPrintf("purge both BSS sets\n");
        cservBssSetPurge(pDevInfo);
        break;
        
    case CSERV_11D_COUNTRY_FOUND:
        cservPrintf("cserv11dHandler return: country code found\n");
        break;

    case CSERV_11D_MAX_SCAN:
        cservPrintf("cserv11dHandler return: MAX_SCAN\n");
        break;

    case CSERV_11D_NO_BSS_FOUND:
    case CSERV_11D_NOT_FOUND:
        cservPrintf("cserv11dHandler return: scan again\n");
        break;

    case CSERV_11D_JOIN_ADHOC:
        cservPrintf("cserv11dHandler return: join adhoc\n");
        break;

    case CSERV_11D_START_ADHOC:
        cservPrintf("cserv11dHandler return: start adhoc\n");
        break;

    case CSERV_11D_SELECT_BSS:
    default:
        cservPrintf("cserv11dHandler return: select BSS\n");
        break;
    }

    /*
     * Figure out which BSS to join, do it ahead of time so that result can
     * be used to start ad-hoc bss.
     *
     * Use default ssid, bssid non-null only if policy permits
     */
    i = cservSelectBss(pDevInfo, &pCsInfo->scanInfo.bssSet, pDevInfo->staConfig.bssType,
                       &pCsInfo->desSsids, NULL, 0);

    if ( (i == -1) && isIbss &&
         (RETRY_SCALE_GET(pCsInfo) >= CSERV_MAX_SCALE_ADHOC) &&
         (pCsInfo->cservOp & CSERV_OP_JOIN) &&
         (pCsInfo->desSsids.elements[0].length > 0) )
    {
        /*
         * Did not find any BSS there; have a non-null SSID; need to join a bss;
         * so for the adhoc/any case start an adhoc one.
         */
        uiPrintf("staSmeScanCallback: Scan found nothing! Starting our own IBSS\n");
        resultCode = cservStartAdhoc(pDevInfo, &pCsInfo->desSsids.elements[0], 
                                     &pCsInfo->scanInfo.bssSet);
        uiPrintf("resultCode %x\n", resultCode);
        pCsInfo->cservOp = 0;
        if (resultCode == MLME_SUCCESS) {
            cservConnectionCompleteNotify(pDevInfo);
        }

        /* Make sure that the next scan is not a hidden one */
        pCsInfo->hiddenScan = FALSE;

        return;
    }

    if (resultCode == MLME_SUCCESS && i >= 0) {
        BSSDESCR *pBss = &pCsInfo->scanInfo.bssSet.bssDescrArray[i];

        /* Reset some CS state after a successful scan and finding a BSS match. */
        pCsInfo->cservPollInterval = CSERV_TIME_AUTOCONN;
        pCsInfo->hiddenScan        = FALSE;
        RETRY_SCALE_RESET(pCsInfo);

        /* Take the next action */
        if (pCsInfo->cservOp & CSERV_OP_JOIN) {
            *pDevInfo->bssDescr = *pBss;

            pCsInfo->scanInfo.bssSet.setIdx = (A_CHAR)i;

            if (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS) {
                /*
                 * Now that we selected IBSS, cancel requests for
                 * auth and assoc
                 */
                pCsInfo->cservOp &= ~(CSERV_OP_AUTH | CSERV_OP_ASSOC | CSERV_OP_REASSOC);
            } else {
                cservPrintf("scanCallback: bss[%d].ch: %d, devInfo ch: %d\n", i,
                         pBss->pChannel->channel, pDevInfo->bssDescr->pChannel->channel);
                if (pDevInfo->staConfig.sku.v.b.cc11d) {
                    cservPrintf("scanCallback: bss[%d] channel %d flags: %x, 11a: %s %d, 11b: %s %d\n",
                             i, pDevInfo->bssDescr->pChannel->channel,
                             pDevInfo->bssDescr->pChannel->channelFlags,
                             pDevInfo->staConfig.sku.v.b.cc11a ? "y" : "n", pCsInfo->bss11d11aIdx,
                             pDevInfo->staConfig.sku.v.b.cc11b ? "y" : "n", pCsInfo->bss11d11bIdx);
                    if (pDevInfo->bssDescr->pChannel->channelFlags & CHANNEL_A) {
                        if (pDevInfo->staConfig.sku.v.b.cc11a &&
                            pbssSet->bssDescrArray[pCsInfo->bss11d11aIdx].ccList.elementID ==
                            ELE_COUNTRY_INFO)
                        {
                            pDevInfo->staConfig.sku.cc11dInfo =
                                pbssSet->bssDescrArray[pCsInfo->bss11d11aIdx].ccList;
                            cservPrintf("scanCallback -- 11b %d %d %d\n",
                                     pDevInfo->staConfig.sku.cc11dInfo.subBand[0].firstChannel,
                                     pDevInfo->staConfig.sku.cc11dInfo.subBand[0].numChannel,
                                     pDevInfo->staConfig.sku.cc11dInfo.subBand[0].maxTxPwr);
                        }
                    } else {
                        if (pDevInfo->staConfig.sku.v.b.cc11b &&
                            pbssSet->bssDescrArray[pCsInfo->bss11d11bIdx].ccList.elementID == ELE_COUNTRY_INFO)
                        {
                            pDevInfo->staConfig.sku.cc11dInfo = pbssSet->bssDescrArray[pCsInfo->bss11d11bIdx].ccList;
                            cservPrintf("scanCallback -- 11b %d %d %d\n",
                                     pDevInfo->staConfig.sku.cc11dInfo.subBand[0].firstChannel,
                                     pDevInfo->staConfig.sku.cc11dInfo.subBand[0].numChannel,
                                     pDevInfo->staConfig.sku.cc11dInfo.subBand[0].maxTxPwr);
                        }
                    }
                }

                wlanClistClnup(pDevInfo);
            }

            resultCode = wlanMlmeJoinRequest(pDevInfo, pDevInfo->bssDescr,
                                             pDevInfo->bssDescr->beaconInterval *
                                             CSERV_JOIN_BEACON_CNT);
        } else if (pCsInfo->joinPolicy != 0) {
            /*
             * there was some join policy, but there's no match.
             * assign some error code. timeout is like not found
             */
            resultCode = MLME_TIMEOUT;
        }
    }

    if (resultCode != MLME_SUCCESS || i < 0) {
        pCsInfo->cservOp = 0;

        if (IS_WZC_ACTIVE(pDevInfo) && !IS_STATE_CONNECTED(pDevInfo) &&
            pCsInfo->hiddenScan)
        {
            /*
             * Continue the scan that may have been interrupted by the hidden
             * SSID scan, or start a new scan.
             */
            cservPrintf("staSmeScanCallback: Last scan was hidden. "
                     "Resuming/Starting normal scan.\n");
            pCsInfo->hiddenScan = FALSE;
            cservScanJoinAuthAssoc(pDevInfo, pDevInfo->staConfig.scanType, FALSE);
            return;
        }

        /* Clear hidden scan in case it is still set. */
        pCsInfo->hiddenScan = FALSE;
        
        if (IS_MAX_RETRY_SCALE(pCsInfo)) {
            pCsInfo->isScanCycleComplete = TRUE;
            /*
             * We've already scanned using all of the dwell time multiples.
             * Go back and try them all again.
             */
            RETRY_SCALE_RESET(pCsInfo);
        } else {
            /* start the next scan cycle immediately. */
            RETRY_SCALE_UP(pCsInfo);
            cservScanJoinAuthAssoc(pDevInfo, pDevInfo->staConfig.scanType, FALSE);
        }
    }
}

/*******************************************************************
 * staSmeJoinCallback
 * Called after wlanMlmeJoinRequest()
 * For infrastructure network, it starts authentication process.
 * For ad-hoc. This is it. It indicates to the OS that it's connected, clean up
 * CS state and return.
 */
void
staSmeJoinCallback(WLAN_DEV_INFO *pDevInfo, MLME_RESULT resultCode)
{
    CSERV_INFO  *pCsInfo = pDevInfo->pCsInfo;

    cservPrintf("joinCallback: Result 0x%x\n", resultCode);

    if (pCsInfo->cservOp) {
        /* We had some CS action in progress */
        /* clear join bit */

        pCsInfo->cservOp &= ~CSERV_OP_JOIN;

        if (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS) {
            /*
             * if we are in ad-hoc network, no need to
             * do auth and assoc anymore
             */
            pCsInfo->cservOp &= ~(CSERV_OP_AUTH |
                                  CSERV_OP_ASSOC |
                                  CSERV_OP_REASSOC);

            if (resultCode == MLME_SUCCESS) {
                if (!GET_CONNECTION_STATUS(pDevInfo)) {
                    /*
                     *  Indicate connect only if we have indicated
                     *  disconnect before
                     */
                    osIndicateConnectStatus(pDevInfo, TRUE);
                    pCsInfo->apConnAttemptCur = 0;
                }

                /* Clear pending timer anyway */
                cservClearDisconnectTimer(pDevInfo);
            }

            cservConnectionCompleteNotify(pDevInfo);
        }

        if (resultCode == MLME_SUCCESS) {
            /* Take the next action only if we are in AP network */
            if (pCsInfo->cservOp & CSERV_OP_AUTH) {
                resultCode = wlanMlmeAuthRequest(pDevInfo,
                                                 &pDevInfo->bssDescr->bssId,
                                                 authAlgNum(pDevInfo),
                                                 RETRY_MUL(pCsInfo,
                                                           CSERV_TIME_AA_MAX));
            }
        }

        if (resultCode != MLME_SUCCESS) {
            /*
             * ResultCode may have changed! Cancel CS operation.
             */
            pCsInfo->cservOp = 0;

            /*
             * If we were in associated state before going into join, that
             * got cleared by mlme, we now need to report to OS
             */
            /* delayed notify OS upper layer */
            if (pCsInfo->disconnectCur == 0) {
                cservStartDisconnectTimer(pDevInfo);
            }
            /* 
             * Failed to receive CSERV_JOIN_BEACON_CNT beacons in a row. 
             * Assume it's gone.
             */
            mlmeBsssetDel(&pCsInfo->scanInfo.bssSet,
                          &pDevInfo->bssDescr->bssId);
        }
    }
}

/*******************************************************************
 * staSmeStartCallback
 * Called after wlanMlmeStartRequest()
 * It doesn't do any further action. It notifies OS of connection.
 */
void
staSmeStartCallback(WLAN_DEV_INFO *pDevInfo, MLME_RESULT resultCode)
{
    CSERV_INFO  *pCsInfo = pDevInfo->pCsInfo;

    if (pCsInfo->cservOp) {
        pCsInfo->cservOp &= ~CSERV_OP_JOIN;
    }

    cservPrintf("startCallback: Entry\n");

    if (resultCode == MLME_SUCCESS) {
#ifndef ADHOC_2STA_B4_CONNECTED
        osIndicateConnectStatus(pDevInfo, TRUE);
        pCsInfo->apConnAttemptCur = 0;
#endif
        uiPrintf("startCallback: BSS startup successful\n");
    } else {
        uiPrintf("startCallback: BSS startup unsuccessful, error code 0x%x\n",
                 resultCode);
    }

    if (resultCode != MLME_SUCCESS) {
        /* resultCode may have changed! */
        /* Cancel CS operation */
        pCsInfo->cservOp = 0;
    }
}

/*******************************************************************
 * staSmePowerCallback
 * Called after wlanMlmePwrmgtRequest()
 * This routine is primarily used to start a background scan because this
 * callback tells CS that it can now switch to another channel safely.
 */
void
staSmePowerCallback(WLAN_DEV_INFO *pDevInfo, A_STATUS status, A_BOOL powerEnable)
{
#ifndef BUILD_AP
    WLAN_SCAN_LIST  *pScanList = NULL;
    CSERV_INFO      *pCsInfo   = pDevInfo->pCsInfo;
    MLME_RESULT     result;

    cservPrintf("staSmePowerCallback\n");
    if (pCsInfo->cservState & CSERV_STATE_BKSCANNING &&
        pCsInfo->bkPowerReq == STATE_BLOCKING_SLEEP)
    {
        pCsInfo->bkPowerReq = 0;

        if (status != MLME_SUCCESS) {
            /* Cancel any pending fake sleep request */
            powerSet(pDevInfo, FAKE_SLEEP, DISABLE, TRUE, 0);
            return;
        }

        if (wlanInitSingleScanList(pCsInfo->bkChannel, &pScanList) != A_OK) {
            /* Cancel any pending fake sleep request */
            uiPrintf("staSmePowerCallBack : wlanInitSingleScanList failed \n");
            powerSet(pDevInfo, FAKE_SLEEP, DISABLE, TRUE, 0);
            return;
        }

        pCsInfo->cservOp = CSERV_OP_SCAN;

        /*
         * Ensure that any previous scans have completed before
         * starting a new one.
         */
        wlanMlmeScanDone(pDevInfo, TRUE);

        result = wlanMlmeScanRequest(pDevInfo, ANY_BSS,
                                     &pDevInfo->bssDescr->bssId,
                                     &pCsInfo->desSsids,
                                     pCsInfo->scanInfo.scantype,
                                     pScanList,
                                     CSERV_TIME_SCANMIN_ACTIVE,
                                     CSERV_TIME_BKSCANMAX_ACTIVE,
                                     CSERV_TIME_SCANMIN_PASSIVE,
                                     CSERV_TIME_BKSCANMAX_PASSIVE,
                                     FALSE);

        cservPrintf("staSmePowerCallback -- wlanMlmeScanRequest %s scan chan %d %dMHz, result: %#x\n",
                 pCsInfo->bkChannel->channelFlags & CHANNEL_PASSIVE ? "passive" : "active",
                 wlanConvertGHztoCh(pCsInfo->bkChannel->channel, pCsInfo->bkChannel->channelFlags),
                 pCsInfo->bkChannel->channel, result);

        if (result != MLME_SUCCESS) {
            wlanFreeScanList(pScanList);
        }
    }
#endif /* BUILD_AP */
}
/*******************************************************************
 * cservPollReset
 * Resets all counters to their initial values. Called by ResetOnError
 */
void
cservPollReset(WLAN_DEV_INFO *pDevInfo)
{
    CSERV_INFO *pCsInfo = pDevInfo->pCsInfo;

    pCsInfo->cservPollCur         = 1; // start connection attempt ASAP
    pCsInfo->rateCheckCur         = pCsInfo->rateCheckInterval;
    pCsInfo->apConnAttemptCur     = 0;
    pCsInfo->bkScanCur            = pCsInfo->bkScanInterval;
    pCsInfo->sleepBetweenScanCur  = 1;
    pCsInfo->scanBeforeSleepCur   = 0;
    pCsInfo->scanPending          = FALSE;
    pCsInfo->resetScanBeforeSleep = TRUE;
    pCsInfo->isScanCycleComplete  = FALSE;
    pCsInfo->cservState           = CSERV_STATE_IDLE;
    pCsInfo->isScanRunDone        = FALSE;
    RETRY_SCALE_RESET(pCsInfo);
}

/*******************************************************************
 * cservBmissNotify
 * Important routine. Detects that station has lost contact to AP.
 * Called by athHandleInterrupt() when hardware generates missing
 * beacon interrupt or by cservPollFunction() when it detects that
 * beacons have not been received for some time.
 * It clears the station state so that cservPollFunction() will start
 * connection sequence again.
 */
void
cservBmissNotify(WLAN_DEV_INFO *pDev)
{
    CSERV_INFO *pCsInfo = pDev->pCsInfo;
    A_BOOL     doUpdate = FALSE, bDisconnected = TRUE;

    ASSERT(pDev->localSta->serviceType == WLAN_STA_SERVICE);
    ASSERT(pDev->staConfig.bssType == INFRASTRUCTURE_BSS);

    if ((pDev->localSta->staState & (STATE_JOINED | STATE_AUTH)) !=
        (STATE_JOINED | STATE_AUTH))
    {
        /*
         * There is no centralized place to disable this when leaving a BSS,
         * so we disable it here if we are no longer "connected" with an AP.
         * TODO: move this when we have a centralized place to disconnect
         *       from a BSS.
         */
        wdcBmissAck(pDev->targetHandle, FALSE);
    }

#if NDIS_WDM
        /*     
        * (LW) Attention !!!
        * This is temp hack, should revisit later
        */
    if (pDev->noApActivity) {
#else
    /* Update the total number of beacons received */
    halMibControl(pDev, UPDATE_SW_ALL);
    if (HACK_TO_PARDEV(pDev)->noApActivity) {
#endif    
        /*
         * We have detected a BMISS previously and there has been no successful
         * transmits or receives in the interrim before receiving this BMISS
         * indication.
         */
        A_BOOL newBeacons;

        newBeacons = (pDev->localSta->stats.RxBeacons !=
                      pDev->pCsInfo->lastRxBeacons);

        if (!newBeacons) {
            A_BOOL isNewApFound;

            /*
             * This is the second time we've received this BMISS indication
             * from HW, and we have neither received traffic of any kind nor have
             * we successfully completed any sends since the last BMISS
             * indication.
             *
             * It's time to give up on this connection.
             */

            /* Giving up, always go back to base mode */
            if (pDev->turboPrimeInfo.turboPrimeAllowed &&
                pDev->turboPrimeInfo.currentBoostState == 1)
            {
                uiPrintf("cservBmissNotify: Turbo Prime back to base\n");
                turboPrimeSwitch(pDev, 0);
                bDisconnected = FALSE;
                wdcBmissAck(pDev->targetHandle, TRUE);
                doUpdate = TRUE;
            }

            if (bDisconnected == TRUE) {
                uiPrintf("cservBmissNotify: Connection is lost.\n");
                pDev->localSta->stats.txRateKb = 0;
                pDev->localSta->stats.RcvRate  = 0;
        
                /*
                 * Make sure we don't get any more of these interrupts until we're
                 * associated again.
                 */
                wdcBmissAck(pDev->targetHandle, FALSE);

                /*
                 * Remove our current BSS from the BSS before calling
                 * cservChooseApAndSwitch().  Also, don't want to send any more
                 * frames (Deauth or Disassoc) to our AP if we were to roam via
                 * cservChooseApAndSwitch(), so we call cservStopBss() first.
                 */
                cservStopBss(pDev, DISCONNECT_DELAYED, REMOVE_BSS, DONT_SEND_DEAUTH);

                /*
                 * See if there's another AP we can switch to.  If
                 * cservChooseApAndSwitch() fails, start scanning for a new AP.
                 */
                isNewApFound = cservChooseApAndSwitch(pDev, 0);
                if (!isNewApFound) {
#ifdef BUILD_AP
                    apCservNotify(pDev);
#else
                    cservScanJoinAuthAssoc(pDev, pDev->staConfig.scanType, FALSE);
#endif
                }
            }
        } else {
            /* We've received some beacons since our last BMISS indication. */
            uiPrintf("cservBmissNotify: New beacons received. AP still alive.\n");
            doUpdate = TRUE;
        }
    } else {
        /*
         * This is either the first time that we've received a BMISS or we
         * have received or sent some traffic since the last one.
         */
        uiPrintf("cservBmissNotify: First BMISS indication.\n");

        /* Toggle turbo prime mode in case missed update notification */
        /* TODO: finer resolution, no activity, signal ahead */
        if (pDev->turboPrimeInfo.turboPrimeAllowed) {
            if (pDev->turboPrimeInfo.currentBoostState == 1) {
                uiPrintf("cservBmissNotify: Turbo Prime back to base\n");
                turboPrimeSwitch(pDev, 0);
            } else {
                uiPrintf("cservBmissNotify: Turbo Prime try turbo\n");
                turboPrimeSwitch(pDev, 1);
            }
        }

        /* Ask the AP if it is still alive */
        if (pDev->staConfig.scanType != PASSIVE_SCAN) {
            wlanMlmeProbeRequest(pDev, &pDev->bssDescr->bssId, &pDev->bssDescr->ssid);
        }

        wdcBmissAck(pDev->targetHandle, TRUE);
        doUpdate = TRUE;
    }

    if (doUpdate) {
        pCsInfo->lastRxBeacons = pDev->localSta->stats.RxBeacons;
#if NDIS_WDM
        /*     
        * (LW) Attention !!!
        * This is temp hack, should revisit later
        */
        pDev->noApActivity     = TRUE;
#else
        HACK_TO_PARDEV(pDev)->noApActivity     = TRUE;
#endif
        /*
         * If the STA is asleep, wake it up to give it a good chance of
         * receiving some traffic that will keep us connected with the
         * AP. The STA will be put back to sleep naturally after its
         * timeout expires. 
         * Don't send a null data frame to AP in ETSI. DFS doesn't like it.
         */
    }
    pDev->localSta->receivedBeaconFromHomeAP = FALSE;
}

void
cservOperationCancel(WLAN_DEV_INFO *pDevInfo)
{
    CSERV_INFO *pCsInfo = pDevInfo->pCsInfo;

    cservPrintf("> cservOperationCancel\n");

    /* couldn't do fake sleep mode, cancel bk scan. */
    pCsInfo->cservState = CSERV_STATE_IDLE;
    pCsInfo->cservOp    = 0;
    pCsInfo->bkPowerReq = 0;
    if (pCsInfo->cservFakeSleep) {
        powerSet(pDevInfo, FAKE_SLEEP, DISABLE, TRUE, 0);
        pCsInfo->cservFakeSleep = FALSE;
    }

    /*
     * mlmeOperationCancel() will cancel SME timer and call timer handler routine,
     * which in turn will, call callback routines and complete the operation.
     */
    mlmeOperationCancel(pDevInfo);

    uiPrintf("< cservOperationCancel\n");

}

#define STATUS_POLL_INTERVAL   1
/*******************************************************************
 * cservInit
 * Initialization of CS module. Called by athInitialize() towards the end.
 */
void
cservInit(WLAN_DEV_INFO *pDev)
{
    CSERV_INFO      *pCsInfo = pDev->pCsInfo;
    WLAN_STA_CONFIG *pConfig = &pDev->staConfig;

    cservPrintf("cservInit: Entered\n");

    // assume that pCsInfo structure is already allocated during driver initialization
    pCsInfo->joinPolicy        = CSERV_JOIN_BESTSIGNAL;
    pCsInfo->apConnAttemptsMax = CSERV_MAX_ATTEMPTS;
    pCsInfo->cservPollInterval = CSERV_TIME_AUTOCONN;  // automatic connection polling interval
    pCsInfo->rateCheckInterval = CSERV_TIME_RATECHECK; // automatic rate control period
    pCsInfo->bkScanInterval    = CSERV_TIME_BKSCAN;
    pCsInfo->cservOp           = 0;                    // Flag bits to indicate which command to do next.
    pCsInfo->cservState        = CSERV_STATE_IDLE;     // current state is idle

    pCsInfo->statusPollInterval = SECS_TO_STAPOLL(STATUS_POLL_INTERVAL);
    pCsInfo->statusPollCount    = 0;

    // load cserv configuration
    pCsInfo->sleepBetweenScanInterval = SECS_TO_STAPOLL(pConfig->sleepTimePostScan);
    pCsInfo->scanBeforeSleepInterval  = SECS_TO_STAPOLL(pConfig->scanTimePreSleep);

    pCsInfo->bkChannelIndex        = 0;
    pCsInfo->bkChannelCnt          = 0;
    pCsInfo->bkScanList            = NULL;
    pCsInfo->scanInfo.bssSet.count = 0;
    RETRY_SCALE_RESET(pCsInfo);

    // Now call this routine to load max values into current counters.
    cservPollReset(pDev);

    // Copy all desired SSIDs
    pCsInfo->desSsids = pConfig->cfgSsids;

    pCsInfo->disconnectInterval = CSERV_DISCONNECT_TIMEOUT;

    // When we first start up the driver, the OS assumes that we're "connected".
    // We need to indicate that we're disconnected within 2 seconds (according
    // to specifications).
    pCsInfo->disconnectCur = CSERV_INIT_DISCONNECT_TIME;
}

/*******************************************************************
 * cservShutDown
 * Shutdown the CS module. Not used yet.
 */
void
cservShutdown(WLAN_DEV_INFO *pDevInfo)
{
    // do all that we need to do to stop connection services.
    // cannot delete the polling timer here, it's outside cs

    if (pDevInfo->pCsInfo) {
        A_DRIVER_FREE(pDevInfo->pCsInfo, sizeof(CSERV_INFO));
        pDevInfo->pCsInfo = NULL;
    }
}

/*******************************************************************
 * cservAssociate
 * Associate with AP
 */
MLME_RESULT
cservAssociate(WLAN_DEV_INFO *pDevInfo, A_UINT32 assocTimeout,
               A_UINT32 listenInterval)
{
    MLME_RESULT     resultCode;
    SIB_ENTRY       *mySib;
    CSERV_INFO      *pCsInfo = pDevInfo->pCsInfo;

    cservPrintf("cservAssociate: Entered\n");
    cservPrintf("cservOp %x, cservState %x\n", pCsInfo->cservOp,
             pCsInfo->cservState);

    mySib = pDevInfo->localSta;

    if (pCsInfo->cservOp) {
        // Some cserv action already in progress, reject
        cservPrintf("cservAssociate: busy\n");
        return MLME_TOOMANY_REQ;
    }

    pCsInfo->apConnAttemptCur++;
    pCsInfo->cservOp = CSERV_OP_ASSOC;
    resultCode = wlanMlmeAssocRequest(pDevInfo, &pDevInfo->bssDescr->bssId,
                                      assocTimeout,
                                      &pDevInfo->localSta->capInfo,
                                      (A_UINT16)listenInterval);

    if (resultCode == MLME_SUCCESS) {
        return MLME_OP_PENDING;
    } else {
        if (pDevInfo->pSmeInfo->assocCallback) {
            (*pDevInfo->pSmeInfo->assocCallback)(pDevInfo, resultCode);
        }
        return resultCode;
    }
}

/*******************************************************************
 * cservDeauthenticate
 * Deauthenticate from AP
 */
MLME_RESULT
cservDeauthenticate(WLAN_DEV_INFO *pDevInfo)
{
    MLME_RESULT     resultCode;

    uiPrintf("cservDeauthenticate: Entered\n");

    resultCode = wlanMlmeDeauthRequest(pDevInfo, &pDevInfo->baseBss,
                                       &pDevInfo->bssDescr->bssId,
                                       REASON_AUTH_EXPIRED, TRUE);

    return resultCode;
}

/*******************************************************************
 * cservScan
 * Start a scan of all channels.  Additional CSERV operations may
 * be passed in via the 'extraOps' parameter.
 */
MLME_RESULT
cservScan(WLAN_DEV_INFO *pDevInfo, SCANTYPE scanType, A_UINT32 extraOps, A_BOOL hiddenOnly)
{
    MLME_RESULT     result;
    SIB_ENTRY       *mySib;
    CSERV_INFO      *pCsInfo   = pDevInfo->pCsInfo;
    WLAN_SCAN_LIST  *pScanList = NULL;

    ASSERT(pCsInfo);

    cservPrintf("cservScan: Entered\n");
    cservPrintf("cservOp %x, cservState %x\n", pCsInfo->cservOp, pCsInfo->cservState);

    if (pCsInfo->cservOp) {
        // Some cserv action already in progress, reject
        cservPrintf("cservScan: busy\n");
        return MLME_TOOMANY_REQ;
    }

    /*
     * Reset our knowledge of BSS existence on other channels before
     * we initialize the scan list.  Only do this after we've successfully
     * associated and then disassociated so that we don't clear out the
     * knowledge too often.  Also, do not wipe out this information if
     * the scan is hidden, as this information is necessary to perform
     * a "hidden", or "temporary", scan.
     */
    if (pCsInfo->wasDisassociated && !hiddenOnly) {
        pCsInfo->wasDisassociated = FALSE;
        wlanClistNewScanCallback(pDevInfo);
    }
    
    if (wlanInitScanList(pDevInfo, pDevInfo->staConfig.pClist, &pScanList, hiddenOnly) != A_OK) {
        uiPrintf("cservScan: wlanInitScanList failed \n");
        return MLME_REFUSED;
    }

    /*
     * If this is the first scan, start the long timeout so that we can scan
     * for a while before finally deciding to go to sleep.
     */
    if (pCsInfo->resetScanBeforeSleep) {
        pCsInfo->scanBeforeSleepCur   = pCsInfo->scanBeforeSleepInterval;
        pCsInfo->resetScanBeforeSleep = FALSE;
    }
    
    if (pCsInfo->cservForcedSleep) {
        /* Return from deep sleep and synch sleep ctr */
        powerSet(pDevInfo, FORCE_SLEEP, DISABLE, FALSE, 0);
        pCsInfo->cservForcedSleep = FALSE;
        pCsInfo->sleepBetweenScanCur = 0;
    }

    mySib                        = pDevInfo->localSta;
    pCsInfo->cservOp             = CSERV_OP_SCAN | extraOps;
    pCsInfo->isScanCycleComplete = FALSE;

    if (hiddenOnly) {
        pCsInfo->hiddenScan = TRUE;
    }
    
    /* Collect all parameters required for mlmeScanRequest */
    if (cservDebugLevel) {
        clistShow(pDevInfo);
    }

    /* Scan for our SSID and any BSS */
    result = wlanMlmeScanRequest(pDevInfo, ANY_BSS, &pDevInfo->bssDescr->bssId,
                                 &pCsInfo->desSsids, scanType,
                                 pScanList,
                                 RETRY_MUL(pCsInfo, CSERV_TIME_SCANMIN_ACTIVE),
                                 RETRY_MUL(pCsInfo, CSERV_TIME_SCANMAX_ACTIVE),
                                 RETRY_MUL(pCsInfo, CSERV_TIME_SCANMIN_PASSIVE),
                                 RETRY_MUL(pCsInfo, CSERV_TIME_SCANMAX_PASSIVE),
                                 hiddenOnly);

    cservPrintf("cservScan -- wlanMlmeScanRequest %s scan result: %#x\n",
             scanType == ACTIVE_SCAN ? "active" :
             scanType == PASSIVE_SCAN ? "passive" : "any", result);

    if (result == MLME_SUCCESS) {
        return MLME_OP_PENDING;
    } else {
        return result;
    }
}

/*******************************************************************
 * cservJoin
 * Join the BSS that is already copied into pDevInfo->bssDescr
 */
MLME_RESULT
cservJoin(WLAN_DEV_INFO *pDevInfo)
{
    MLME_RESULT     resultCode;
    SIB_ENTRY       *mySib;
    CSERV_INFO      *pCsInfo = pDevInfo->pCsInfo;

    ASSERT(pCsInfo);

    cservPrintf("cservJoin: Entered\n");
    cservPrintf("cservOp %x, cservState %x\n", pCsInfo->cservOp, pCsInfo->cservState);

    if (pCsInfo->cservOp) {
        // Some cserv action already in progress, reject
        cservPrintf("cservJoin: busy\n");
        return MLME_TOOMANY_REQ;
    }

    /*
     * Scan confirmation has stored the correct description
     * in staBssDescr. Use it to join the bss.
     */
    mySib            = pDevInfo->localSta;
    pCsInfo->cservOp = CSERV_OP_JOIN;

    /* collect all parameters required for mlmeJoinRequest */
    resultCode = wlanMlmeJoinRequest(pDevInfo, pDevInfo->bssDescr,
                     pDevInfo->bssDescr->beaconInterval *
                     CSERV_JOIN_BEACON_CNT);
    cservPrintf("cservJoin: Join request result = %x\n", resultCode);

    if (resultCode == MLME_SUCCESS) {
        return MLME_OP_PENDING;
    } else {
        return resultCode;
    }
}

/*******************************************************************
 * cservStartAdhoc
 * Start a new IBSS, build our own bssDescr
 */
MLME_RESULT
cservStartAdhoc(WLAN_DEV_INFO *pDevInfo, SSID *pSsid, BSSDESCR_SET *pBssSet)
{
    CSERV_INFO      *pCsInfo = pDevInfo->pCsInfo;
    WLAN_STA_CONFIG *pConfig = &pDevInfo->staConfig;
    BSSDESCR        *pBss    = pDevInfo->bssDescr;
    WLAN_CFLAGS     cflags;
    MLME_RESULT     resultCode;

    ASSERT(pCsInfo);

    A_MEM_ZERO(pBss, sizeof(*pBss));

    if (pConfig->privacyInvoked &&
        pDevInfo->keyTable[pDevInfo->staConfig.defaultKey].keyLength == 0)
    {
        // asking for WEP enabled frames, but key is not set.
        // cannot start this SSID.
        return MLME_INVALID_PARAM;
    }

    if (mlmeBsssetIsFull(&pCsInfo->scanInfo.bssSet) == TRUE) {
        return MLME_INVALID_PARAM;
    }
        
    if (pConfig->AdhocBand == ATH_ADHOCBAND_NONE) {
        return MLME_INVALID_PARAM;
    }

    /* Determine wireless mode to start adhoc */
    switch (pConfig->AdhocBand) {
        case ATH_ADHOCBAND_BONLY:
            cflags = CHANNEL_B;
            break;

        case ATH_ADHOCBAND_AONLY:
            cflags = CHANNEL_A;
            break;

        case ATH_ADHOCBAND_TONLY:
            cflags = CHANNEL_T;
            break;

        case ATH_ADHOCBAND_GONLY:
            cflags = CHANNEL_G;
            break;

        case ATH_ADHOCBAND_108GONLY:
            cflags = CHANNEL_108G;
            break;

        default:
            /* Should never come here, but default is 11a */
            cflags = CHANNEL_A;
    }

    /*
     * Check if adhoc is allowed or not.
     * If not, then check if adhoc is allowed in 11b.
     * If so, then start adhoc in 11b. Otherwise, quit.
     */
    if (IS_CHAN_5GHZ(cflags)) {
        if (pDevInfo->staConfig.sku.adHocMode == START_ADHOC_PER_11D &&
            pDevInfo->staConfig.sku.v.b.cc11d == FALSE) {
            cflags = CHANNEL_B;
        }
        if (wlanIsAdHocAllowed(pDevInfo, cflags) == FALSE) {
            cflags = CHANNEL_B;
            if (wlanIsAdHocAllowed(pDevInfo, cflags) == FALSE) {
                /* Adhoc is not allowed */
                uiPrintf("cservStartAdhoc: IBSS is not allowed\n");
                return MLME_INVALID_PARAM;
            }
            pConfig->AdhocBand = ATH_ADHOCBAND_BONLY;
        }
    }
    
    /*
     * In the absence of country info for global mode SKU,
     * no 108g allowed in adhoc.
     */
    if (wlanIsWwrSKU (pDevInfo) == TRUE &&
        pConfig->AdhocBand == ATH_ADHOCBAND_108GONLY &&
        pDevInfo->staConfig.sku.v.b.commonMode == TRUE)
    {
        /* Adhoc is not allowed */
        uiPrintf("cservStartAdhoc: Not allowed to start 108g IBSS\n");
        return MLME_INVALID_PARAM;
    }

    cservSelectAdhocChannel(pDevInfo, pCsInfo);
    if (pConfig->adhocChannelFreq != 0 &&
        (pConfig->AdhocBand == ATH_ADHOCBAND_BONLY ||
         pConfig->AdhocBand == ATH_ADHOCBAND_GONLY ||
         pConfig->AdhocBand == ATH_ADHOCBAND_108GONLY))
    {
        pBss->pChannel = wlanFindChannelMatch(pDevInfo, pConfig->pClist,
                                              cflags, pConfig->adhocChannelFreq);
        if (pBss->pChannel == NULL) {
            uiPrintf("cservStartAdhoc: Wrong channel for starting IBSS.\n");
            return MLME_INVALID_PARAM;
        }
    } else {
        /* TODO: Scan possible interfering wireless mode for better channel placement */
        pBss->pChannel = wlanFindChannel(pDevInfo, &pCsInfo->scanInfo.bssSet,
                                         pConfig->pClist, NULL, cflags);

        resultCode = wlanSelectAdHocChannel(pDevInfo, pBssSet, cflags);
        if (resultCode || pBss->pChannel == NULL) {
            // couldn't find a valid channel to start adhoc. punt.
            uiPrintf("cservStartAdhoc: Can't find channel to start ad Hoc\n");
            return MLME_INVALID_PARAM;
        }
    }

    if (wlanIsAdHocAllowed(pDevInfo, cflags) == FALSE) {
        /*
         * Cannot start adhoc if active scan is not allowed. We don't want to start transmitting
         * when we are not sure about radar present on the channel. There is no
         * radar detection capability on station yet. We do passive scan in ETSI.
         */
        uiPrintf("cservStartAdhoc: Can't start IBSS because active scan is not allowed.\n");
        return MLME_INVALID_PARAM;
    }

    /*
     * Added to allow 11G capable station to start in 11B mode.
     */
    if (pConfig->AdhocBand == ATH_ADHOCBAND_BONLY) {
        pBss->pChannel->channelFlags &= ~(CHANNEL_G | CHANNEL_108G);
        pBss->pChannel->channelFlags |= CHANNEL_B;
        pBss->opMode = wlanCFlagsToWirelessMode(pDevInfo, CHANNEL_B);
    }

    // create different elements. Cannot assume any preinitialized values in bssdescr.
    mlmeElementCopy((INFO_ELEMENT *)pSsid, (INFO_ELEMENT *)&pBss->ssid);
    pBss->bsstype                 = INDEPENDENT_BSS;
    pBss->capabilityInfo.ess      = 0;
    pBss->capabilityInfo.ibss     = 1;
    pBss->capabilityInfo.privacy  = pConfig->privacyInvoked;
    pBss->rssi                    = 0;
    pBss->ibssParamSet.elementID  = ELE_IBSS_PARAM_SET;
    pBss->ibssParamSet.length     = ELE_IBSS_PARAM_SIZE;

    if (IS_CHAN_CCK(pBss->pChannel->channelFlags)) {
        pBss->capabilityInfo.shortPreamble = pConfig->shortPreamble;
    } else {
        pBss->capabilityInfo.shortPreamble = 0;
    }

    /* Don't yet support power management in adhoc. Reset atimWindow to default */
    pBss->ibssParamSet.atimWindow = pDevInfo->defaultStaConfig.atimWindow;
    pConfig->atimWindow           = pDevInfo->defaultStaConfig.atimWindow;
    pBss->beaconInterval          = pDevInfo->staConfig.beaconInterval;
    pBss->DTIMPeriod              = DEFAULT_DTIM_PERIOD;

    /* make sure CF_PARAM_SET element is invalid */
    pBss->cfParamSet.length = 0;

    cservCreateAdhocBssid(pDevInfo, &pBss->bssId);

    resultCode = wlanMlmeStartRequest(pDevInfo, pBss);

    pBss->scanTime = A_MS_TICKGET(); // save the time of start
    mlmeBsssetAdd(pDevInfo, pBss, &pCsInfo->scanInfo.bssSet);

    uiPrintf("cservStartAdhoc: result %x\n", resultCode);

    return resultCode;
}

/*******************************************************************
 * cservAuthAssoc - authenticate and associate with bss
 *
 * DESCRIPTION: This routine performs authentication followed by
 * association.
 */
MLME_RESULT
cservAuthAssoc(WLAN_DEV_INFO *pDevInfo)
{
    MLME_RESULT     resultCode;
    CSERV_INFO      *pCsInfo = pDevInfo->pCsInfo;

    ASSERT(pDevInfo->pCsInfo);

    cservPrintf("cservAuthAssoc: Entered\n");
    cservPrintf("cservOp %x, cservState %x\n", pCsInfo->cservOp, pCsInfo->cservState);

    if (pCsInfo->cservOp) {
        // Some cserv action already in progress, reject
        cservPrintf("cservAuthAssoc: busy\n");
        return MLME_TOOMANY_REQ;
    }

    pCsInfo->apConnAttemptCur++;
    pCsInfo->cservOp = CSERV_OP_AUTH | CSERV_OP_ASSOC;
    resultCode = wlanMlmeAuthRequest(pDevInfo, &pDevInfo->bssDescr->bssId,
                                     authAlgNum(pDevInfo),
                                     RETRY_MUL(pCsInfo, CSERV_TIME_AA_MAX));

    if (resultCode == MLME_SUCCESS) {
        return MLME_OP_PENDING;
    } else {
        return resultCode;
    }
}

/*******************************************************************
 * cservScanJoinAuthAssoc - Perform scan-join-auth-assoc operation
 *
 * DESCRIPTION: This routine performs active or passive scan,
 * collects beacons, then it chooses the bss using joinPolicy and
 * performs a join operation. Then it performs authentication followed
 * by association. This is an equivalent of bringing up the station
 * on the access point.
 */
MLME_RESULT
cservScanJoinAuthAssoc(WLAN_DEV_INFO *pDevInfo, SCANTYPE scanType, A_BOOL hiddenOnly)
{
    MLME_RESULT     result;
    CSERV_INFO      *pCsInfo   = pDevInfo->pCsInfo;

    ASSERT(pCsInfo);

    cservPrintf("cservScanJoinAuthAssoc: Entered\n");
    cservPrintf("cservOp %x, cservState %x\n", pCsInfo->cservOp, pCsInfo->cservState);

    if (pCsInfo->cservOp) {
        // Some cserv action already in progress, reject
        cservPrintf("cservScanJoinAuthAssoc: busy\n");
        return MLME_TOOMANY_REQ;
    }

    pCsInfo->apConnAttemptCur++;

    result = cservScan(pDevInfo, scanType,
                       CSERV_OP_JOIN | CSERV_OP_AUTH | CSERV_OP_ASSOC, hiddenOnly);

    return result;
}

/*
 * canStaRessoc
 * This function returns a boolean result which indicates whether
 * the station can reassociate with the newly selected AP
 */
LOCAL A_BOOL 
canStaReassoc(WLAN_DEV_INFO *pDev, BSSDESCR *pbssDescr) 
{
    ASSERT(pDev->bssDescr);
    ASSERT(pDev->bssDescr->bsstype == INFRASTRUCTURE_BSS);

    /*
     * Station can re-associate only if 
     *    - global reassocEnable is true
     *    - station is already associated
     *    - the ssid of the old AP and the new AP are the same
     */
    if (pDev->staConfig.reAssocEnable            &&
        (pDev->localSta->staState & STATE_ASSOC) &&
        isSsidEqual(&pbssDescr->ssid, &pDev->bssDescr->ssid))
    {
        return TRUE;
    }

    return FALSE;
}

/*******************************************************************
 * cservSwitchAP
 * Assuming that we are connected to one AP, move over to other.
 * Reassoc is not fully functional. Hence, we just disassoc/deauth
 * from current, connect to the other.
 */
MLME_RESULT
cservSwitchAP(WLAN_DEV_INFO *pDevInfo, BSSDESCR *pbssDescr)
{
    MLME_RESULT     resultCode;
    CSERV_INFO      *pCsInfo = pDevInfo->pCsInfo;
    WLAN_CFLAGS     chanFlags;
    A_BOOL          reassoc;

    ASSERT(pCsInfo);

    cservPrintf("cservSwitchAP: Entered\n");

    chanFlags = pDevInfo->bssDescr->pChannel->channelFlags & CHANNEL_ALL;

    if (A_MACADDR_COMP(&pDevInfo->bssDescr->bssId, &pbssDescr->bssId) == 0 &&
        (pbssDescr->pChannel->channelFlags & CHANNEL_ALL) == chanFlags)
    {
        // don't need to switch. We are already on that AP.
        uiPrintf("cservSwitchAP: We're already with the AP that we want\n");
        return MLME_SUCCESS;
    }

    // cancel current cserv operation. Background scan may still be in progress
    cservOperationCancel(pDevInfo);

    uiPrintf("cservSwitchAP: switching from ");
    printMacAddress(pDevInfo->bssDescr->bssId);
    uiPrintf(" to ");
    printMacAddress(pbssDescr->bssId);
    uiPrintf("\n");

    // wake up chip until new assoc is made
    powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_1, FALSE, 0);

    reassoc = canStaReassoc(pDevInfo, pbssDescr);
    
    // are we currently associated? disconnect.
    if (pDevInfo->localSta->staState & STATE_AUTH) {
        resultCode =
            wlanMlmeDeauthRequest(pDevInfo, &pDevInfo->baseBss,
                                  &pDevInfo->bssDescr->bssId,
                                  REASON_AUTH_LEAVING, reassoc ? FALSE : TRUE);
    }

    cservStartDisconnectTimer(pDevInfo);
    pDevInfo->localSta->staState &= ~STATE_JOINED;

    // now start the operation to join new AP and auth and assoc
    pCsInfo->cservOp = CSERV_OP_JOIN | CSERV_OP_AUTH;
    if (reassoc) {
        // Copy the bssid of the current AP which will be used in the reassoc request frame
        A_MACADDR_COPY(&pDevInfo->bssDescr->bssId, &pDevInfo->lastApAddr);
        pCsInfo->cservOp |= CSERV_OP_REASSOC;
    } else {
        pCsInfo->cservOp |= CSERV_OP_ASSOC;
    }

    resultCode = wlanMlmeJoinRequest(pDevInfo, pbssDescr,
                                     pDevInfo->bssDescr->beaconInterval *
                                     CSERV_JOIN_BEACON_CNT);

    cservPrintf("cservSwitchAP: Join Request result %x\n", resultCode);

    // chip will sleep after assoc is successful
    powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_1, FALSE, 0);

    if (resultCode == MLME_SUCCESS) {
        return MLME_OP_PENDING;
    } else {
        return resultCode;
    }
}

/*******************************************************************
 * cservSwitchSsid
 * Assuming that we may be connected to one SSID, move to another.
 */
MLME_RESULT
cservSwitchSsid(WLAN_DEV_INFO *pDevInfo, SSID_ARRAY *pSsids,
                A_BOOL fromWzc)
{
    MLME_RESULT resultCode;
    CSERV_INFO  *pCsInfo = pDevInfo->pCsInfo;
    A_INT32     i;
    A_BOOL      isIbss     = (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS);
    A_BOOL      isJoined   = (pDevInfo->localSta->staState & STATE_JOINED);

    ASSERT(pCsInfo);

    cservPrintf("cservSwitchSsid: Entered\n");

    /*
     * Strange requirement from MS OIDs: even if we reject the SSID, we should
     * disconnect from previous network connection.
     */

    /* Cancel current connection services operation. */
    cservOperationCancel(pDevInfo);

    pDevInfo->staConfig.cfgSsids = *pSsids;
    pCsInfo->desSsids = *pSsids;

    if (fromWzc &&
        GET_CONNECTION_STATUS(pDevInfo) &&
        ssidArrayFind(pSsids, &pDevInfo->bssDescr->ssid) != -1)
    {
        /*
         * The following is to address the case where autoconnect caused us to
         * connect with a different WEP status than WZC gave us subsequently for
         * the same SSID.  We need to disconnect and reconnect with the right
         * CSN elements in that case.  We discern the mode we connected in the
         * last time around based on the presence of a valid UCSE element.
         */
#define privacyInvokedLastAssoc(_pDev) \
        (VALID_CIPHER_OUI(((_pDev)->localSta->uCipher)))

        if (! ((pDevInfo->staConfig.privacyInvoked &&
                !privacyInvokedLastAssoc(pDevInfo))
               || ((!pDevInfo->staConfig.privacyInvoked) &&
                   privacyInvokedLastAssoc(pDevInfo))) )
        {
            return MLME_SUCCESS;
        }
    }

    /* 
     * If SSIDs are changed we have to disconnect from the old one.
     * Must be awake to access hardware
     */
    powerSet(pDevInfo, WAKE_UP, SET_PWR_TO_1, FALSE, 0);

    if (IS_STATE_CONNECTED(pDevInfo)) {
        cservStopBss(pDevInfo, DISCONNECT_NOW, DONT_REMOVE_BSS, SEND_DEAUTH);
    } else {
         /* Remove the Adhoc BSS on switch ssid */
        if (isIbss && isJoined) {
            mlmeBsssetDel(&pCsInfo->scanInfo.bssSet, &pDevInfo->bssDescr->bssId);
        }
    }

    if (fromWzc) {
        sibFreeTargetConn(pDevInfo, pDevInfo->pSibTable);
    }

    powerSet(pDevInfo, WAKE_UP, REDUCE_PWR_FROM_1, FALSE, 0);

    // figure out which BSS to join, do it ahead of time so that result can
    // use new ssid
    i = cservSelectBss(pDevInfo, &pCsInfo->scanInfo.bssSet,
                       pDevInfo->staConfig.bssType, &pCsInfo->desSsids,
                       NULL, 0);

    if (i < 0) {
        // We did not find any matching BSS in the list. In order to locate
        // hidden SSIDs using Global Mode and Wireless Zero Config we start
        // a "temporary" scan here which will re-scan the channels on which
        // we've previously seen a hidden SSID beacon.  If that fails to
        // match the new SSID just handed down, the previous scan will be
        // restarted or, if none was interrupted, a new one will commence.
        //uiPrintf("cservSwitchSsid: No matching BSS.  Starting hidden SSID scan.\n");
        resultCode = cservScanJoinAuthAssoc(pDevInfo, pDevInfo->staConfig.scanType, fromWzc);

        return resultCode;
    }

    // we can proceed with new SSID
    // now start the operation to join new AP and auth and assoc
    pCsInfo->scanInfo.bssSet.setIdx = (A_CHAR)i;


    *pDevInfo->bssDescr         = pCsInfo->scanInfo.bssSet.bssDescrArray[i];
    pDevInfo->staConfig.bssType = pDevInfo->bssDescr->bsstype;

    uiPrintf("cservSwitchSsid: BSS[%d] ", i);
    printSsid(&pDevInfo->bssDescr->ssid);
    uiPrintf("\n");

    if (pCsInfo->cservForcedSleep) {
        /* Return from deep sleep and synch sleep ctr */
        powerSet(pDevInfo, FORCE_SLEEP, DISABLE, FALSE, 0);
        pCsInfo->cservForcedSleep = FALSE;
        pCsInfo->sleepBetweenScanCur = 0;
    }
    resultCode = wlanMlmeJoinRequest(pDevInfo, pDevInfo->bssDescr,
                                     pDevInfo->bssDescr->beaconInterval *
                                     CSERV_JOIN_BEACON_CNT *
                                     RETRY_SCALE_GET(pCsInfo));

    cservPrintf("cservSwitchSsid: Join Request result %x\n", resultCode);

    if (resultCode == MLME_SUCCESS) {
        pCsInfo->cservOp = CSERV_OP_JOIN | CSERV_OP_AUTH | CSERV_OP_ASSOC;
        return MLME_OP_PENDING;
    } else {
        return resultCode;
    }
}

void
cservStopBss(WLAN_DEV_INFO *pDev, DISCONNECT_ENUM discMode,
             REMOVE_BSS_ENUM removeBss, SEND_DEAUTH_ENUM deauth)
{
    CSERV_INFO  *pCsInfo   = pDev->pCsInfo;
    A_BOOL      isIbss     = (pDev->bssDescr->bsstype == INDEPENDENT_BSS);
    A_BOOL      isJoined   = (pDev->localSta->staState & STATE_JOINED);
    CHAN_VALUES *pChannel  = pDev->staConfig.pChannel;

    ASSERT(pDev->pCsInfo);

    /*
     * Since the use of the channelFlags is overloaded as both a
     * capability indicator and a mode selector.  Need to 
     * restore the channelFlags value based on the NetBand.  For
     * now just restore current channel in use. 
     */
    if (pChannel) {
        if (IS_CHAN_2GHZ(pChannel->channelFlags) &&
            (pDev->staConfig.NetBand & (MODE_SELECT_11G | MODE_SELECT_108G)))
        {
            pChannel->channelFlags &= ~CHANNEL_B;
            pChannel->channelFlags |= IS_CHAN_108G(pChannel->channelFlags) ?
                                      CHANNEL_108G : CHANNEL_G;
        }
    }
    
    /*
     * Call wlanResetRequest() before removing our BSS from the list as the
     * aforementioned call may cancel a scan in progress which includes our
     * current BSS.
     */
    wlanResetRequest(pDev, deauth);

    if ((isIbss && isJoined) || removeBss == REMOVE_BSS) {
        mlmeBsssetDel(&pCsInfo->scanInfo.bssSet, &pDev->bssDescr->bssId);
    }


    switch (discMode) {
    case DISCONNECT_NOW:
        /* Indicate to the OS that we're disconnected immediately. */
        osIndicateConnectStatus(pDev, FALSE);
        break;

    case DISCONNECT_DELAYED:
        /*
         * Start the disconnect coundown timer
         * (for indicating disconnect to the OS).
         */
        cservStartDisconnectTimer(pDev);
        break;

    default:
        /* Unknown mode. */
        ASSERT(0);
    }
}

/*******************************************************************
 * cservBkScan
 * Perform a background scan on one channel.
 */
MLME_RESULT
cservBkScan(WLAN_DEV_INFO *pDevInfo, CHAN_VALUES *pChval, SCANTYPE scanType)
{
#ifdef BUILD_AP
    if (pDevInfo->staConfig.bkScanEnable == 0) {
        cservPrintf("cservBkScan: not enabling background scanning\n");
    }
    return MLME_REFUSED;
#else
    SIB_ENTRY   *mySib;
    CSERV_INFO  *pCsInfo = pDevInfo->pCsInfo;
    A_UINT32    status   = A_OK;

    ASSERT(pCsInfo);
    ASSERT(pChval);

    cservPrintf("cservBkScan: Entered, channel %d\n", pChval->channel);
    cservPrintf("cservOp %x, cservState %x\n", pCsInfo->cservOp, pCsInfo->cservState);

    if (pDevInfo->staConfig.bkScanEnable == 0) {
        uiPrintf("cservBkScan: not enabling background scanning\n");
        return MLME_REFUSED;
    }

    if (pCsInfo->cservOp) {
        // Some cserv action already in progress, reject
        cservPrintf("cservBkScan: busy\n");
        return MLME_TOOMANY_REQ;
    }

    if (pCsInfo->bkPowerReq != 0) {
        uiPrintf("csrevBkScan: failed, power request already in progress\n");
        return MLME_TOOMANY_REQ;
    }

        

    // Step 1: preparing for BKSCAN
    // set a flag saying I'm doing background scan
    // enable power save mode
    // put a null data frame with PS sleep

    mySib                      = pDevInfo->localSta;
    pCsInfo->scanInfo.scantype = scanType;
    pCsInfo->bkChannel         = pChval;
    pCsInfo->cservState        = CSERV_STATE_BKSCANNING;
    pCsInfo->bkPowerReq        = 0;

    if ((mySib->staState & STATE_CONNECTED) == STATE_CONNECTED) {
        /* Tell the AP that we're going to sleep before background scanning */
        status = wlanMlmePwrmgtRequest(pDevInfo, PS_MODE_POWER_SAVE, TRUE);
        if (MLME_SUCCESS == status || MLME_OP_PENDING == status) {
            pCsInfo->cservFakeSleep = TRUE;
        }

    } else {
        /*
         * no need to pretend sleep since we are not doing power save or not associated.
         * directly call the routine that kicks off scan.
         * also, no need to block-sleep on our own channel
         */
        ASSERT(pDevInfo->powerMgmt.powerState != D2_STATE);
        // wake up the NIC to scan
        status = wlanMlmePwrmgtRequest(pDevInfo, PS_MODE_ACTIVE, TRUE);
    }

    switch (status) {
    case MLME_OP_PENDING:
        // Wait for ps frame to go out, staSmePowerCallback() will be called by pwrmgt.
        pCsInfo->bkPowerReq = STATE_BLOCKING_SLEEP;
        return MLME_SUCCESS;

    case MLME_SUCCESS:
        // It's ok to change channel and go scanning.
        pCsInfo->bkPowerReq = STATE_BLOCKING_SLEEP;
        staSmePowerCallback(pDevInfo, A_OK, TRUE);
        return MLME_SUCCESS;

    case MLME_DRIVER_ERROR:
    default:
        return MLME_DRIVER_ERROR;
    }
#endif
}

/*******************************************************************
 * cservBkScanAll
 * Start a sequence of background scan on all channels.
 */
MLME_RESULT
cservBkScanAll(WLAN_DEV_INFO *pDevInfo, BKSCAN_ENUM scanType)
{
    CSERV_INFO      *pCsInfo = pDevInfo->pCsInfo;
    MLME_RESULT     status;
    WIRELESS_MODE   mode;
    
    mode = wlanCFlagsToWirelessMode(pDevInfo, pDevInfo->staConfig.pChannel->channelFlags);

    ASSERT(pCsInfo);

    cservPrintf("cservBkScanAll: Entered\n");
    cservPrintf("cservOp %x, cservState %x\n", pCsInfo->cservOp, pCsInfo->cservState);

    if (pCsInfo->cservState & CSERV_STATE_BKSCANNING) {
        /* we are already doing a bk scan. */
        cservPrintf("cservBkScanAll: busy\n");
        return MLME_TOOMANY_REQ;
    }

    /* Defer background scan while in turbo phase of turbo prime */
    if (pDevInfo->turboPrimeInfo.turboPrimeAllowed && ((mode == WIRELESS_MODE_108g) || (mode == WIRELESS_MODE_TURBO))) {
        cservPrintf("cservBkScanAll: In Turbo Mode, Bk Scan Deferred\n");
        pCsInfo->scanPending = TRUE;
        return MLME_REFUSED;
    }

    /*
     * Check to see if we've had send/receive activity recently if the
     * caller has opted for a less-intrusive background scan.
     */
    if ( scanType == SCAN_WHEN_IDLE &&
         (A_MS_TICKGET() - pCsInfo->lastActivityTime) <
         (A_UINT32)(pDevInfo->noActivityInterval * CSERV_TIME_STAPOLL) )
    {
        cservPrintf("cservBkScanAll: Too busy to scan right now\n");
        pCsInfo->scanPending = TRUE;
        return MLME_REFUSED;
    }

    pCsInfo->scanPending = FALSE;

    /* Clear the current scan list, if configured to do so. */
    if (pDevInfo->staConfig.clearListOnScan) {
        mlmeBsssetAge(&pCsInfo->scanInfo.bssSet, 0, &pDevInfo->bssDescr->bssId);
    }

    pCsInfo->bkScanCur = pCsInfo->bkScanInterval;

    /* look for the same ssid that we have right now. */
    pCsInfo->desSsids = pDevInfo->staConfig.cfgSsids;

    /*
     * Previously, the background channel scan index
     * (pCsInfo->bkChannelIndex) is used to perform two functions.
     * To obtain the background scan channel by using its value
     * to index to the channel list points to by pDevInfo->staConfig.pClist.
     * Its value is also used to terminate the background scan sequence.
     * In this implementation, the background scan sequence is
     * controlled by a separate background channel scan counter.
     * The background channel scan index is used only to index
     * to the channel list table in pDevInfo->staConfig.
     * The background scan sequence begins by calling wlanBkSlistSort()
     * to re-arrange the channel list table in pDevInfo->staConfig.
     * Because wlanBkSlistSort() will always put the current channel first,
     * the current channel will be the very first one used by cservBkScan
     * to perform the background scan. Afterwards, cservPollFunction() will
     * use the background channel scan index (pCsInfo->bkChannelIndex)
     * to scan the next channel. The scan index will wrap around if
     * it reaches the end of the channel table. By always starting with
     * the first channel, we would terminate the background scan
     * when its value reaches (max channel # + 2).
     */
    pCsInfo->bkChannelIndex = 0;
    pCsInfo->bkChannelCnt   = 0;
    status = cservBkScan(pDevInfo, wlanBkSlistSort(pDevInfo),
                         pDevInfo->staConfig.scanType == PASSIVE_SCAN ? PASSIVE_SCAN :
                         (wlanIsActiveScanAllowed(pDevInfo) ? ACTIVE_SCAN : PASSIVE_SCAN));

    if (status != MLME_SUCCESS) {
        return status;
    } else {
        // bk scan successfully started. Let the called wait for completion.
        return MLME_OP_PENDING;
    }
}


/*******************************************************************
 * ccFinder
 *
 * Return code:
 * - CCFINDER_NO_BSS if no BSS is found
 * - CCFINDER_11D_FOUND if valid country code 802.11d is found
 * - CCFINDER_IMPLICIT_RD_FOUND if implicit regulatory domain is found
 * - CCFINDER_11D_NOT_FOUND if no 802.11d is found in BSS
 *
 * If no 802.11d element is found, but the channel identifies an
 * implicit regulatory domain, then the implicit regulatory domain
 * is used.
 * If 802.11d elements are found with different country code, then
 * the majority of the country code element will be used. Else the
 * first country code element is used.
 *
 */
A_INT32
ccFinder(WLAN_DEV_INFO *pDevInfo, BSSDESCR_SET *pbssSet, CC_FINDER *pFind)
{
    COUNTRY_INFO_LIST *pCcInfo;

    CSERV_INFO  *pCsInfo = pDevInfo->pCsInfo;
    int         i, ij, k;
    int         ccount, match, cmatch, tblIdx, tblSize;
    A_INT32     status;
    char        *pStr;
    CTRY_CODE   ccode;
    A_BOOL      adHocMode;
    struct bss_cc {
        int     bssIdx;     /* BSS table index */
        int     match;      /* matching country code count */
    } *pTbl;

    cservPrintf("ccFinder: BSS num: %d, cache cnt: %d\n", pbssSet->count, pCsInfo->scanInfo.bssSet.count);

    pCsInfo->bss11d11aIdx = pCsInfo->bss11d11bIdx = 0;

    status = CCFINDER_11D_NOT_FOUND;
    A_MEM_ZERO (pFind, sizeof(CC_FINDER));

    if (pbssSet == NULL || (pbssSet != NULL && pbssSet->count <= 0)) {
        return CCFINDER_NO_BSS; /* No BSS */
    }

    adHocMode = (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS) ? TRUE : FALSE;
    pFind->numBss = pbssSet->count;

    /*
     * Create a table to hold country code in BSS
     */
    tblSize = sizeof(struct bss_cc) * pbssSet->count;
    if ((pTbl = A_DRIVER_MALLOC(tblSize)) == NULL) {
        /* Default status is set to CCFINDER_11D_NOT_FOUND */
        return status;
    }
    
    A_MEM_ZERO (pTbl, tblSize);

    /*
     * Look for 802.11d country code element.
     * If can't find in table, then store it in a new table entry.
     * If found in table, then increment the match count.
     */
    ccount = 0;

    for (i = 0; i < pbssSet->count; i++) {
        /* Count ad Hoc beacon */
        cservPrintf("BSS[%2d]: ", i);
        cservPrintMacAddress(pbssSet->bssDescrArray[i].bssId);
        cservPrintf("\n");

        if (pbssSet->bssDescrArray[i].bsstype == INDEPENDENT_BSS) {
            if (!pFind->bssAdHoc.count) {
                pFind->bssAdHoc.idx = i;
            }

            pFind->bssAdHoc.count++;
            pFind->adHocFound = TRUE;
        }

        if (IS_CHAN_5GHZ(pbssSet->bssDescrArray[i].pChannel->channelFlags)) {
            pFind->num11a++;
        } else {
            pFind->num11b++;
        }

        pCcInfo = &pbssSet->bssDescrArray[i].ccList;
        ccode = CTRY_INVALID;

        if (pCcInfo->elementID == ELE_COUNTRY_INFO) {
            ccode = wlanGetCountryCodeByName(pCcInfo->countryString, FALSE);
        }

        if (ccode != CTRY_INVALID) {
            /*
             * country code is valid. Check the table for duplcate entry.
             * Increment the match count if the country code is found in
             * the table.
             */
            if (!pFind->bss11d.count) {
                pFind->bss11d.idx = i;
            }

            pFind->bss11d.count++;
            cservPrintf("802.11d BSS[%2d]: ", i);
            cservPrintMacAddress(pbssSet->bssDescrArray[i].bssId);
            cservPrintf(", %s\n",
                     wlanGetCountryCodeName(wlanGetCountryCodeByName(pbssSet->bssDescrArray[i].ccList.countryString, FALSE)));

            match = 0;

            for (ij = 0; ij < ccount; ij++) {
                k = (pTbl + ij)->bssIdx;
                pStr = pbssSet->bssDescrArray[k].ccList.countryString;

                cservPrintf("ccnt: %d, ij: %d, k: %d\n", ccount, ij, k);

                if (ccode == wlanGetCountryCodeByName(pStr, FALSE)) {
                    match++;
                    (pTbl + ij)->match++;
                    break;
                }
            }

            if (!match) {   /* New table entry? */
                cservPrintf("tbl[%d]: %d\n", ccount, i);

                (pTbl + ccount)->bssIdx = i;
                (pTbl + ccount)->match = 1;
                ++ccount;
            }
        } else {
            /*
             * Country code not found or invalid country code
             */
            if (!pFind->bssNo11d.count) {
                pFind->bssNo11d.idx = i;
            }

            pFind->bssNo11d.count++;
        }
    }

    /*
     * If country code element is found, scan table to find the country
     * code with the highest match count and build a channel list for it.
     * Note: No 802.11d in ad Hoc beacon/probe request.
     */
    if (ccount) {
        tblIdx = pTbl->bssIdx;
        if (ccount > 1) {
            cmatch = pTbl->match;
            for (ij = 1; ij < ccount; ij++) {
                if ((pTbl + ij)->match > cmatch) { /* New majority found? */
                    cmatch = (pTbl + ij)->match;
                    tblIdx = (pTbl + ij)->bssIdx;
                }
            }
        }

        pStr = pbssSet->bssDescrArray[tblIdx].ccList.countryString;
        ccode = wlanGetCountryCodeByName(pStr, FALSE);

        pFind->country_code = ccode;

        /* Find the first occurrence of 11a & 11b country code */
        for (i = pbssSet->count-1; i >= 0; i--) {
            pCcInfo = &pbssSet->bssDescrArray[i].ccList;
            if (pCcInfo->elementID == ELE_COUNTRY_INFO &&
                ccode == wlanGetCountryCodeByName(pCcInfo->countryString, FALSE)) {
                if (IS_CHAN_5GHZ(pbssSet->bssDescrArray[i].pChannel->channelFlags)) {
                    pDevInfo->staConfig.sku.v.b.cc11a = TRUE;   /* Mark 11a found */
                    pCsInfo->bss11d11aIdx = (A_UINT16)i;
                    cservPrintf ("ccFinder: 11d/11aIdx: %d\n", i);
                } else {
                    pDevInfo->staConfig.sku.v.b.cc11b = TRUE;   /* Mark 11b found */
                    pCsInfo->bss11d11bIdx = (A_UINT16)i;
                    cservPrintf ("ccFinder: 11d/11bIdx: %d\n", i);
                }
            }
        }

        cservPrintf ("ccFinder: 11d/11aIdx: %d, 11d/11bIdx: %d\n", pCsInfo->bss11d11aIdx, pCsInfo->bss11d11bIdx);
        cservPrintf("ccFinder: 11d found, count: %d\n", pFind->bss11d.count);

        pFind->ccFound = TRUE;
        status = CCFINDER_11D_FOUND;
    } else {
        cservPrintf("ccFinder: No 11d found\n");

        status = CCFINDER_11D_NOT_FOUND;

        if (!adHocMode || (adHocMode && pDevInfo->staConfig.sku.adHocMode == START_ADHOC_PER_11D)) {
            /*
             * Can't find 802.11d country code element, then check if BSS
             * belongs to any implicit regulatory domain.
             */
            for (i = 0; i < pbssSet->count; i++) {
                if (wlanIsImplicitRegDmn(pbssSet->bssDescrArray[i].pChannel, &ccode) == TRUE) {
                    cservPrintf("ccFinder: implicit regulatory domain found, country code: %d, %s\n",
                             ccode, wlanGetCountryName(pDevInfo, ccode));
                    pFind->country_code = ccode;
                    pFind->implicitRd = TRUE;
                    status = CCFINDER_IMPLICIT_RD_FOUND;
                    break;
                }
            }
        }
    }

    A_DRIVER_FREE(pTbl, tblSize);

    return status;
}


/*******************************************************************
 * cserv11dHandler
 * Routine to search the BSS set for country code
 *
 * Return code:
 * - CSERV_11D_NO_BSS if no BSS is found
 * - CSERV_11D_COUNTRY_FOUND if country code is found
 * - CSERV_11D_NOT_FOUND if no country code is found
 * - CSERV_11D_START_ADHOC to start ad-hoc network
 * - CSERV_11D_JOIN_ADHOC to join existing ad-hoc network
 * - CSERV_11D_JOINT_NETWORK to join existing network
 *
 */
A_INT32
cserv11dHandler(WLAN_DEV_INFO *pDevInfo, BSSDESCR_SET *pbssSet)
{
    CSERV_INFO  *pCsInfo = pDevInfo->pCsInfo;
    CC_FINDER   findPkt;
    A_UINT16    ccode;
    A_INT32     status = 0;
    A_BOOL      adHocMode;

    ASSERT(pCsInfo);

    cservPrintf("cserv11dHandler: Entry\n");

    adHocMode = (pDevInfo->bssDescr->bsstype == INDEPENDENT_BSS) ? TRUE : FALSE;

    pDevInfo->staConfig.sku.v.w = 0;

    if (wlanIsWwrSKU(pDevInfo) == FALSE || pDevInfo->staConfig.ignore11dBeacon) {
        return CSERV_11D_SELECT_BSS;
    }

    /*
     * Call ccFinder to look for any 802.11d country code element
     * If found, then parse the BSS sets and return the result.
     */
    status = ccFinder(pDevInfo, pbssSet, &findPkt);

    switch (status) {
        case CCFINDER_NO_BSS:
            return CSERV_11D_NO_BSS_FOUND;

        case CCFINDER_IMPLICIT_RD_FOUND:
        case CCFINDER_11D_FOUND:
            /* 802.11d found */
            ccode = findPkt.country_code;
            pDevInfo->staConfig.sku.v.b.commonMode = FALSE;   /* Don't use common mode */

            cservPrintf("cserv11dHandler: ccode %d(old) %d(new) \"%s\", scale: %d\n",
                     pDevInfo->staConfig.countryCode,
                     ccode, wlanGetCountryName(pDevInfo, ccode),
                     RETRY_SCALE_GET(pCsInfo));

            if (pDevInfo->staConfig.countryCode != ccode) {
                pDevInfo->staConfig.countryCode        = ccode;
                pDevInfo->defaultStaConfig.countryCode = ccode;

                cservPrintf("cserv11dHandler: country code %d\n", pDevInfo->staConfig.countryCode);

                switch (wlanInitCcChannelList(pDevInfo)) {
                    case WLAN_INIT_CC_CLIST_ERR:
                        return CSERV_11D_NOT_FOUND;      /* No country code found */

                    case WLAN_INIT_CC_CLIST_NO_CHG:     /* No channel list change */
                        return CSERV_11D_SELECT_BSS;     /* Go join network */

                    case WLAN_INIT_CC_CLIST_NEW:        /* Channel list has been changed */
                        pDevInfo->staConfig.sku.v.b.cc11d = TRUE;   /* Mark 11d found */
                        return CSERV_11D_NEW_COUNTRY_FOUND;      /* New country code found */
                }
            }

            if (status == CCFINDER_11D_FOUND) {
                cservPrintf("cserv11dHandler: mark 11d found\n");
                pDevInfo->staConfig.sku.v.b.cc11d = TRUE;   /* Mark 11d found */
            }

            /*
             * Country code found. Clear passive flag if not ETSI
             */
            wlanClistClnup(pDevInfo);
            return CSERV_11D_COUNTRY_FOUND;

        case CCFINDER_11D_NOT_FOUND:
            /*
             * Can't find any 802.11d.
             * Can't determine regulatory domain, use common mode.
             */
            pDevInfo->staConfig.sku.v.b.commonMode = TRUE;      /* Use common mode */
            pDevInfo->staConfig.sku.v.b.cc11d = FALSE;          /* No 11d found */
            cservPrintf("cserv11dHandler: %s, scale: %d\n", adHocMode ? "adHoc" : "AP",
                      RETRY_SCALE_GET(pCsInfo));

            if (adHocMode) {
                if (findPkt.bssAdHoc.count) {
                    cservPrintf("cserv11dHandler: adHoc beacon found\n");
                    return CSERV_11D_JOIN_ADHOC;
                }

                if (RETRY_SCALE_GET(pCsInfo) >= CSERV_MAX_SCALE_ADHOC) {
                    cservPrintf("cserv11dHandler: Timeout! Start adHoc\n");
                    pDevInfo->staConfig.sku.v.b.startAdhoc = TRUE;
                    return CSERV_11D_START_ADHOC;
                }

                return CSERV_11D_NOT_FOUND;
            } else {
                return CSERV_11D_SELECT_BSS;
            }

        default:
            if (!adHocMode) {
                return CSERV_11D_MAX_SCAN;
            }

            return CSERV_11D_NO_BSS_FOUND;
    }

    return CSERV_11D_NO_BSS_FOUND;
}

/*******************************************************************
 * cservSelectBss
 * Routine to select a BSS that matches criteria.
 * Specific or null (any) ssid
 * Specific or null (any) bssid
 * bssType specific or any
 * minimum rssi that the BSS must have.
 */
A_INT32
cservSelectBss(WLAN_DEV_INFO *pDevInfo, BSSDESCR_SET *pbssSet, BSSTYPE bssType,
               SSID_ARRAY *pSsids, BSSID *pBssid, A_UINT32 minRssi)
{
    int             i;
    int             choice   = -1;
    WLAN_STA_CONFIG *pConfig = &pDevInfo->staConfig;

    /*
     * We should be given a valid BSS set, and either pSsids or
     * pBssid should be a valid pointer.
     */
    ASSERT(pbssSet);
    ASSERT(pSsids || pBssid);

    if (pbssSet == NULL || pbssSet->count <= 0 || !(pBssid || pSsids)) {
        return -1;
    }
    

    if (pSsids && A_ATOMIC_READ(&pSsids->numValid) == 0 && !pBssid) {
        /*
         * There are no SSIDs to check, and a BSSID has not
         * been given.  Nothing to do, so return.
         */
        return -1;
    }

    /* Give CCX code a chance to process the BSS cache for preferred APs */
    ccxProcessPreferredAps(pDevInfo, pbssSet);

    for (i = 0; i < pbssSet->count; i++) {
        A_INT32             searchRssi, rssiToBeat; // needs to be signed
        BSSDESCR            *pCurrBss;
        BSSDESCR            *pBss  = &pbssSet->bssDescrArray[i];
        WLAN_CFLAGS         iflags = pBss->pChannel->channelFlags & CHANNEL_ALL;
        WIRELESS_MODE       cMode;
        CSERV_MARGIN const  *pMargin;
        WLAN_RATE_SET       testRateSet;
        int                 j;

        pBss->rank = IS_CHAN_2GHZ(iflags) ? 0 : 10;

        /* Check the criteria one by one */
        if (pBssid) {
            /* We are looking for a specific BSSID. It must be unique. */
            if (A_MACADDR_COMP(&pBss->bssId, pBssid) == 0) {
                if (bssType != INDEPENDENT_BSS && (pBss->apState & BAD_AP)) {
                    /* found it but AP crashed or not responding - try later */
                    pBss->apState = 0;
                    printMacAddress(pBss->bssId);
                    uiPrintf(" ");
                    printSsid(&pBss->ssid);
                    uiPrintf(": Bad AP found!\n");
                    return -1;
                }
                return i;
            }
            continue;
        }
        pBss->rank++;

        /* If AP is not in proper state, look for another */
        if (pDevInfo->bssDescr->bsstype != INDEPENDENT_BSS &&
            (pBss->apState & BAD_AP))
        {
            printMacAddress(pBss->bssId);
            uiPrintf(" ");
            printSsid(&pBss->ssid);
            uiPrintf(" is a bad AP\n");
            continue;
        }
        pBss->rank++;

        ASSERT(A_ATOMIC_READ(&pSsids->count) >= A_ATOMIC_READ(&pSsids->numValid));
        for (j = 0; j < A_ATOMIC_READ(&pSsids->numValid); j++) {
            SSID *pThisSsid = &pSsids->elements[j];

            if (isNullSsid(pThisSsid) || isSsidEqual(&pBss->ssid, pThisSsid)) {
                break;
            }

            cservPrintMacAddress(pBss->bssId);
            cservPrintf(" BSS[%d]: \"", i);
            cservPrintSsid(&pBss->ssid);
            cservPrintf("\" != \"");
            cservPrintSsid(pThisSsid);
            cservPrintf("\"\n");
        }

        if (j == A_ATOMIC_READ(&pSsids->numValid)) {
            continue;
        }
        pBss->rank++;

        if (bssType != ANY_BSS && pBss->bsstype != bssType) {
            // not the network type we are looking for
            printMacAddress(pBss->bssId);
            uiPrintf(" ");
            printSsid(&pBss->ssid);
            uiPrintf(" BSS[%d]: wrong BSS type (BSS is type %d,"
                     "looking for type %d)\n", i, pBss->bsstype, bssType);
            continue;
        }
        pBss->rank++;

        if (!pConfig->privacyInvoked && pBss->capabilityInfo.privacy) {
            // station did not enable encryption but AP requires it. .
            // Don't chose this one. WECA violation.
            printMacAddress(pBss->bssId);
            uiPrintf(" ");
            printSsid(&pBss->ssid);
            uiPrintf(" BSS[%d]: BSS requires privacy\n", i);
            continue;
        }
        pBss->rank++;

        if (pConfig->privacyInvoked && !pBss->capabilityInfo.privacy) {
            uiPrintf("BSS[%d]:  BSS is clear; we want privacy", i);
            if (ccxMixedPrivacyPolicyCheck(pDevInfo) == A_OK) {
                uiPrintf("...might join anyway\n");
            } else {
                uiPrintf("...do not join\n");
                continue;
            }
        }
        pBss->rank++;

        /* Don't accept non-WPA AP if we're told to find one */
        if (pDevInfo->staConfig.wpaEnabled &&
            !VALID_WPA_ELEMENT(&pBss->wpaIe))
        {
            printMacAddress(pBss->bssId);
            uiPrintf(" ");
            printSsid(&pBss->ssid);
            uiPrintf(" BSS[%d]: BSS doesn't support WPA\n", i);
            continue;
        }
        pBss->rank++;

#ifdef XR_HACKERY
        if (IS_CHAN_XR(pBss->pChannel->channelFlags) && (pBss->opMode != WIRELESS_MODE_XR)) {
            printMacAddress(pBss->bssId);
            uiPrintf(" ");
            printSsid(&pBss->ssid);
            uiPrintf(" BSS[%d]: BSS doesn't support XR\n", i);
            continue;
        }
        if ((!IS_CHAN_XR(pBss->pChannel->channelFlags)) && (pBss->opMode == WIRELESS_MODE_XR)) {
            printMacAddress(pBss->bssId);
            uiPrintf(" ");
            printSsid(&pBss->ssid);
            uiPrintf(" BSS[%d]: BSS is XR\n", i);
            continue;
        }
#endif

        /* If STA does not support all BSS basic rates, do not attempt to join. */
#if NDIS_WDM
        /*     
        * (LW) Attention !!!
        * This is temp hack, should revisit later
        */
        wlanRateTbl2RateSet(pDevInfo->hwRateTable[pBss->opMode], &testRateSet);
#else
        wlanRateTbl2RateSet(HACK_TO_PARDEV(pDevInfo)->hwRateTable[pBss->opMode], &testRateSet);
#endif
        if (!basicRateSubsetCheck(&pBss->supportedRateSet, &testRateSet)) {
            printMacAddress(pBss->bssId);
            uiPrintf(" ");
            printSsid(&pBss->ssid);
            uiPrintf(" BSS[%d]: STA does not support all Basic Rates of BSS\n", i);
            continue;
        }

        /*
         * Verify whether it is ok to join existing adhoc network
         * in this wireless mode
         */
        if (pBss->bsstype == INDEPENDENT_BSS && IS_CHAN_5GHZ(iflags) &&
            wlanIsWwrSKU(pDevInfo) &&
            iflags == wlanUnsupportedAdHocBand(pDevInfo))
        {
            printMacAddress(pBss->bssId);
            uiPrintf(" ");
            printSsid(&pBss->ssid);
            uiPrintf(" BSS[%d]: Can't join IBSS in this wireless mode\n", i);            
            continue;
        }

        /* Now we have an acceptable BSS. Check signal strength. */
        pBss->rank += pBss->rssi;

        /*
         * Disclaimer: Following code is written for simplicity,
         *             not optimization or performance.
         */

        if (choice == -1) {
            /* No BSS chosen yet */
            if (minRssi == 0) {
                /* Caller hasn't given RSSI to beat */
                choice  = i;
                minRssi = pBss->rssi;
                cservPrintMacAddress(pBss->bssId);
                cservPrintf(" ");
                cservPrintSsid(&pBss->ssid);
                cservPrintf(" BSS[%d]: CHOSEN: RSSI %d is new RSSI to beat\n",
                            i, minRssi);
                continue;
            }
        }

        pBss->rank += 256;

        /*
         * Determine wireless mode for current choice (cMode).
         */

        /* cMode is dependent on the current BSSDescr */
        pCurrBss = NULL;
        if (choice == -1) {
            if (IS_CONN_OR_JOINED(pDevInfo)) {
                pCurrBss = pDevInfo->bssDescr;
            }
        } else {
            pCurrBss = &pbssSet->bssDescrArray[choice];
        }

        /* If no current AP, use pBss->opMode. This will prevent the use of margins */
        cMode = pCurrBss ? pCurrBss->opMode : pBss->opMode;

        /* Lookup appropriate margin table entry for this rssi */
        searchRssi = A_MAX(minRssi, pBss->rssi);

        pMargin = &cservMarginTable[0];
        for (j = 0; j < NUM_MARGIN_TABLE; j++) {
            if ((searchRssi >= cservMarginTable[j].startRssi) &&
                (searchRssi <= cservMarginTable[j].endRssi)) 
            {
                pMargin = &cservMarginTable[j];
                break;
            }
        }
        ASSERT(j < NUM_MARGIN_TABLE);

        /* Update rank of this BSS */
        for (j = 0; j < NUM_MARGIN_TABLE; j++) {
            pBss->rank += pMargin->margin[j];
        }

        rssiToBeat = minRssi;

        /* Apply margins to rssiToBeat */
        if (cMode != pBss->opMode) {
            if (pMargin->pref[cMode] > pMargin->pref[pBss->opMode]) {
                /* Current mode is prefered => add margin of new mode */
                rssiToBeat += pMargin->margin[pBss->opMode];
            } else if (pMargin->pref[cMode] < pMargin->pref[pBss->opMode]) {
                /* New mode is prefered => subtract margin of current mode */
                rssiToBeat -= pMargin->margin[cMode];
            } else {
                /* modes have same preference => no margin applied */
            }
        }

        if ((A_INT32)(pBss->rssi + pBss->rssiAdder) > rssiToBeat) {
            choice  = i;
            minRssi = pBss->rssi;
            cservPrintMacAddress(pBss->bssId);
            cservPrintf(" ");
            cservPrintSsid(&pBss->ssid);
            cservPrintf(" BSS[%d]: CHOSEN: RSSI %d (%d adder) "
                     "is new RSSI to beat\n", i, minRssi, pBss->rssiAdder);
        } else {
            cservPrintMacAddress(pBss->bssId);
            cservPrintf(" ");
            cservPrintSsid(&pBss->ssid);
            cservPrintf(" BSS[%d]: RSSI %d <= %d (adjusted, if necc., for mode"
                        " diffs)\n", i, pBss->rssi + pBss->rssiAdder,
                        rssiToBeat);
        }
    } /* for() loop through BSS list */

    if (choice >= 0) {
        BSSDESCR *pBss = &pbssSet->bssDescrArray[choice];
        
        pBss->rank += 256;
        /* Mark it as good AP */
        pBss->apState = 0; 
        uiPrintf("cservSelectBss: Selected \"");
        printSsid(&pBss->ssid);
        uiPrintf("\" (");
        printMacAddress(pBss->bssId);
        uiPrintf(")\n");
    } else {
        //uiPrintf("cservSelectBSS: No matching BSS\n");
        /*
         * Now that we have tried all AP's by skipping over bad
         * ones, give them a fair chance for the next round...unless
         * we have some reason to delay doing so.
         */
        for (i = 0; i < pbssSet->count; i++) {
            if (pbssSet->bssDescrArray[i].badAPTimeout != 0) {
                if (A_MS_TICKGET() < pbssSet->bssDescrArray[i].badAPTimeout) {
                    ASSERT(pbssSet->bssDescrArray[i].apState & BAD_AP);
                    continue;
                } else {
                    pbssSet->bssDescrArray[i].badAPTimeout = 0;
                }
            }
            pbssSet->bssDescrArray[i].apState = 0;
        }
    }

    return choice;
}

/*******************************************************************
 * cservPollFunction
 * Engine that drives connection services. Called by staPollFunction
 * at a periodic interval.
 */
void
cservPollFunction(WLAN_DEV_INFO *pDevInfo)
{
    CSERV_INFO *pCsInfo          = pDevInfo->pCsInfo;
    BSSID      *pBssid           = NULL;
    A_BOOL     isTimeToScanAgain = FALSE;
    A_UINT32   rssi;
    WIRELESS_MODE   mode;

    ASSERT(pCsInfo && pDevInfo->localSta);

    // cservPrintf("cservPollFunction: cservOp=0x%x, cservState=0x%x\n",
    //          pCsInfo->cservOp, pCsInfo->cservState);

    rssi = A_RSSI_OUT(pDevInfo->localSta->stats.rssi);
    
    if (pCsInfo->disconnectCur > 0 &&
        --pCsInfo->disconnectCur == 0)
    {
        // waited long enough to see if we have reassociated.
        osIndicateConnectStatus(pDevInfo, FALSE);
    }

    // If we've previously held off background scanning to avoid negatively
    // impacting throughput, attempt to do the scan now
    if (pCsInfo->scanPending) {
        cservPrintf("cservPollFunction: Background scan pending. Trying now....\n");
        cservBkScanAll(pDevInfo, SCAN_WHEN_IDLE);
    }

    if (pDevInfo->flushInProgress == TRUE) {
        return;
    }

    // do we need to move bk scan to the next channel?
    if (pCsInfo->cservState & CSERV_STATE_BKSCANNING) {
        /* Do not scan while in turbo phase of turbo prime */
        mode = wlanCFlagsToWirelessMode(pDevInfo, pDevInfo->staConfig.pChannel->channelFlags);
        if ((--pCsInfo->bkScanCur <= 0) && 
            !(pDevInfo->turboPrimeInfo.turboPrimeAllowed && ((mode == WIRELESS_MODE_108g) || (mode == WIRELESS_MODE_TURBO))))
        {
            WLAN_CHANNEL_LIST *pClist = pDevInfo->staConfig.pClist;
            CHAN_VALUES       *pCv    = NULL;

            // time to advance to next channel in bk scan list
            pCsInfo->bkChannelIndex = (pCsInfo->bkChannelIndex + 1) % pClist->listSize;

            if (++pCsInfo->bkChannelCnt <= pClist->listSize) {
                if (pCsInfo->bkScanList) {
                    pCv = pCsInfo->bkScanList->pChanArray[pCsInfo->bkChannelIndex];
                } else {
                    pCv = wlanGetBksChannel(pClist, pCsInfo->bkChannelIndex);
                }
            }

            if (pCv) {
                SCANTYPE type = pDevInfo->staConfig.scanType == PASSIVE_SCAN ?
                                PASSIVE_SCAN : (wlanIsActiveScanAllowed(pDevInfo) ?
                                               ACTIVE_SCAN : PASSIVE_SCAN);

                cservBkScan(pDevInfo, pCv, type);
            } else {
                /* Done with background scanning */
                pCsInfo->bkChannelIndex = 0;
                pCsInfo->cservState     = CSERV_STATE_IDLE;

                /*
                 * After we've received a command to background scan, see if we've
                 * found something better that we should switch to.
                 */
                cservChooseApAndSwitch(pDevInfo, rssi);
            }

            pCsInfo->bkScanCur = pCsInfo->bkScanInterval;
        }

        /*
         * If we're background scanning (or just completed one), no need to
         * bother with any of the other stuff below.
         */
        pCsInfo->cservPollCur = pCsInfo->cservPollInterval;
        return;
    }

    // Keep track of how long we continuously scan before attempting to save power.
    if (pCsInfo->scanBeforeSleepCur > 0) {
        pCsInfo->scanBeforeSleepCur--;
    }

    // Keep track of how long we sleep before starting the next scan.
    if (pCsInfo->sleepBetweenScanCur > 0) {
        pCsInfo->sleepBetweenScanCur--;
        if (pCsInfo->sleepBetweenScanCur == 0) {
            isTimeToScanAgain = TRUE;
        }
    }

    /*
     * If rate control is enabled, we check periodically to see if we should
     * roam from our current connection to one that might be better.  This only
     * applies when we're in infrastructure mode.
     */
    if (pDevInfo->staConfig.rateCtrlEnable &&
        --pCsInfo->rateCheckCur <= 0 &&
        pDevInfo->bssDescr->bsstype == INFRASTRUCTURE_BSS &&
        IS_STATE_CONNECTED(pDevInfo))
    {
        A_UINT32 rate = A_RATE_OUT(pDevInfo->localSta->stats.txRateKb);
        
        cservTxRateNotify(pDevInfo, rate, rssi);

        pCsInfo->rateCheckCur = pCsInfo->rateCheckInterval;
    }

    // Poll the connection status of station
    if (--pCsInfo->cservPollCur > 0 && !isTimeToScanAgain) {
        return;
    }

    /**
     ** Everything below is done at cservPollInterval intervals
     ** or when just waking up from sleeping in between scans.
     **/

    if (IS_CONN_OR_JOINED(pDevInfo)) {
        BSSDESCR *pBss;

        pBssid = &pDevInfo->bssDescr->bssId;

        /* Update the RSSI of our BSS in the BSS list */
        pBss = mlmeBsssetFind(&pCsInfo->scanInfo.bssSet, pBssid,
                              pDevInfo->staConfig.pChannel->channelFlags);

        if (pBss != NULL) {
            pBss->rssi = A_RSSI_OUT(pDevInfo->localSta->stats.rssi);
        }
    }

    /*
     * Age out old BSS entries in our list.  Keep our own BSSID in the list
     * if we're joined/associated.
     */
    mlmeBsssetAge(&pCsInfo->scanInfo.bssSet,
                  pDevInfo->staConfig.bssAgingPeriod, pBssid);

    ASSERT(pDevInfo->localSta->serviceType == WLAN_STA_SERVICE);
    
    // Attempt to connect.
    //
    // Don't attempt if:
    // 1. Background scanning, or
    // 2. Sleeping waiting for the next scan cycle to start
    //    
    if (!IS_CONN_OR_JOINED(pDevInfo) &&
        !(pCsInfo->cservState & CSERV_STATE_BKSCANNING) &&
        pCsInfo->sleepBetweenScanCur <= 0)
    {
        A_UINT16 state            = pDevInfo->localSta->staState;
        A_BOOL   isTimeToSleep    = (pCsInfo->scanBeforeSleepCur <= 0 &&
                                     !pCsInfo->resetScanBeforeSleep   &&
                                     pCsInfo->isScanCycleComplete     &&
                                     !isTimeToScanAgain               &&
                                     pCsInfo->cservOp == 0);

        cservPrintf("cservPollFunction: autoconnect\n");       
        if (pCsInfo->apConnAttemptCur >= pCsInfo->apConnAttemptsMax || isTimeToSleep) {
            // We have either tried auth/assoc too many times (bad AP?)
            // or we've scanned for a long time and found nothing. Now we clear
            // some state so that driver can start from scan-join again.

            if (pCsInfo->cservForcedSleep) {
                // Return from deep sleep
                powerSet(pDevInfo, FORCE_SLEEP, DISABLE, FALSE, 0);
                pCsInfo->cservForcedSleep = FALSE;
            }

            pCsInfo->apConnAttemptCur = 0;
            uiPrintf("cservPollFunction: too many connection attempts or "
                     "it's time to sleep.\n");
            cservOperationCancel(pDevInfo);
            RETRY_SCALE_RESET(pCsInfo);

            if (state & STATE_AUTH) {
                cservDeauthenticate(pDevInfo); // in case AP is not responding to assoc requests
            }

            pDevInfo->localSta->staState &= ~STATE_CONNECTED;
            state = STATE_QUIET;

            if (isTimeToSleep) {
                // If we've continuously scanned for long enough and the chip is
                // asleep, i.e., we're not plugged in and power save is enabled,
                // sleep for a while before starting the next scan cycle.
                powerSet(pDevInfo, FORCE_SLEEP, ENABLE, FALSE, 0);
                pCsInfo->cservForcedSleep = TRUE;
                if (!powerEnabled(pDevInfo)) {
                    pCsInfo->sleepBetweenScanCur = pCsInfo->sleepBetweenScanInterval;
                }
            }
        }

        // Don't start any new connections if we just decided to go to sleep
        if (pCsInfo->sleepBetweenScanCur <= 0) {
            if (state == (STATE_JOINED | STATE_AUTH)) {
                A_UINT32 listenInt = pDevInfo->staConfig.listenInterval;
                A_UINT32 beaconInt = pDevInfo->staConfig.beaconInterval;

                // join and auth is done, but no association. Try again.
                // express listen interval in beacon interval units, rounded up.
                cservAssociate(pDevInfo, RETRY_MUL(pCsInfo, CSERV_TIME_AA_MAX),
                               (listenInt + beaconInt - 1) / beaconInt);
            } else if (state == STATE_JOINED) {
                // joined but no auth or assoc
                if (pDevInfo->bssDescr->bsstype == INFRASTRUCTURE_BSS) {
                    // Not required for ad-hoc
                    BSSDESCR *pBssdescr = mlmeBsssetFind(&pCsInfo->scanInfo.bssSet, 
                                                         &pDevInfo->bssDescr->bssId,
                                                         pDevInfo->staConfig.pChannel->channelFlags);

                    if (pBssdescr != NULL && (pBssdescr->apState & BAD_AP)) {
                        // We've tried too many times to associate with this
                        // one.  Time to move on.  cservChooseApAndSwitch will
                        // ignore the bad AP.
                        // Note: Not from WZC (last param), but want to maintain
                        // previous logic. (MDS)
                        cservSwitchSsid(pDevInfo, &pCsInfo->desSsids, TRUE);
                    } else {
                        // Try again
                        cservAuthAssoc(pDevInfo);
                    }
                }
            } else if (state == (STATE_AUTH | STATE_ASSOC) || state == STATE_AUTH) {
                // strange case of auth and assoc but not joined, happens after hw reset
                cservPrintf("cservJoin -- called from cservPollFunction\n");
                cservJoin(pDevInfo);
            } else {
                // Start the scan process
                cservScanJoinAuthAssoc(pDevInfo, pDevInfo->staConfig.scanType, FALSE);
            }
        }
    }

    pCsInfo->cservPollCur = pCsInfo->cservPollInterval;
}

/*******************************************************************
 * cservTxRateNotify
 * Called by transmit rate adaptation routine to tell that we are
 * lowering the rate or it is already low.
 */
void
cservTxRateNotify(WLAN_DEV_INFO *pDev, A_UINT32 rate, A_UINT32 rssi)
{
    CSERV_INFO      *pCsInfo = pDev->pCsInfo;
    WLAN_STA_CONFIG *pConfig = &pDev->staConfig;
    A_UINT32        now      = A_MS_TICKGET();
    A_UINT32        lastScan = pCsInfo->scanInfo.scanEndTime;
    WLAN_CFLAGS     cf       = pConfig->pChannel->channelFlags;
    A_UINT32        roamRate, roamRssi;

    if (pConfig->NetBand == MODE_SELECT_11B) {
        /* We're in 11b only mode, so we use a different set of thresholds */
        roamRate = pConfig->roamRateBOnly;
        roamRssi = pConfig->roamRssiBOnly;
    } else {
        if (IS_CHAN_CCK(cf) || (pDev->bssDescr->opMode == WIRELESS_MODE_11b)) {
            /* AP is 11b, but channel flag is configured for 11g */
            roamRate = pConfig->roamRateB;
            roamRssi = pConfig->roamRssiB;
        } else {
            roamRate = pConfig->roamRateA;
            roamRssi = pConfig->roamRssiA;
        }
    }

    if (rate >= roamRate && rssi >= roamRssi) {
        return;
    }

    cservPrintf("cservTxRateNotify: rate %d < %d, or rssi %d < %d: "
             "looking for a better BSS...\n", rate, roamRate, rssi, roamRssi);

    // if rssi is low or dropping the rate below 18 mbps level, do a scan
    // to find a better ap. If a recent scan list is available, then use it.
    if (pConfig->bkScanEnable &&
        (now - lastScan) >= CSERV_VALID_SCAN_TIME)
    {
        if (pDev->turboPrimeInfo.friendlyTurboAllowed && 
            ((now - pDev->turboPrimeInfo.timer) > 0) && 
            ((now - pDev->turboPrimeInfo.timer) < 5000)){
          return;
        } else {
          // last bk scan expired. start again
          cservPrintf("cservTxRateNotify: kicking off a new background scan\n");
          cservBkScanAll(pDev, SCAN_NOW);
          return;
        }
    }

    // we do have a recent scanned list of AP's.
    cservChooseApAndSwitch(pDev, rssi);
}

/*******************************************************************
 * cservChooseApAndSwitch
 *
 * Selects an AP from the scan list.  If we find a winner, we
 * switch to that AP.
 *
 * Returns: TRUE if we switched APs, FALSE if not
 */
A_BOOL
cservChooseApAndSwitch(WLAN_DEV_INFO *pDev, A_UINT32 rssiToBeat)
{
    CSERV_INFO      *pCsInfo = pDev->pCsInfo;
    WLAN_STA_CONFIG *pConfig = &pDev->staConfig;
    A_UINT32        hysAdjust = 0;
    A_INT32         i;

    cservPrintf("cservChooseApAndSwitch -- rssiToBeat: %d\n", rssiToBeat);

    /*
     * Ensure that there is some hysteresis when switching between
     * APs so that we're not constantly switching back and forth.  
     */
    if (rssiToBeat > 0) {
        for (i = 0; i < HYS_TABLE_SIZE; i++) {
            if (rssiToBeat >= cservHysTable[i].thresh) {
               hysAdjust = cservHysTable[i].hysAdjust;
               break;
            }
        }
        ASSERT(i < HYS_TABLE_SIZE);
        rssiToBeat += hysAdjust;
    }

    i = cservSelectBss(pDev, &pCsInfo->scanInfo.bssSet, pDev->bssDescr->bsstype,
                       &pConfig->cfgSsids, NULL, rssiToBeat);

    if (i < 0) {
        /* Didn't find a winner */
        return FALSE;
    }

    cservSwitchAP(pDev, &pCsInfo->scanInfo.bssSet.bssDescrArray[i]);
    pCsInfo->scanInfo.bssSet.setIdx = (A_CHAR)i;

    return TRUE;
}

/*******************************************************************
 * cservSelectAdhocChannel
 * Finds a channel where there is no traffic.
 */
void
cservSelectAdhocChannel(WLAN_DEV_INFO *pDevInfo, CSERV_INFO *pCsInfo)
{
    A_UINT32 j, bssSetCount;
    SSID     *pSsid = &pCsInfo->desSsids.elements[0];

    // Delete all instances of our own SSID in the BSS set.
    while (1) {
        bssSetCount = pCsInfo->scanInfo.bssSet.count;

        for (j = 0; j < bssSetCount; j++) {
            BSSDESCR *pBss = &pCsInfo->scanInfo.bssSet.bssDescrArray[j];
            if (isSsidEqual(pSsid, &pBss->ssid) &&
                pBss->bsstype == INDEPENDENT_BSS)
            {
                // Break out of the loop and restart because mlmeBsssetDel
                // modifies the bssSet.
                mlmeBsssetDel(&pCsInfo->scanInfo.bssSet, &pBss->bssId);
                break;
            }
        }

        if (j == bssSetCount) {
            // didn't find any of our BSSes to delete, exit.
            break;
        }
    }
}

/*******************************************************************
 * cservCreateAdhocBssid
 * Create a random bssid
 */
void
cservCreateAdhocBssid(WLAN_DEV_INFO *pDevInfo, WLAN_MACADDR *pMacaddr)
{
    int i;
    union {
        WLAN_MACADDR tmac;
        A_UINT32     tstamp;
    } randVal;

    randVal.tstamp = A_MS_TICKGET(); // timestamp in ms should be random enough

    // lower 4 bytes filled with timestamp (kind of random),
    // upper 2 bytes uninitialized trash from stack (kind of random)
    // XOR with permanent MAC address (unique if set by manufacturing or 0)
    // hopefully this will reduce the chance of 2 stations generating same bssid

    for (i = 0; i < 6; i++) {
        pMacaddr->octets[i] = randVal.tmac.octets[i] ^
                              pDevInfo->staConfig.macPermAddr.octets[i];
    }

    // If the random value doesn't change the BSS - then XOR the word portion to make
    // sure the new BSS is different from our MAC.
    if (A_MACADDR_COMP(pMacaddr, &(pDevInfo->staConfig.macAddr)) == 0) {
        pMacaddr->st.word = pMacaddr->st.word ^ 0xFFFFFFFF;
    }

    // now set the important bits 802.11 spec sections 7.1.3.3.3, 11.1.3
    // LSB 0 = group/individual address bit, 0 means individual address
    // LSB 1 = local/universal address bit, 1 means it is locally administrated
    pMacaddr->octets[0] = (pMacaddr->octets[0] & 0xfc) | 0x02;
}

/*******************************************************************
 * cservAdHocTrafficReceived
 *
 * IBSS traffic (a beacon or other traffic from another STA) was
 * received, so this function is called to see if we need to
 * indicate to the OS that we're connected, and to reset
 * our disconnect countdown timer.
 */
#ifdef ADHOC_2STA_B4_CONNECTED
void
cservAdHocTrafficReceived(WLAN_DEV_INFO *pDev)
{
    A_BOOL isConnected = GET_CONNECTION_STATUS(pDev);

    ASSERT(pDev->bssDescr->bsstype == INDEPENDENT_BSS);

    if (!isConnected) {
        /*
         * We were disconnected but just received a Beacon with our
         * BSSID. That means our IBSS network is back in action.
         * Indicate to the OS that we're connected.
         */
        osIndicateConnectStatus(pDev, TRUE);
        pDev->pCsInfo->apConnAttemptCur = 0;
    } else {
        cservStartDisconnectTimer(pDev);
    }
}
#endif

/*******************************************************************
 * cservStartDisconnectTimer
 *
 * Starts/Resets a timer which, upon expiration, will indicate to
 * the OS that we are no longer connected.
 */
void
cservStartDisconnectTimer(WLAN_DEV_INFO *pDev)
{
    pDev->pCsInfo->disconnectCur = pDev->pCsInfo->disconnectInterval;
}

/*******************************************************************
 * cservCancelDisconnectTimer
 *
 * Cancels a delayed indication to the OS that we are no longer
 * connected. This may happen when we reset the hw and join again quickly.
 */void
cservClearDisconnectTimer(WLAN_DEV_INFO *pDev)

{
    pDev->pCsInfo->disconnectCur = 0;
}

/*******************************************************************
 * cservUpdateLastActivityTime
 *
 * Updates the last time we've sent/received traffic.  Used to stave
 * off background scan during high traffic times
 */
void
cservUpdateLastActivityTime(WLAN_DEV_INFO *pDev)
{
    pDev->pCsInfo->lastActivityTime = A_MS_TICKGET();
}

/*******************************************************************
 * cservConnectionCompleteNotify
 *
 * Here is where we put stuff that needs to be done once we're finally
 * "connected" (i.e., associated in infrastructure mode or joined in
 * ad hoc mode).
 */
void
cservConnectionCompleteNotify(WLAN_DEV_INFO *pDev)
{
    CSERV_INFO *pCsInfo = pDev->pCsInfo;

    /*
     * Make sure that our next foreground scan is using the first
     * duration multiple and that we revert back to the mode where
     * we scan for a while before going in and out of sleep.
     */
    RETRY_SCALE_RESET(pCsInfo);
    pCsInfo->resetScanBeforeSleep = TRUE;

    /*
     * Need to clear any cached SSIDs from the OS so we don't
     * "roam" to any other SSIDs.
     */
    ssidArrayInit(&pDev->pOSHandle->ssids);

    /*
     * If we've previously told the OS that we're out of resources (to hold
     * off packets to send), let it know that it can now send us the queued
     * frames.
     */
    osIndicateResourcesAvailable(pDev);
    
    /*
     * Update our RSSI stat so that if the OS asks for it right away,
     * we can return a reasonably-valid estimate.
     */
    pDev->localSta->stats.rssi = A_RSSI_LPF(pDev->localSta->stats.rssi,
                                            pDev->bssDescr->rssi);
}

A_UINT32
cservAdHocBand2Wmode (A_UINT16 adhocBand)
{
    A_UINT32 wMode;

    ASSERT(adhocBand < ATH_ADHOCBAND_END);
 
    /* Determine wireless mode to start adhoc */
    switch (adhocBand) {
        case ATH_ADHOCBAND_BONLY:
            wMode = MODE_SELECT_11B;
            break;

        case ATH_ADHOCBAND_AONLY:
            wMode = MODE_SELECT_11A;
            break;

        case ATH_ADHOCBAND_TONLY:
            wMode = MODE_SELECT_TURBO;
            break;

        case ATH_ADHOCBAND_GONLY:
            wMode = MODE_SELECT_11G;
            break;

        case ATH_ADHOCBAND_108GONLY:
            wMode = MODE_SELECT_108G;
            break;

        case ATH_ADHOCBAND_NONE:
            wMode = 0;
            break;

        default:
            /* Should never come here, but default is 11a */
            wMode = MODE_SELECT_11A;
    }
    return wMode;
}

static void
cservBssSetPurge(WLAN_DEV_INFO *pDevInfo)
{
    int i, j;
    BSSDESCR_SET *pSet;
    BSSDESCR_SET *pSet1;
    BSSDESCR_SET *pSet2;
    CSERV_INFO *pCsInfo = pDevInfo->pCsInfo;
    CHAN_VALUES *p1, *p2;
    
    
    if (!pDevInfo || !pCsInfo) {
        return;
    }

    /*
     * Use the channel list to purge the BSS set in pCsInfo
     */
    pSet1 = &pCsInfo->scanInfo.bssSet;
    pSet2 = &pCsInfo->scanInfo.bssSet;
    /*
     * Use the channel list again to purge the BSS scan in pDevInfo
     */
    if (pDevInfo->pscanInfo) {
        pSet2 = &pDevInfo->pscanInfo->bssSet;
    }

    pSet = pSet1 ? pSet1 : pSet2;
    while (pSet) {
        cservPrintf("BSS cache/scan cnt: %d (before)\n", pSet->count);

        for (i = 0; i < pDevInfo->staConfig.pClist->listSize; i++) {
            p1 = &pDevInfo->staConfig.pClist->chanArray[i];
            j = 0;
            while (j < pSet->count) {
                p2 = pSet->bssDescrArray[j].pChannel;
                if ((p1->channel == p2->channel) &&
                    ((p1->channelFlags & CHANNEL_ALL) ==
                    (p2->channelFlags & CHANNEL_ALL)) &&
                    (p1->channelFlags & CHANNEL_DONT_SCAN))
                {
                    cservPrintf("purge BSS[%d]: %d ", j, p2->channel);
                    printSsid(&pSet->bssDescrArray[j].ssid);
                    cservPrintf("\n");
                    if (j < pSet->count - 1) {
                        /* No need to move last entry. */
                        pSet->bssDescrArray[j] =
                            pSet->bssDescrArray[pSet->count-1];
                    }
                    pSet->count--;
                } else {
                    j++;
                }
            }
        }
        cservPrintf("BSS cache/scan cnt: %d (after)\n", pSet->count);
        pSet = (pSet == pSet2) ? NULL : pSet2;
    }
}

/*
 * The following processes information received about a nearby AP.
 * Some APs send this information upon association in order to
 * facilitate agile roaming behavior.
 */
void
cservNearbyApReport(WLAN_DEV_INFO *pDev, A_UINT16 channel,
                    WLAN_MACADDR *pBssid, SSID *pSsid)
{
    /* TODO: DO SOMETHING HERE */
}

