/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/wdc/hostDev.c#5 $
 *
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 */
#include "wlandrv.h"
#include "wlanchannel.h"
#include "wlantype.h"
#include "wdcApi.h"
#include "wdcMsg.h"
#include "wdcMsgApi.h"
#include "hostSend.h"
#include "hostReceive.h"
#include "tgtInfo.h"
#include "usbstub.h"
#if NDIS_WDM
#include "athusbapi.h"
#endif
#include "wdcPriv.h"
#include "wlanvers.h"

TARGET_MSG_HANDLE TargetMsgHandle;

/*
 * There is an event table associated with PER_TARGET_INFO
 * for most events.  But the devAvailEvent is not
 * associated with any particular target; it's global.
 */
WDCEVT_ITEM devAvailEvent;

/* Forward references */
static WDCMSG_DISPATCH_FN wdcMsgDispatch;
static void wdcDeviceAvailable(TARGET_HANDLE targetHandle, A_STATUS *pStatus);

/*
 * Initialize the host-side WDC layer.
 * This must be called before any other WDC interface is used.
 */
A_BOOL
wdcInit(void)
{
    PER_TARGET_INFO     *targetHandle;
    A_UINT32            i;

    targetHandle = (PER_TARGET_INFO *)A_DRIVER_MALLOC(sizeof(PER_TARGET_INFO));
    if (targetHandle == NULL) {
        return A_NO_RESOURCE;
    }

    A_MEM_ZERO(targetHandle, sizeof(*targetHandle));

    /*
     * Tell the messaging layer what to use for our TargetHandle,
     * and determine what targetMsgHandle to use when talking to
     * the target, and tell it how to find our Dispatch function.
     */
    targetHandle->targetMsgHandle =
        wdcMsgInit((TARGET_HANDLE)targetHandle, wdcMsgDispatch);

    targetHandle->bTargetHang = FALSE;
    targetHandle->NoRespCount = 0;


#if NDIS_WDM
    for (i = 0; i < WDCMSG_MAX; i ++) {
        NdisInitializeEvent(&targetHandle->msgCompEvent[i]);
    }
    for (i = 0; i < MAX_MSG_ID_COUNT; i ++) {
        NdisInitializeEvent(&targetHandle->rxMsgCompEvent[i]);
        NdisResetEvent(&targetHandle->rxMsgCompEvent[i]);
        NdisInitializeEvent(&targetHandle->txMsgCompEvent[i]);
        NdisResetEvent(&targetHandle->txMsgCompEvent[i]);
    }
#endif

    return A_OK;
}

void
wdcRegisterEventHandler(
    TARGET_HANDLE       targetHandle,
    WDC_EVENT           event,
    EVENT_HANDLER       *pHandler,
    void *              param,
    A_STATUS            *pStatus
)
{
    A_STATUS        status;
    WDCEVT_ITEM     *pEvtItem;    

    if (event >= WDCEVT_MAX) {
        status = A_EINVAL;
        goto regevtExit;
    }

    if (event == WDCEVT_DEVICE_AVAILABLE) {
        ASSERT(targetHandle == NULL);
        pEvtItem = &devAvailEvent;
    } else {
        PER_TARGET_INFO     *pTargInfo;

        ASSERT(targetHandle != NULL);
        pTargInfo = (PER_TARGET_INFO *)targetHandle;
        pEvtItem = &pTargInfo->wdcHandlerTable[event];
    }

    pEvtItem->pHandler = pHandler;
    pEvtItem->param    = param;

    status = A_OK;

regevtExit:
    if (pStatus) {
        *pStatus = status;
    }
}

void
wdcGetEventHandler(
    TARGET_HANDLE       targetHandle,
    WDC_EVENT           event,
    EVENT_HANDLER       **ppHandler,
    void *              *pParam,
    OUT A_STATUS        *pStatus
)
{
    A_STATUS        status;
    WDCEVT_ITEM     *pEvtItem;

    if (event >= WDCEVT_MAX) {
        status = A_EINVAL;
        goto getevtExit;
    }

    if (event == WDCEVT_DEVICE_AVAILABLE) {
        ASSERT(targetHandle == NULL);
        pEvtItem = &devAvailEvent;
    } else {
        PER_TARGET_INFO     *pTargInfo;

        ASSERT(targetHandle != NULL);
        pTargInfo = (PER_TARGET_INFO *)targetHandle;
        pEvtItem = &pTargInfo->wdcHandlerTable[event];
    }

    if (ppHandler) {
        *ppHandler = pEvtItem->pHandler;
    }
    if (pParam) {
        *pParam  = pEvtItem->param;
    }

    status = A_OK;

getevtExit:
    if (pStatus) {
        *pStatus = status;
    }
}

A_STATUS
wdcTriggerEventHandler(
    TARGET_HANDLE       targetHandle,
    WDC_EVENT           event,
    A_UINT32            infoSize,
    A_UCHAR             *pInfo
)
{
    A_STATUS        status;
    WDCEVT_ITEM     *pEvtItem;

    ASSERT(event < WDCEVT_MAX);
    ASSERT(targetHandle != NULL);

    if (event == WDCEVT_DEVICE_AVAILABLE) {
        pEvtItem = &devAvailEvent;
    } else {
        PER_TARGET_INFO     *pTargInfo;

        pTargInfo = (PER_TARGET_INFO *)targetHandle;
        pEvtItem = &pTargInfo->wdcHandlerTable[event];
    }

    if (pEvtItem->pHandler != NULL) {
        /* Call the pre-registered Event Handler */
        pEvtItem->pHandler(targetHandle, event, pEvtItem->param, infoSize, pInfo);
    }

    status = A_OK;

    return status;
}


/*
 * This function is called when the host becomes
 * aware of the presence of a new device.
 *
 * It is intended for use on systems where the host
 * itself detects new hardware (e.g. PCI).  On systems
 * with an independent target, this will never be called.
 */
void
wdcDeviceDiscovered(
    DEVICE_DATA devData,
    A_STATUS *pStatus
)
{
    A_STATUS          status;
    PER_TARGET_INFO   *pTargInfo;

#if NDIS_WDM
    extern struct target_msg_info theTargetMsgStruct;
    A_UINT32            eventNum;
    A_UINT32            i;

    pTargInfo = (PER_TARGET_INFO *)theTargetMsgStruct.targetHandle;

    if (pStatus) {
        *pStatus = A_ERROR;
    }

    gepBufSize[0][DATA_PIPE] = WDC_DATA_MSG_LENGTH_MAX; /* Receive */
    gepBufSize[1][DATA_PIPE] = WDC_DATA_MSG_LENGTH_MAX; /* Send */

    status = athUsbDrvInit(0,
                           devData,
                           (void *)pTargInfo,
                           athUsbRecvIndication,
                           athUsbSendConfirm,
                           athUsbStatusIndication,
                           &pTargInfo->usbHandle,
                           &pTargInfo->totalInPipe,
                           &pTargInfo->totalOutPipe);

    if (status == A_OK) {
        /* Give the messaging layer buffers for Control Messages */
        for (i = 0; i < MAX_RCV_MSG_COUNT; i ++) {
            pTargInfo->pRcvMsgBuf[i] = 
                            (WDC_MSG *)A_DRIVER_MALLOC(WDC_MSG_LENGTH_MAX + 
                                                       sizeof(WDC_MSG_OOB_DATA));
            if (pTargInfo->pRcvMsgBuf[i]) {            
                wdcCtrlMsgRecv(pTargInfo->targetMsgHandle,pTargInfo->pRcvMsgBuf[i]);
            } else {
                break;
            }
        }

        pTargInfo->numRxMsgBuf = i;

        wdcHostAvailable(pTargInfo->hostHandle, 
                         (TARGET_HANDLE) pTargInfo,
                         &status);

        eventNum = WDCMSG_DEVICE_AVAILABLE - 1;
        if (!NdisWaitEvent(&pTargInfo->msgCompEvent[eventNum], MAX_MSGREAD_WAIT_TIME)) {
            athUsbDrvExit(0,pTargInfo->usbHandle);
            uiPrintf("Recevie WDCMSG_DEVICE_AVAILABLE message timeout\n");
            return;
        }

        wdcDeviceAvailable(pTargInfo, &status);
        pTargInfo->busStatus = ATHUSB_BUS_STATE_NORMAL;
    }

    if (pStatus) {
        (*pStatus) = status;
    }
#endif
}

/*
 * This function is called as soon as the host becomes
 * aware of the absence of a target device.  
 */
void
wdcSurpriseRemoved(
    TARGET_HANDLE       targetHandle
)
{
    PER_TARGET_INFO *pTargInfo = (PER_TARGET_INFO *)targetHandle;
	/* WDC layer should know about the device removal, otherwise the host
     * can still send WDC commands and will wait unsuccessfully for
     * 20 seconds. delays driver/device removal a lot */
    pTargInfo->busStatus = ATHUSB_BUS_STATE_SURPRISE_REMOVED;
    athUsbSurpriseRemoved(0,pTargInfo->usbHandle);
}


/*
 * When a host-side driver decides to manage a target,
 * it binds to the target.
 */
void
wdcBind(
    HOST_HANDLE         hostHandle,
    TARGET_HANDLE       targetHandle,
    A_UINT32            hostApiVersion,
    A_STATUS            *pStatus
)
{
    PER_TARGET_INFO *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    WDC_MSG         *wdcMsg;
    A_STATUS        status;

    ASSERT(pTargInfo);

    if (pStatus) {
        (*pStatus) = A_ERROR;
    }

    pTargInfo->hostHandle = hostHandle;

    /*
     * Send a WDCMSG_BIND message to the target.
     */
    wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle, WDCMSG_BIND);
    if (wdcMsg == NULL) {
        uiPrintf("No resource available\n");
        return;
    }

    wdcMsgAddParam(wdcMsg, fixEndian(hostApiVersion));

    status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);
    ASSERT(status != A_PENDING);

    if (status == A_OK) {
        wdcCtrlMsgFree(wdcMsg);
        pTargInfo->bBind = TRUE;
    } else {
        pTargInfo->bBind = FALSE;
    }

    if (pStatus) {
        *pStatus = status;
    }
}

void 
wdcUnbind(
    HOST_HANDLE         hostHandle,
    TARGET_HANDLE       targetHandle
)
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    WDC_MSG             *wdcMsg;
    A_STATUS            status;
    A_UINT32            i;
    KIRQL               curIrql;
    A_UINT32            msgId;

    athGetCurrentIrql(curIrql);

    if (pTargInfo->bBind == TRUE &&
        pTargInfo->busStatus == ATHUSB_BUS_STATE_NORMAL)
    {
        /*
        * Send a WDCMSG_UNBIND message to the target.
        */
        wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle,WDCMSG_UNBIND);
        if (wdcMsg == NULL) {
            uiPrintf("No resource available\n");
            return;
        }

        if (curIrql < DISPATCH_LEVEL) {
            wdcMsg->r.responseDesired = TRUE;
        }

        msgId = wdcMsg->msgId;
        NdisResetEvent(&pTargInfo->rxMsgCompEvent[msgId]);

        status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);
        if (status == A_OK) {
            wdcCtrlMsgFree(wdcMsg);
        }

#if NDIS_WDM
        if (curIrql < DISPATCH_LEVEL) {
            if (!NdisWaitEvent(&pTargInfo->rxMsgCompEvent[msgId], MAX_MSGREAD_WAIT_TIME)) {
                uiPrintf("No response for WDCMSG_UNBIND\n");                
            } else {
                wdcMsg = pTargInfo->rcvMsg[msgId];
                wdcCtrlMsgRecv(pTargInfo->targetMsgHandle,wdcMsg);
            }
        }
#endif
    }

    pTargInfo->hostHandle = NULL;

#if NDIS_WDM
    athUsbDrvExit(0,pTargInfo->usbHandle);
    for (i = 0; i < pTargInfo->numRxMsgBuf; i ++) {
        if (pTargInfo->pRcvMsgBuf[i]) {
            A_DRIVER_FREE(pTargInfo->pRcvMsgBuf[i],
                      WDC_MSG_LENGTH_MAX + sizeof(WDC_MSG_OOB_DATA));
        }
    }
#endif

    /*
     * Free our own memory
     */    
    hsShutdownTx(targetHandle);
    hsShutdownRx(targetHandle);

    A_DRIVER_FREE(pTargInfo,sizeof(PER_TARGET_INFO));
}

/*
 * After the host understands capabilities of this target and
 * after the host has configured this target as desired, it
 * starts the device.
 */
void
wdcTargetStart(
    TARGET_HANDLE       targetHandle,
    A_STATUS            *pStatus
)
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    WDC_MSG             *wdcMsg;
    A_STATUS            status = A_ERROR;
    A_UINT32            msgId, seed;
    int                 i;
    KIRQL               curIrql;
    A_UINT32            cursor = 0;

    ASSERT(pTargInfo);
    ASSERT(pStatus);


    pTargInfo->bStart = FALSE;

    /* Initialize intermediate tx buffers */
    status = hsInitTx(targetHandle);
    if (status != A_OK) {
        uiPrintf ("hsInitTx Failed\n");
        goto exitError;
    }

    /* Allocate rx buffers */
    status = hsInitRx(targetHandle);
    if (status != A_OK) {
        uiPrintf ("hsInitRx Failed\n");
        goto exitError;
    }

    athGetCurrentIrql(curIrql);
    /*
     * Send a WDCMSG_TARGET_START message to the target.
     */
    wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle,WDCMSG_TARGET_START);
    if (wdcMsg == NULL) {
        uiPrintf("No resource available\n");
        return;
    }

    if (curIrql < DISPATCH_LEVEL) {
        wdcMsg->r.responseDesired = TRUE;
    }

    status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);    

    msgId = wdcMsg->msgId;
    NdisResetEvent(&pTargInfo->rxMsgCompEvent[msgId]);

    if (status == A_OK) {
        wdcCtrlMsgFree(wdcMsg);
    }

#if NDIS_WDM
    if (curIrql < DISPATCH_LEVEL) {
        if (!NdisWaitEvent(&pTargInfo->rxMsgCompEvent[msgId], MAX_MSGREAD_WAIT_TIME)) {
            uiPrintf("No response for WDCMSG_TARGET_START\n");    
            status = A_ERROR;
            goto exitError;
        }    

        wdcMsg = pTargInfo->rcvMsg[msgId];
        if (wdcMsg) {
            status = wdcMsg->r.targetStatus;
            if (status == A_OK) {     
                pTargInfo->bStart = TRUE;
            }
            wdcMsgReadParam(wdcMsg,&cursor,&seed);
            randomInit(seed);

            wdcCtrlMsgRecv(pTargInfo->targetMsgHandle,wdcMsg);
        }
    } else {
        pTargInfo->bStart = TRUE;
    }

#endif

    *pStatus = status;

    return;

exitError:
    *pStatus = status;
    
    /* Cleanup */
    hsShutdownTx(targetHandle);
    hsShutdownRx(targetHandle);
}

void
wdcTargetStop(
    TARGET_HANDLE       targetHandle,
    A_STATUS            *pStatus
)
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    WDC_MSG             *wdcMsg;
    A_STATUS            status;
    KIRQL               curIrql;
    A_UINT32            msgId;

    ASSERT(pTargInfo);

    if (pStatus) {
        *pStatus = A_OK;
    }

    athGetCurrentIrql(curIrql);

    if (pTargInfo->bStart == TRUE &&
        pTargInfo->busStatus == ATHUSB_BUS_STATE_NORMAL)
    {
        /*
         * Send a WDCMSG_TARGET_STOP message to the target.
         */
        wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle,WDCMSG_TARGET_STOP);
        if (wdcMsg == NULL) {
            uiPrintf("No resource available\n");
            return;
        }

        if (curIrql < DISPATCH_LEVEL) {
            wdcMsg->r.responseDesired = TRUE;
        }

        msgId = wdcMsg->msgId;
        NdisResetEvent(&pTargInfo->rxMsgCompEvent[msgId]);

        status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);
        if (status == A_OK) {
            wdcCtrlMsgFree(wdcMsg);
        }

#if NDIS_WDM
        if (curIrql < DISPATCH_LEVEL) {
            if (!NdisWaitEvent(&pTargInfo->rxMsgCompEvent[msgId], MAX_MSGREAD_WAIT_TIME)) {
                uiPrintf("No response for WDCMSG_TARGET_STOP\n");                
            } else {
                wdcMsg = pTargInfo->rcvMsg[msgId];
                wdcCtrlMsgRecv(pTargInfo->targetMsgHandle,wdcMsg);
            }
        }
#endif
    }
}

/*
 * wdcTargetGetCapability
 *
 * Allows the host to learn about the target's capabilities. 
 */
void
wdcTargetGetCapability(
    IN  TARGET_HANDLE       targetHandle,
    IN  TARGET_CAP_ID       capId,
    OUT A_UINT32            *capSz,
    OUT TARGET_CAP_DATA     *capData,
    OUT A_STATUS            *pStatus)
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    WDC_MSG             *wdcOutMsg,*wdcInMsg;
    A_UINT32            msgId;
    A_UINT32            rcvCapSz;
    A_STATUS            status;
    A_UINT8             *pData;
    A_UINT32            cursor = 0;
    WDC_MSG_OOB_DATA    *pOobData;
    KIRQL               curIrql;

    ASSERT(pTargInfo);

    if (pStatus) {
        (*pStatus) = A_ERROR;
    }

    if (capSz) {
        (*capSz) = 0;
    }

    /*
     * Send a WDCMSG_TARGET_GET_CAPABILITY message to the target.
     */
    wdcOutMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle,WDCMSG_TARGET_GET_CAPABILITY);    
    if (wdcOutMsg == NULL) {
        uiPrintf("No resource available\n");
        return;
    }
#ifdef DEBUG
    wdcOutMsg->msgDebug[0] = capId;
#endif

#if NDIS_WDM    
    msgId = wdcOutMsg->msgId;
    NdisResetEvent(&pTargInfo->rxMsgCompEvent[msgId]);    
    /* 
     * This function may be called from dispatch_level, we can't wait for
     * receive message complete on dispatch_level, so we need save all capSz
     * and capData pointer on here and we will fill up them on message 
     * completion routine. Even we can wait receive complete, it will be better
     * to let completion routine fill up data because this function may be called
     * from different thread to get different cap.
     *
     * Please Note: if this function is called from dispatch_level,capSz and capData 
     * should still be exist even after exit from this function, that means capData 
     * and capSz should be some resident memory, for example, the data inside 
     * pDevInfo, otherwise, callback function will access invalid memory
     */
    athGetCurrentIrql(curIrql);
    if (curIrql < DISPATCH_LEVEL) {
        pTargInfo->pPendCapSz[msgId] = NULL;
        pTargInfo->pendCapBuf[msgId] = 0;
    } else {
        pTargInfo->pPendCapSz[msgId] = capSz;
        pTargInfo->pendCapBuf[msgId] = (A_UINT32) capData;        
    }
#endif

    wdcMsgAddParam(wdcOutMsg, fixEndian(capId));

    status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcOutMsg);    

    if (status == A_TIMEOUT) {
        uiPrintf("Send WDCMSG_TARGET_GET_CAPABILITY message timeout\n");
        return;
    }

    if (status == A_OK) {
        wdcCtrlMsgFree(wdcOutMsg);
    }

#if NDIS_WDM      

    if (curIrql < DISPATCH_LEVEL) {
        if (!NdisWaitEvent(&pTargInfo->rxMsgCompEvent[msgId], MAX_MSGREAD_WAIT_TIME)) {
            uiPrintf("Receive WDCMSG_TARGET_GET_CAPABILITY message timeout\n");        
            return;
        } else {
            wdcInMsg = pTargInfo->rcvMsg[msgId];
            ASSERT(wdcInMsg);
        }
    } else {
        if (pStatus) {
            (*pStatus) = A_PENDING;
        }
        return;
    }
#endif

    ASSERT(!pTargInfo->bDebugEnabled || (wdcInMsg->msgDebug[0] == capId));
    
    wdcMsgReadParam(wdcInMsg,&cursor,&rcvCapSz);
    ASSERT(rcvCapSz <= WDC_MSG_LENGTH_MAX - WDCMSG_HEADER_LENGTH - sizeof(A_UINT32));

    uiPrintf("OpCode = %x, msgLen = %d, capId = %x, capSz = %x\n",
            wdcInMsg->msgOpcode,wdcInMsg->msgLength,wdcInMsg->msgId,rcvCapSz);

    if (rcvCapSz) {
        /* Variable-sized configuration data */
        wdcMsgSeekData(wdcInMsg,&cursor,(void **)&pData);
        A_BCOPY(pData,capData,rcvCapSz);

        if (capSz) {
            (*capSz) = rcvCapSz;
        }
    } else {
        /* Fixed-sized configuration data */
        wdcMsgReadParam(wdcInMsg,&cursor,(A_UINT32 *)capData);
    
        if (capSz) {
            (*capSz) = 4;
        }
        uiPrintf("capData = %x\n",(*capData));
    }
#if NDIS_WDM
    wdcCtrlMsgRecv(pTargInfo->targetMsgHandle,wdcInMsg);
    pTargInfo->rcvMsg[msgId] = NULL;
#endif

    if (pStatus) {
        (*pStatus) = A_OK;
    }
}

/*
 * Information that will be needed in order to free a variable-sized
 * status buffer.  This is added into a received status message just
 * after the status data itself.
 */
struct statusFreeInfo{
    TARGET_HANDLE      targetHandle;
    A_UINT32           msgId;
    WDC_MSG            *wdcInMsg;
};

/*
 * wdcTargetGetStatus
 *
 * Allows the host to fetch target status information. 
 */
void
wdcTargetGetStatus(
    IN  TARGET_HANDLE       targetHandle,
    IN  TARGET_STATUS_ID    statusId,
    OUT A_UINT32            *statusSz,
    OUT TARGET_STATUS_DATA  *statusData,
    OUT A_STATUS            *pStatus)
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    WDC_MSG             *wdcOutMsg,*wdcInMsg;
    A_UINT32            msgId;
    A_UINT32            rcvStatSz;
    A_STATUS            status;
    A_UINT8             *pData;
    A_UINT32            cursor = 0;
    WDC_MSG_OOB_DATA    *pOobData;

    ASSERT(pTargInfo);
    ASSERT(statusData);

    /*
     * Some status items are handled entirely on the Host.
     */
    switch(statusId) {

    case ST_TX_QUEUE_DEPTH:
    {
        *statusData = (TARGET_STATUS_DATA)pTargInfo->intTxPendQueue.qFrameCount;
        if (statusSz) {
            (*statusSz) = 0;
        }
        if (pStatus) {
            (*pStatus) = A_OK;
        }
        return;
    }
    };

    /*
     * Send a WDCMSG_TARGET_GET_STATUS message to the target.
     */
    wdcOutMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle,
                                 WDCMSG_TARGET_GET_STATUS);

    if (wdcOutMsg == NULL) {
        uiPrintf("No resource available\n");
        return;
    }
#ifdef DEBUG
    wdcOutMsg->msgDebug[0] = statusId;
#endif

#if NDIS_WDM    
    msgId = wdcOutMsg->msgId;
    NdisResetEvent(&pTargInfo->rxMsgCompEvent[msgId]);
#endif

    wdcMsgAddParam(wdcOutMsg, fixEndian(statusId));

    status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcOutMsg);
    ASSERT(status != A_PENDING);

    if (status == A_TIMEOUT) {
        uiPrintf("Send WDCMSG_TARGET_GET_STATUS message timeout\n");
        goto getStatusExit;
    }

    if (status == A_OK) {
        wdcCtrlMsgFree(wdcOutMsg);
    }

#if NDIS_WDM
    NDIS_ASSERT(KeGetCurrentIrql() < DISPATCH_LEVEL);
    if (!NdisWaitEvent(&pTargInfo->rxMsgCompEvent[msgId], MAX_MSGREAD_WAIT_TIME)) {
        uiPrintf("Receive WDCMSG_TARGET_GET_STATUS message timeout\n");        
        status = A_TIMEOUT;
        goto getStatusExit;
    } else {
        wdcInMsg = pTargInfo->rcvMsg[msgId];
        ASSERT(wdcInMsg);
    }
#endif

    ASSERT(!pTargInfo->bDebugEnabled || (wdcInMsg->msgDebug[0] == statusId));

    wdcMsgReadParam(wdcInMsg, &cursor, &rcvStatSz);
    ASSERT(rcvStatSz <= WDC_MSG_LENGTH_MAX - WDCMSG_HEADER_LENGTH - sizeof(A_UINT32));

    uiPrintf("OpCode = %x, msgLen = %d, statId = 0x%x, statLen = %d\n",
            wdcInMsg->msgOpcode,wdcInMsg->msgLength,wdcInMsg->msgId,rcvStatSz);

    if (rcvStatSz) {
        /* Variable-sized status data */
        struct statusFreeInfo *freeInfo;
        A_UINT32               roundedSz;

        wdcMsgSeekData(wdcInMsg, &cursor, (void **)&pData);

        /*
         * Store backpointers that will be needed by wdcStatusDataFree
         * when the Host driver is done looking at status data.
         */
        roundedSz = ROUND_UP(rcvStatSz, 4);
        freeInfo = (struct statusFreeInfo *)((A_UINT32)pData + roundedSz);
        ASSERT(WDCMSG_HEADER_LENGTH + roundedSz + sizeof(*freeInfo) < WDC_MSG_LENGTH_MAX);
        freeInfo->targetHandle = targetHandle;
        freeInfo->msgId        = msgId;
        freeInfo->wdcInMsg     = wdcInMsg;

        *statusData = pData;
    } else {
        /* Fixed-sized configuration data */
        wdcMsgReadParam(wdcInMsg, &cursor, (A_UINT32 *)statusData);
        rcvStatSz = 4;
#if NDIS_WDM
        wdcCtrlMsgRecv(pTargInfo->targetMsgHandle, wdcInMsg);
        pTargInfo->rcvMsg[msgId] = NULL;
#endif
    }    

    ASSERT(status == A_OK);
    if (statusSz) {
        (*statusSz) = rcvStatSz;
    }

getStatusExit:
    if (pStatus) {
        (*pStatus) = status;
    }
}

void
wdcStatusDataFree(
    IN  TARGET_STATUS_DATA  statusData,
    IN  A_UINT32            statusSz
)
{
    struct statusFreeInfo *freeInfo;
    PER_TARGET_INFO       *pTargInfo;
    WDC_MSG               *wdcInMsg;
    A_UINT32              msgId;
    A_UINT32              roundedSz;

    roundedSz = ROUND_UP(statusSz, 4);
    freeInfo = (struct statusFreeInfo *)((A_UINT32)statusData + roundedSz);

    pTargInfo = (PER_TARGET_INFO *)(freeInfo->targetHandle);
    wdcInMsg  = freeInfo->wdcInMsg;
    msgId     = freeInfo->msgId;

#if NDIS_WDM
    wdcCtrlMsgRecv(pTargInfo->targetMsgHandle, wdcInMsg);
    pTargInfo->rcvMsg[msgId] = NULL;
#endif
}


/*
 * Extract a single status item from a ST_ALL status buffer.
 */
A_STATUS
wdcGetStatusItem(
    IN  TARGET_STATUS_DATA  *allStatusData,
    IN  TARGET_STATUS_ID    statusId,
    OUT A_UINT32            *statusSz,
    OUT TARGET_STATUS_DATA  *statusItem
)
{
    WDC_ALL_STATUS     *allStatus;
    AR_DEV_STATUS      *devStatus;
    AR_DEV_CONFIG      *devConfig;
    A_UINT32           sz = 0;
    A_STATUS           status = A_OK;
    TARGET_STATUS_DATA data;

    allStatus = (WDC_ALL_STATUS *)allStatusData;

    devStatus = &allStatus->devStatus;
    devConfig = &allStatus->devConfig;

    ASSERT(devStatus);
    ASSERT(devConfig);

    switch(statusId) {
    case ST_SERVICE_TYPE:
        data = (TARGET_STATUS_DATA)devConfig->serviceType;
        break;

    case ST_WLAN_MODE:
        data = (TARGET_STATUS_DATA)devConfig->chanDesc.wlanMode;
        break;

    case ST_FREQ:
        data = (TARGET_STATUS_DATA)devConfig->chanDesc.channelFreq;
        break;

    case ST_BAND:
        data = (TARGET_STATUS_DATA)devConfig->chanDesc.channelBand;
        break;

    case ST_LAST_RSSI:
        data = (TARGET_STATUS_DATA)devStatus->s_phyHwLastRssi;
        break;

    case ST_PS_FRAMES_DROPPED:
        data = (TARGET_STATUS_DATA)devStatus->s_psFramesDropped;
        break;

    case ST_CACHED_DEF_ANT:
        data = (TARGET_STATUS_DATA)devStatus->s_cachedDefAnt;
        break;

    case ST_COUNT_OTHER_RX_ANT:
        data = (TARGET_STATUS_DATA)devStatus->s_countOtherRxAnt;
        break;

    case ST_USE_FAST_DIVERSITY:
        data = (TARGET_STATUS_DATA)devStatus->s_useFastDiversity;
        break;

    case ST_MAC_ADDR: {
        data = (TARGET_STATUS_DATA)&devConfig->macAddr;
        sz    = sizeof(devConfig->macAddr);
        break;
    }

    case ST_SERIAL_NUMBER: {
        data = (TARGET_STATUS_DATA)devConfig->serialNumber;
        sz    = sizeof(devConfig->serialNumber);
        break;
    }

    default:
        status = A_EINVAL; /* unknown status item */
    }

    if (status == A_OK) {
        if (statusSz) {
            *statusSz = sz;
        }
        if (statusItem) {
            *statusItem = data;
        }
    }

    return status;
}


/*
 * wdcTargetSetConfiguration
 *
 * specifies a new value for a target-wide configuration attribute. 
 */
void
wdcTargetSetConfiguration(
    IN  TARGET_HANDLE       targetHandle,
    IN  TARGET_CONFIG_ID    attrId,
    IN  A_UINT32            cfgSz,
    IN  TARGET_CONFIG_VAL   cfgVal,
    OUT A_STATUS            *pStatus)
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    WDC_MSG             *wdcMsg;
    A_UINT32            eventNum;
    A_UINT32            cfgData;
    A_UINT32            maxLength;
    A_STATUS            status;
    A_UINT8             *pData;    

    ASSERT(pTargInfo);

    if (pStatus) {
        (*pStatus) = A_ERROR;
    }
    
    /*
     * Send a WDCMSG_TARGET_SET_CONFIGURATION message to the target.
     */
    wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle,WDCMSG_TARGET_SET_CONFIGURATION);
    if (wdcMsg == NULL) {
        uiPrintf("No resource available\n");
        return;
    }    

    wdcMsgAddParam(wdcMsg, fixEndian(attrId));
    
    switch (attrId) {
        /*
         * All the following attrID use the fixed size
         */
        case CFG_REG_DOMAIN:
        case CFG_RATE_CONTROL_ENABLE:
        case CFG_HW_TX_RETRIES:
        case CFG_SW_TX_RETRIES:
        case CFG_CALIBRATION_INTERVAL:
        case CFG_SLOW_CLOCK_ENABLE:
        case CFG_COMP_PROC:
        case CFG_USER_RTS_THRESHOLD:
        case CFG_PROTECTION_TYPE:
        case CFG_BURST_SEQ_THRESHOLD:
        case CFG_ABOLT:
        case CFG_MODE_CTS:
        case CFG_WME_ENABLED:
        case CFG_GPRS_CBR_PERIOD:
        case CFG_SERVICE_TYPE:
#if defined(PCI_INTERFACE)
        case CFG_DEBUG_EAR:
#endif
        case CFG_DEBUG_ID:
        case CFG_COMP_WIN_SZ:
        case CFG_DIVERSITY_CTL:
        case CFG_TP_SCALE:
        case CFG_TPC_HALF_DBM5:
        case CFG_TPC_HALF_DBM2:
        case CFG_OVERRD_TX_POWER:
        case CFG_USE_32KHZ_CLOCK:
        case CFG_GMODE_PROTECTION:
        case CFG_GMODE_PROTECT_RATE_INDEX:
        case CFG_GMODE_NON_ERP_PREAMBLE:

            /*
             * Set cfg size first
             */
            wdcMsgAddParam(wdcMsg, 0);
            cfgData = *((A_UINT32 *)cfgVal);

            wdcMsgAddParam(wdcMsg, fixEndian(cfgData));
            break;

        /*
         * All the following attrID use the variable size
         */
        case CFG_MAC_ADDR:

            /*
             * Set cfg size first
             */
            wdcMsgAddParam(wdcMsg, fixEndian(cfgSz));
            wdcMsgPrepData(wdcMsg,(void **)&pData, &maxLength);
            ASSERT(cfgSz <= maxLength);            
            A_BCOPY(cfgVal, pData, cfgSz);
            wdcMsgCommitData(wdcMsg,cfgSz);
            break;

        case CFG_INIT_REGS:
            wdcMsgAddParam(wdcMsg, fixEndian(cfgSz));
            wdcMsgPrepData(wdcMsg, (void **)&pData, &maxLength);
            ASSERT(cfgSz <= maxLength);
            A_BCOPY(cfgVal, pData, cfgSz);
            wdcDataSwizzle(pData, cfgSz);
            wdcMsgCommitData(wdcMsg, cfgSz);
            break;

    }    

    status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);
    if ((status != A_PENDING) && (status != A_TIMEOUT)) {
        wdcCtrlMsgFree(wdcMsg);
    }   

    if (pStatus) {
        (*pStatus) = status;
    }
}


/*
 * wdcTargetGetStats
 *
 * Retrieve Per Target statistics.  
 * Host must register for the WDCEVT_WLAN_STATS_UPDATE event.
 *
 * See also wdcGetStatsItem
 * See also wdcGetConnectionStats (per connection statistics)
 */

void
wdcTargetGetStats(
    IN  TARGET_HANDLE       targetHandle
)
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    WDC_MSG             *wdcMsg;
    A_STATUS            status;

    ASSERT(pTargInfo);

    /*
     * Send a WDCMSG_TARGET_GET_STATS message to the target.
     */
    wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle,
                                 WDCMSG_TARGET_GET_STATS);

    if (wdcMsg == NULL) {
        uiPrintf("No resource available\n");
        return;
    }

    status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);

    if (status == A_OK) {
        wdcCtrlMsgFree(wdcMsg);
    }
}


typedef struct statsParam {
    A_UINT32      statsId;
    A_UINT32      size;
    A_UINT32      offset;
} STATS_PARAM;

/* 
 * NOTE: this table must be kept in the same order as
 * the TARGET_STATS_ID enum 
 */
static STATS_PARAM statsTable[] = {
    /* enum                         size                  offset */
    { STATS_NONE,                   0,                    0},
    { STATS_ALL,                    0,                    0},

    { STATS_RX_FCS_FAIL_CNT,        sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, FcsFailCnt) },
    { STATS_RX_DMA_OVERRUN_ERRORS,  sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, RcvDmaOverrunErrors) },
    { STATS_RX_BEACON_COUNT,        sizeof(A_LONGSTATS),  FIELD_OFFSET(AR_STATS, RxBeacons) },

    { STATS_TX_GOOD_FRAMES,         sizeof(A_LONGSTATS),  FIELD_OFFSET(AR_STATS, GoodTransmits) },
    { STATS_TX_UNICAST_FRAMES,      sizeof(A_LONGSTATS),  FIELD_OFFSET(AR_STATS, TxUnicastFrames) },
    { STATS_TX_BROADCAST_FRAMES,    sizeof(A_LONGSTATS),  FIELD_OFFSET(AR_STATS, TxBroadcastFrames) },
    { STATS_TX_MUTLICAST_FRAMES,    sizeof(A_LONGSTATS),  FIELD_OFFSET(AR_STATS, TxMulticastFrames) },
    { STATS_TX_FRAMES_DROPPED,      sizeof(A_LONGSTATS),  FIELD_OFFSET(AR_STATS, TxFramesDropped) },
    { STATS_TX_GOOD_BYTES,          sizeof(A_LONGSTATS),  FIELD_OFFSET(AR_STATS, GoodTransmitBytes) },

    { STATS_TX_ERRORS,              sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, TransmitErrors) },
    { STATS_TX_EXCESSIVE_RETRIES,   sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, TxExcessiveRetries) },
    { STATS_TX_DMA_UNDERRUN,        sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, TxDmaUnderrun) },
    { STATS_TX_ACK_RCV_FAILURES,    sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, AckRcvFailures) },
    { STATS_TX_RTS_SUCCESS_CNT,     sizeof(A_LONGSTATS),  FIELD_OFFSET(AR_STATS, RtsSuccessCnt) },
    { STATS_TX_RTS_FAIL_CNT,        sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, RtsFailCnt) },

    { STATS_TX_COMP_SUCCESS_CNT,    sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, compSuccessCnt) },
    { STATS_TX_COMP_CPC0_CNT,       sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, CompCPC0Cnt) },
    { STATS_TX_COMP_CPC1_CNT,       sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, CompCPC1Cnt) },
    { STATS_TX_COMP_CPC2_CNT,       sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, CompCPC2Cnt) },
    { STATS_TX_COMP_CPC3_CNT,       sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, CompCPC3Cnt) },

    { STATS_TX_FILTERED,            sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, TxFiltered) },
    { STATS_TX_TOTAL_RETRIES,       sizeof(A_LONGSTATS),  FIELD_OFFSET(AR_STATS, TotalRetries) },
//xxxxPGSxxxx - TODO    STATS_TX_GOOD,  sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, RetryBins[16]) },              // long frame bin
//xxxxPGSxxxx - TODO    STATS_TX_GOOD,  sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, shortFrameRetryBins[16]) },
    { STATS_TX_SW_RETRY_TOTAL_CNT,  sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, swRetryTotalCnt) },            // sw retries for this connection
    { STATS_TX_SW_RETRY_MAX_RETRIES_EXCEEDED, sizeof(A_UINT32),  FIELD_OFFSET(AR_STATS, swRetryMaxRetriesExceeded) },

    /* Rates/Antenna/RSSI etc. */

    { STATS_ACK_RSSI,               sizeof(A_INT32),      FIELD_OFFSET(AR_STATS, ackRssi) },
    { STATS_TX_RATE_KB,             sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, txRateKb) },                   // Averaged rate used for transmit
//xxxxPGSxxxx TODO    STATS_ANT_CNT,  sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, AntCnt[2]) },
    { STATS_ANT_SW_CNT,             sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, AntSwCnt) },
    { STATS_CURR_TX_POWER,          sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, tx6PowerInHalfDbm) },
    { STATS_MAX_TX_POWER,           sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, maxTxPowerAvail) },
    { STATS_NOISE_FLOOR,            sizeof(A_UINT32),     FIELD_OFFSET(AR_STATS, noiseFloor) },
};

/*
 * wdcGetStatsItem
 *
 * Called during WDCEVT_WLAN_STATS_UPDATE event to retreive
 * individual statistics items.
 *
 */
A_STATUS
wdcGetStatsItem(
    IN  TARGET_STATS_DATA   allStatsData,
    IN  TARGET_STATS_ID     statsId,
    OUT A_UINT32            *statsSz,
    OUT TARGET_STATS_DATA   *statsItem
)
{
    AR_STATS            *pStats;
    A_UINT32            sz = 0;
    A_STATUS            status = A_OK;
    TARGET_STATS_DATA   data = NULL;

    ASSERT(allStatsData != NULL);
    ASSERT(statsId < STATS_MAX);
    ASSERT(statsTable[statsId].statsId == statsId);

    pStats = (AR_STATS *)allStatsData;

    if (statsId < STATS_MAX) {
        data = (TARGET_STATS_DATA) (((A_UCHAR *)pStats) + statsTable[statsId].offset);
        sz   = statsTable[statsId].size;
    } else {
        status = A_EINVAL; /* unknown status item */
    }

    if (status == A_OK) {
        if (statsSz) {
            *statsSz = sz;
        }
        if (statsItem) {
            *statsItem = data;
        }
    }

    return status;
}


/*
 * Announce availability of a target to the host driver.
 * This is called when a "device available" message is received
 * by the host.
 */
static void
wdcDeviceAvailable(
    TARGET_HANDLE       targetHandle,
    A_STATUS            *pStatus
)
{
    PER_TARGET_INFO   *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    A_UINT32          targetDebugEnabled;
    A_STATUS          status;

    /* Determine whether or not Target-side WDC_MSG Debugging is available */
    wdcTargetGetCapability(targetHandle, CAP_DEBUG_WDCMSG_SUPPORT, NULL,
                           (TARGET_CAP_DATA *)&targetDebugEnabled, NULL);
    pTargInfo->bDebugEnabled = targetDebugEnabled ? TRUE : FALSE;

    status = wdcTriggerEventHandler(targetHandle,
                                    WDCEVT_DEVICE_AVAILABLE,
                                    0,
                                    NULL);
    ASSERT(status == A_OK);

    if (pStatus) {
        *pStatus = A_OK;
    }
}

A_BOOL
wdcTargetPresent(TARGET_HANDLE  targetHandle)
{
#if NDIS_WDM
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)targetHandle;

    ASSERT(pTargInfo);

    /*     
     * We should get target present from PnP message istead of check from function
     * The following function only can be called from passive level, but most time
     * we check target status from dispatch_level
     */
    return TRUE;

    //return athUsbDetectCardPresent(pTargInfo->usbHandle);
#endif

    return TRUE;
}
/*
 * Host WDC message dispatcher.
 */
void
wdcMsgDispatch(
    TARGET_HANDLE targetHandle,
    WDC_MSG *wdcMsg
)
{
    A_UINT32          msgLength  = wdcMsg->msgLength;
    WDC_MSG_OPCODE    msgOpcode  = wdcMsg->msgOpcode;
    char              *msgBuffer = (char *)wdcMsg->msgBuffer;
    A_STATUS          status;
    PER_TARGET_INFO   *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    A_BOOL            bQueueBack = TRUE;
    A_UINT32          msgId = wdcMsg->msgId;
    A_UINT32          cursor = 0;

    /* Validate message */
    ASSERT(msgLength <= WDC_MSG_LENGTH_MAX);
    ASSERT(msgOpcode > WDCMSG_INVAL);
    ASSERT(msgOpcode < WDCMSG_MAX);

    switch(msgOpcode) {
    case WDCMSG_TARGET_STATS_UPDATE: {
        TARGET_STATS_DATA statsData = NULL;
        A_UINT32          statsSize = 0;
        AR_STATS          *pStats;
        TARGET_STATS_ID   statsId;
        A_UINT32          sz = 0;
        TARGET_STATS_DATA data = NULL;


        wdcMsgReadParam(wdcMsg,&cursor, &statsSize);
        ASSERT(statsSize != 0);

        if (statsSize) {
            wdcMsgSeekData(wdcMsg, &cursor, &statsData);

            pStats = (AR_STATS *) statsData;

            /* Fix endianness */
            for (statsId = 0; statsId < STATS_MAX; statsId++) {

                data = (TARGET_STATS_DATA) (((A_UCHAR *)pStats) +
                       statsTable[statsId].offset);
                sz   = statsTable[statsId].size;

                switch (sz) {
                case 0:
                    break;
                case sizeof(A_UINT32):
                    *((A_UINT32 *)data) = fixEndian(*((A_UINT32 *)data));
                    break;
                case sizeof(A_LONGSTATS):
                    *((A_LONGSTATS *)data) = fixEndian64(*((A_LONGSTATS *)data));
                    break;
                default:
                    /* TODO - variable length  */
                    ASSERT(0);
                }
            }

#if 0
            /* Receive related stats */
            uiPrintf(">> WDCMSG_TARGET_STATS_UPDATE\n");

            uiPrintf("FcsFailCnt            %u\n", pStats->FcsFailCnt);
            uiPrintf("RcvDmaOverrunErrors   %u\n", pStats->RcvDmaOverrunErrors);
            uiPrintf("RxBeacons             %u\n", pStats->RxBeacons);


            /* Transmit related stats */
            uiPrintf("GoodTransmits         %llu\n", pStats->GoodTransmits);
            uiPrintf("TxUnicastFrames       %llu\n", pStats->TxUnicastFrames);
            uiPrintf("TxBroadcastFrames     %llu\n", pStats->TxBroadcastFrames);
            uiPrintf("TxMulticastFrames     %llu\n", pStats->TxMulticastFrames);
            uiPrintf("TxFramesDropped       %llu\n", pStats->TxFramesDropped);
            uiPrintf("GoodTransmitBytes     %llu\n", pStats->GoodTransmitBytes);

            uiPrintf("TransmitErrors        %u\n", pStats->TransmitErrors);
            uiPrintf("TxExcessiveRetries    %u\n", pStats->TxExcessiveRetries);
            uiPrintf("TxDmaUnderrun         %u\n", pStats->TxDmaUnderrun);
            uiPrintf("AckRcvFailures        %u\n", pStats->AckRcvFailures);
            uiPrintf("RtsSuccessCnt         %llu\n", pStats->RtsSuccessCnt);
            uiPrintf("RtsFailCnt            %u\n", pStats->RtsFailCnt);

            uiPrintf("compSuccessCnt        %u\n", pStats->compSuccessCnt);
            uiPrintf("CompCPC0Cnt           %u\n", pStats->CompCPC0Cnt);
            uiPrintf("CompCPC1Cnt           %u\n", pStats->CompCPC1Cnt);
            uiPrintf("CompCPC2Cnt           %u\n", pStats->CompCPC2Cnt);
            uiPrintf("CompCPC3Cnt           %u\n", pStats->CompCPC3Cnt);

            uiPrintf("TxFiltered            %u\n", pStats->TxFiltered);
            uiPrintf("TotalRetries          %llu\n", pStats->TotalRetries);
            //xxxxPGSxxxx uiPrintf("RetryBins             %u\n", pStats->RetryBins);
            //xxxxPGSxxxx uiPrintf("shortFrameRetryBins   %u\n", pStats->shortFrameRetryBins);
            uiPrintf("swRetryTotalCnt       %u\n", pStats->swRetryTotalCnt);
            uiPrintf("swRetryMaxRetriesExceeded %u\n", pStats->swRetryMaxRetriesExceeded);

            /* Rates/Antenna/RSSI etc. */

            uiPrintf("ackRssi               %d\n", pStats->ackRssi);
            uiPrintf("txRateKb              %u\n", pStats->txRateKb);
            //xxxxPGSxxxx uiPrintf("AntCnt                %u\n", pStats->AntCnt);
            uiPrintf("AntSwCnt              %u\n", pStats->AntSwCnt);
            uiPrintf("tx6PowerInHalfDbm     %u\n", pStats->tx6PowerInHalfDbm);
            uiPrintf("maxTxPowerAvail       %u\n", pStats->maxTxPowerAvail);
            uiPrintf("noiseFloor            %u\n", pStats->noiseFloor);

            uiPrintf("<< WDCMSG_TARGET_STATS_UPDATE\n");
#endif

            status = wdcTriggerEventHandler(pTargInfo,
                                            WDCEVT_WLAN_STATS_UPDATE,
                                            0, statsData);
        }
        break;
    }

    case WDCMSG_CONNECTION_STATS_UPDATE: {
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;
    }

    case WDCMSG_BEACON_MISS_INTERRUPT: {
        A_UINT32       length;
        A_LONGSTATS    RxBeacons;

        wdcMsgReadData(wdcMsg, &cursor, &length, &RxBeacons);
        
        RxBeacons = fixEndian64(RxBeacons);
        status = wdcTriggerEventHandler(pTargInfo,
                                        WDCEVT_BEACON_MISS,
                                        0, (UCHAR *)&RxBeacons);
        break;
    }

    case WDCMSG_STORE_INFO: {
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;
    }

    case WDCMSG_RETRIEVE_INFO: {
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;
    }

    case WDCMSG_TARGET_BEACON_UPDATE: {
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;
    }

    case WDCMSG_ADHOC_MERGE_UPDATE: {
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;
    }

    case WDCMSG_TARGET_START:
    case WDCMSG_TARGET_STOP:
    case WDCMSG_UNBIND:
    case WDCMSG_TARGET_DISABLE:
    {
#if NDIS_WDM
        bQueueBack = FALSE;
        pTargInfo->rcvMsg[msgId] = wdcMsg;
        NdisSetEvent(&pTargInfo->rxMsgCompEvent[msgId]);
#endif
        break;
    }

    case WDCMSG_DEVICE_AVAILABLE: {
#if NDIS_WDM
        NdisSetEvent(&pTargInfo->msgCompEvent[msgOpcode - 1]);
#endif
        break;
    }

    case WDCMSG_DEVICE_UNAVAILABLE: {
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;
    }

    case WDCMSG_DEBUG_MESSAGE_NOTIFY: {
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;
    }

    case WDCMSG_TARGET_GET_CAPABILITY: {
#if NDIS_WDM
        A_UINT32            rcvCapSz;
        A_STATUS            status;
        A_UINT8             *pData;
        A_UINT32            *capSz;
        TARGET_CAP_DATA     *capData;
        A_UINT32            cursor = 0;

        capSz = pTargInfo->pPendCapSz[wdcMsg->msgId];
        capData = (TARGET_CAP_DATA *) pTargInfo->pendCapBuf[wdcMsg->msgId];

        if (capData) {
            /* The original caller is already gone because it is called from 
             * dispatch_level, we should maker sure the all memory we access in
             * here is resident memory instead of any temporary memory */
            wdcMsgReadParam(wdcMsg,&cursor,&rcvCapSz);
            ASSERT(rcvCapSz <= WDC_MSG_LENGTH_MAX - WDCMSG_HEADER_LENGTH - sizeof(A_UINT32));

            uiPrintf("OpCode = %x, msgLen = %d, msgId = %x, capSz = %x\n",
                wdcMsg->msgOpcode,wdcMsg->msgLength,wdcMsg->msgId,rcvCapSz);

            if (rcvCapSz) {
                /* Variable-sized capability data */
                wdcMsgSeekData(wdcMsg,&cursor,(void **)&pData);

                A_BCOPY(pData,capData,rcvCapSz);
                if (capSz) {
                    (*capSz) = rcvCapSz;
                }
            } else {
                /* Fixed-sized capability data */            
                wdcMsgReadParam(wdcMsg,&cursor,(A_UINT32 *)capData);            
                if (capSz) {
                    (*capSz) = 4;
                }        
            }
            pTargInfo->pPendCapSz[wdcMsg->msgId] = NULL;
            pTargInfo->pendCapBuf[wdcMsg->msgId] = 0;
        } else {
            pTargInfo->rcvMsg[wdcMsg->msgId] = wdcMsg;
            bQueueBack = FALSE;
        }

        NdisSetEvent(&pTargInfo->rxMsgCompEvent[wdcMsg->msgId]);        

#endif
        break;
    }
    case WDCMSG_TARGET_GET_STATUS: {
#if NDIS_WDM
        pTargInfo->rcvMsg[wdcMsg->msgId] = wdcMsg;

        NdisSetEvent(&pTargInfo->rxMsgCompEvent[wdcMsg->msgId]);
        bQueueBack = FALSE;
#endif
        break;
    }
    case WDCMSG_SEND_COMPLETE: {
	    TRANSMIT_INFO  txInfo = {0};
            A_UINT32       cursor = 0;
	    A_UINT8       *pData;  
	    A_UINT32       length;

	    wdcMsgReadData(wdcMsg, &cursor, &length, &txInfo);
	    ASSERT(sizeof(txInfo) == length);
	    ASSERT(!(sizeof(txInfo) % 4));

	    wdcDataSwizzle(&txInfo, sizeof(txInfo));
	    txInfo.hostFrameHandle  = (HOST_FRAME_HANDLE) wdcMsg->msgId;

	    status = wdcTriggerEventHandler(pTargInfo,
					    WDCEVT_SEND_COMPLETE,
					    sizeof(txInfo),
					    (A_UCHAR *) &txInfo);
	    ASSERT(bQueueBack == TRUE);
	    break;
    }
    case WDCMSG_FLUSH: {
        A_UINT32    disableEnableCounter;
        wdcMsgReadParam(wdcMsg,&cursor,&disableEnableCounter);

        status = wdcTriggerEventHandler(pTargInfo,
                                        WDCEVT_FLUSH_COMPLETE,
                                        0,
                                        (A_CHAR*)(&disableEnableCounter));
        break;
    }
    default: {
        ASSERT(0);
    }
    }
    /*
     * We're done with this message buffer.
     * Recycle it so that it can be used to receive the
     * next Control Message.
     */
     if (bQueueBack) {
        wdcCtrlMsgRecv(pTargInfo->targetMsgHandle,wdcMsg);
     }
}

#if NDIS_WDM

VOID
athUsbRecvIndication(IN  void     *pStubHandle,
                     IN  A_UINT8   epNum,
                     IN  A_UINT8  *buffer,
                     IN  A_UINT32  bytesReceived)
{
    extern void *theTargetMsgHandle;

    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)pStubHandle;
    WDC_MSG             *wdcMsg;
    WDC_MSG_OOB_DATA    *pOobData;

    if (epNum == MESSAGE_PIPE) {
        wdcMsg = (WDC_MSG *)buffer;
        pOobData = (WDC_MSG_OOB_DATA *)((A_UINT8 *)wdcMsg + WDC_MSG_LENGTH_MAX); 
        
        if (bytesReceived) {            
            /*
            * USB target side will use big endian
            */
            wdcMsgSwapHeader(wdcMsg);
            wdcCtrlMsgRecvInd(theTargetMsgHandle,wdcMsg);            
        } else {
            /*
             * To Do, need handle bus error
             */
        }
	    return;
    }

    if (epNum == DATA_PIPE) {
	    WDC_RXMSG     *wdcRxMsg = (WDC_RXMSG *) buffer;

	    if (bytesReceived) {
	        wdcRxMsgSwapHeader(wdcRxMsg);
	        ASSERT(wdcRxMsg->msgOpcode == WDCMSG_DATA_INDICATION);
	        wdcDataMsgRecvInd(theTargetMsgHandle, wdcRxMsg);
	    }  else {
	        /* XXX TBD */ 
	    }
	    return;
    }

    ASSERT(0);
}

VOID
athUsbSendConfirm(IN  void     *pStubHandle,
                  IN  A_UINT8   epNum,
                  IN  A_UINT8  *buffer,
                  IN  A_UINT32  bytesSent)
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)pStubHandle;

    if (epNum == MESSAGE_PIPE) {
	    WDC_MSG             *wdcMsg = (WDC_MSG *) buffer;
	    WDC_MSG_OOB_DATA    *pOobData;

        pOobData = (WDC_MSG_OOB_DATA *)((A_UINT8 *)wdcMsg + 
					WDC_MSG_LENGTH_MAX); 

	    // If a request was asynchronous (i.e. A_PENDING), we free
	    // the buffer.

        if (pOobData->bTimeOut == TRUE) {
            wdcCtrlMsgFree(wdcMsg);            
        } else {
            wdcMsgSwapHeader(wdcMsg);
            NdisSetEvent(&pTargInfo->txMsgCompEvent[wdcMsg->msgId]);
            if (bytesSent) {
                pOobData = (WDC_MSG_OOB_DATA *)((A_UINT8 *)wdcMsg + 
						WDC_MSG_LENGTH_MAX);
                if (pOobData->bAsynCall == TRUE) {
                    wdcCtrlMsgFree(wdcMsg);
                }
            }
	    }

    } else {
        WDC_TXMSG      *txMsg;
        TRANSMIT_INFO  txInfo = {0};

	    ASSERT(epNum == DATA_PIPE);

        txMsg = (WDC_TXMSG *) buffer;

        wdcTxMsgSwapHeader(txMsg);

        if ((bytesSent == 0 || txMsg->responseDesired == FALSE) &&
            (txMsg->msgOpcode == WDCMSG_SEND)) {
            
            txInfo.hostFrameHandle = (HOST_FRAME_HANDLE)txMsg->msgId;
            
            if (bytesSent == 0) {
                txInfo.status = TX_ERR;
            } else if (txMsg->responseDesired == FALSE) {
                txInfo.status = TX_OK;
            }

            wdcTriggerEventHandler(pTargInfo,
                                   WDCEVT_SEND_COMPLETE,
                                   sizeof(txInfo),
                                   (A_UCHAR *) &txInfo);
        }

	    wdcDataMsgSendCnf((TARGET_HANDLE) pTargInfo,
                          (WDC_TXMSG *) buffer);
    }
}

VOID
athUsbStatusIndication(IN  void     *pStubHandle,
                       IN  A_UINT32  status)
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)pStubHandle;

    wdcTriggerEventHandler(pTargInfo,
                    WDCEVT_STATUS_NOTIFY,
                    sizeof(A_UINT32),
                    (A_UCHAR *) &status);

    if ((status == ATHUSB_BUS_STATE_NORMAL) && (pTargInfo->busStatus != status)){
        A_UINT32            i;

        for (i = 0; i < pTargInfo->numRxMsgBuf; i ++) {
            wdcCtrlMsgRecv(pTargInfo->targetMsgHandle,pTargInfo->pRcvMsgBuf[i]);
        }

        for (i = 0; i < pTargInfo->numRxBufs; i++) {
            wdcDataMsgRecv(pTargInfo->targetMsgHandle, 
        	               pTargInfo->pRxMsgBuf[i]);
        }
    }

    pTargInfo->busStatus = status;
}

#endif

VOID athConvertChanValue(IN   CHAN_VALUES     *pChannel,
                        OUT  A_UINT16        *pChannelBand,
                        OUT  CHANNEL_FREQ    *pChannelFrequency,
                        OUT  PHY_MODS        *pPhyModulations)
{
    (*pChannelFrequency) = pChannel->channel;

    (*pChannelBand) = 0;
    if (IS_CHAN_2GHZ(pChannel->channelFlags)) {
        (*pChannelBand) |= CHANNEL_BAND_2GHz;
    }

    if (IS_CHAN_5GHZ(pChannel->channelFlags)) {
        (*pChannelBand) |= CHANNEL_BAND_5GHz;
    }

    (*pPhyModulations) = PHY_MOD_NONE;
    if (IS_CHAN_TURBO(pChannel->channelFlags)) {
        (*pPhyModulations) |= PHY_MOD_TURBO;
    }

    if (IS_CHAN_CCK(pChannel->channelFlags)) {
        (*pPhyModulations) |= PHY_MOD_CCK;
    }

    if (IS_CHAN_OFDM(pChannel->channelFlags)) {
        (*pPhyModulations) |= PHY_MOD_OFDM;
    }

    if (IS_CHAN_XR(pChannel->channelFlags)) {
        (*pPhyModulations) |= PHY_MOD_XR;
    }
}

void
wdcHostAvailable(
    IN HOST_HANDLE    hostHandle, 
    IN TARGET_HANDLE  targetHandle, 
    OUT A_STATUS     *pStatus) 
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    WDC_MSG             *wdcOutMsg;
    A_STATUS            status;
    ASSERT(pTargInfo);

    if (pStatus) {
        (*pStatus) = A_ERROR;
    }

    wdcOutMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle,
                                 WDCMSG_HOST_AVAILABLE);
    if (wdcOutMsg == NULL) {
        uiPrintf("No resource available\n");
        return;
    }

    /*
     * Send host driver version to the target
     */
    wdcMsgAddParam(wdcOutMsg, fixEndian(ATH_SW_VER_MAJOR));
    wdcMsgAddParam(wdcOutMsg, fixEndian(ATH_SW_VER_MINOR));
    wdcMsgAddParam(wdcOutMsg, fixEndian(ATH_SW_VER_PATCH));
    wdcMsgAddParam(wdcOutMsg, fixEndian(ATH_SW_VER_BUILD));

    status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcOutMsg);    

    if (status == A_TIMEOUT) {
        uiPrintf("Send WDCMSG_HOST_AVAILABLE message timeout\n");
        return;
    }

    if (status == A_OK) {
        wdcCtrlMsgFree(wdcOutMsg);
    }

    if (pStatus)
        (*pStatus) = status;

    return;
}


/*
 * wdcTargetSetPowerMode
 *
 * specifies a new value for a target-wide configuration attribute. 
 */
void
wdcTargetSetPowerMode(
    IN  TARGET_HANDLE       targetHandle,
    IN  TARGET_POWER_MODE   pwrMode,
    OUT A_STATUS            *pStatus)
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    WDC_MSG             *wdcMsg;
    A_STATUS            status;
    A_UINT32            i;

    ASSERT(pTargInfo);

    if (pStatus) {
        (*pStatus) = A_ERROR;
    }
    switch(pwrMode) {
        
        case TARGET_DEVICE_PWRSAVE : 
            /*
             * Send a WDCMSG_TARGET_SET_CONFIGURATION message to the target.
             */
            wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle, WDCMSG_TARGET_SET_PWR_MODE);
            if (wdcMsg == NULL) {
                uiPrintf("No resource available\n");
                return;
            }    

            wdcMsgAddParam(wdcMsg, fixEndian(pwrMode));

            /* Send the message to the device */
            status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);

            if ((status != A_PENDING) && (status != A_TIMEOUT)) {
                wdcCtrlMsgFree(wdcMsg);
            }   
            if ((status == A_PENDING) || (status == A_OK)) {
                /* 
                 * TODO : when target is capable of remote 
                 * wake up, put the USB in selective suspend, and prepare the 
                 * Stub layer for remote wakeup . From this state the device 
                 * wakeup by itself or could be wokenup by the host itself. 
                 * 
                 */
                status = A_OK;
            }
            break;
        
        case TARGET_DEVICE_SLEEP : 
            /*
             * Send a WDCMSG_TARGET_SET_CONFIGURATION message to the target.
             */
            wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle, WDCMSG_TARGET_SET_PWR_MODE);
            if (wdcMsg == NULL) {
                uiPrintf("No resource available\n");
                return;
            }    

            wdcMsgAddParam(wdcMsg, fixEndian(pwrMode));

            /* Send the Message to the target device */
            status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);

            if ((status != A_PENDING) && (status != A_TIMEOUT)) {
                wdcCtrlMsgFree(wdcMsg);
            }   
            if ((status == A_PENDING) || (status == A_OK)) {
                /* 
                 * TODO : Put the USB in selective suspend the device 
                 * will be waken up the host when it wants.
                 */
                status = A_OK;
            }
            break;

        case TARGET_DEVICE_PWRDN : 
            /*
             * Send a WDCMSG_TARGET_SET_CONFIGURATION message to the target.
             */
            wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle, WDCMSG_TARGET_SET_PWR_MODE);
            if (wdcMsg == NULL) {
                uiPrintf("No resource available\n");
                return;
            }    

            wdcMsgAddParam(wdcMsg, fixEndian(pwrMode));

            /* 
             * The host about to hibernate just inform the device 
             * about a pendind shudown 
             */
            status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);
            if ((status != A_PENDING) && (status != A_TIMEOUT)) {
                wdcCtrlMsgFree(wdcMsg);
            }   
            if ((status == A_PENDING) || (status == A_OK)) {
                status = A_OK;
            }
            break;
        
        case TARGET_DEVICE_AWAKE : 
            /*
             * Send a WDCMSG_TARGET_SET_CONFIGURATION message to the target.
             */
            wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle, WDCMSG_TARGET_SET_PWR_MODE);
            if (wdcMsg == NULL) {
                uiPrintf("No resource available\n");
                return;
            }    

            wdcMsgAddParam(wdcMsg, fixEndian(pwrMode));

            /* 
             * TODO: If the target was put in selective suspend wake up 
             * the device and then send the message, if the device was 
             * able wake up the host, then cancel the pending device
             * wakeup operation
             */ 
            status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);
            if ((status != A_PENDING) && (status != A_TIMEOUT)) {
                wdcCtrlMsgFree(wdcMsg);
            }   
            if ((status == A_PENDING) || (status == A_OK)) {
                status = A_OK;
            }
            break;

        case TARGET_DEVICE_SUSPEND :
            status = athUsbSuspend(0,pTargInfo->usbHandle);
            pTargInfo->busStatus = ATHUSB_BUS_STATE_SUSPEND;
            break;

        case TARGET_DEVICE_RESUME:
            status = athUsbResume(0,pTargInfo->usbHandle);
            if (status == A_OK) {
                for (i = 0; i < pTargInfo->numRxMsgBuf; i ++) {
                    wdcCtrlMsgRecv(pTargInfo->targetMsgHandle,pTargInfo->pRcvMsgBuf[i]);
                }

                for (i = 0; i < pTargInfo->numRxBufs; i++) {
                    wdcDataMsgRecv(pTargInfo->targetMsgHandle, 
        	                       pTargInfo->pRxMsgBuf[i]);
                }

                pTargInfo->busStatus = ATHUSB_BUS_STATE_NORMAL;
            } else {
                pTargInfo->busStatus = ATHUSB_BUS_STATE_SUSPEND;
            }

            break;
        
        default:
            ASSERT(0);
    }

    if (pStatus) {
        (*pStatus) = status;
    }
}

void
wdcSetLedState(
    IN  TARGET_HANDLE       targetHandle,
    IN  TARGET_LED_STATE    ledState)
{
    PER_TARGET_INFO  *pTargInfo;
    A_STATUS          status;
    WDC_MSG          *wdcMsg;

    pTargInfo = (PER_TARGET_INFO *) targetHandle;

    wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle,
                              WDCMSG_TARGET_SET_LED_STATE);
    if (wdcMsg == NULL) {
        uiPrintf("No resource available\n");
        return;
    }

    wdcMsgAddParam(wdcMsg, fixEndian(ledState));

    status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);

    if (status == A_OK) {
        wdcCtrlMsgFree(wdcMsg);
    }

    return;
}

void
wdcBmissAck(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_BOOL              bEnable)
{
    PER_TARGET_INFO  *pTargInfo;
    A_STATUS          status;
    WDC_MSG          *wdcMsg;
    A_UINT32          data = (A_UINT32)bEnable;

    pTargInfo = (PER_TARGET_INFO *) targetHandle;

    wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle,
                              WDCMSG_BEACON_MISS_ACK);
    if (wdcMsg == NULL) {
        uiPrintf("No resource available\n");
        return;
    }

    wdcMsgAddParam(wdcMsg, fixEndian(data));

    status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);

    if (status == A_OK) {
        wdcCtrlMsgFree(wdcMsg);
    }

    return;
}

/**************************************************************
 * wdcApplyMKKIrreversibleUpdate
 *
 */
A_STATUS
wdcApplyMKKIrreversibleUpdate(TARGET_HANDLE targetHandle, A_UINT32 enOddU1, A_UINT32 undoUpdate)
{
    PER_TARGET_INFO  *pTargInfo;
    A_STATUS          status;
    WDC_MSG          *wdcMsg;

    pTargInfo = (PER_TARGET_INFO *) targetHandle;

    wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle,
                              WDCMSG_TARGET_APPLY_MKK_IRREV_UPDATE);
    if (wdcMsg == NULL) {
        uiPrintf("No resource available\n");
        return A_ERROR;
    }

    wdcMsgAddParam(wdcMsg, fixEndian(enOddU1));
    wdcMsgAddParam(wdcMsg, fixEndian(undoUpdate));

    status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);

    if (status == A_OK) {
        wdcCtrlMsgFree(wdcMsg);
    }
    return A_OK;
}
