/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/wdc/hostMsg.c#2 $
 *
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 */

#include "wlantype.h"
#include "wdcApi.h"
#include "wdcMsg.h"
#include "wdcMsgApi.h"
#include "hostSend.h"
#include "tgtInfo.h"
#if NDIS_WDM
#include "athusbapi.h"
#endif

struct target_msg_info theTargetMsgStruct;
TARGET_MSG_HANDLE theTargetMsgHandle = &theTargetMsgStruct;

TARGET_MSG_HANDLE
wdcMsgInit(TARGET_HANDLE targetHandle, WDCMSG_DISPATCH_FN *pDispatchFn)
{
    theTargetMsgStruct.targetHandle = targetHandle;
    theTargetMsgStruct.pDispatchFn = pDispatchFn;

    return theTargetMsgHandle;
}

void
wdcCtrlMsgRecvInd(TARGET_MSG_HANDLE targetMsgHandle, WDC_MSG *wdcMsg)
{
    ASSERT(targetMsgHandle == theTargetMsgHandle);

    theTargetMsgStruct.pDispatchFn(theTargetMsgStruct.targetHandle, wdcMsg);
}

LOCAL A_UINT32 lastGenerationNum = 0;

void
wdcDataMsgRecvInd(TARGET_MSG_HANDLE targetMsgHandle, WDC_RXMSG *wdcRxMsg) 
{
    EVENT_HANDLER *pHandler = NULL;
    A_STATUS      status;
    WDC_RX_INFO   *wdcRxInfo;

    ASSERT(targetMsgHandle == theTargetMsgHandle);

    /*
     * The Generation Number changes across SetChannel boundaries.
     * Before we notify the Host driver of the new data, let it
     * know that all future Data Indications are for the new channel.
     */

    /* Doesn't consider wrap; but that's not of practical concern. */
    ASSERT(wdcRxMsg->generationNum >= lastGenerationNum);

    if (wdcRxMsg->generationNum != lastGenerationNum) {
        (void)wdcTriggerEventHandler(targetMsgHandle->targetHandle,
	                             WDCEVT_RX_IN_SYNC,
	                             0,
	                             NULL);
        lastGenerationNum = wdcRxMsg->generationNum;
    }
    /*
     * Tricky: Re-use the start of the Rx Msg buffer to hold 
     * WDC_RX_INFO.  We're just taking advantage of some available
     * space so that we can avoid malloc'ing WDC_RX_INFO structures.
     */
    ASSERT(WDC_RXMSG_HEADER_LENGTH >= sizeof(WDC_RX_INFO));
    wdcRxInfo = (WDC_RX_INFO *)wdcRxMsg;

    wdcRxInfo->dataLength = wdcRxMsg->rxMsgLength - WDC_RXMSG_HEADER_LENGTH;
    wdcRxInfo->receiveInfo= &wdcRxMsg->receiveInfo;
    wdcRxInfo->rxData     = (A_CHAR *)wdcRxMsg->rxData;

    ASSERT(wdcRxInfo->receiveInfo);
    ASSERT(wdcRxInfo->rxData);
    
    status = wdcTriggerEventHandler(targetMsgHandle->targetHandle,
	                            WDCEVT_DATA_INDICATION,
	                            sizeof(*wdcRxInfo),
	                            (A_UCHAR *)wdcRxInfo);
    ASSERT(status == A_OK);

    return;
}

void
wdcCtrlMsgRecv(TARGET_MSG_HANDLE targetMsgHandle, WDC_MSG *wdcMsg)
{
#if NDIS_WDM
    PER_TARGET_INFO   *targetHandle;

    targetHandle = targetMsgHandle->targetHandle;

    athUsbDrvReceive(targetHandle->usbHandle,
                     MESSAGE_PIPE,
                     (A_UINT8 *)wdcMsg);
#endif
}


void
wdcDataMsgRecv(TARGET_MSG_HANDLE targetMsgHandle, WDC_RXMSG *wdcRxMsg) 
{
#if NDIS_WDM
    PER_TARGET_INFO   *targetHandle;

    targetHandle = targetMsgHandle->targetHandle;

    athUsbDrvReceive(targetHandle->usbHandle,
		     DATA_PIPE,
                     (A_UINT8 *)wdcRxMsg);
#endif
}


A_STATUS
wdcDataMsgSend(TARGET_MSG_HANDLE targetMsgHandle, WDC_TXMSG *wdcTxMsg)
{
#if NDIS_WDM
    KIRQL              curIrql;
    PER_TARGET_INFO   *targetHandle;
    A_UINT32           msgId;

    athGetCurrentIrql(curIrql);
    targetHandle = targetMsgHandle->targetHandle;

    /*
     * USB target side will use big endian
     */
    wdcTxMsgSwapHeader(wdcTxMsg);

#if !defined(DO_ZERO_TERM)
if (athUsbDrvSend(targetHandle->usbHandle,
                      DATA_PIPE,
                      (A_UINT8 *)wdcTxMsg) != A_OK)
    {
        return A_ERROR;
    }
#else
    if (athUsbDrvSend(targetHandle->usbHandle,
                      DATA_PIPE,
                      (A_UINT8 *)wdcTxMsg,
                      fixEndian(wdcTxMsg->txMsgLength)) != A_OK)
    {
        return A_ERROR;
    }
#endif

    return A_PENDING;
#endif
}


A_STATUS
wdcCtrlMsgSend(TARGET_MSG_HANDLE targetMsgHandle, WDC_MSG *wdcMsg)
{
#if NDIS_WDM
    KIRQL              curIrql;
    PER_TARGET_INFO   *targetHandle;
    A_UINT32           msgId;
    WDC_MSG_OOB_DATA  *pOobData;

    athGetCurrentIrql(curIrql);
    targetHandle = targetMsgHandle->targetHandle;

    msgId = wdcMsg->msgId;    

    NdisResetEvent(&targetHandle->txMsgCompEvent[msgId]);

    /*
     * USB target side will use big endian
     */
    wdcMsgSwapHeader(wdcMsg);

    /*
     * Use out of band data to record whether we use sync or async call
     */
    pOobData = (WDC_MSG_OOB_DATA *)((A_UINT8 *)wdcMsg + WDC_MSG_LENGTH_MAX);
    if (curIrql < DISPATCH_LEVEL) {
        pOobData->bAsynCall = FALSE;
    } else {
        pOobData->bAsynCall = TRUE;
    }

    pOobData->bTimeOut = FALSE;
#if !defined(DO_ZERO_TERM)
    if (athUsbDrvSend(targetHandle->usbHandle,
                      MESSAGE_PIPE,
                      (A_UINT8 *)wdcMsg) != A_OK)
    {
        return A_ERROR;
    }
#else
    if (athUsbDrvSend(targetHandle->usbHandle,
                      MESSAGE_PIPE,
                      (A_UINT8 *)wdcMsg,
                      fixEndian(wdcMsg->msgLength)) != A_OK)
    {
        return A_ERROR;
    }
#endif

    /* If we are calling from passive_level, we can wait send complete before we return*/
    if (curIrql < DISPATCH_LEVEL) {
        if (NdisWaitEvent(&targetHandle->txMsgCompEvent[msgId],MAX_MSGWRITE_WAIT_TIME)) {            
            return A_OK;
        } else {
            pOobData->bTimeOut = TRUE;
            return A_TIMEOUT;
        }        
    } else {
        return A_PENDING;
    }
#endif
}

WDC_MSG *
wdcCtrlMsgCreate(IN TARGET_MSG_HANDLE targetMsgHandle,
                 IN WDC_MSG_OPCODE msgOpcode)
{
    WDC_MSG *wdcMsg;
    A_UINT32 size;
    PER_TARGET_INFO   *pTargInfo;

    ASSERT(msgOpcode > WDCMSG_INVAL);
    ASSERT(msgOpcode < WDCMSG_MAX);

    pTargInfo = (PER_TARGET_INFO *)targetMsgHandle->targetHandle;

    size = WDC_MSG_LENGTH_MAX + sizeof(WDC_MSG_OOB_DATA);
    wdcMsg = (WDC_MSG *)A_DRIVER_MALLOC(size);
    if (wdcMsg == NULL) {
        return wdcMsg;
    }

    A_MEM_ZERO(wdcMsg,size);
    wdcMsg->msgOpcode = msgOpcode;
    wdcMsg->msgLength = WDCMSG_HEADER_LENGTH;

#if NDIS_WDM
    wdcMsg->msgId = A_ATOMIC_READ(&pTargInfo->msgId);

    InterlockedIncrement(&pTargInfo->msgId);
    if (athReadAtomic(pTargInfo->msgId) >= MAX_MSG_ID_COUNT) {
        InterlockedExchange(&pTargInfo->msgId,0);
    }
#endif

    return wdcMsg;
}

void
wdcCtrlMsgFree(WDC_MSG *wdcMsg)
{
    A_DRIVER_FREE(wdcMsg, WDC_MSG_LENGTH_MAX + sizeof(WDC_MSG_OOB_DATA));
}

void
wdcMsgAddParam(WDC_MSG *wdcMsg, A_UINT32 param)
{
    A_UINT32 *pParam;

    pParam = (A_UINT32 *)((A_UINT32)wdcMsg + wdcMsg->msgLength);

    /* Store data */
    *pParam = param;

    /* Adjust message length */
    wdcMsg->msgLength += sizeof(A_UINT32);
}

void
wdcMsgAddData(WDC_MSG *wdcMsg, A_UINT32 length, void *data)
{
    char       *pData;
    A_UINT32   paddedLength;

    /* Round length up 4-bytes */
    paddedLength = (length + 3) & ~0x3;

    ASSERT(paddedLength + sizeof(A_UINT32) + wdcMsg->msgLength <= WDC_MSG_LENGTH_MAX);
    pData = (char *)((A_UINT32)wdcMsg + wdcMsg->msgLength);

    /* Save length of this data item */
    *((A_UINT32 *)pData) = fixEndian(length);
    pData += sizeof(A_UINT32);

    /* Save data itself */
    A_BCOPY(data, pData, length);

    /* Adjust total message length to account for size, data, and padding. */
    wdcMsg->msgLength += sizeof(A_UINT32) + paddedLength;
}

void 
wdcMsgSwapHeader(WDC_MSG *wdcMsg)
{
    wdcMsg->msgOpcode = fixEndian(wdcMsg->msgOpcode);
    wdcMsg->msgLength = fixEndian(wdcMsg->msgLength);
}

void
wdcTxMsgSwapHeader(WDC_TXMSG *wdcTxMsg)
{
    wdcTxMsg->txMsgLength = fixEndian(wdcTxMsg->txMsgLength);
    wdcTxMsg->bufferLength = fixEndian(wdcTxMsg->bufferLength);
    wdcTxMsg->msgOpcode   = fixEndian(wdcTxMsg->msgOpcode);
    wdcTxMsg->txQueueId   = fixEndian(wdcTxMsg->txQueueId);
    wdcTxMsg->txConnId    = fixEndian(wdcTxMsg->txConnId);
}


void
wdcRxMsgSwapHeader(WDC_RXMSG *wdcRxMsg) 
{
    wdcDataSwizzle((void *) wdcRxMsg,  WDC_RXMSG_HEADER_LENGTH);
}

void
wdcDataSwizzle(void *pData, A_UINT32 length) {
    int i;
    int msg32Length = length >> 2;
    A_UINT32 *msg32 = (A_UINT32 *)pData;

    ASSERT(!(length % 4));

    for (i=0; i<msg32Length; i++) {
        msg32[i] = fixEndian(msg32[i]);
    }
    return;
}



/*
 * Prepare a WDC Message for data, which will be copied into the
 * buffer by the caller.  Caller (and receiver) are responsible
 * for handling any endianness issues.  The data must subsequently
 * be commited via TARG_wdcMsgCommitData.
 *
 * Returns:
 *   pData     - the start address where caller's data should be copied
 *   maxLength - the maximum length of data that can be accommodated
 */
void
wdcMsgPrepData(WDC_MSG   *wdcMsg,
               void *    *pData,
               A_UINT32  *pMaxLength)
{
    char *msgData;

    msgData = (char *)((A_UINT32)wdcMsg + wdcMsg->msgLength);

    if (pData) {
        *pData = msgData;
    }

    if (pMaxLength) {
        *pMaxLength = WDC_MSG_LENGTH_MAX - wdcMsg->msgLength;
    }
}

void
wdcMsgCommitData(WDC_MSG   *wdcMsg,
                 A_UINT32  length)
{
    A_UINT32   paddedLength;

    /* Round length up 4-bytes */
    paddedLength = (length + 3) & ~0x3;

    ASSERT(paddedLength + wdcMsg->msgLength <= WDC_MSG_LENGTH_MAX);

    /* Adjust total message length to account for size, data, and padding. */
    wdcMsg->msgLength += paddedLength;
}



/************************************************************
 * Functions to extract data from a message buffer.
 *
 * For fixed-sized data, use TARG_wdcMsgReadParam.
 * For variable-sized data, there are two choices:
 * 1) Use TARG_wdcMsgReadData, which copies data from
 *    the message buffer into a user-supplied buffer
 *    and updates your "position" in the buffer.
 *    OR
 * 2) Use TARG_wdcMsgSeekData, manually copy the data,
 *    and then TARG_wdcMsgConsumeData.  Seek returns a
 *    pointer into the message buffer, but it doesn't
 *    copy any data and it doesn't update your position.
 *    Consume updates your position.
 */


/*
 * Read a single parameter from a message.
 */
void
wdcMsgReadParam(WDC_MSG        *wdcMsg,
                A_UINT32       *pCursor,
                A_UINT32       *pParam)
{
    A_UINT32       param;
    A_UINT32       newCursor;

    ASSERT(pParam);

    if (pCursor) {
        newCursor = *pCursor;
    } else {
        newCursor = 0;    
    }

    ASSERT(newCursor <= wdcMsg->msgLength - WDCMSG_HEADER_LENGTH);

    param = *(A_UINT32 *)((A_UINT32)wdcMsg + WDCMSG_HEADER_LENGTH + newCursor);

    newCursor += sizeof(A_UINT32);

    param = fixEndian(param);

    *pCursor = newCursor;
    *pParam = param;
}

/*
 * Read variable length data from a message.
 */
void
wdcMsgReadData(WDC_MSG        *wdcMsg,
               A_UINT32       *pCursor,
               A_UINT32       *pLength,
               void           *pData)
{
    A_UINT32       length;
    A_UINT32       paddedLength;
    A_UINT32       newCursor;
    char           *msgData;

    ASSERT(pData);

    if (pCursor) {
        newCursor = *pCursor;
    } else {
        newCursor = 0;
    }

    wdcMsgReadParam(wdcMsg, &newCursor, &length);
    ASSERT(length > 0);

    /* Round length up 4-bytes */
    paddedLength = (length + 3) & ~0x3;

    msgData = (char *)((A_UINT32)wdcMsg + WDCMSG_HEADER_LENGTH + newCursor);
    A_BCOPY(msgData, (char *)pData, length);
    newCursor += paddedLength;

    if (pLength) {
        *pLength = length;
    }

    if (pCursor) {
        *pCursor = newCursor;
    }
}

/*
 * Seek to start of variable-sized data, which will be copied from the
 * buffer by the caller.  Caller and sender are responsible for handling
 * any endianness issues.  The data must subsequently be consumed via
 * wdcMsgConsumeData.  The cursor is modified by ConsumeData, not
 * by SeekData.
 *
 * Returns:
 *   ppData    - a pointer into the message buffer where data begins
 */
void
wdcMsgSeekData(WDC_MSG        *wdcMsg,
               A_UINT32       *pCursor,
               void           **ppData)
{
    A_UINT32       cursor;
    char           *msgData;

    ASSERT(pCursor);

    cursor = *pCursor;

    msgData = (char *)((A_UINT32)wdcMsg + WDCMSG_HEADER_LENGTH + cursor);

    if (ppData) {
        *ppData = msgData;
    }
}

/*
 * Consume variable-sized data -- in other words, move the cursor --
 * without copying any data.
 */
void
wdcMsgConsumeData(WDC_MSG        *wdcMsg,
                  A_UINT32       *pCursor,
                  A_UINT32       length)
{
    A_UINT32       paddedLength;
    A_UINT32       newCursor;

    ASSERT(pCursor);

    newCursor = *pCursor;

    /* Round length up 4-bytes */
    paddedLength = (length + 3) & ~0x3;

    newCursor += paddedLength;

    *pCursor = newCursor;
}

