
/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/host/wdc/hostSend.c#3 $
 *
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 */
#include "wlandrv.h"
#include "wlanchannel.h"
#include "wlantype.h"
#include "wdcApi.h"
#include "wdcMsg.h"
#include "wdcMsgApi.h"
#include "hostSend.h"
#include "tgtInfo.h"
#include "usbstub.h"
#if NDIS_WDM
#include "athusbapi.h"
#include "wlanext.h"
#endif

/* 
 * Local Definitions 
 */

#define NUM_INT_TX_DESC     128                   /* Number of intermediate tx buffers */

/* 
 * Local data structures 
 */



/* 
 * Local function prototypes 
 */

LOCAL A_STATUS
allocateIntTxBuffers(
    IN TARGET_HANDLE targetHandle);

LOCAL void
freeIntTxBuffers(
    IN TARGET_HANDLE targetHandle);

LOCAL A_STATUS
intDescQInit(INT_QUEUE_DETAILS *pQueue);

LOCAL INT_DESC *
intDescQGetHead(INT_QUEUE_DETAILS *pQueue);

LOCAL INT_DESC *
intDescQGetTail(INT_QUEUE_DETAILS *pQueue);

LOCAL INT_DESC *
intDescQPopHead(INT_QUEUE_DETAILS *pQueue);

LOCAL void
intDescQPushTail(INT_QUEUE_DETAILS *pQueue, INT_DESC *pDesc, INT_DESC *pTail);

LOCAL void
intDescQPushHead(INT_QUEUE_DETAILS *pQueue, INT_DESC *pDesc);

LOCAL INT_DESC *
getIntDescFromQueue(INT_QUEUE_DETAILS *pQueue, INT_DESC *pDesc);

void wdcDataMsgSendCnf(
    IN TARGET_HANDLE   targetHandle,
    IN WDC_TXMSG      *wdcTxMsg)
{
    PER_TARGET_INFO   *pTargInfo = (PER_TARGET_INFO *) targetHandle;
    INT_DESC          *pDesc;
    INT_DESC          *pDescPendQ; 

    pDesc = (INT_DESC *) ((A_UINT32) wdcTxMsg - sizeof(INT_DESC));

#ifdef DEBUG
    A_MEM_ZERO(wdcTxMsg, WDC_TXMSG_LENGTH);
#endif

    ASSERT((WDC_TXMSG *) pDesc->pBufferVirtPtr == wdcTxMsg);
	
    pDescPendQ = getIntDescFromQueue(&pTargInfo->intTxPendQueue,pDesc);
    
    if (pDescPendQ == pDesc) {
        intDescQPushTail(&pTargInfo->intTxFreeQueue, pDesc, pDesc);
    }    

    pTargInfo->bTargetHang = FALSE;
    pTargInfo->NoRespCount = 0;

    if (pTargInfo->intTxPendQueue.qFrameCount == 0) {
        wdcTriggerEventHandler(targetHandle,
                               WDCEVT_NO_TX_DATA,
                               0,
                               NULL);
    }
}


/*
 * Host-Side implementation of transmit related functions.
 */

/******************************************************************************
 * WDC API implementation
 */


/*
 * wdcSend
 * 
 * Initiate a frame transmission aynchronously. The bIndicateStatus controls
 * whether the completion status needs to be indicated.
 * If bIndicateStatus = TRUE the callback function will be invoked until 
 * the WLAN target transmission completed. Otherwise the callback function
 * is invoked after the bus transmission is completed and the frame resource
 * is freed.
 *
 * IN PARAMETERS:
 *  hostFrameHandle     - handle for the host description of the frame,
 *                        used when indicating the tx compleiton to the host
 *                        and as reference for host to release the frame resource
 *  wdcTxQueueId        - handle for the target queue for transmission
 *  wdcConnId           - the target connection handle for the destionation
 *  pBufferDesc         - frame scatter-gether buffer link-list
 *  bIndicateTransmitStatus - completion status indication flag
 */
A_STATUS
wdcSend(IN  TARGET_HANDLE       targetHandle,
	IN  HOST_FRAME_HANDLE   hostFrameHandle,
	IN  A_UINT32            wdcTxQueueId,
	IN  A_UINT32            wdcConnId,
	IN  WDC_BUFFER_DESC     *pBufferDesc,
	IN  A_BOOL              bIndicateTransmitStatus) 
{
    PER_TARGET_INFO   *pTargInfo       = (PER_TARGET_INFO *) targetHandle;
    WDC_BUFFER_DESC   *pBufferDescTmp;
    INT_DESC          *pIntDesc;
    A_UINT32          *txData;
    WDC_TXMSG         *txMsg;
    
    pIntDesc = intDescQPopHead(&pTargInfo->intTxFreeQueue);
    ASSERT(pIntDesc);    

    if (!pIntDesc) 
	    return A_ERROR;

    ASSERT(pIntDesc->pNextVirtPtr == NULL);

    txMsg  = (WDC_TXMSG *) pIntDesc->pBufferVirtPtr;
    txData = txMsg->txData;

    pBufferDescTmp = pBufferDesc;
    txMsg->bufferLength = 0;

    do {
        /* Verify that buffer data will fit in the intermediate TX buffer */
        if (txMsg->bufferLength + pBufferDescTmp->bufferLength > WDC_TXMSG_DATA_LENGTH) {
            ASSERT(0); /* Should never happen in practice */
            intDescQPushTail(&pTargInfo->intTxFreeQueue, pIntDesc, pIntDesc);
            return A_ERROR;
        }

        A_DRIVER_BCOPY((void *) pBufferDescTmp->pBuffer,
                       (void *) txData, 
                       pBufferDescTmp->bufferLength);
        txData = (A_UINT32 *) ((A_UINT32) txData + 
                               pBufferDescTmp->bufferLength);
        txMsg->bufferLength += pBufferDescTmp->bufferLength;
        pBufferDescTmp = pBufferDescTmp->pNext;
    } while(pBufferDescTmp != NULL);
    
    txMsg->txMsgLength = pBufferDesc->frameLength + WDC_TXMSG_HEADER_LENGTH;
    txMsg->msgId       = (A_UINT32) hostFrameHandle;
    txMsg->msgOpcode   = WDCMSG_SEND;
    txMsg->txQueueId   = wdcTxQueueId;
    txMsg->txConnId    = wdcConnId;
    txMsg->responseDesired = bIndicateTransmitStatus;

    intDescQPushTail(&pTargInfo->intTxPendQueue, pIntDesc, pIntDesc);
    
    return wdcDataMsgSend(pTargInfo->targetMsgHandle, txMsg);
}

/*
 * Check whether our target is hang or not
 */
A_BOOL
wdcCheckForHang(TARGET_HANDLE  targetHandle)
{
    PER_TARGET_INFO     *pTargInfo   = (PER_TARGET_INFO *)targetHandle;

    if (pTargInfo->busStatus != ATHUSB_BUS_STATE_NORMAL) {
        /*
         * Device isn't ready, don't need check for hange
         */
        return FALSE;
    }

    if (pTargInfo->bTargetHang) {
        if (intDescQGetHead(&pTargInfo->intTxPendQueue)) {
            pTargInfo->NoRespCount ++;

            if (pTargInfo->NoRespCount > 30) {
                /*
                 * still have some data in the queu and we don't receive
                 * completion in pass 60 second, target appear hang
                 */
                athUsbAbortAndReset(0,pTargInfo->usbHandle);
                pTargInfo->bTargetHang = FALSE;
                pTargInfo->NoRespCount = 0;
                uiPrintf("No Response from target over 60 second\n");
                return TRUE;
            }
        } else {
            pTargInfo->bTargetHang = FALSE;
            pTargInfo->NoRespCount = 0;
            return FALSE;
        }
    } else {
        if (intDescQGetHead(&pTargInfo->intTxPendQueue)) {
            pTargInfo->bTargetHang = TRUE;
        }
    }

    return FALSE;
}

/*
 * Target "barrier" support.  This is all rather hacky.
 * The USB transport uses multiple pipes -- one for WDC Control messages
 * and one (or more) for data messages.  Messages sent from Host to Target
 * always arrive in order as long as they're all on a single pipe.  Between
 * pipes, USB provides no guarantees.
 */

/*
 * Block Target processing of wdcSends until do_wdcCtrlBarrier is called.
 */
A_STATUS
do_wdcDataBarrier(TARGET_HANDLE  targetHandle)
{
    PER_TARGET_INFO   *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    INT_DESC          *pIntDesc;
    A_STATUS          status;
    WDC_TXMSG         *txMsg; 

    pIntDesc = intDescQPopHead(&pTargInfo->intTxFreeQueue);
    ASSERT(pIntDesc);    

    if (!pIntDesc) 
	    return A_ERROR;

    ASSERT(pIntDesc->pNextVirtPtr == NULL);

    txMsg  = (WDC_TXMSG *) pIntDesc->pBufferVirtPtr;

    txMsg->txMsgLength     = WDC_TXMSG_HEADER_LENGTH;
    txMsg->msgId           = 0;
    txMsg->msgOpcode       = WDCMSG_FLUSH;
    txMsg->txQueueId       = 0;
    txMsg->txConnId        = 0;
    txMsg->responseDesired = FALSE;

    intDescQPushTail(&pTargInfo->intTxPendQueue, pIntDesc, pIntDesc);
    
    status = wdcDataMsgSend(pTargInfo->targetMsgHandle, txMsg);

    if (status != A_PENDING) {
        intDescQPushTail(&pTargInfo->intTxFreeQueue, pIntDesc, pIntDesc);
        return A_ERROR;
    }

    return A_OK;
}

/* 
 * Block Target processing of Control messages until do_wdcDataBarrier
 * is called.
 */
A_STATUS
do_wdcCtrlBarrier(TARGET_HANDLE  targetHandle, A_UINT32 disableEnableCounter)
{
    PER_TARGET_INFO   *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    WDC_MSG           *wdcMsg;
    A_STATUS          status;

    wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle, WDCMSG_FLUSH);
    if (wdcMsg == NULL) {
        return A_ERROR;
    }    

    wdcMsgAddParam(wdcMsg, fixEndian(disableEnableCounter));

    status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);
    if (status != A_PENDING) {
        wdcCtrlMsgFree(wdcMsg);
        return A_ERROR;
    }   

    return A_OK;
}

/* Guts of wdcFlush.  Intended to be called from within the WDC itself. */
A_STATUS
do_wdcFlush(TARGET_HANDLE  targetHandle, A_UINT32 disableEnableCounter)
{
    A_STATUS   status;

    status = do_wdcDataBarrier(targetHandle);
    if (status != A_OK) {
        return status;
    }

    status = do_wdcCtrlBarrier(targetHandle, disableEnableCounter);
    if (status != A_OK) {
        return status;
    }

    return A_OK;
}

A_STATUS
wdcFlush(TARGET_HANDLE  targetHandle, A_UINT32 disableEnableCounter)
{
    return do_wdcFlush(targetHandle, disableEnableCounter);
}


/*
 * wdcSetupTxQueue
 *  
 * Set up a target TX queue with specified configuration attributes
 *
 */
void
wdcSetupTxQueue(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcTxQueueId,
    IN  TXQ_ATTRIBUTES      *pTxQueueInfo)
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    TARGET_MSG_HANDLE   targetMsgHandle = pTargInfo->targetMsgHandle;
    WDC_MSG             *wdcMsg;
    A_STATUS            status;
    A_UINT32            value;

    /*
     * Send a WDCMSG_SETUP_TX_QUEUE message to the target.
     */
    wdcMsg = wdcCtrlMsgCreate(pTargInfo->targetMsgHandle, WDCMSG_SETUP_TX_QUEUE);
    if (wdcMsg == NULL) {
        return;
    }

    value = (A_UINT32)wdcTxQueueId;
    wdcMsgAddParam(wdcMsg, fixEndian(value));

    /* xxxxPGSxxxx- TEMP
     * Currently pass the TXQ_ATTRIBUTES as separate fields.
     * This doesn't work well for when TXQ_ATTRIBUTES is modified, but 
     * we will be moving away from TXQ_ATTRIBUTES to a single parameter model.
     * So this is ok for now (and I don't have to worry about endianess problems).
     */
    value = (A_UINT32)pTxQueueInfo->priority;
    wdcMsgAddParam(wdcMsg, fixEndian(value));

    value = (A_UINT32)pTxQueueInfo->aifs;
    wdcMsgAddParam(wdcMsg, fixEndian(value));

    value = (A_UINT32)pTxQueueInfo->logCwMin;
    wdcMsgAddParam(wdcMsg, fixEndian(value));

    value = (A_UINT32)pTxQueueInfo->logCwMax;
    wdcMsgAddParam(wdcMsg, fixEndian(value));

    value = (A_UINT32)pTxQueueInfo->burstTime;
    wdcMsgAddParam(wdcMsg, fixEndian(value));

    value = (A_UINT32)pTxQueueInfo->compression;
    wdcMsgAddParam(wdcMsg, fixEndian(value));

    value = (A_UINT32)pTxQueueInfo->qosMode;
    wdcMsgAddParam(wdcMsg, fixEndian(value));

    status = wdcCtrlMsgSend(pTargInfo->targetMsgHandle, wdcMsg);

    if (status == A_OK) {
        wdcCtrlMsgFree(wdcMsg);
    } 
}


#if 0
/* xxxxPGSxxx- TODO */
/*
 * wdcUpdateTxQueueAttribute
 * 
 * Update a TX queue with a new (set) of attributes
 */
void
wdcUpdateTxQueueAttribute(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcTxQueueId,
    IN  TXQ_ATTRIBUTE       attribute,
    IN  A_UINT32            value
)
{

//    AR_DEV_INFO        *pArDev = (AR_DEV_INFO *) targetHandle;
    TXQ_ATTRIBUTES     *pTxQueueAttributes;
//    HAL_TX_QUEUE_INFO  queueParam;
    AR_QUEUE_INFO       *pArQueueInfo;

    ASSERT(wdcTxQueueId < HAL_NUM_TX_QUEUES);

    pArQueueInfo = &pArDev->txHwQueues[wdcTxQueueId];
    pTxQueueAttributes = &((AR_QUEUE_INFO *)pArQueueInfo)->wdc;

    switch (attribute) {
    case PRIORITY:
        pTxQueueAttributes->priority = value;
        break;

    case AIFS:
        pTxQueueAttributes->aifs = value;
        break;

    case LOG_CW_MIN:
        pTxQueueAttributes->logCwMin = value;
        break;

    case LOG_CW_MAX:
        pTxQueueAttributes->logCwMax = value;
        break;

    case BURST_TIME:
        pTxQueueAttributes->burstTime = value;
        break;

    case COMPRESSION:
        pTxQueueAttributes->compression = value;
        break;

    default:
        break;
    }
   
}
#endif

/*****************************************************************************
 * Host Stub functions (public, but not part of the WDC API)
 */

/*
 * hsInitTx - initialize host stub for transmit.
 *
 * RETURNS: status
 */
A_STATUS hsInitTx(
    IN TARGET_HANDLE targetHandle)
{
    PER_TARGET_INFO  *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    A_STATUS          status;

	intDescQInit(&pTargInfo->intTxFreeQueue);
	intDescQInit(&pTargInfo->intTxPendQueue);

    /* Allocate intermediate tx buffers */
    status = allocateIntTxBuffers(targetHandle);

    return status;

}

/*
 * hsShutdownTx - host stub transmit shutdown
 *
 * RETURNS: status
 */
void hsShutdownTx(
    IN TARGET_HANDLE targetHandle)
{
    /* Free intermediate tx buffers */
    freeIntTxBuffers(targetHandle);
}


/*****************************************************************************
 * Local functions (private to this module)
 */

/*
 * allocateIntTxBuffers - allocate memory for intermediate tx descriptors/buffers
 *
 * RETURNS: status
 */
A_STATUS
allocateIntTxBuffers(
    IN TARGET_HANDLE targetHandle)
{
    PER_TARGET_INFO     *pTargInfo = (PER_TARGET_INFO *)targetHandle;
    A_STATUS            status;
    int                 i;
    INT_DESC            *pDesc;
    int                 bufferSize = sizeof(INT_DESC) + WDC_TXMSG_LENGTH;

	/* Instead of allocating a big chunk of memory distribute it as small
     * chunks */
#ifndef Linux
    /* Allocate intermediate tx descriptors/buffers */
    pTargInfo->pIntTxMemory = A_DRIVER_MALLOC(NUM_INT_TX_DESC * bufferSize);

    if (pTargInfo->pIntTxMemory == NULL) {
        return A_NO_MEMORY;
    }
#endif

    /* Now go ahead and build the descriptor's queue */
    for (i = 0; i < NUM_INT_TX_DESC; i++) {
#ifdef Linux
        pDesc = A_DRIVER_MALLOC(bufferSize); 
        if (pDesc == NULL) {
            return A_NO_MEMORY;
        }
#else
        pDesc = (INT_DESC *) (((A_UINT8 *) pTargInfo->pIntTxMemory) + (bufferSize * i));
#endif
        pDesc->pNextVirtPtr   = NULL;
        pDesc->pBufferVirtPtr = ((A_UINT8 *) pDesc) + sizeof(INT_DESC);
        intDescQPushTail(&pTargInfo->intTxFreeQueue, pDesc, pDesc);
    }
    return A_OK;
}

/*
 * freeIntTxBuffers - free Tx descriptor memory
 *
 * RETURNS: status
 */
void
freeIntTxBuffers(
    IN TARGET_HANDLE targetHandle)
{
    PER_TARGET_INFO *pTargInfo = (PER_TARGET_INFO *)targetHandle;

#ifdef Linux
    INT_DESC          *pIntDesc;
    while (pIntDesc = intDescQPopHead(&pTargInfo->intTxFreeQueue)) {
        A_DRIVER_FREE(pIntDesc, 0);
    }
#else
    /* Free descriptors */
    if (pTargInfo->pIntTxMemory) {
        A_DRIVER_FREE(pTargInfo->pIntTxMemory,
                      (NUM_INT_TX_DESC * (sizeof(INT_DESC) + WDC_TXMSG_LENGTH)));
        pTargInfo->pIntTxMemory = NULL;
    }
#endif
}


/* xxxxPGSxxxx - TODO - it would be nice to have common queue manipulation functions */

/*
 * intDescQInit - initialize the data structure corresponding to a descriptor
 *  queue
 *
 * RETURNS: success or failure
 */
A_STATUS
intDescQInit(INT_QUEUE_DETAILS *pQueue)
{
    ASSERT(pQueue);

    A_MEM_ZERO(pQueue, sizeof(INT_QUEUE_DETAILS));   
    A_SEM_INIT(pQueue->qSem, 0, CREATE_UNLOCKED);

    return A_SEM_VALID(pQueue->qSem) ? A_OK : A_ERROR;
}

/*
 * intDescQGetHead - get pointer to first descriptor on the queue,
 *                don't remove it
 *
 * RETURNS: ATHEROS_DESC *
 */
INT_DESC *
intDescQGetHead(INT_QUEUE_DETAILS *pQueue)
{
    return pQueue->pDescQueueHead;
}

/*
 * intDescQGetTail - Get pointer to last descriptor on the queue,
 *                don't remove it
 *
 * RETURNS: ATHEROS_DESC *
 */
INT_DESC *
intDescQGetTail(INT_QUEUE_DETAILS *pQueue)
{
    return pQueue->pDescQueueTail;
}

/*
 * intDescQPopHead - get a descriptor from the given queue
 *
 * RETURNS: ATHEROS_DESC *
 */
INT_DESC *
intDescQPopHead(INT_QUEUE_DETAILS *pQueue)
{
    INT_DESC *pDesc;

    A_SEM_LOCK(pQueue->qSem, WAIT_FOREVER);

    pDesc = pQueue->pDescQueueHead;
    if (pDesc) {
        pQueue->pDescQueueHead = pDesc->pNextVirtPtr;

        if (pQueue->pDescQueueHead == NULL) {
            pQueue->pDescQueueTail = NULL;
        }

        pDesc->pNextVirtPtr = NULL;
        ASSERT(pQueue->qFrameCount > 0);
        pQueue->qFrameCount --;
    }

    A_SEM_UNLOCK(pQueue->qSem);

    return pDesc;
}

/*
 * intDescQPushTail - add the descriptor to the tail of queue struct
 *
 * RETURNS: N/A
 */
void
intDescQPushTail(INT_QUEUE_DETAILS *pQueue, INT_DESC *pDesc, INT_DESC *pTail)
{
    ASSERT(pDesc);
    ASSERT(pQueue->pDescQueueTail != pDesc);

    A_SEM_LOCK(pQueue->qSem, WAIT_FOREVER);

    /* check to see if anything is in the queue */
    if (pQueue->pDescQueueTail == NULL) {
        ASSERT(pQueue->pDescQueueHead == NULL);
        pQueue->pDescQueueHead = pDesc;
    } else {
        pQueue->pDescQueueTail->pNextVirtPtr = pDesc;
    }
    pQueue->pDescQueueTail = pTail;

#if 0
    /* FIX: This ASSERT isn't quite right....wdcFlush calls can throw us over */
    ASSERT(pQueue->qFrameCount < NUM_INT_TX_DESC);
#endif
    pQueue->qFrameCount ++;
    A_SEM_UNLOCK(pQueue->qSem);

    return;
}

/*
 * intDescQPushHead - add the descriptor to the head of queue struct
 *                 (non conventional!)
 *
 * RETURNS: N/A
 */
void
intDescQPushHead(INT_QUEUE_DETAILS *pQueue, INT_DESC *pDesc)
{
    ASSERT(pDesc);
    ASSERT(pQueue->pDescQueueHead != pDesc);

    A_SEM_LOCK(pQueue->qSem, WAIT_FOREVER);

    /* check to see if anything is in the queue */
    if (pQueue->pDescQueueHead == NULL) {
        ASSERT(pQueue->pDescQueueTail == NULL);
        pQueue->pDescQueueTail = pDesc;
    } else {
        pDesc->pNextVirtPtr = pQueue->pDescQueueHead;
    }
    pQueue->pDescQueueHead = pDesc;

    ASSERT(pQueue->qFrameCount < NUM_INT_TX_DESC);
    pQueue->qFrameCount ++;
    A_SEM_UNLOCK(pQueue->qSem);

    return;
}

INT_DESC *
getIntDescFromQueue(INT_QUEUE_DETAILS *pQueue, INT_DESC *pDesc)
{
    A_SEM_LOCK(pQueue->qSem, WAIT_FOREVER);
    if (pDesc) {
        INT_DESC  **ppTail = &pQueue->pDescQueueHead;
        INT_DESC  *pTail  = pDesc;

        while (*ppTail != pDesc) {
            if ((*ppTail) == NULL) {
                A_SEM_UNLOCK(pQueue->qSem);
                return NULL;
            }
            if (!pTail->pNextVirtPtr) {
                pQueue->pDescQueueTail = *ppTail;
            }
            ppTail = &((*ppTail)->pNextVirtPtr);
        }

        ASSERT(*ppTail == pDesc);

        *ppTail = pTail->pNextVirtPtr;
        if (!pQueue->pDescQueueHead) {
            pQueue->pDescQueueTail = NULL;
        }        

        pTail->pNextVirtPtr = NULL;
    }

    ASSERT(pQueue->qFrameCount > 0);
    pQueue->qFrameCount --;

    A_SEM_UNLOCK(pQueue->qSem);
    return pDesc;
}
