/*
 * Copyright (c) 2000-2004 Atheros Communications, Inc., All Rights Reserved
 *
 * Definitions for core driver
 * This is a common header file for all platforms and operating systems.
 */

#ifndef _WDCAPI_H_
#define _WDCAPI_H_

#ifdef __cplusplus
extern "C" {
#endif

/*******************************************************************************
 * WDC API : Wireless Driver Control API
 *   The WDC API partitions the Atheros Wireless LAN driver
 *   into host and target halves. The host and the target driver
 *   are independent modules. The WDC API defines the complete
 *   interface for the host driver module to access the target.
 *   For remote targets, WDC API stubs are implemented over
 *   the transport layer. The stubs and the transport driver
 *   on the host side serve as a target.
 *
 *   Similarily, the transport driver on the client side,
 *   and the stubs serve as a host for the target. Therefore
 *   the following two models are possible:
 *   - The classic model for PCI-like solutions
 *                  -------------------
 *                 |   (Host driver)   |
 *                 |------WDC API------|
 *                 |  (Target driver)  |
 *                  -------------------
 *   - The remote (e.g. USB) model
 *                  -------------------
 *                 |   (Host driver)   |
 *                 |------WDC API------|
 *                 | WDC Stubs for USB |
 *                 |-------------------|
 *                 |  USB Host driver  |
 *                  ===================
 *                 | USB Target driver |
 *                 |-------------------|
 *                 | WDC Stubs'for USB |
 *                 |------WDC API------|
 *                 |  (Target driver)  |
 *                  -------------------
 *   Similarily, WDC Stubs can be implemented for any other
 *   transport medium like - Ethernet, SDIO, IEEE-1394 etc.
 *   The model is designed for maximal reuse of modules for
 *   various solutions.
 *
 * Programming Model:
 *   - All WDC API calls without any OUT parameters or return
 *     codes are asynchronous - i.e. non-blocking calls
 *   - All WDI API calls with OUT parameters are considered
 *     synchronous (blocking) calls
 *   - Providing NULL references for all the OUT parameters of
 *     an otherwise blocking WDC API call makes the call non-
 *     blocking or asynchronous
 *   - Serialization: all API calls are expected to execute in
 *     the order in which they are made; all callbacks are
 *     expected to execute in the order in which they are made
 *   - The API transport is considered to be reliable
 *
 *   HANDLES are references to objects in one context and
 *   opaque in the other.
 *
 *   TBD: It is intended that the API will be versioned
 */


#include <wlantype.h>
#include <wlanproto.h>

#define IN
#define OUT


/*******************************************************************************
 * The REF_COUNT mechanism is used to track pending references to
 * various objects - so that objects can be freed up in a clean
 * manner. It is anticipated that for each handle which refers
 * to an object in one context, say target (or vice-versa), and
 * is opaque in the other context, say host (or vice-versa), the
 * following is done:
 *   - the host maintains a refcount corresponding to the
 *     opaque target handle; the refcount is incremented
 *     each time a WDC API call using the specific handle
 *     is made by the host
 *   - the target maintains an internal count corresponding
 *     to the number of times it has received the handle from
 *     the host through WDC API calls
 *   - on starting a detach operation corresponding to the
 *     object, the host will pass its refcount to the target
 *     as part of the WDC API
 *   - the target can free up the object once its internal
 *     count is the same as the refcount received from the
 *     host with the detach call
 * The mechanism is generic and can/should be used with all
 * object handles across the WDC API.
 * The refcount mechanism is not used with the main device
 * handle(s), as the entire set of resources is expected to
 * be freed up at device detach
 */

typedef A_UINT32 REF_COUNT;


/*******************************************************************************
 * WLAN Device Part of the WDC API
 * The HOST_HANDLE is the host driver cookie - and opaque to the WDC.
 * The TARGET_HANDLE is the target driver cookie - a handle supplied
 * by the WDC which is used in WDC calls to access a target.
 *
 * Before the host and target can communicate with each other, they need to know
 * how to address each other. The wdcBind call creates that binding.
 *
 * The wdcUnbind call breaks the binding between the host and the target.
 *
 * The host can determine capabilities of the target through
 * wdcTargetGetCapability.  The host can set configuration attributes
 * associated with the target through wdcTargetSetConfiguration.  The host
 * can get current status about the target and statistics through
 * wdcTargetGetStatus.
 *
 * Whenever the wireless device has to be set to a particular channel,
 * wdcTargetSetChannel API is used. This API sets the hardware in a state that
 * allows the reception of beacons and probe responses for that channel.
 * This API is used as part the SCAN functionality, change to TURBO & XR
 * modes. As an outcome of the background channel scan, if the SCAN
 * functionality goes back to the original channel, then it needs to
 * re-issue calls to set the target to the right state.  The target drops
 * any frames received from the host and the air, and changes the channel
 * after the queues have been drained.
 *
 * The host can reset a WLAN device by issuing the wlanReset call. The
 * target drops all the info received from the host and from the air. The
 * target uses the cached information to re-program the hardware after the
 * queues have been drained.
 *
 * The Target is enabled/disabled with wlanTargetEnable/wlanTargetDisable
 * calls.
 *
 * STARTUP:
 * Typical host/target startup interactions are as follows:
 *
 * The host uses wdcRegisterEventHandler(...WDCEVT_DEVICE_AVAILABLE...) in
 * order to express an interest in controlling a target, and to register a
 * callback to be used when a device is discovered.
 *
 * When target hardware is discovered, the software that discovers it
 * calls wdcDeviceDiscovered and passes device-specific (perhaps PCI
 * or USB specific) device context.  The WDC layer than invokes the hosts's
 * registered callback function for WDCEVT_DEVICE_AVAILABLE events.
 *
 * Within the callback, the host binds itself to the discovered target
 * (wdcBind).
 *
 * The host then determines capabilites of the target (wdcTargetGetCapability),
 * and configures the target (wdcTargetSetConfiguration) as appropriate.
 *
 * Finally, the host starts the target (wdcTargetStart).
 *
 * NDIS usage: With NDIS, the driver's Initialize entry point is called
 * when hardware is discovered.  The driver should call wdcDeviceDiscovered
 * in order to let the WDC layer know.  The driver's DEVICE_AVAILABLE callback
 * can simply remember the device's targetHandle in a per-device data structure.
 * So by the time wdcDeviceDiscovered completes, the driver's Initialize
 * function has the targetHandle stored in a per-device structure.
 */

typedef void * HOST_HANDLE;
typedef void * TARGET_HANDLE;

typedef enum {
    CFG_NONE,                  /* Sentinal to indicate "no config" */
    CFG_REG_DOMAIN,            /* Regulatory Domain */
    CFG_RATE_CONTROL_ENABLE,
    CFG_HW_TX_RETRIES,
    CFG_SW_TX_RETRIES,
    CFG_CALIBRATION_INTERVAL,
    CFG_SLOW_CLOCK_ENABLE,
    CFG_COMP_PROC,
    CFG_USER_RTS_THRESHOLD,
    CFG_PROTECTION_TYPE,
    CFG_BURST_SEQ_THRESHOLD,
    CFG_ABOLT,
    CFG_MODE_CTS,
    CFG_WME_ENABLED,
    CFG_GPRS_CBR_PERIOD,
    CFG_SERVICE_TYPE,
    CFG_MAC_ADDR,              /* MAC Address to use.  Overrides EEPROM */
    CFG_DEBUG_EAR,
    CFG_INIT_REGS,
    CFG_DEBUG_ID,              /* An ID for use in error & debug messages */
    CFG_COMP_WIN_SZ,
    CFG_DIVERSITY_CTL,
    CFG_TP_SCALE,
    CFG_TPC_HALF_DBM5,
    CFG_TPC_HALF_DBM2,
    CFG_OVERRD_TX_POWER,
    CFG_USE_32KHZ_CLOCK,
    CFG_GMODE_PROTECTION,
    CFG_GMODE_PROTECT_RATE_INDEX,
    CFG_GMODE_NON_ERP_PREAMBLE,
//  ...
} TARGET_CONFIG_ID;

typedef enum {
    CAP_NONE,                   /* Sentinal to indicate "no capability" */
    CAP_ALL,                    /* ALL capabilities */
    CAP_TARGET_VERSION,
    CAP_TARGET_REVISION,
    CAP_MAC_VERSION,
    CAP_MAC_REVISION,
    CAP_PHY_REVISION,
    CAP_ANALOG_5GHz_REVISION,
    CAP_ANALOG_2GHz_REVISION,
    CAP_DEBUG_WDCMSG_SUPPORT,    /* Target supports WDC message debug features */

    CAP_REG_DOMAIN,
    CAP_COUNTRY_CODE,
    CAP_REG_CAP_BITS,

    CAP_WIRELESS_MODES,
    CAP_CONNECTION_ID_MAX,      /* Should absorb CAP_KEY_CACHE_SIZE */
    CAP_TX_QUEUE_ID_MAX,        /* Should absorb CAP_NUM_TX_QUEUES */



    /* xxxxLMWxxxx NOTE: Capabilities below this line have been added
     * without much thought in order to ease the porting process.
     * Each of these must either be REMOVED or PROMOTED to above this
     * line, if we think they should stay.
     */


    CAP_CHAN_SPREAD_SUPPORT,
    CAP_COMPRESS_SUPPORT,
    CAP_BURST_SUPPORT,
    CAP_FAST_FRAMES_SUPPORT,
    CAP_CHAP_TUNING_SUPPORT,
    CAP_TURBOG_SUPPORT,
    CAP_TURBO_PRIME_SUPPORT,
    CAP_DEVICE_TYPE,

    CAP_LOW_2GHZ_CHAN,
    CAP_HIGH_2GHZ_CHAN,
    CAP_LOW_5GHZ_CHAN,
    CAP_HIGH_5GHZ_CHAN,

    CAP_CIPHER_WEP,
    CAP_CIPHER_AES_OCB,
    CAP_CIPHER_AES_CCM,
    CAP_CIPHER_TKIP,
    CAP_CIPHER_CKIP,
    CAP_MIC_CKIP,
    CAP_MIC_TKIP,
    CAP_MIC_AES_CCM,
    CAP_TWICE_ANTENNAGAIN_5G,
    CAP_TWICE_ANTENNAGAIN_2G,
//  ...
} TARGET_CAP_ID;

typedef enum {
    ST_NONE,                    /* Sentinal to indicate "no status" */
    ST_ALL,
    ST_SERVICE_TYPE,
    ST_WLAN_MODE,
    ST_FREQ,
    ST_BAND,
    ST_LAST_RSSI,
    ST_PS_FRAMES_DROPPED,
    ST_CACHED_DEF_ANT,
    ST_COUNT_OTHER_RX_ANT,
    ST_USE_FAST_DIVERSITY,
    ST_MAC_ADDR,
    ST_RX_GENERATION_NUM,
    ST_TX_QUEUE_DEPTH,
    ST_SERIAL_NUMBER,
//  ...
} TARGET_STATUS_ID;

typedef enum {
    STATS_NONE,                  /* Sentinal to indicate "no stats" */
    STATS_ALL,

    /* Receive related stats  */
    STATS_RX_FCS_FAIL_CNT,                      // 4 bytes      
    STATS_RX_DMA_OVERRUN_ERRORS,                // 4 bytes      
    STATS_RX_BEACON_COUNT,                      // 4 bytes      

    /* Transmit related stats */

    STATS_TX_GOOD_FRAMES,                       // 8 bytes    
    STATS_TX_UNICAST_FRAMES,                    // 8 bytes    
    STATS_TX_BROADCAST_FRAMES,                  // 8 bytes    
    STATS_TX_MUTLICAST_FRAMES,                  // 8 bytes    
    STATS_TX_FRAMES_DROPPED,                    // 8 bytes    
    STATS_TX_GOOD_BYTES,                        // 8 bytes    

    STATS_TX_ERRORS,                            // 4 bytes      
    STATS_TX_EXCESSIVE_RETRIES,                 // 4 bytes      
    STATS_TX_DMA_UNDERRUN,                      // 4 bytes      
    STATS_TX_ACK_RCV_FAILURES,                  // 4 bytes      
    STATS_TX_RTS_SUCCESS_CNT,                   // 8 bytes      
    STATS_TX_RTS_FAIL_CNT,                      // 4 bytes      

    STATS_TX_COMP_SUCCESS_CNT,                  // 4 bytes      
    STATS_TX_COMP_CPC0_CNT,                     // 4 bytes      
    STATS_TX_COMP_CPC1_CNT,                     // 4 bytes      
    STATS_TX_COMP_CPC2_CNT,                     // 4 bytes      
    STATS_TX_COMP_CPC3_CNT,                     // 4 bytes      

    STATS_TX_FILTERED,                          // 4 bytes      
    STATS_TX_TOTAL_RETRIES,                     // 8 bytes      
//xxxxPGSxxxx - TODO    STATS_TX_GOOD,              // 4 bytes      
//xxxxPGSxxxx - TODO    STATS_TX_GOOD,              // 4 bytes      
    STATS_TX_SW_RETRY_TOTAL_CNT,                // 4 bytes      
    STATS_TX_SW_RETRY_MAX_RETRIES_EXCEEDED,     // 4 bytes      

    /* Rates/Antenna/RSSI etc. */

    STATS_ACK_RSSI,                             // 4 bytes      
    STATS_TX_RATE_KB,                           // 4 bytes      
//xxxxPGSxxxx - TODO    STATS_ANT_CNT,              // 4 bytes      
    STATS_ANT_SW_CNT,                           // 4 bytes   
    STATS_CURR_TX_POWER,                        // 4 bytes   
    STATS_MAX_TX_POWER,                         // 4 bytes   
    STATS_NOISE_FLOOR,                          // 4 bytes   
//  ...
    STATS_MAX
} TARGET_STATS_ID;


typedef enum {
    CHANNEL_BAND_2GHz,
    CHANNEL_BAND_5GHz
} CHANNEL_BAND;

typedef A_UINT16 CHANNEL_FREQ;

typedef enum {
    PHY_MOD_NONE   = 0x00,
    PHY_MOD_CCK    = 0x01,
    PHY_MOD_OFDM   = 0x02,
    PHY_MOD_XR     = 0x04,
    PHY_MOD_TURBO  = 0x08
} PHY_MOD_FLAGS;
typedef A_UINT8 PHY_MODS;


typedef enum {
    TARGET_DEVICE_AWAKE,
    TARGET_DEVICE_SLEEP,
    TARGET_DEVICE_PWRDN,
    TARGET_DEVICE_PWRSAVE,
    TARGET_DEVICE_SUSPEND,
    TARGET_DEVICE_RESUME,
} TARGET_POWER_MODE;

typedef enum {
    TARGET_LED_DISCONNECT_STATE,
    TARGET_LED_CONNECT_STATE,
}TARGET_LED_STATE;
/*
 * wdcInit
 *
 * This API is used to initialize the WDC software.  It must be
 * called exactly once before any other WDC interface is used.
 */
A_BOOL
wdcInit(void);


/*
 * wdcHostAvailable
 *
 * This API is used.
 *
 * INPUT PARAMETERS
 *
 * hostHandle     - Cookie identifying the host, used by the target
 *                  to address the host
 * targetHandle   - Cookie identifying the target.
 *
 * OUTPUT PARAMETERS
 *
 * pStatus        - Status of the wdcHostAvailable call.
 *
 */

void
wdcHostAvailable(
    IN HOST_HANDLE    hostHandle,
    IN TARGET_HANDLE  targetHandle,
    OUT A_STATUS     *pStatus
);


/*
 * wdcBind
 *
 * This API is used to create a binding between the host and the target device.
 * It is generally invoked from the host's  WDCEVT_DEVICE_AVAILABLE handler.
 *
 * INPUT PARAMETERS
 *
 * hostHandle     - Cookie identifying the host, used by the target to address the
 *                  host
 * targetHandle   - Cookie identifying the target.
 * hostApiVersion - TODO
 *
 * OUTPUT PARAMETERS
 *
 * pStatus        - Status of the wdcBind call.
 */
void
wdcBind(
    IN  HOST_HANDLE         hostHandle,
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            hostApiVersion,
    OUT A_STATUS            *pStatus
);

/*
 * wdcUnbind
 *
 * The binding between the host and the target device is broken using this API.
 * The host and target release any resources that were allocated in the bind
 * call. This API does not have an OUT parameter to take care of architectures,
 * where the target can be remote from the host.
 *
 * INPUT PARAMETERS
 *
 * hostHandle     - Cookie identifying the host
 * targetHandle   - Cookie identifying the target.
 */
void
wdcUnbind(
    IN  HOST_HANDLE         hostHandle,
    IN  TARGET_HANDLE       targetHandle
);


/*
 * OS-dependent, connection-dependent per-device communications handle.
 * For example, a PCI-connected device would use PCI info here.
 * A USB-connected device would use a USB messaging handle.
 */
typedef void *DEVICE_DATA;


/*
 * wdcDeviceDiscovered
 *
 * Informs the WDC layer that a new hardware device has been discovered.
 * The WDC layer allocates structures to track the new target, and it
 * invokes the pre-registered WDCEVT_DEVICE_AVAILABLE callback, if any.
 * If there is no such callback, the presense of this device is remembered
 * so that if a callback is ever registered, it will be called immediately.
 *
 * Intended for use on the target side; but used by the host
 * when target and host are colocated.
 *
 * INPUT PARAMETERS
 *
 * devData -   OS-dependent and connection-dependent device data,
 *             which also serves as a unique identifier
 */
void
wdcDeviceDiscovered(
    IN  DEVICE_DATA      devData,
    OUT A_STATUS         *pStatus
);

/*
 * wdcDeviceRemoved
 *
 * Informs the WDC layer that a hardware device that was previously
 * discovered is no longer available to WDC.  This causes WDC to
 * invoke any pre-registered WDCEVT_DEVICE_UNAVAILABLE callback.
 * Then WDC erases all knowledge of the device.
 *
 * Intended for use on the target side; but used by the host
 * when target and host are colocated.
 *
 * INPUT PARAMETERS
 *
 * devData -    The same unique identifier that was earlier
 *              passed to wdcDeviceDiscovered
 */
void
wdcSurpriseRemoved(
    TARGET_HANDLE       targetHandle
);

typedef void * TARGET_CAP_DATA;


/*
 * wdcTargetGetCapability
 *
 * Allows the host to learn about the target's capabilities.
 *
 * INPUT PARAMETERS
 * targetHandle   - Cookie to identify target
 * capId          - Identify which capability we are interested in
 *
 * OUTPUT PARAMETERS
 *
 * capSize        - Size of the capability returned
 * capData        - Pointer to capability data, or the capability data
 *                  itself, if capSz=0.
 * pStatus        - Status of the call.
 */
void
wdcTargetGetCapability(
    IN  TARGET_HANDLE       targetHandle,
    IN  TARGET_CAP_ID       capId,
    OUT A_UINT32            *capSz,
    OUT TARGET_CAP_DATA     *capData,
    OUT A_STATUS            *pStatus
);

/*
 * wdcCapDataFree
 *
 * If wdcTargetGetCapability is used to fetch a variable-size capability,
 * wdc allocates a buffer to hold the result.  The host must free that buffer
 * with wdcCfgDataFree when it is done examining the data.
 *
 * INPUT PARAMETERS
 * capSz          - The capSz value from a previous wdcTargetGetCapability
 * capData        - The capData value from a previous wdcTargetGetCapability
 */
void
wdcCapDataFree(
    IN  A_UINT32            capSz,
    IN  TARGET_CAP_DATA     *capData
);

/*
 * When wdcTargetGetCap is used to fetch CAP_ALL, then wdcGetCapItem is
 * used to extract individual capability elements that were fetched.  After
 * all interesting items are extracted, the capability data must be freed
 * with a call to wdcCapDataFree.
 *
 * The CAP_ALL support is intended to support efficient use of the
 * underlying WDC Target/Host Transport.
 *
 * Usage example:
 *     wdcTargetGetCapability(targetHandle, CAP_ALL, &sz, &all, NULL);
 *     wdcGetCapItem(all, CAP_TARGET_VERSION, NULL, &targVersion);
 *     wdcGetCapItem(all, CAP_TARGET_REVERSION, NULL, &targRevision);
 *     wdcGetCapItem(all, CAP_MAC_VERSION, NULL, &MACVersion);
 *     wdcCapDataFree(targetHandle, all, sz);
 *
 * INPUT PARAMETERS:
 * allCapData     - Value of capData, returned from a previous call
 *                  to wdcTargetGetCap
 * capId          - Capability item of interest
 *
 * OUTPUT PARAMETERS:
 * capSz          - Size, in bytes, of capability item
 * capItem        - Capability item, or pointer to capability item
 *                  if capSz > 0.
 */
A_STATUS
wdcGetCapItem(
    IN  TARGET_CAP_DATA     *allCapData,
    IN  TARGET_CAP_ID       capId,
    OUT A_UINT32            *capSz,
    OUT TARGET_CAP_DATA     *capItem
);


typedef void * TARGET_CONFIG_VAL;

/*
 * wdcTargetSetConfiguration
 *
 * This API specifies a new value for a target-wide configuration attribute.
 *
 * INPUT PARAMETERS
 * targetHandle   - Cookie to identify target
 * cfgId          - Identify which configuration attribute we will set.
 * cfgSz          - Size in bytes of the config data
 * cfgVal         - Pointer to configuration value, or the value itself,
 *                  if cfgSz=0.
 *
 * OUTPUT PARAMETERS
 *
 * pStatus        - Status of the call.
 */
void
wdcTargetSetConfiguration(
    IN  TARGET_HANDLE       targetHandle,
    IN  TARGET_CONFIG_ID    attrId,
    IN  A_UINT32            cfgSz,
    IN  TARGET_CONFIG_VAL   cfgVal,
    OUT A_STATUS            *pStatus
);

typedef void *TARGET_STATUS_DATA;

/*
 * wdcTargetGetStatus
 *
 * This API allows the host to fetch target status information.
 *
 * INPUT PARAMETERS
 * targetHandle   - Cookie to identify target
 * statusId       - Identify which status attribute to fetch.
 *
 * OUTPUT PARAMETERS
 * statusSz       - Size in bytes of the status data
 * statusData     - Status data, or pointer to status data if statusSz>0.
 * pStatus        - Status of the call itself.
 */

void
wdcTargetGetStatus(
    IN  TARGET_HANDLE       targetHandle,
    IN  TARGET_STATUS_ID    statusId,
    OUT A_UINT32            *statusSz,
    OUT TARGET_STATUS_DATA  *statusData,
    OUT A_STATUS            *pStatus
);


/*
 * wdcStatusDataFree
 *
 * If wdcTargetGetStatus is used to fetch a status of variable size,
 * wdc allocates a buffer to hold the result.  The host must free that buffer
 * with wdcStatusDataFree when it is done examining the data.
 *
 * INPUT PARAMETERS
 * statusData     - The statusData value from a previous wdcTargetGetStatus
 * statusSz       - The statusSz value from a previous wdcTargetGetStatus
 */
void
wdcStatusDataFree(
    IN  TARGET_STATUS_DATA  statusData,
    IN  A_UINT32            statusSz
);

/*
 * When wdcTargetGetStatus is used to fetch ST_ALL,
 * then wdcGetStatusItem is used to extract individual
 * status elements that were fetched.  After all interesting
 * items are extracted, the status data is freed with call
 * to wdcStatusDataFree.
 *
 * The ST_ALL support is intended to support two things:
 * 1) efficient use of the underlying WDC Target/Host Transport
 * 2) allows status data to be fetched "atomically" so that it
 *    represents a status snapshot at some point in time.
 *
 * Usage example:
 *     wdcTargetGetStatus(targetHandle, ST_ALL, &sz, &all, NULL);
 *     wdcGetStatusItem(all, ST_TX_FRAMES_DROPPED, NULL, &TxFramesDropped);
 *     wdcGetStatusItem(all, ST_TX_FRAMES_OK,      NULL, &TxFramesOk);
 *     wdcGetStatusItem(all, ST_RX_FRAMES_DROPPED, NULL, &RxFramesDropped);
 *     wdcStatusDataFree(targetHandle, all, sz);
 *
 * INPUT PARAMETERS:
 * allStatusData  - Value of statusData, returned from a previous call
 *                  to wdcTargetGetStatus
 * statusId       - Status item of interest
 *
 * OUTPUT PARAMETERS:
 * statusSz       - Size, in bytes, of status item
 * statusItem     - Status item, or pointer to status item if statusSz > 0.
 */
A_STATUS
wdcGetStatusItem(
    IN  TARGET_STATUS_DATA  *allStatusData,
    IN  TARGET_STATUS_ID    statusId,
    OUT A_UINT32            *statusSz,
    OUT TARGET_STATUS_DATA  *statusItem
);

A_BOOL wdcCheckForHang(TARGET_HANDLE  targetHandle);

/*
 * wdcTargetStart
 *
 * Puts the target into an operational and enabled mode.
 * Note: This API is intended for use just after a wdcBind.
 */
void
wdcTargetStart(
    IN  TARGET_HANDLE       targetHandle,
    OUT A_STATUS            *pStatus
);


/*
 * wdcTargetStop
 *
 * Stops the Target and puts it into a non-operational mode,
 * freeing all data structures.  When the Target is Stopped,
 * it may lose configuration, so the Host driver should
 * re-establish Target Configuration and then re-start the
 * Target with wdcTargetStart in order to send/transmit.
 *
 * Note: This API is intended for use just before a wdcUnbind.
 * See also wdcTargetDisable.
 */
void
wdcTargetStop(
    IN  TARGET_HANDLE       targetHandle,
    OUT A_STATUS            *pStatus
);

/*
 * wdcTargetPresent
 *
 * Check target device still present or not
 */
A_BOOL
wdcTargetPresent(TARGET_HANDLE  targetHandle);

/*
 * wdcTargetSetChannel
 *
 * This API sets the target to a particul channel. A channel is defined by the
 * channel band, the channel frequency and the PHY modes. The valid PHY modes
 * combinatios  are CCK,OFDM, CCK+OFDM, OFDM + XR, OFDM + TURBO, CCK + OFDM +
 * TURBO, CCK + OFDM + XR.
 * This API is used to perform the SCAN function as well as change from/into
 * TURBO and XR modes. In case of background scan, when the host driver decides
 * to go back to the original channel, it must re-issue APIs to set the target
 * back in the original configration, since the state of the target had been
 * modified to enable the background scan.
 */
void
wdcTargetSetChannel(
    IN  TARGET_HANDLE       targetHandle,
    IN  CHANNEL_BAND        channelBand,
    IN  CHANNEL_FREQ        channelFrequency,
    IN  PHY_MODS            phyModulations,
    IN  A_UINT32            maxRDPower,
    IN  A_UINT32            cfgCtl,
    IN  A_UINT32            twiceAntennaReduction,
    IN  A_BOOL              keepRCContent,
    OUT A_STATUS            *pStatus
);

/*
 * wdcTargetReset
 *
 * The Target is reset in the most vigorous way possible.
 * The Target drops all frames received from the host and the air.
 * Configuration and State are lost, and must be re-established by
 * the Host driver.
 *
 * This interface is not needed for normal operation.
 */
void
wdcTargetReset(
    IN  TARGET_HANDLE       targetHandle,
    OUT A_STATUS            *pStatus
);

/*
 * wdcTargetEnable
 *
 * Restarts a target that has been wdcTargetDisable'd.
 */
void
wdcTargetEnable(
    IN  TARGET_HANDLE       targetHandle,
    OUT A_STATUS            *pStatus
);

/*
 * wdcTargetDisable
 *
 * Temporarily stops the Target from transmitting/receiving.
 * In order to re-start, use wdcTargetEnable.
 * No Target configuration state is lost when the Target is
 * Disabled and re-Enabled.
 *
 * Note: The Host may continue to receive some WDCEVT_DATA_INDICATION's
 * after the Target has been disabled.  These represent data that was
 * received before the Target was disabled.  The Host may process them,
 * ignore them, or register a NULL event handler for WDCEVT_DATA_INDICATION
 * in order to avoid seeing them.
 *
 * See also wdcTargetStop.
 */
void
wdcTargetDisable(
    IN  TARGET_HANDLE       targetHandle,
    OUT A_STATUS            *pStatus
);

/*
 * wdcTargetGetStats
 *
 * Retrieve Per Target statistics.  
 * Host must register for the WDCEVT_WLAN_STATS_UPDATE event.
 *
 * See also wdcGetStatsItem
 * See also wdcGetConnectionStats (per connection statistics)
 */
void
wdcTargetGetStats(
    IN  TARGET_HANDLE       targetHandle
);

/*
 * wdcGetStatsItem
 *
 * Called during WDCEVT_WLAN_STATS_UPDATE event to retreive
 * individual statistics items.
 *
 */

typedef void *TARGET_STATS_DATA;

A_STATUS
wdcGetStatsItem(
    IN  TARGET_STATS_DATA   allStatsData,
    IN  TARGET_STATS_ID     statsId,
    OUT A_UINT32            *statsSz,
    OUT TARGET_STATS_DATA   *statsItem
);


/*******************************************************************************
 * WLAN state and BSS related API
 *
 *   Host uses these APIs to configure the target for the WLAN mode it wants
 *   to operate on and also to update the mode with the required parameters,
 *   during operation. The APIs use the TARGET_HANDLE to address the target
 *   device, which is obtained during device attach process.
 *
 *   Station Mode Operation:
 *   =======================
 *      In station mode, the host after deciding on the BSS to join, would
 *      issue a Join request using the wdcStaJoinBss which is used to
 *      configure the target in joined mode either to an infrastructure or
 *      an adhoc network.
 *
 *      In the case of station joining an infrastructure BSS, the host --
 *      after the joining process -- would then associate with the AP:
 *      it would issue an Assoc Mode request using the wdcStaAssocBss
 *      API, which is used to set the target station mode to "associated"
 *      in an infrastructure network.
 *
 *      In case of a station joining an independent BSS, the host -- after the
 *      join process -- issues a Start command using the wdcStartBss API to
 *      allow the target to start beacons for the independent BSS.
 *
 *   AP Mode Operation:
 *   ==================
 *      The AP mode operation will be similar to the adhoc station - the AP
 *      will issue a Start request to the target using the wdcStartBss API,
 *      subsequent to using the wdcStaJoinBss for the join process. The start
 *      bss operation is used by the target to configure the Beacon and the
 *      hardware parameters to start a BSS. The start BSS API will take a
 *      wdcBssId parameter which will be used as a bssHandle for the host
 *      to communicate with the target - i.e. the host uses the wdcBssId to
 *      configure the BSS parameters started with this API for further operations.
 *
 *      The AP can update the Information element parameters for the network it
 *      has started by using the wdcUpdateBssIe API.
 *
 *   Host uses the wdcUpdateBssAttribute API to configure the target with other
 *   configuration parameters after it has set the target in associated or
 *   started mode. The target modes are reset by the device operations
 *   API based on the type of operations.
 *
 *   <Open Issue: API corresponding to the mulitcast key for the BSS>
 */

typedef enum {
    WLAN_MODE_NONE = 0,
    WLAN_MODE_11b,
    WLAN_MODE_11a,
    WLAN_MODE_11g,
    WLAN_MODE_11a_TURBO,
    WLAN_MODE_11g_TURBO,
    WLAN_MODE_11a_TURBO_PRIME,
    WLAN_MODE_11g_TURBO_PRIME,
    WLAN_MODE_11a_XR,
    WLAN_MODE_11g_XR,
    WLAN_MODE_MAX                 // Must be last
} WLAN_MODE;

typedef struct {
    BSSTYPE     bssType;
    WLAN_MODE   wlanMode;
    A_UINT32    beaconInterval;
    A_UINT32    dtimInterval;
    A_UINT32    cfpInterval;
    A_UINT32    cfpMaxDuration;
    A_UINT32    atimWindow;
    A_UINT32    defaultRateIndex;
    A_UINT32    shortSlotTime11g;
    A_UINT32    sleepDuration;
    A_UINT32    bmissThreshold;
    A_UINT32    tpcPowerLimit;
    A_UINT32    quietDuration;
    A_UINT32    quietOffset;
    A_UINT32    quietAckCtsAllow;
    A_UINT32    bssDefaultKey;
} WDC_BSS_ATTRIBUTES __ATTRIB_ALIGN(8);

typedef enum {
    BEACON_INTERVAL,
    DTIM_INTERVAL,
    CFP_INTERVAL,
    CFP_MAX_DURATION,
    ATIM_WINDOW,
    DEFAULT_RATE_INDEX,
    SHORT_SLOT_TIME_11g,
    SLEEP_DURATION,
    BMISS_THRESHOLD,
    TPC_POWER_LIMIT,
    BSS_KEY_UPDATE,
} WDC_BSS_ATTRIBUTE;


/* xxxxPGSxxxx - make "more" a bitfield??? */
typedef struct bufferDesc {
    struct bufferDesc       *pNext;
    A_UCHAR                 *pBuffer;
    A_UINT32                bufferPhysAddr;
    A_UINT32                bufferLength;
    A_UINT32                frameLength; /* must be valid for first descriptor in frame */
    A_UINT32                more;        /* Set if packet continues in next descriptor */
} WDC_BUFFER_DESC __ATTRIB_ALIGN(8);


/*
 * wdcStaJoinBss
 *
 * This API sets the target in the joined mode. The join process
 * involves synchronizing the TSF with the BSS's Timestamp, and setting
 * the MAC filters for the BSS and programming other
 * the hardware parameters for this mode.
 */

void
wdcStaJoinBss(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcBssId,
    IN  WLAN_MACADDR        *pBssId,
    IN  WDC_BSS_ATTRIBUTES  *pBssConfig
);

/*
 * wdcStaAssocBss
 *
 * This API sets the target in the associated mode. Target at this point
 * programs the hardware for assciated mode operation which involves
 * setting the assoc ID and programming the queues for Ps-Polls
 * transmission etc
 */

void
wdcStaAssocBss(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcBssId,
    IN  A_UINT16            assocId
);

/*
 * wdcStartBss
 *
 * This API sets the target in the start mode for adhoc and AP.
 * Target at this point initializes beacons for the requested mode
 * corresponding to the requested wdcBssId handle.
 */

void
wdcStartBss(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcBssId,
    IN  WDC_BUFFER_DESC     *pBeaconTemplate
);

/*
 * wdcUpdateBssAttribute
 *
 * This API is used to update the bss mode set on the target,
 * called with the bssHandle returned during the start API
 */

void
wdcUpdateBssAttribute(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcBssId,
    IN  WDC_BSS_ATTRIBUTE   attribute,
    IN  A_UINT32            cfgSz,
    IN  TARGET_CONFIG_VAL   cfgVal
);

/*
 * wdcUpdateBssIe
 *
 * This API is used to update the information element of the BSS
 * started at the target using the start command.
 */

void
wdcUpdateBssIe(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcBssId,
    IN  INFO_ELEMENT        *pIe
);

/*
 * wdcDetachBss
 *
 * This API is used to leave a previously joined/associated/started
 * BSS
 */

void
wdcDetachBss(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcBssId
);

/*******************************************************************************
 * Connection/SIB related API
 *
 * A connection needs to be established before the target will transmit
 * any data packets or encrypt any packets to a particular
 * destination.  The target will still forward received packets to the
 * host even if a connection is not present.  If the received packet
 * is encrypted the frame will not be properly decrypted.
 *
 * The host uses the wdcCreateConnection call to create a connection
 * on the target.  With this call the host can (and should whenever possible)
 * reserve encryption resources by specifying the expected encryption type of
 * the unicast key.  Examples of when the host will probably want to
 * create a connection:
 *      - 802.11 auth time in infrastructure mode with shared key auth.
 *      - association time in infrastructure mode in open key auth.
 *      - join time in ad-hoc mode.
 *
 * The target will not correctly send or receive encrypted frames until
 * a key is added.  The host uses the wdcSetConnectionKey to
 * add a key in the target. Only the unicast key is maintained with a
 * connection information.  The multicast case is associated to the BSS.
 *
 * The target can fail either the wdcCreationConnection or the
 * wdcUpdateConnectionAttribute requests if it is unable to allocate resources
 * for this connection (eg. memory, key cache, etc).  The host should
 * free up resources on the target by deleting a similar connection when
 * such failure occurs.  If the wdcCreationConnection fails then
 * the connection is never created on the target. If the
 * wdcUpdateConnectionAttribute fails then the attribute will not have
 * been modified by the target but the connection will remain.
 *
 * The host uses the wdcDeleteConnection() call to delete a connection.
 * It is expected that the host will delete the connection when no more
 * data communication is expected with the destination or when resources
 * in the target need to be freed up to make room for a separate connection.
 *
 * Some connection parameter definitions:
 *
 * wdcConnId - 32 bit value defined by the host than uniquely indentifies the
 * target.
 *
 * WDC_INVALID_CONN_ID is used to transmit a frame before a connection is created.
 * e.g. to transmit probe requests while scanning for a BSS to join.
 *
 *   <Open issue: encoding for the rate and the antenna
 *                in RATE_ANT_CODE>
 */

typedef void * TARGET_CONN_HANDLE;
typedef void * HOST_CONN_HANDLE;

typedef A_UINT32    RATE_ANT_CODE;
typedef A_UINT32    CONN_TYPE;

#define MAX_CONN_RATE_SIZE  32

typedef struct {
    A_UINT8     length;                     /* Length field */
    A_UINT8     rates[MAX_CONN_RATE_SIZE];  /* Rates array of octets */
} CONN_RATE_SET;

typedef struct {
    A_UINT32        keytype;
    A_UINT32        compression;
    A_UINT32        longPreambleOnly;
    RATE_ANT_CODE   currentTxRateAnt;
    CONN_TYPE       serviceType;
    CONN_RATE_SET   workingRateSet;     /* Working rates to be used for current tx */
    WLAN_MODE       wlanMode;
//  ...
} CONN_ATTRIBUTES;

typedef enum {
    CONN_KEYTYPE,
    CONN_COMPRESSION,
    CONN_LONG_PREAMBLE_ONLY,
    CONN_CURRENT_TX_RATE_ANT,
    CONN_SERVICE_TYPE,
    CONN_WORKING_RATE_SET,
    CONN_WORKING_RATE_SET_KEEPRC,
    CONN_WLAN_MODE,
    CONN_KEY_UPDATE,
//  ...
} CONN_ATTRIBUTE;

typedef struct {
//  ...
    A_UINT32    connStateOK;
} CONN_STATS;


#define WDC_INVALID_CONN_ID 0xFFFFFFFF

/*
 * Used by the host to create a connection in the target
 */
void
wdcCreateConnection(
    IN  TARGET_HANDLE       targetHandle,
    IN  WLAN_MACADDR        *pMacAddr,
    IN  A_UINT32            wdcConnId,
    IN  A_UINT32            wdcBssId,
    IN  CONN_ATTRIBUTES     *pOptions
);

/*
 * Is used by the host to set/clear/modify an attribute related to a
 * particular connection. Whether it makes sense to modify an attribute
 * is attribute specific. The target is not required (but it could)
 * perform any validation of whether an attribute should be changed
 * or not (eg. it currently doesn't make sense to change compression after
 * associating but the target might not reject such request).
 */
void
wdcUpdateConnectionAttribute(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcConnId,
    IN  CONN_ATTRIBUTE      attribute,
    IN  A_UINT32            cfgSz,
    IN  TARGET_CONFIG_VAL   cfgVal
);

/*
 * Used by the host to add/modify/clear a key for a particular connection
 */
void
wdcSetConnectionKey(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcConnId,
    IN  A_UINT32             keyType,
    IN  A_UINT32            keySize,            /* in bytes */
    IN  A_UCHAR             *pKeyMaterial
);


/*
 * TODO - need async event to report connection stats
 */
void
wdcGetConnectionStats(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcConnId,
    OUT CONN_STATS          *pConnStats
);

void
wdcDeleteConnection(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcConnId
);


/*******************************************************************************
 * Transmit/Receive Part of the API
 *
 * Transmit Queue Management:
 *
 * Queue management functions enable the host to support QOS features by
 * directly control the target TX queues. It minimizes the requirements
 * for target to preserve all the queue configuration. The queue operations
 * include dynamically create, update the target TX queues. All configuration
 * parameters of target TX queues are maintained by the host.
 *
 * Target directly controls the configuration/creation/update/operations for
 * the following queues locally: BEACON queue, CAB queue, PS-POLL queue, RX queue.
 *
 * TARGET_QUEUE_HANDLE: is opaque to the host. The host should pass this to
 * the target when transmitting a frame.
 *
 *
 * Transmit Flow:
 *
 * Host driver initiates the transmission by calling wdcSend(), which also
 * passes a completion callback function. Upon the target completes the
 * transmission the callback function is invoked by the host driver to initiate
 * the transmit post processing i.e. releasing all the resources associated
 * with the transmitted frames and indicating completion and status to the
 * sender dependent on the completion policies.
 * The completion policy is that for some special frames (management frames)
 * the host driver invokes the callback function upon the frames complete final
 * transmission and the the transmission status is propagated back to the host
 * driver by target via sending a message.
 *
 * Receive Flow:
 *
 * The WDC owns receive buffers.
 * The WDC calls the host driver's pre-registered event handler for
 * WDCEVT_DATA_INDICATION to process received frames upon reception completion.
 * Once the buffer has been handled, the driver calls wdcRxDone to return the
 * buffer back to the WDC.
 *
 * Rate Control:
 *   Some level of rate control is expected to be performed in
 *   the target driver - only the target can be reactive enough
 *   for fast control of the transmit rates. The host driver could
 *   implement a more sophisticated, slightly slower moving rate
 *   adaptation - and provide contraints/advisory to the target
 *   on a per connection, or on a frame by frame basis.
 *
 *   <Open issue: multicast/CAB traffic from the AP>
 *   <Open issue: where to combine frames into Fast-Frames?
 *                working assumption is on the host>
 */

typedef void *  HOST_FRAME_HANDLE;

typedef struct {
    A_UINT32    priority;
    A_UINT32    aifs;
    A_UINT32    logCwMin;
    A_UINT32    logCwMax;
    A_UINT32    burstTime;
    A_UINT32    compression;
    A_UINT32    qosMode;
//  ...
} TXQ_ATTRIBUTES;

typedef enum {
    PRIORITY,
    AIFS,
    LOG_CW_MIN,
    LOG_CW_MAX,
    BURST_TIME,
    COMPRESSION,
    QOS_MODE
//  ...
} TXQ_ATTRIBUTE;

typedef enum {
    RX_UCAST     = 0x00000001,
    RX_MCAST     = 0x00000002,
    RX_BCAST     = 0x00000004,
    RX_CONTROL   = 0x00000008,
    RX_BEACON    = 0x00000010,
    RX_PROM      = 0x00000020,
    RX_PHY_ERR   = 0x00000040,
    RX_PHY_RADAR = 0x00000080,
    RX_XR_POLL   = 0x00000400,
    RX_PROBE_REQ = 0x00000800,

    RX_COMMON    = ( RX_UCAST   | RX_MCAST  | RX_BCAST |
                     RX_CONTROL | RX_BEACON | RX_PROM  )

} RX_FILTER_FLAGS;

typedef enum {
    TX_OK,
    TX_ERR
} TX_STATUS_ENUM;

typedef struct {
    HOST_FRAME_HANDLE  hostFrameHandle;
    TX_STATUS_ENUM     status;
    A_UINT32           timeStamp;
    A_UINT32           retryCount;
    A_UINT32           xmitRate;
    A_UINT32           ackRssi;
} TRANSMIT_INFO;

typedef enum {
    RX_STATUS_OK,
    RX_STATUS_STOP_IN_PROGRESS,
    RX_STATUS_CRC_ERR,
    RX_STATUS_PHY_ERR,
    RX_STATUS_DECRYPT_CRC_ERR,
    RX_STATUS_DECRYPT_MIC_ERR,
    RX_STATUS_DECOMP_ERR,
    RX_STATUS_KEY_ERR,
    RX_STATUS_ERR
} RX_STATUS_ENUM;

typedef struct {
    RX_STATUS_ENUM  status;
    WLAN_TIMESTAMP  timestamp;
    A_UINT32        frameLength;
    A_UINT32        rate;
    A_UINT32        antenna;
    A_INT32         rssi;
    A_UINT32        channel;
    A_UINT32        phyError;
    A_UINT32        connIndex; /* Also the key table index for BSS traffic */
    A_UINT32        decryptError;
    A_UINT32        keyCacheMiss;
} RECEIVE_INFO;


/*
 * wdcSetupTxQueue
 *
 * Set up a target TX queue with specified configuration attributes
 *
 */
void
wdcSetupTxQueue(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcTxQueueId,
    IN  TXQ_ATTRIBUTES      *pTxQueueInfo
);


/*
 * wdcUpdateTxQueueAttribute
 *
 * Update a TX queue with a new (set) of attributes
 */
void
wdcUpdateTxQueueAttribute(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcTxQueueId,
    IN  TXQ_ATTRIBUTE       attribute,
    IN  A_UINT32            value
);

/*
 * wdcSetRxFilter
 *
 * Change the RX filter
 */
void
wdcSetRxFilter(
    IN  TARGET_HANDLE       targetHandle,
    IN  RX_FILTER_FLAGS     rxFilter
);


/*
 * wdcInitRxMulticastFilter
 *
 * Initialize the multicast filter.
 *
 * If bReset = TRUE, all filter bits are 0 (all multicast are filtered)
 *      Individual multicast filter bits are enabled via wdcAddRxMulticastFilter
 * If bReset = FALSE, all filter bits are 1 (all multicast are allowed)
 *      This is useful for software multicast filtering.
 *
 */
void
wdcInitRxMulticastFilter(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_BOOL              bReset
);

/*
 * wdcSetRxMulticastFilter
 *
 * Add the specified MAC address to the list of multicast addresses
 * allowed to be received.
 */
void
wdcSetRxMulticastFilter(
    IN  TARGET_HANDLE       targetHandle,
    IN  WLAN_MACADDR        *pMcastAddr
);

/*
 * wdcClearRxMulticastFilter
 *
 * Remove the specified MAC address from the list of multicast addresses
 * allowed to be received.
 */
void
wdcClearRxMulticastFilter(
    IN  TARGET_HANDLE       targetHandle,
    IN  WLAN_MACADDR        *pMcastAddr
);

/*
 * wdcSend
 *
 * Initiate a frame transmission aynchronously.
 *
 * IN PARAMETERS:
 *  hostFrameHandle     - handle for the host description of the frame,
 *                        used when indicating the tx compleiton to the host
 *                        and as reference for host to release the frame resource
 *  wdcTxQueueId        - ID of the target queue for transmission
 *  wdcConnId           - the target connection ID for the destination
 *  pBufferDesc         - frame scatter-gether buffer link-list
 *  bIndicateStatus     - completion status indication flag.  If set, a
 *                        WDCEVT_SEND_COMPLETE event is sent when the buffer
 *                        has been processed.
 */
A_STATUS
wdcSend(
    IN  TARGET_HANDLE       targetHandle,
    IN  HOST_FRAME_HANDLE   hostFrameHandle,
    IN  A_UINT32            wdcTxQueueId,
    IN  A_UINT32            wdcConnId,
    IN  WDC_BUFFER_DESC     *pBufferDesc,
    IN  A_BOOL              bIndicateStatus);

/*******************************************************************************
 * Receive Handling:
 * The Host uses wdcRegisterEventHandler to register an event handler
 * for the WDCEVT_DATA_INDICATION event.  Once the Host has done a
 * TargetStart, it begins to receive Data Indications from the Target.
 * The EVENT_HANDLER's "pInfo" field is a pointer to a WDC_RX_INFO structure,
 * stored in persistent memory.  Once the Host driver is done processing a
 * received buffer, it must call wdcRxDone to release the buffer.  After it
 * calls wdcRxDone, the driver must no longer touch the WDC_RX_INFO structure,
 * nor any of the members that the structure contains or points to [including
 * the data buffer].
 */


typedef struct {
    A_UINT32            dataLength;     /* Length in bytes of received data */
    RECEIVE_INFO        *receiveInfo;   /* Information about the receipt */
    char                *rxData;        /* The data buffer itself */
} WDC_RX_INFO;

/*
 * Called when the driver has completed processing received data.
 * This hands the buffer, RECEIVE_INFO, and WDC_RX_INFO back to
 * the WDC layer for re-use.
 */
void
wdcRxDone(
    IN  TARGET_HANDLE       targetHandle,
    IN  WDC_RX_INFO         *rxInfo
);


/*******************************************************************************
 * Event handling part of the API
 *
 * Some target events have to be propagated to the host since the host has all
 * the smarts and maintains the connection service states. The events could be
 * beacon miss interrupts, SWBA (software beacon alert) interrupts. Additionally
 * the host may want to monitor target by querying the target state periodically,
 * such as getting WLAN device stats.
 * For event that could potentially interrupt at any time, there must be an
 * active event handler which is registered during initialization time by
 * wdcRegisterHandler() as a callback function. Upon an event happens the target
 * sends a notification to the host and the host calls the event callback handler
 * to response based on the event type and information passed along with the event.
 *
 * STORE_INFO and RETRIEVE_INFO are provided for the benefit of the target
 *   to use the host memory as a repository for any opaque data (in case
 *   the target is running low on memory and wants the host to cache or
 *   backup some state on its behalf). The first byte of the info is
 *   used as a cookie to pattern-match a subsequent retrieve operation.
 *   Note that the RETRIEVE_INFO callback used by the target to retrieve
 *   previously stored information from the host is non-blocking; the event
 *   is treated as an asynchronous request from the target by the host - in
 *   repose to which it generates the wdcRestoreInfo call to push the
 *   requested information back to the target
 *
 * wdcFlush causes a WDCEVT_FLUSH_COMPLETE event to be generated when all
 *   previously issued WDC commands have completed.  This call is not needed
 *   for normal operation, but it provided to assist with diagnostics and
 *   for other uses when a driver may need complex synchronization.
 */

typedef enum {
    WDCEVT_NONE,
    WDCEVT_WLAN_STATS_UPDATE,
    WDCEVT_CONNECTION_STATS_UPDATE,
    WDCEVT_BEACON_MISS,
    WDCEVT_STORE_INFO,
    WDCEVT_RETRIEVE_INFO,
    WDCEVT_WLAN_BEACON_UPDATE,
    WDCEVT_ADHOC_MERGE_UPDATE,
    WDCEVT_DEVICE_AVAILABLE,
    WDCEVT_DEVICE_UNAVAILABLE,
    WDCEVT_DEBUG_MESSAGE_NOTIFY,
    WDCEVT_DATA_INDICATION,
    WDCEVT_SEND_COMPLETE,
    WDCEVT_STATUS_NOTIFY,
    WDCEVT_RX_IN_SYNC,
    WDCEVT_FLUSH_COMPLETE,
    WDCEVT_NO_TX_DATA,
    WDCEVT_STATS_NOTIFY,
//  ...
    WDCEVT_MAX
} WDC_EVENT;

typedef void (EVENT_HANDLER)(
    IN  TARGET_HANDLE targetHandle,
    IN  WDC_EVENT     event,
    IN  void *        param,
    IN  A_UINT32      infoSize,
    IN  A_UCHAR       *pInfo
);

/*
 * wdcRegisterEventHandler
 *
 * Register WDC event handlers.
 * To UNregister, register a NULL pHandler.
 * When registering a WDCEVT_DEVICE_AVAILABLE event, use a NULL targetHandle.
 * If a previous handler/parameter was registered, they are overridden.
 */
void
wdcRegisterEventHandler(
    IN  TARGET_HANDLE       targetHandle,
    IN  WDC_EVENT           event,
    IN  EVENT_HANDLER       *pHandler,
    IN  void *              param,
    OUT A_STATUS            *pStatus
);

/*
 * wdcGetEventHandler
 *
 * Returns a registered event handler and parameter.
 */
void
wdcGetEventHandler(
    IN  TARGET_HANDLE       targetHandle,
    IN  WDC_EVENT           event,
    OUT EVENT_HANDLER       **ppHandler,
    OUT void *              *pParam,
    OUT A_STATUS            *pStatus
);

void
wdcRestoreInfo(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            infoSize,
    IN  A_UCHAR             *pInfo
);

A_STATUS
wdcFlush(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            disableEnableCounter
);

/*
 * wdcTargetSetPowerMode
 *
 * specifies a target Power Save Mode.
 */
void
wdcTargetSetPowerMode(
    IN  TARGET_HANDLE       targetHandle,
    IN  TARGET_POWER_MODE   pwrMode,
    OUT A_STATUS            *pStatus
);

void wdcSetLedState(
    IN  TARGET_HANDLE       targetHandle,
    IN  TARGET_LED_STATE    ledSate
);

void
wdcBmissAck(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_BOOL              bEnable);


void
wdcSetLedSteadyMode(
    IN  TARGET_HANDLE       targetHandle, 
	IN  A_UINT32            gpioNumber,
    IN  A_UINT32            mode);


void
wdcSetLedBlinkMode(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32			ledNumber,
    IN  A_UINT32			ledMode,
    IN  A_UINT32			blinkRate,
    IN  A_UINT32			slowMode);

#include "shrcfg.h"

/*******************************************************************************
 * LED operation, ANI operation, RADAR detection, STA Power Save
 *
 *   <Open issue: corresponding API>
 *
 */

#ifdef __cplusplus
}
#endif

#endif /* _WDCAPI_H_ */
