/*
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 * WDC Message Definitions
 * This is a common header file for all platforms and operating systems.
 */

#ifndef _WDCMSG_H_
#define _WDCMSG_H_

#ifdef __cplusplus
extern "C" {
#endif

/*
 * Messages are passed in Target Endianness.  All fixed-size fields of
 * a WDC Control Message are treated as 32-bit values, and
 * Control Messages are guaranteed to be 32-bit aligned.
 *
 * The format of a WDC Control Message is as follows:
 *   Message Length  32 bits
 *   Message Opcode  32 bits
 *   Message ID      32 bits
 *   parameter1
 *   parameter2
 *      .
 *      .
 *      .
 * A variable-length parameter, or a parameter that is larger than
 * 32 bits is passed as a (length, data) pair, where length is a
 * 32-bit quantity and data is padded to 32-bits.
 *
 * The message format should remain independent of compile-time
 * constants, so do not add any DEBUG-only members to this structure.
 */
typedef struct {
    A_UINT32      msgLength;         /* The length in bytes of the entire
                                        message, including the WDC_MSG header.  
                                        It is padded to a 32-bit boundary.  */

    A_UINT32      msgOpcode;         /* One of the WDC_MSG_OPCODE's, below. */

    A_UINT32      msgId;             /* An arbitrary 32-bit ID.  It is intended 
                                        that the host can supply an ID with a
                                        request message, and the target
                                        response uses that same ID.  It allows
                                        the host to easily relate a response to 
                                        the request that generated it.  */

    union {
        A_UINT32  responseDesired;   /* In a message from the Host, this member
                                        is treated like a boolean: If non-zero,
                                        the Host desires a status response
                                        message from the Target. */

        A_STATUS  targetStatus;      /* In a response message from the Target,
                                        this member is an A_STATUS indicator. */
    } r;
    
#define WDC_CTRL_MSG_DEBUG_CNT 4
    A_UINT32      msgDebug[WDC_CTRL_MSG_DEBUG_CNT];
                                     /* The Host can put arbitrary debug data
                                        here, and the Target reflects it back
                                        in the associated Response (if any).
                                        This should be used for debug purposes
                                        only -- production target code may
                                        ignore this field. */

    A_UINT32      msgBuffer[0];       /* The message data itself */

} WDC_MSG __ATTRIB_ALIGN(8);

#define WDCMSG_HEADER_LENGTH (A_UINT32)(&((WDC_MSG *)0)->msgBuffer)

/*
 * This is the maximum size of any message.
 * In other words, it is guaranteed that msgLength <= WDC_MSG_MAX_LENGTH.
 */
#define WDC_MSG_LENGTH_MAX 512

/*
 * These are the messaging opcodes between host and target.
 * There is a separate message opcode for each WDC interface,
 * each WDC event, and each WDC callback.
 */
typedef enum {
    WDCMSG_INVAL = 0,

                    /* From Host to Target */
    /*
     * The message formats for each of these opcodes mimics
     * the IN parameters of the corresponding WDC API call.
     * For example, the message buffer for the
     * WDCMSG_TARGET_SET_CHANNEL message includes channelBand,
     * channelFrequency, and phyModulations.
     *
     * In place of the IN/OUT "A_STATUS *pStatus" parameter is
     * a boolean field, "responseDesired", that is part of the
     * message header.  If the host wants a status response,
     * it should use a non-zero responseDesired, and if the
     * host doesn't want a response, it should use a zero.
     */
    WDCMSG_HOST_AVAILABLE,
    WDCMSG_BIND,
    WDCMSG_UNBIND,
    WDCMSG_TARGET_GET_CAPABILITY,
    WDCMSG_TARGET_SET_CONFIGURATION,
    WDCMSG_TARGET_GET_STATUS,
    WDCMSG_TARGET_GET_STATS,
    WDCMSG_TARGET_START,
    WDCMSG_TARGET_STOP,
    WDCMSG_TARGET_SET_CHANNEL,
    WDCMSG_TARGET_RESET,
    WDCMSG_TARGET_ENABLE,
    WDCMSG_TARGET_DISABLE,
    WDCMSG_STA_JOIN_BSS,
    WDCMSG_STA_ASSOC_BSS,
    WDCMSG_STA_START_BSS,
    WDCMSG_STA_DETACH_BSS,
    WDCMSG_UPDATE_BSS_ATTRIBUTE,
    WDCMSG_UPDATE_BSS_IE,
    WDCMSG_CREATE_CONNECTION,
    WDCMSG_UPDATE_CONNECTION_ATTRIBUTE,
    WDCMSG_SET_CONNECTION_KEY,
    WDCMSG_GET_CONNECTION_STATS,
    WDCMSG_DELETE_CONNECTION,
    WDCMSG_SETUP_TX_QUEUE,
    WDCMSG_UPDATE_TX_QUEUE_ATTRIBUTE,
    WDCMSG_SET_RX_FILTER,
    WDCMSG_SET_RX_MULTICAST_FILTER,
    WDCMSG_INIT_RX_MULTICAST_FILTER,
    WDCMSG_CLEAR_RX_MULTICAST_FILTER,
    WDCMSG_SEND,
    WDCMSG_RESTORE_INFO,
    WDCMSG_FLUSH,

                    /* From Target to Host */
    /*
     * If the Host issues a request that needs a response from
     * the Target, the Target uses the same Message Opcode that
     * the host used.  In other words, if the Host sends a
     * WDCMSG_TARGET_GET_CAPABILITY message to the target, the
     * target responds with a WDCMSG_TARGET_GET_CAPABILITY
     * message which contains the response data.  [In this case,
     * the response data includes the capId, capSz, capData, and
     * status.]
     *
     * The message formats for each of these opcodes mimics
     * the OUT parameters of the corresponding WDC API call.
     * For example, the message buffer for the
     * WDCMSG_TARGET_GET_CAPABILITY response message includes
     * capSz, and capData.  Status is returned via the targetStatus
     * field of the message header.  It is only guaranteed to be
     * meaningful if "responseDesired" was indicated in the
     * original request message from the host.
     *
     */
    WDCMSG_TARGET_STATS_UPDATE,
    WDCMSG_CONNECTION_STATS_UPDATE,
    WDCMSG_BEACON_MISS_INTERRUPT,
    WDCMSG_STORE_INFO,
    WDCMSG_RETRIEVE_INFO,
    WDCMSG_TARGET_BEACON_UPDATE,
    WDCMSG_ADHOC_MERGE_UPDATE,
    WDCMSG_DEVICE_AVAILABLE,
    WDCMSG_DEVICE_UNAVAILABLE,
    WDCMSG_DEBUG_MESSAGE_NOTIFY,

    /* WDCMSG_CAPABILITY_UPDATE, */
    /* WDCMSG_STATUS_UPDATE, */
    WDCMSG_SEND_COMPLETE,
    WDCMSG_DATA_INDICATION,
    WDCMSG_TARGET_SET_PWR_MODE,

    WDCMSG_TARGET_SET_LED_STATE,
    WDCMSG_BEACON_MISS_ACK,
	
    WDCMSG_TARGET_SET_LED_STEADY_MODE,
    WDCMSG_TARGET_SET_LED_BLINK_MODE,
    WDCMSG_TARGET_APPLY_MKK_IRREV_UPDATE,
    WDCMSG_TARGET_MKK_LOCK_EEPROM,
    WDCMSG_TARGET_RAW_EEPROM_READ,

    WDCMSG_MAX
} WDC_MSG_OPCODE;




#if defined(ECOS) 
#define WDCMSG_PRINT(msg)                                    \
do {                                                         \
    int i;                                                   \
    int numWords = msg->msgLength / sizeof(A_UINT32);        \
                                                             \
    if (numWords > 16) {                                     \
        numWords = 16;                                       \
    }                                                        \
    uiPrintf("    [MSG:");                                   \
    for (i = 0; i < numWords; i++) {                         \
        uiPrintf(" 0x%04x", ((A_UINT32 *)msg)[i]);           \
    }                                                        \
    uiPrintf("]\n");                                         \
} while (0)
#else
#define WDCMSG_PRINT(msg)
#endif

#if defined(FAST_FRAMES_ENABLED)
/*
 * The message size must be large enough to hold multiple 802.11 frames
 * plus various network headers (e.g. WLAN_DATA_MAC_HEADERS,
 * LLC_SNAP_HEADER, ATHL2P_802_3_TUNNEL_HDR, LAN_FRAME_HEADERs) plus
 * WDC message headers.
 */
#define WDC_DATA_MSG_LENGTH_MAX 3584
#else
#define WDC_DATA_MSG_LENGTH_MAX 2048
#endif /* FAST_FRAMES_ENABLED */


/*
 * The message format for DATA INDICATION messages from Target to Host.
 * 
 * [NB: The current Host WDC implementation re-uses the start of the
 * WDC_RXMSG to hold WDC_RX_INFO.]
 */
typedef struct {
    A_UINT32          rxMsgLength;       /* The length in bytes of the entire
                                            message, including the WDC_RXMSG
                                            header.  It is padded to a 32-bit
                                            boundary.  */

    A_UINT32          msgOpcode;         /* WDCMSG_DATA_INDICATION */

    A_UINT32          generationNum;     /* Synchronization support */

    RECEIVE_INFO      receiveInfo;       /* Ancillary information about the
                                            data received.  Extracted from
                                            the hardware descriptor, and passed
                                            back to the Host. */

    A_UINT32          rxData[0];          /* The received data itself */

} WDC_RXMSG __ATTRIB_ALIGN(8);

#define WDC_RXMSG_HEADER_LENGTH ((A_UINT32)(&((WDC_RXMSG *)0)->rxData))
#define WDC_RXMSG_LENGTH_MAX WDC_DATA_MSG_LENGTH_MAX

/* Convert between RX Buffers and RX Messages */
#define WDC_RXBUF_TO_RXMSG(rxbuf) \
    (WDC_RXMSG *)((A_UINT32)(rxbuf) - WDC_RXMSG_HEADER_LENGTH)

#define WDC_RXMSG_TO_RXBUF(rxmsg) \
    (char *)((A_UINT32)(rxmsg) + WDC_RXMSG_HEADER_LENGTH)

#define WDC_RXMSG_LENGTH WDC_RXMSG_LENGTH_MAX




typedef struct {
    A_UINT32          txMsgLength;        /* The length in bytes of the entire
					     message, including the WDC_TXMSG
					     header.  It is padded to a 32-bit
					     boundary. */
    
    A_UINT32          msgId;

    A_UINT32          msgOpcode;          /* WDCMSG_SEND or WDCMSG_FLUSH */

    A_UINT32          txQueueId;

    A_UINT32          txConnId;

    A_UINT32          responseDesired;

    A_UINT32          bufferLength;

    A_UINT32          txData[0];           /* TX data */

} WDC_TXMSG __ATTRIB_ALIGN(8);

#define WDC_TXMSG_HEADER_LENGTH ((A_UINT32)(&((WDC_TXMSG *)0)->txData))
#define WDC_TXMSG_LENGTH_MAX  WDC_DATA_MSG_LENGTH_MAX

/* Convert between TX Buffers and TX Messages */

#define WDC_TXBUF_TO_TXMSG(txbuf) \
       ((WDC_TXMSG *)((A_UINT32)(txbuf) - WDC_TXMSG_HEADER_LENGTH)) 

#define WDC_TXMSG_TO_TXBUF(txmsg) \
       ((A_CHAR *)((A_UINT32)(txmsg) + WDC_TXMSG_HEADER_LENGTH))

#define WDC_TXMSG_LENGTH        (WDC_TXMSG_LENGTH_MAX)
#define WDC_TXMSG_DATA_LENGTH   (WDC_TXMSG_LENGTH - WDC_TXMSG_HEADER_LENGTH)


#ifdef __cplusplus
}
#endif

#endif /* _WDCMSG_H_ */
