/*
 *  Copyright (c) 2000-2002 Atheros Communications, Inc., All Rights Reserved
 *
 *  Chips specific device attachment and device info collection
 *  Connects Init Reg Vectors, EEPROM Data, and device Functions to pArDev
 */

#ifdef BUILD_AR5212

#include "arDev.h"

/* Standard HAL Headers */
#include "halApi.h"
#include "hal.h"
#include "halUtil.h"
#include "halDevId.h"

/* Headers for HW private items */
#include "ar5212/ar5212Reg.h"
#include "ar5212/ar5212Attach.h"
#include "ar5212/ar5212Phy.h"
#include "ar5212/ar5212Receive.h"
#include "ar5212/ar5212.h"
#include "arEvent.h"

#if defined(AR5523)
#include "cyg/hal/ar5523reg.h"
#endif

#if defined(ECOS)
#include "cyg/hal/plf_intr.h"

// xxxJDCxxx temporary hack
A_HANDLE           IntHandle;
A_INTERRUPT_OBJECT IntObject;

#endif

#if defined(ECOS_NOTDONE)
// taken from wlanchannel.h
#define MODE_SELECT_11A   0x01
#define MODE_SELECT_TURBO 0x02
#define MODE_SELECT_11B   0x04
#define MODE_SELECT_11G   0x08
#define MODE_SELECT_108G  0x10
/* extra mode bits for dynamic turbo prime */
#define MODE_SELECT_TPA   0x20
#define MODE_SELECT_TPG   0x40
#define MODE_SELECT_2GHZ  (MODE_SELECT_11B | MODE_SELECT_11G | MODE_SELECT_108G)
#define MODE_SELECT_ALL   (MODE_SELECT_11A | MODE_SELECT_TURBO | MODE_SELECT_2GHZ)
#endif /* ECOS_NOTDONE */

/* Header files exporting vector mapped functions */
#include "ar5212/ar5212Attach.h"
#include "ar5212/ar5212Reset.h"
#include "ar5212/ar5212Misc.h"
#include "ar5212/ar5212KeyCache.h"
#include "ar5212/ar5212Power.h"
#include "ar5212/ar5212Beacon.h"
#include "ar5212/ar5212Transmit.h"
#include "ar5212/ar5212Receive.h"
#include "ar5212/ar5212Interrupts.h"

/* Forward Declarations */

static A_STATUS
ar5212RecordSerialNumber(AR_DEV_INFO *pArDev, A_CHAR *pSerNum);

static A_STATUS
ar5212VerifyEepromChecksum(A_UINT16 *pRawEeprom, A_UINT32 eepEndLoc);

static A_STATUS
ar5212AllocateProm(EEP_MAP *pEepMap);

static A_STATUS
ar5212ReadEepromIntoDataset(AR_DEV_INFO *pArDev, A_UINT16 *pRawEeprom);

static void
ar5212ReadHeaderInfo(AR_DEV_INFO *pArDev, EEP_HEADER_INFO *pHeaderInfo, A_UINT16 *pRawEeprom);

static void
getPcdacInterceptsFromPcdacMinMax(A_UINT16 version, A_UINT16 pcdacMin,
    A_UINT16 pcdacMax, A_UINT16 *pPcdacValues);

static A_UINT16
fbin2freq(A_UINT16 version, A_UINT16 fbin, A_BOOL is2GHz);

#if defined(EAR_SUPPORT)
static int
ar5212EarPreParse(A_UINT16 *in, int numLocs, EAR_ALLOC *earAlloc);

static A_BOOL
ar5212EarAllocate(EAR_ALLOC *earAlloc, EAR_HEADER **ppEarHead);

static A_BOOL
ar5212EarCheckAndFill(EAR_HEADER *earHead, A_UINT16 *in, int totalLocs, EAR_ALLOC *pEarAlloc);

#ifdef DEBUG
static void
printEar(EAR_HEADER *earHead, EAR_ALLOC *earAlloc);

static void
printModeString(A_UINT16 mode);

static void
showEarAlloc(EAR_ALLOC *earAlloc);
#endif /* DEBUG */
#endif /* EAR_SUPPORT */

static A_UINT16
ar5212EnableMKKNew11a(AR_DEV_INFO *pDev);

#if !defined(HAL_CODE_REDUCTION)
/* Constants */
static const HW_FUNCS ar5212Funcs = {
    ar5212DevInit,
    ar5212DevFini,
    ar5212FillCapabilityInfo,

    /* Reset Functions */
    ar5212Reset,
    ar5212PhyDisable,
    ar5212Disable,
    ar5212PerCalibration,
    ar5212GetRfgain,
    ar5212SetTxPowerLimit,

    /* TX Functions */
    ar5212UpdateTxTrigLevel,
    ar5212SetupTxQueue,
    ar5212ReleaseTxQueue,
    ar5212GetTxDP,
    ar5212SetTxDP,
    ar5212StartTxDma,
    ar5212NumTxPending,
    ar5212StopTxDma,

    ar5212SetupTxDesc,
    ar5212ProcessTxDesc,
    ar5212GetTxDescDone,
#if defined(DEBUG) || defined(_DEBUG)
    ar5212DebugPrintTxDesc,

    /* RX Functions */
    ar5212GetRxDP,
    ar5212SetRxDP,
    ar5212EnableReceive,
    ar5212StopDmaReceive,
    ar5212StartPcuReceive,
    ar5212StopPcuReceive,
    ar5212SetMulticastFilter,
    ar5212MulticastFilterIndex,
    ar5212SetRxFilter,
    ar5212ProcessRxDesc,
#ifdef UPSD
    NULL,
#endif
    ar5212SetupRxDesc,

    /* Misc Functions */
    ar5212SetRegulatoryDomain,
    ar5212SetLedState,
    ar5212WriteAssocid,
    ar5212GpioCfgInput,
    ar5212GpioCfgOutput,
    ar5212GpioGet,
    ar5212GpioSet,
    ar5212GpioSetIntr,
    ar5212SetStaBeaconTimers,
    ar5212GetTsf,
    ar5212ResetTsf,
    ar5212SetAdhocMode,
    ar5212SetBasicRate,
    ar5212GetRandomSeed,
    ar5212DetectCardPresent,
    ar5212MibControl,
    ar5212EnableRadarDetection,
    ar5212AniControl,
    ar5212GetCurRssi,
    ar5212GetDefAntenna,
    ar5212SetDefAntenna,
    ar5212SetAntennaSwitch,
    ar5212UpdateAntenna,
    ar5212UseShortSlotTime,
    ar5212DmaDebugDump,
    ar5212SetDecompMask,
    ar5212GetApSwitchHelper,
    ar5212SetApSwitchHelper,
    ar5212ApplyMKKIrreversibleUpdate,
    ar5212MkkLockEeprom,
    ar5212RawEepromRead,

    /* Key Cache Functions */
    ar5212GetKeyCacheSize,
    ar5212ResetKeyCacheEntry,
    ar5212SetKeyCacheEntry,
    ar5212SetKeyCacheBssid,
    ar5212KeyCacheAlloc,
    ar5212KeyCacheFree,

    /* Power Management Functions */
    ar5212SetPowerMode,
    ar5212GetPowerMode,
    ar5212GetPowerStatus,
    ar5212SetupPsPollDesc,

    /* Beacon Functions */
    ar5212SetupBeaconDesc,
    ar5212BeaconInit,
    ar5212SetupGrpPollChain,

    /* Interrupt Functions */
    ar5212IsInterruptPending,
    ar5212GetInterrupts,
    ar5212EnableInterrupts,
    ar5212DisableInterrupts,
#if (defined(UPSD) && defined(BUILD_AP))
    ar5212UpsdResponse,
#endif
#endif /* ECOS_NOTDONE */

};

#endif /* HAL_CODE_REDUCTION */

static const A_UINT16 channels11b[] = {2412, 2447, 2484};
static const A_UINT16 channels11g[] = {2312, 2412, 2484};

/**************************************************************
 * ar5212Probe
 *
 * ar5212Probe performs the device specific functions for halProbe()
 */
A_STATUS
ar5212Probe(AR_DEV_INFO *pArDev)
{
    A_STATUS  status;
    A_UINT16  i;
    EEP_MAP   *pEMap;
    A_UINT32  revId;
    A_UINT16  eepVersion;
    A_UINT16  *pRawEeprom = NULL, *pData16, data;
    A_UINT32  addr, loop, wrData, rdData, pattern, eepEndLoc;
    A_UINT32  regAddr[2] = {MAC_STA_ID0, PHY_BASE+(8 << 2)};
    A_UINT32  regHold[2];
    A_UINT32  patternData[4] = {0x55555555, 0xaaaaaaaa, 0x66666666, 0x99999999};

    ASSERT(pArDev);

    /* Pull Chip out of Reset */
    status = ar5212ChipReset(pArDev, NULL);
    if (status != A_OK) {
        return status;
    }

    /* Read Revisions from Chips */
#if defined(PCI_INTERFACE) || defined(AR5523)
    {
        A_UINT32  macRev;

        macRev                     = A_REG_RD(pArDev, MAC_SREV) & MAC_SREV_ID_M;
#if defined(PREDATOR_FB24)
        macRev = 0x53;
#endif
        pArDev->hwInfo.macVersion  = (A_UINT16) (macRev >> MAC_SREV_ID_S);
        pArDev->hwInfo.macRevision = (A_UINT16) (macRev & MAC_SREV_REVISION_M);
    }
#else
#error "Need to implement"
#endif

    /*
     * Complain when unrecognized MAC SREV's are probed;
     * but don't fail probe in production.
     */
    if (pArDev->hwInfo.macVersion < MAC_SREV_VERSION_VENICE) {
        uiPrintf("ar5212Probe: Mac Chip Ver 0x%02X.%X is not supported by this driver\n",
                 pArDev->hwInfo.macVersion, pArDev->hwInfo.macRevision);
        ASSERT(0);
    }
    if ((pArDev->hwInfo.macVersion == MAC_SREV_VERSION_VENICE) &&
	(pArDev->hwInfo.macRevision < MAC_5312_2_0)) {
        uiPrintf("ar5212Probe: Mac Chip Rev 0x%02X.%X is not supported by this driver\n",
                 pArDev->hwInfo.macVersion, pArDev->hwInfo.macRevision);
        return A_HARDWARE;
    }

    pArDev->hwInfo.phyRevision = (A_UINT16)A_REG_RD(pArDev, PHY_CHIP_ID);

    /* Self test PHY & MAC registers */
    for (i = 0; i < 2; i++) {
        addr       = regAddr[i];
        regHold[i] = A_REG_RD(pArDev, addr);

        for (loop = 0; loop < 0x100; loop++) {

            wrData = (loop << 16) | loop;
            A_REG_WR(pArDev, addr, wrData);
            rdData = A_REG_RD(pArDev, addr);
            if (rdData != wrData) {
                uiPrintf("ar5212Probe: address test failed addr: 0x%08x - wr:0x%08x != rd:0x%08x\n",
                         addr, wrData, rdData);
                return A_HARDWARE;
            }
        }
        for (pattern = 0; pattern < 4; pattern++) {
            wrData = patternData[pattern];
            A_REG_WR(pArDev, addr, wrData);
            rdData = A_REG_RD(pArDev, addr);
            if (wrData != rdData) {
                uiPrintf("ar5212Probe: address test failed addr: 0x%08x - wr:0x%08x != rd:0x%08x\n",
                         addr, wrData, rdData);
                return A_HARDWARE;
            }
        }
        A_REG_WR(pArDev, regAddr[i], regHold[i]);
    }
    udelay(100);

    /* Set correct Baseband to analog shift setting to access analog chips. */
    A_REG_WR(pArDev, PHY_BASE, 0x00000007);

    /* Read Radio Chip Rev Extract */
    A_REG_WR(pArDev, (PHY_BASE + (0x34 << 2)), 0x00001c16);
    for (i = 0; i < 8; i++) {
        A_REG_WR(pArDev, (PHY_BASE + (0x20 << 2)), 0x00010000);
    }
    revId = (A_REG_RD(pArDev, PHY_BASE + (256 << 2)) >> 24) & 0xff;
    revId = ((revId & 0xf0) >> 4) | ((revId & 0x0f) << 4);

    /*
     * Complain when unrecognized Analog REV's are probed;
     * but don't fail attach in production.
     */
    pArDev->hwInfo.analog5GhzRevision = (A_UINT16)reverseBits(revId, 8);
    if (((pArDev->hwInfo.analog5GhzRevision & 0xF0) != RAD5112_SREV_MAJOR) &&
        ((pArDev->hwInfo.analog5GhzRevision & 0xF0) != RAD2112_SREV_MAJOR))
    {
        uiPrintf("ar5212Probe: Radio Chip Rev 0x%02X is unknown to this driver\n",
                 pArDev->hwInfo.analog5GhzRevision);
        ASSERT(0);
    }

    if (IS_RAD5112_REV1(pArDev)) {
        uiPrintf("ar5212Probe: 5112 Rev 1 is no longer supported - failing probe\n");
        return A_HARDWARE;
    }

    /* EEPROM version and size checking */
    status = ar5212EepromRead(pArDev, ATHEROS_EEPROM_OFFSET + 1, &eepVersion);
    if (status != A_OK) {
        uiPrintf("ar5212Probe: Could not access EEPROM\n");
        return A_HARDWARE;
    }

#if defined(PCI_INTERFACE)
    {
        A_UINT32 promSize;

        promSize = (A_REG_RD(pArDev, MAC_PCICFG) & MAC_PCICFG_EEPROM_SIZE_M) >>
                                                   MAC_PCICFG_EEPROM_SIZE_S;
        if (promSize != MAC_PCICFG_EEPROM_SIZE_16K) {
            uiPrintf("ar5212Probe: A 16k calibrated EEPROM is required for correct operation\n");
            return A_HARDWARE;
        }
    }
#endif

    if (eepVersion < EEPROM_VER3_2) {
        uiPrintf("ar5212Probe: A Rev 3.2 or greater EEPROM is required for correct operation\n");
        return A_HARDWARE;
    }

    /* Read sizing information */
    status = ar5212EepromRead(pArDev, EEPROM_SIZE_UPPER, &data);
    if (status != A_OK) {
        uiPrintf("ar5212Probe: Could not access EEPROM\n");
        return A_HARDWARE;
    }

    if (data == 0) {
        eepEndLoc = ATHEROS_EEPROM_END_DEFAULT;
    } else {
        eepEndLoc = (data & EEPROM_SIZE_UPPER_M) << 12;
        status = ar5212EepromRead(pArDev, EEPROM_SIZE_LOWER, &data);
        if (status != A_OK) {
            uiPrintf("ar5212Probe: Could not access EEPROM\n");
            return A_HARDWARE;
        }
        eepEndLoc |= data;
    }
    ASSERT(eepEndLoc > ATHEROS_EEPROM_OFFSET);

    /* Cache EEPROM for checksum verification and multiple parse passes */
    pArDev->pHalInfo->rawEepromSz =
        sizeof(A_UINT16) * (eepEndLoc - ATHEROS_EEPROM_OFFSET);
    pRawEeprom = (A_UINT16 *)A_DRIVER_MALLOC(pArDev->pHalInfo->rawEepromSz);
    if (!pRawEeprom) {
        uiPrintf("ar5212Probe: Could not allocate space to cache the EEPROM\n");
        return A_NO_MEMORY;
    }
    A_MEM_ZERO(pRawEeprom, pArDev->pHalInfo->rawEepromSz);
    pArDev->pHalInfo->pRawEeprom = pRawEeprom;

    pData16 = pRawEeprom;
    for (addr = ATHEROS_EEPROM_OFFSET; addr < eepEndLoc; addr++) {
        status = ar5212EepromRead(pArDev, addr, &data);
        if (status != A_OK) {
            goto probeError;
        }
        *(pData16++) = data;
    }

    /* Checksum validation */
    status = ar5212VerifyEepromChecksum(pRawEeprom, eepEndLoc);
    if (status != A_OK) {
        goto probeError;
    }

    /* Allocate EEPROM Map */
    pEMap = (struct eepMap *)A_DRIVER_MALLOC(sizeof(struct eepMap));
    if (!pEMap) {
        uiPrintf("ar5212Probe: Could not allocate memory for version 3 EEPROM\n");
        goto probeError;
    }
    A_MEM_ZERO(pEMap, sizeof(struct eepMap));
    pArDev->pHalInfo->pEepData = pEMap;

    status = ar5212AllocateProm(pEMap);
    if (status != A_OK) {
        goto probeError;
    }

    /* Set Version so EEPROM Header can be correctly interpreted */
    pEMap->version = eepVersion;

    /* Parse the calibration portion of EEPROM into driver structures */
    status = ar5212ReadEepromIntoDataset(pArDev, pRawEeprom);
    if (status != A_OK) {
        goto probeError;
    }

    status = ar5212EepromRead(pArDev, EEPROM_PROTECT_OFFSET, &pEMap->protect);
    if (status != A_OK) {
        goto probeError;
    }

    status = ar5212RecordSerialNumber(pArDev, pArDev->pHalInfo->serialNumber);
    if (status != A_OK) {
        goto probeError;
    }

    status = ar5212FillCapabilityInfo(pArDev);
    if (status != A_OK) {
        goto probeError;
    }

    /*
     * Try to fetch the default MAC address from from EEPROM.
     * Later, this may be overridden by a user-supplied MAC Address.
     */
    (void) ar5212GetMacAddr(pArDev, &pArDev->config.macAddr);


    if (((pArDev->config.macAddr.st.word == 0) &&
         (pArDev->config.macAddr.st.half == 0)) ||
        ((pArDev->config.macAddr.st.word == 0xFFFFFFFF) &&
         (pArDev->config.macAddr.st.half == 0xFFFF))) {
        // xxxJDCxxx XXX
        // HARD-CODED MAC ADDRESS until we have EEPROM
        pArDev->config.macAddr.st.word = 0x00037FBF;
        pArDev->config.macAddr.st.half = 0xAAAA;
    }

    A_MACADDR_COPY(&pArDev->config.macAddr,&pArDev->config.macPermAddr);

#ifdef DEBUG
    uiPrintf("ar5212Probe: wlan %d revisions: mac %d.%d phy %d.%d analog %d.%d eeprom %d.%d\n",
             pArDev->config.id,
             pArDev->hwInfo.macVersion,
             pArDev->hwInfo.macRevision,
             pArDev->hwInfo.phyRevision >> 4,
             pArDev->hwInfo.phyRevision & 0xF,
             pArDev->hwInfo.analog5GhzRevision >> 4,
             pArDev->hwInfo.analog5GhzRevision & 0xF,
             eepVersion >> 12, eepVersion & 0xFFF);
#endif

#if !defined(HAL_CODE_REDUCTION)
    /* Associate device-specific functions for use by hal */
    pArDev->pHwFunc = &ar5212Funcs;
#endif /* HAL_CODE_REDUCTION */

    status = A_OK;
    goto probeExit;

probeError:
    uiPrintf("ar5212Probe: EEPROM-related failure\n");
    if (pRawEeprom) {
        A_DRIVER_FREE(pRawEeprom, pArDev->pHalInfo->rawEepromSz);
        pArDev->pHalInfo->pRawEeprom = NULL;
    }
    status = A_HARDWARE;

probeExit:
    return status;
}

/**************************************************************
 * ar5212DevInit
 *
 * ar5212DevInit performs the device specific functions for halDevInit()
 */
A_STATUS
ar5212DevInit(AR_DEV_INFO *pArDev)
{
    A_STATUS  status;
    struct gainValues *pGainValues;
    A_UINT16  *pRawEeprom;
    EEP_MAP   *pEMap;
    int i;
#if defined(EAR_SUPPORT)
    A_UINT16  *earBuffer = NULL;
    int       earLocs = 0, parsedLocs;
    EAR_ALLOC earAlloc;
#endif /* EAR_SUPPORT */

    ASSERT(pArDev);

    pRawEeprom = pArDev->pHalInfo->pRawEeprom;
    ASSERT(pRawEeprom);

    pEMap = pArDev->pHalInfo->pEepData;
    ASSERT(pEMap);

#ifdef DEBUG
    uiPrintf("%s AR_DEV_INFO: 0x%p\n", __FUNCTION__, (void *)pArDev);
#endif

    /* Allocate thermal gain adjustment structure */
    pGainValues = (struct gainValues *)A_DRIVER_MALLOC(sizeof(*pGainValues));
    if (!pGainValues) {
        uiPrintf("ar5212DevInit: Could not allocate memory for gain optimization values\n");
        goto devInitError;
    }

    A_MEM_ZERO(pGainValues, sizeof(*pGainValues));
    pArDev->pHalInfo->pGainValues = pGainValues;
    /* Initialize gain ladder thermal calibration structure */
    ar5212InitializeGainValues(pArDev, pGainValues);

    /* Initialize Key cache */
    for (i = 0; i < MAC_KEY_CACHE_SIZE; i++) {
        ar5212ResetKeyCacheEntry(pArDev, i);
    }

    /* Allocate analog bank scratch buffer */
    if (ar5212AllocateRfBanks(pArDev, pArDev->pHalInfo) == FALSE) {
        uiPrintf("ar5212DevInit: Could not allocate memory for RF Banks\n");
        goto devInitError;
    }

    /* Attach rate table for various wireless modes to the hal */
    ar5212AttachRateTables(pArDev);

    /* ANI Initialization */
    pArDev->arDoAni = ar5212AniAttach(pArDev);

    /* Attach the RF Hal */
    ASSERT(IS_5112(pArDev));
    pArDev->pHalInfo->pRfHal = &ar5112Funcs;

#if defined(EAR_SUPPORT)
    if (pEMap->pEepHeader->earStart) {
        A_UINT32 eepEndLoc =
           (pArDev->pHalInfo->rawEepromSz/sizeof(A_UINT16))+ATHEROS_EEPROM_OFFSET;
        ASSERT(pEMap->pEepHeader->earStart > ATHEROS_EEPROM_OFFSET);
        ASSERT(pEMap->pEepHeader->earStart < eepEndLoc);
        earBuffer = &pRawEeprom[pEMap->pEepHeader->earStart - ATHEROS_EEPROM_OFFSET];
        earLocs = eepEndLoc - pEMap->pEepHeader->earStart;
        ASSERT(earBuffer);
        ASSERT(earLocs);
    }
#ifdef DEBUG
    /* Debug EAR overrides normal EAR */
    if (pArDev->config.EarDebug.earLength > 0) {
        earBuffer = pArDev->config.EarDebug.earBuffer;
        earLocs = pArDev->config.EarDebug.earLength / sizeof(A_UINT16);
    }
#endif

    if (earBuffer) {
        parsedLocs = ar5212EarPreParse(earBuffer, earLocs, &earAlloc);

#ifdef DEBUG
        if (EarDebugLevel >= EAR_DEBUG_VERBOSE) {
            showEarAlloc(&earAlloc);
        }
#endif

        /* Do not allocate EAR if no registers exist */
        if (earAlloc.numRHs) {
            if (ar5212EarAllocate(&earAlloc, &(pArDev->pHalInfo->pEarHead)) == FALSE) {
                uiPrintf("ar5212DevInit: Could not allocate memory for EAR structures\n");
                goto devInitError;
            }

            if(!ar5212EarCheckAndFill(pArDev->pHalInfo->pEarHead, earBuffer, parsedLocs, &earAlloc)) {
                /* Ear is bad */
                uiPrintf("ar5212DevInit: An unrecoverable EAR error was detected\n");
                goto devInitError;
            }
#ifdef DEBUG
            if (EarDebugLevel >= EAR_DEBUG_BASIC) {
                printEar(pArDev->pHalInfo->pEarHead, &earAlloc);
            }
#endif
        }
    }
#endif /* EAR_SUPPORT */

#ifdef ECOS

    A_CREATE_INTERRUPT(CYGNUM_HAL_INT_WMAC_VEC,
		       pArDev,
		       ar_wmac_isr,
		       ar_wmac_dsr,
		       &IntHandle,
		       &IntObject);

    A_ATTACH_INTERRUPT(IntHandle);

    A_ENABLE_INTERRUPT(CYGNUM_HAL_INT_WMAC_VEC);

#endif /* ECOS */

    status = A_OK;
    goto devInitExit;

devInitError:
    uiPrintf("ar5212DevInit: EEPROM-related failure\n");
    ar5212DevFini(pArDev);
    status = A_HARDWARE;

devInitExit:
    /* Cleanup cached EEPROM */
    A_DRIVER_FREE(pRawEeprom, pArDev->pHalInfo->rawEepromSz);
    pArDev->pHalInfo->pRawEeprom = NULL;
    pArDev->hwInfo.numTxQueues   = HAL_NUM_TX_QUEUES;

    return status;
}

/**************************************************************
 * ar5212DevFini
 *
 * DevFini removes all hwLayer allocated data structures and
 * places the device into reset.
 */
A_STATUS
ar5212DevFini(AR_DEV_INFO *pArDev)
{
    HAL_INFO       *pInfo;
    struct eepMap  *pData;
    HEADER_WMODE   headerMode;
    int            numChannels;

    ASSERT(pArDev && pArDev->pHalInfo);

    pInfo = pArDev->pHalInfo;
    pData = pInfo->pEepData;

    if (pData) {
        ASSERT(pData->version >= EEPROM_VER3);

        if (pData->pEepHeader) {
            A_DRIVER_FREE(pData->pEepHeader, sizeof(EEP_HEADER_INFO));
        }

        if (pData->pPcdacInfo) {
            A_DRIVER_FREE(pData->pPcdacInfo, sizeof(PCDACS_ALL_MODES));
        }

        if (pData->pTrgtPowerInfo) {
            A_DRIVER_FREE(pData->pTrgtPowerInfo, sizeof(TRGT_POWER_ALL_MODES));
        }

        if (pData->pRdEdgesPower) {
            A_DRIVER_FREE(pData->pRdEdgesPower,
                          sizeof(RD_EDGES_POWER) * NUM_EDGES * NUM_CTLS_3_3);
        }

        for (headerMode = headerInfo11A; headerMode <= headerInfo11G; headerMode++) {
            numChannels = pData->modePowerArray5112[headerMode].numChannels;
            if (pData->modePowerArray5112[headerMode].pChannels) {
                A_DRIVER_FREE(pData->modePowerArray5112[headerMode].pChannels,
                              sizeof(A_UINT16) * numChannels);
            }
            if (pData->modePowerArray5112[headerMode].pDataPerChannel) {
                A_DRIVER_FREE(pData->modePowerArray5112[headerMode].pDataPerChannel,
                              sizeof(EXPN_DATA_PER_CHANNEL_5112) * numChannels);
            }
        }

        A_DRIVER_FREE(pData, sizeof(struct eepMap));
        pInfo->pEepData = NULL;
    }

    if (pInfo->pGainValues) {
        A_DRIVER_FREE(pInfo->pGainValues, sizeof(struct gainValues));
        pInfo->pGainValues = NULL;
    }

    ar5212FreeRfBanks(pArDev, pInfo);

#if defined(EAR_SUPPORT)
    if (pInfo->pEarHead) {
        if (pInfo->pEarHead->numRHs) {
            A_DRIVER_FREE(pInfo->pEarHead->pRH, pInfo->pEarHead->earSize);
            pInfo->pEarHead->pRH = NULL;
        }

        A_DRIVER_FREE(pInfo->pEarHead, sizeof(EAR_HEADER));
        pInfo->pEarHead = NULL;
    }
#endif /* EAR_SUPPORT */

    /* Free HAL info struct */
    A_DRIVER_FREE(pArDev->pHalInfo, sizeof(*pArDev->pHalInfo));
    pArDev->pHalInfo = NULL;

    return A_OK;
}

/**************************************************************
 * ar5212FillCapabilityInfo
 *
 * Fill all software cached or static hardware state information
 *
 * Returns failure for various EEPROM reading failure cases
 */
A_STATUS
ar5212FillCapabilityInfo(AR_DEV_INFO *pArDev)
{
    A_STATUS            status = A_OK;
    AR_DEV_CAPABILITIES *arCap = &(pArDev->hwInfo);
    EEP_HEADER_INFO     *pEepHeader = pArDev->pHalInfo->pEepData->pEepHeader;
    A_UINT16            regDmn, regCapBits;
    A_UINT32            wMode = 0;
    struct eepMap        *pMap;

    pMap = pArDev->pHalInfo->pEepData;

    /* Read and save EEPROM regDomain */
    status = ar5212EepromRead(pArDev, REGULATORY_DOMAIN_OFFSET, &regDmn);
    if (status != A_OK) {
        return status;
    }
    arCap->regDomain = regDmn;

    /* Read and save the EEPROM capability bits */
    if (pMap && pMap->version >= EEPROM_VER5_3) {
        status = ar5212EepromRead(pArDev, EEPROM_REG_CAPABILITIES_OFFSET, &regCapBits);
        if (status != A_OK) {
            return status;
        }
        arCap->regCapBits = (regCapBits & EEPCAP_REG_MASK) >> (EEPCAP_REG_EN_FCC_MID_S - 1);
        arCap->regCapBits |= 1;
    } else {
        status = ar5212EepromRead(pArDev, EEPROM_REG_CAPABILITIES_OFFSET, &regCapBits);
        if (status != A_OK) {
            return status;
        }
        regCapBits &= (EEPCAP_REG_EN_KK_NEW_11A_M | EEPCAP_REG_EN_KK_U1_ODD_M);
        arCap->regCapBits = (regCapBits & EEPCAP_REG_MASK) >> (EEPCAP_REG_EN_FCC_MID_S - 1);
	}

    /* Get wireless mode */
    if (pEepHeader->Amode) {
        wMode |= MODE_SELECT_11A;
        if (!pEepHeader->turbo5Disable) {
            wMode |= MODE_SELECT_TURBO;
        }
    }
    if (pEepHeader->Bmode) {
        wMode |= MODE_SELECT_11B;
    }
    if (pEepHeader->Gmode) {
        wMode |= MODE_SELECT_11G;
        if (!pEepHeader->turbo2Disable) {
            wMode |= MODE_SELECT_108G;
        }
    }
    arCap->wirelessModes = wMode;

    arCap->low2GhzChan        = 2312;
    if (IS_5112(pArDev)) {
        arCap->high2GhzChan   = 2500;
    } else {
        arCap->high2GhzChan   = 2732;
    }
    arCap->low5GhzChan        = 4920;
    arCap->high5GhzChan       = 6100;

    arCap->supportCipherCKIP    = FALSE;
    arCap->supportCipherTKIP    = TRUE;

    arCap->supportCipherAES_CCM = TRUE;

    arCap->supportMicCKIP       = FALSE;
    arCap->supportMicTKIP       = TRUE;
    arCap->supportMicAES_CCM    = TRUE;

    arCap->chanSpreadSupport    = TRUE;

    arCap->compressSupport     =
    arCap->burstSupport        =
    arCap->fastFramesSupport   =
    arCap->chapTuningSupport   =
    arCap->turboGSupport       =
    arCap->turboPrimeSupport   = 
        (pArDev->hwInfo.macVersion > MAC_SREV_VERSION_VENICE) ?
            1 : (pArDev->hwInfo.macRevision > 1);

    arCap->deviceType          = pEepHeader->deviceType;
    arCap->twiceAntennaGain5G  = pEepHeader->antennaGainMax[0];
    arCap->twiceAntennaGain2G  = pEepHeader->antennaGainMax[1];

    return status;
}

/**************************************************************
 * ar5212RecordSerialNumber
 *
 * Copy EEPROM locations with serial number into provided string
 *
 * Returns failure for various EEPROM reading failure cases
 */
static A_STATUS
ar5212RecordSerialNumber(AR_DEV_INFO *pArDev, A_CHAR *pSerNum)
{
    A_STATUS status = A_OK;
    A_UINT16 eepVal;
    int      i;

    for (i = 0; i < (EEPROM_SERIAL_NUM_SIZE / 2); i++) {
        status = ar5212EepromRead(pArDev, EEPROM_SERIAL_NUM_OFFSET + i, &eepVal);
        if (status != A_OK) {
            break;
        }
        pSerNum[2 * i]     = (A_CHAR)(eepVal & 0xFF);
        pSerNum[2 * i + 1] = (A_CHAR)((eepVal >> 8) & 0xFF);
    }
    pSerNum[EEPROM_SERIAL_NUM_SIZE] = '\0';

    return status;
}

/************** EEPROM REV 3 ATTACH FUNCTIONS *****************/

/**************************************************************
 * ar5212VerifyEepromChecksum
 *
 * Verifies the EEPROM calibration data XOR checksum
 */
A_STATUS
ar5212VerifyEepromChecksum(A_UINT16 *pRawEeprom, A_UINT32 eepEndLoc)
{

    A_STATUS status;
    A_UINT16 *pData16 = pRawEeprom;
    A_UINT16 chkSum = 0;
    A_UINT32 addr;

    /* EEPROM checksum validation */
    for (addr = ATHEROS_EEPROM_OFFSET; addr < eepEndLoc; addr++) {
        chkSum ^= *(pData16++);
    }

    if (chkSum != 0xffff) {
        uiPrintf("ar5212VerifyEepromChecksum: Checksum %04x != 0xffff\n", chkSum);
        status = A_HARDWARE;
    } else {
        status = A_OK;
    }

    return status;
}

/**************************************************************
 * ar5212AllocateProm
 *
 * Allocates memory for the EEPROM structures
 */
A_STATUS
ar5212AllocateProm(EEP_MAP *pEepMap)
{
    /* TODO: only allocate 2.4 info if configured */
    int i;

    /* allocate the struct to hold the header info */
    pEepMap->pEepHeader = (EEP_HEADER_INFO *)A_DRIVER_MALLOC(sizeof(EEP_HEADER_INFO));
    if (!pEepMap->pEepHeader) {
        uiPrintf("Unable to allocate eeprom structure for header info\n");
        return A_NO_MEMORY;
    }
    A_MEM_ZERO(pEepMap->pEepHeader, sizeof(EEP_HEADER_INFO));

    /* allocate the struct to hold the pcdac/power info */
    pEepMap->pPcdacInfo = (PCDACS_ALL_MODES *)A_DRIVER_MALLOC(sizeof(PCDACS_ALL_MODES));
    if (!pEepMap->pPcdacInfo) {
        uiPrintf("Unable to allocate eeprom structure for pcdac/power info\n");
        return A_NO_MEMORY;
    }
    A_MEM_ZERO(pEepMap->pPcdacInfo, sizeof(PCDACS_ALL_MODES));

    pEepMap->pPcdacInfo->numChannels11a = NUM_11A_EEPROM_CHANNELS;
    pEepMap->pPcdacInfo->numChannels2_4 = NUM_2_4_EEPROM_CHANNELS;

    for (i = 0; i < NUM_11A_EEPROM_CHANNELS; i ++) {
        pEepMap->pPcdacInfo->DataPerChannel11a[i].numPcdacValues = NUM_PCDAC_VALUES;
    }

    /* the channel list for 2.4 is fixed, fill this in here */
    for (i = 0; i < NUM_2_4_EEPROM_CHANNELS; i++) {
        pEepMap->pPcdacInfo->Channels11b[i] = channels11b[i];
        pEepMap->pPcdacInfo->Channels11g[i] = channels11g[i];
        pEepMap->pPcdacInfo->DataPerChannel11b[i].numPcdacValues = NUM_PCDAC_VALUES;
        pEepMap->pPcdacInfo->DataPerChannel11g[i].numPcdacValues = NUM_PCDAC_VALUES;
    }

    /* allocate the structure to hold target power info */
    pEepMap->pTrgtPowerInfo = (TRGT_POWER_ALL_MODES *)A_DRIVER_MALLOC(sizeof(TRGT_POWER_ALL_MODES));
    if (!pEepMap->pTrgtPowerInfo) {
        uiPrintf("Unable to allocate eeprom structure for target power info\n");
        return A_NO_MEMORY;
    }
    A_MEM_ZERO(pEepMap->pTrgtPowerInfo, sizeof(TRGT_POWER_ALL_MODES));

    /* allocate structure for RD edges */
    pEepMap->pRdEdgesPower = (RD_EDGES_POWER *)A_DRIVER_MALLOC(sizeof(RD_EDGES_POWER) *
        NUM_EDGES * NUM_CTLS_3_3);
    if (!pEepMap->pRdEdgesPower) {
        uiPrintf("Unable to allocate eeprom structure for RD edges info\n");
        return A_NO_MEMORY;
    }
    A_MEM_ZERO(pEepMap->pRdEdgesPower, sizeof(RD_EDGES_POWER) * NUM_EDGES * NUM_CTLS_3_3);
    return A_OK;
}

/**************************************************************
 * ar5212ReadEepromFreqPierInfo
 *
 * Now copy EEPROM frequency pier contents into the allocated space
 */
INLINE void
ar5212ReadEepromFreqPierInfo(AR_DEV_INFO *pArDev, A_UINT16 *pRawEeprom, A_UINT16 offsetIn)
{
    A_UINT16             tempValue;
    A_UINT32             i;
    A_UINT16             offset = offsetIn;
    PCDACS_ALL_MODES     *pEepromData;
    struct eepMap        *pMap;

    ASSERT(pArDev->pHalInfo && pArDev->pHalInfo->pEepData);

    pMap           = pArDev->pHalInfo->pEepData;
    pEepromData    = pMap->pPcdacInfo;

    if (pMap->version >= EEPROM_VER3_3) {
        for (i = 0; i < pEepromData->numChannels11a; i += 2) {
            tempValue = pRawEeprom[offset++];
            pEepromData->Channels11a[i] = (A_UINT16)(( tempValue >> 8 ) & FREQ_MASK_3_3);
            pEepromData->Channels11a[i + 1] = (A_UINT16)( tempValue & FREQ_MASK_3_3);
        }
    } else {
        tempValue = pRawEeprom[offset++];
        pEepromData->Channels11a[0] = (A_UINT16)(( tempValue >> 9 ) & FREQ_MASK);
        pEepromData->Channels11a[1] = (A_UINT16)(( tempValue >> 2 ) & FREQ_MASK);
        pEepromData->Channels11a[2] = (A_UINT16)(( tempValue << 5 ) & FREQ_MASK);

        tempValue = pRawEeprom[offset++];
        pEepromData->Channels11a[2] = (A_UINT16)((( tempValue >> 11 ) & 0x1f) | pEepromData->Channels11a[2]);
        pEepromData->Channels11a[3] = (A_UINT16)(( tempValue >> 4 ) & FREQ_MASK);
        pEepromData->Channels11a[4] = (A_UINT16)(( tempValue << 3 ) & FREQ_MASK);

        tempValue = pRawEeprom[offset++];
        pEepromData->Channels11a[4] = (A_UINT16)((( tempValue >> 13 ) & 0x7) | pEepromData->Channels11a[4]);
        pEepromData->Channels11a[5] = (A_UINT16)(( tempValue >> 6 ) & FREQ_MASK);
        pEepromData->Channels11a[6] =  (A_UINT16)(( tempValue << 1 ) & FREQ_MASK);

        tempValue = pRawEeprom[offset++];
        pEepromData->Channels11a[6] = (A_UINT16)((( tempValue >> 15 ) & 0x1) | pEepromData->Channels11a[6]);
        pEepromData->Channels11a[7] = (A_UINT16)(( tempValue >> 8 ) & FREQ_MASK);
        pEepromData->Channels11a[8] =  (A_UINT16)(( tempValue >> 1 ) & FREQ_MASK);
        pEepromData->Channels11a[9] =  (A_UINT16)(( tempValue << 6 ) & FREQ_MASK);

        tempValue = pRawEeprom[offset++];
        pEepromData->Channels11a[9] = (A_UINT16)((( tempValue >> 10 ) & 0x3f) | pEepromData->Channels11a[9]);
    }

    for (i = 0; i < pEepromData->numChannels11a; i++ ) {
        pEepromData->Channels11a[i] = fbin2freq(pMap->version, pEepromData->Channels11a[i], FALSE);
    }
}

/**************************************************************
 * ar5212ReadEepromRawPowerCalInfo
 *
 * Now copy EEPROM Raw Power Calibration per frequency contents
 * into the allocated space
 */
INLINE void
ar5212ReadEepromRawPowerCalInfo(AR_DEV_INFO *pArDev, A_UINT16 *pRawEeprom, A_UINT16 offSetIn)
{
    A_UINT16             tempValue;
    A_UINT32             i, j;
    A_UINT32             offset = 0;
    HEADER_WMODE         mode;
    A_UINT16             enable24;
    A_UINT16             numChannels = 0;
    DATA_PER_CHANNEL     *pChannelData = NULL;
    PCDACS_ALL_MODES     *pEepromData;
    struct eepMap        *pMap;

    ASSERT(pArDev->pHalInfo && pArDev->pHalInfo->pEepData);

    pMap            = pArDev->pHalInfo->pEepData;
    pEepromData     = pMap->pPcdacInfo;

    enable24        = pMap->pEepHeader->Bmode;
    /*
     * Group 2:  read raw power data for all frequency piers
     *
     * NOTE: Group 2 contains the raw power calibration information
     * for each of the channels that we recorded in ar5212ReadEepromFreqPiersInfo()
     */
    for (mode = headerInfo11A; mode <= headerInfo11G; mode++) {
        A_UINT16 *pChannels = NULL;
        offset = offSetIn;

        switch (mode) {
        case headerInfo11A:
            offset      += GROUP2_OFFSET;
            numChannels  = pEepromData->numChannels11a;
            pChannelData = pEepromData->DataPerChannel11a;
            pChannels    = pEepromData->Channels11a;
            break;

        case headerInfo11B:
            if (!enable24) {
                continue;
            }
            offset      += GROUP3_OFFSET;
            numChannels  = pEepromData->numChannels2_4;
            pChannelData = pEepromData->DataPerChannel11b;
            pChannels    = pEepromData->Channels11b;

            break;

        case headerInfo11G:
            if (!enable24) {
                continue;
            }
            offset      += GROUP4_OFFSET;
            numChannels  = pEepromData->numChannels2_4;
            pChannelData = pEepromData->DataPerChannel11g;
            pChannels    = pEepromData->Channels11g;
            break;

        default:
            ASSERT(0);
        }

        for (i = 0; i < numChannels; i++) {
            pChannelData->channelValue = pChannels[i];

            tempValue = pRawEeprom[offset++];
            pChannelData->pcdacMax     = (A_UINT16)(( tempValue >> 10 ) & PCDAC_MASK);
            pChannelData->pcdacMin     = (A_UINT16)(( tempValue >> 4  ) & PCDAC_MASK);
            pChannelData->PwrValues[0] = (A_UINT16)(( tempValue << 2  ) & POWER_MASK);

            tempValue = pRawEeprom[offset++];
            pChannelData->PwrValues[0] = (A_UINT16)(((tempValue >> 14 ) & 0x3 ) | pChannelData->PwrValues[0]);
            pChannelData->PwrValues[1] = (A_UINT16)((tempValue >> 8 ) & POWER_MASK);
            pChannelData->PwrValues[2] = (A_UINT16)((tempValue >> 2 ) & POWER_MASK);
            pChannelData->PwrValues[3] = (A_UINT16)((tempValue << 4 ) & POWER_MASK);

            tempValue = pRawEeprom[offset++];
            pChannelData->PwrValues[3] = (A_UINT16)(( ( tempValue >> 12 ) & 0xf ) | pChannelData->PwrValues[3]);
            pChannelData->PwrValues[4] = (A_UINT16)(( tempValue >> 6 ) & POWER_MASK);
            pChannelData->PwrValues[5] = (A_UINT16)(tempValue  & POWER_MASK);

            tempValue = pRawEeprom[offset++];
            pChannelData->PwrValues[6] = (A_UINT16)(( tempValue >> 10 ) & POWER_MASK);
            pChannelData->PwrValues[7] = (A_UINT16)(( tempValue >> 4  ) & POWER_MASK);
            pChannelData->PwrValues[8] = (A_UINT16)(( tempValue << 2  ) & POWER_MASK);

            tempValue = pRawEeprom[offset++];
            pChannelData->PwrValues[8]  = (A_UINT16)(( ( tempValue >> 14 ) & 0x3 ) | pChannelData->PwrValues[8]);
            pChannelData->PwrValues[9]  = (A_UINT16)(( tempValue >> 8 ) & POWER_MASK);
            pChannelData->PwrValues[10] = (A_UINT16)(( tempValue >> 2 ) & POWER_MASK);

            getPcdacInterceptsFromPcdacMinMax(pMap->version, pChannelData->pcdacMin,
                        pChannelData->pcdacMax, pChannelData->PcdacValues) ;

            for (j = 0; j < pChannelData->numPcdacValues; j++) {
                pChannelData->PwrValues[j] = (A_UINT16)(PWR_STEP * pChannelData->PwrValues[j]);
                /* Note these values are scaled up. */
            }
            pChannelData++;
        }
    }
}

/************** EEPROM REV 4 5112 Power Extract FUNCTIONS *****************/
/**************************************************************
 * ar5212readPowerDataFromEeprom5112
 *
 * Pulls the 5112 power data from EEPROM and stores it in a temporary structure
 * that records only the EEPROM calibraiton
 */
static void
ar5212readPowerDataFromEeprom5112(AR_DEV_INFO *pArDev, EEPROM_POWER_5112 *pPowerSet,
                                  A_UINT16 startOffset, A_UINT16 maxPiers, A_UINT16 *pRawEeprom,
                                  HEADER_WMODE headerMode)
{
    A_UINT16    i;
    A_UINT16    dbmmask             = 0xff;
    A_UINT16    pcdac_delta_mask    = 0x1f;
    A_UINT16    pcdac_mask          = 0x3f;
    A_UINT16    freqmask            = 0xff;
    A_UINT16    offset, numPiers = 0;
    A_UINT16    freq[NUM_11A_EEPROM_CHANNELS];
    A_UINT16    *pCalPiers;
    A_UINT16    version = pArDev->pHalInfo->pEepData->version;
    EEP_HEADER_INFO *pHeader = pArDev->pHalInfo->pEepData->pEepHeader;

    offset = startOffset;
    if (headerMode == headerInfo11A) {
        while (numPiers < maxPiers) {
            if ((pRawEeprom[offset] & freqmask) == 0) {
                offset++;
                break;
            }
            freq[numPiers] = fbin2freq(version, (pRawEeprom[offset] & freqmask), FALSE);
            numPiers++;

            if (((pRawEeprom[offset] >> 8) & freqmask) == 0) {
                offset++;
                break;
            }
            freq[numPiers] = fbin2freq(version, ((pRawEeprom[offset] >> 8) & freqmask), FALSE);
            numPiers++;
            offset++;
        }
        offset = startOffset + (maxPiers / 2);
    } else {
        pCalPiers = (headerMode == headerInfo11B) ? (pHeader->calPier11b) : (pHeader->calPier11g);
        for (i = 0; i < NUM_2_4_EEPROM_CHANNELS; i++) {
            if (pCalPiers[i] != CHANNEL_UNUSED) {
                freq[numPiers++] = pCalPiers[i];
            }
        }
    }

    pPowerSet->numChannels = numPiers;

    for (i = 0; i < numPiers; i++) {
        pPowerSet->pChannels[i] = freq[i];
        pPowerSet->pDataPerChannel[i].channelValue = freq[i];

        pPowerSet->pDataPerChannel[i].pwr1_xg0 = (A_INT16)((pRawEeprom[offset] & dbmmask) - ((pRawEeprom[offset] >> 7) & 0x1)*256);
        pPowerSet->pDataPerChannel[i].pwr2_xg0 = (A_INT16)(((pRawEeprom[offset] >> 8) & dbmmask) - ((pRawEeprom[offset] >> 15) & 0x1)*256);

        offset++;
        pPowerSet->pDataPerChannel[i].pwr3_xg0 = (A_INT16)((pRawEeprom[offset] & dbmmask) - ((pRawEeprom[offset] >> 7) & 0x1)*256);
        pPowerSet->pDataPerChannel[i].pwr4_xg0 = (A_INT16)(((pRawEeprom[offset] >> 8) & dbmmask) - ((pRawEeprom[offset] >> 15) & 0x1)*256);

        offset++;
        pPowerSet->pDataPerChannel[i].pcd2_delta_xg0 = (A_UINT16)(pRawEeprom[offset] & pcdac_delta_mask);
        pPowerSet->pDataPerChannel[i].pcd3_delta_xg0 = (A_UINT16)((pRawEeprom[offset] >> 5) & pcdac_delta_mask);
        pPowerSet->pDataPerChannel[i].pcd4_delta_xg0 = (A_UINT16)((pRawEeprom[offset] >> 10) & pcdac_delta_mask);

        offset++;
        pPowerSet->pDataPerChannel[i].pwr1_xg3 = (A_INT16)((pRawEeprom[offset] & dbmmask) - ((pRawEeprom[offset] >> 7) & 0x1)*256);
        pPowerSet->pDataPerChannel[i].pwr2_xg3 = (A_INT16)(((pRawEeprom[offset] >> 8) & dbmmask) - ((pRawEeprom[offset] >> 15) & 0x1)*256);

        offset++;
        pPowerSet->pDataPerChannel[i].pwr3_xg3 = (A_INT16)((pRawEeprom[offset] & dbmmask) - ((pRawEeprom[offset] >> 7) & 0x1)*256);
        if (version >= EEPROM_VER4_3) {
            pPowerSet->pDataPerChannel[i].maxPower_t4 = pPowerSet->pDataPerChannel[i].pwr4_xg0;
            pPowerSet->pDataPerChannel[i].pcd1_xg0 = (A_UINT16)((pRawEeprom[offset] >> 8) & pcdac_mask);
        } else {
            pPowerSet->pDataPerChannel[i].maxPower_t4 = (A_INT16)(((pRawEeprom[offset] >> 8) & dbmmask) - ((pRawEeprom[offset] >> 15) & 0x1)*256);
            pPowerSet->pDataPerChannel[i].pcd1_xg0 = 1;
        }

        offset++;
    }
    pPowerSet->xpdMask = pHeader->xgain[headerMode];
}

/**************************************************************
 * ar5212AllocExpnPower5112
 *
 * Allocate the power information based on the number of channels
 * recorded by the calibration.  These values are then initialized.
 */
static A_STATUS
ar5212AllocExpnPower5112(EEPROM_POWER_EXPN_5112 *pPowerExpn, A_UINT16 numChannels,
                               A_UINT16 *pChanList)
{
    A_UINT16    i, j, channelValue;

    /* Allocate the channel array */
    pPowerExpn->pChannels = (A_UINT16 *)A_DRIVER_MALLOC(sizeof(A_UINT16) * numChannels);
    if (NULL == pPowerExpn->pChannels) {
        uiPrintf("unable to allocate raw data struct (gen3)\n");
        return(A_NO_MEMORY);
    }

    /* Allocate the Power Data for each channel */
    pPowerExpn->pDataPerChannel = (EXPN_DATA_PER_CHANNEL_5112 *)A_DRIVER_MALLOC(sizeof(EXPN_DATA_PER_CHANNEL_5112) * numChannels);
    if (NULL == pPowerExpn->pDataPerChannel) {
        uiPrintf("unable to allocate raw data struct data per channel(gen3)\n");
        A_DRIVER_FREE(pPowerExpn->pChannels, sizeof(A_UINT16) * numChannels);
        return(A_NO_MEMORY);
    }

    pPowerExpn->numChannels = numChannels;

    for (i = 0; i < numChannels; i++) {
        channelValue = pChanList[i];

        pPowerExpn->pChannels[i] = channelValue;
        pPowerExpn->pDataPerChannel[i].channelValue = channelValue;

        for (j = 0; j < NUM_XPD_PER_CHANNEL; j++) {
            pPowerExpn->pDataPerChannel[i].pDataPerXPD[j].xpd_gain = j;
            pPowerExpn->pDataPerChannel[i].pDataPerXPD[j].numPcdacs = 0;
        }
        pPowerExpn->pDataPerChannel[i].pDataPerXPD[0].numPcdacs = 4;
        pPowerExpn->pDataPerChannel[i].pDataPerXPD[3].numPcdacs = 3;
    }

    return(A_OK);
}

/**************************************************************
 * ar5212ExpandPower5112
 *
 * Expand the dataSet from the calibration information into the
 * final power structure for 5112
 */
static A_BOOL
ar5212ExpandPower5112(EEPROM_POWER_5112 *pCalDataset, EEPROM_POWER_EXPN_5112 *pPowerExpn)
{
    A_UINT16                     ii, jj, kk;
    A_INT16                      maxPower_t4;
    EXPN_DATA_PER_XPD_5112       *pExpnXPD;
    EEPROM_DATA_PER_CHANNEL_5112 *pCalCh;   /* ptr to array of info held per channel */
    A_UINT16                     xgainList[2];
    A_UINT16                     xpdMask;

    pPowerExpn->xpdMask = pCalDataset->xpdMask;

    xgainList[0] = 0xDEAD;
    xgainList[1] = 0xDEAD;

    kk = 0;
    xpdMask = pPowerExpn->xpdMask;

    for (jj = 0; jj < NUM_XPD_PER_CHANNEL; jj++) {
        if (((xpdMask >> jj) & 1) > 0) {
            if (kk > 1) {
                uiPrintf("ar5212ExpandPower5112: Error - A maximum of 2 xpdGains supported in dataset\n");
                return FALSE;
            }
            xgainList[kk++] = (A_UINT16) jj;
        }
    }

    pPowerExpn->numChannels = pCalDataset->numChannels;
    for (ii = 0; ii < pPowerExpn->numChannels; ii++) {
        pCalCh = &(pCalDataset->pDataPerChannel[ii]);
        pPowerExpn->pDataPerChannel[ii].channelValue = pCalCh->channelValue;
        pPowerExpn->pDataPerChannel[ii].maxPower_t4  = pCalCh->maxPower_t4;
        maxPower_t4 = pPowerExpn->pDataPerChannel[ii].maxPower_t4;

        if (xgainList[1] == 0xDEAD) {
            for (jj=0; jj < NUM_XPD_PER_CHANNEL; jj++) {
                pPowerExpn->pDataPerChannel[ii].pDataPerXPD[jj].numPcdacs = 0;
            }

            jj = xgainList[0];
            pExpnXPD = &(pPowerExpn->pDataPerChannel[ii].pDataPerXPD[jj]);
            pExpnXPD->numPcdacs = 4;
            pExpnXPD->pcdac[0] = pCalCh->pcd1_xg0;
            pExpnXPD->pcdac[1] = (A_UINT16)(pExpnXPD->pcdac[0] + pCalCh->pcd2_delta_xg0);
            pExpnXPD->pcdac[2] = (A_UINT16)(pExpnXPD->pcdac[1] + pCalCh->pcd3_delta_xg0);
            pExpnXPD->pcdac[3] = (A_UINT16)(pExpnXPD->pcdac[2] + pCalCh->pcd4_delta_xg0);

            pExpnXPD->pwr_t4[0] = pCalCh->pwr1_xg0;
            pExpnXPD->pwr_t4[1] = pCalCh->pwr2_xg0;
            pExpnXPD->pwr_t4[2] = pCalCh->pwr3_xg0;
            pExpnXPD->pwr_t4[3] = pCalCh->pwr4_xg0;

        } else {
            for (jj=0; jj < NUM_XPD_PER_CHANNEL; jj++) {
                pPowerExpn->pDataPerChannel[ii].pDataPerXPD[jj].numPcdacs = 0;
            }

            pPowerExpn->pDataPerChannel[ii].pDataPerXPD[xgainList[0]].pcdac[0] = pCalCh->pcd1_xg0;
            pPowerExpn->pDataPerChannel[ii].pDataPerXPD[xgainList[1]].pcdac[0] = 20;
            pPowerExpn->pDataPerChannel[ii].pDataPerXPD[xgainList[1]].pcdac[1] = 35;
            pPowerExpn->pDataPerChannel[ii].pDataPerXPD[xgainList[1]].pcdac[2] = 63;

            jj = xgainList[0];
            pExpnXPD = &(pPowerExpn->pDataPerChannel[ii].pDataPerXPD[jj]);
            pExpnXPD->numPcdacs = 4;
            pExpnXPD->pcdac[1] = (A_UINT16)(pExpnXPD->pcdac[0] + pCalCh->pcd2_delta_xg0);
            pExpnXPD->pcdac[2] = (A_UINT16)(pExpnXPD->pcdac[1] + pCalCh->pcd3_delta_xg0);
            pExpnXPD->pcdac[3] = (A_UINT16)(pExpnXPD->pcdac[2] + pCalCh->pcd4_delta_xg0);
            pExpnXPD->pwr_t4[0] = pCalCh->pwr1_xg0;
            pExpnXPD->pwr_t4[1] = pCalCh->pwr2_xg0;
            pExpnXPD->pwr_t4[2] = pCalCh->pwr3_xg0;
            pExpnXPD->pwr_t4[3] = pCalCh->pwr4_xg0;

            jj = xgainList[1];
            pExpnXPD = &(pPowerExpn->pDataPerChannel[ii].pDataPerXPD[jj]);
            pExpnXPD->numPcdacs = 3;

            pExpnXPD->pwr_t4[0] = pCalCh->pwr1_xg3;
            pExpnXPD->pwr_t4[1] = pCalCh->pwr2_xg3;
            pExpnXPD->pwr_t4[2] = pCalCh->pwr3_xg3;
        }
    }
    return TRUE;
}

/**************************************************************
 * ar5212ReadEepromPowerCal5112
 *
 * Allocate, expand and fill expanded power structure for 5112
 * cal'ed EEPROM across all modes.
 */
static A_STATUS
ar5212ReadEepromPowerCal5112(AR_DEV_INFO *pArDev, A_UINT16 *pRawEeprom,
                             A_UINT16 offSetIn, EEPROM_POWER_EXPN_5112 *pPowerExpn)
{
    A_STATUS          status = A_OK;
    A_UINT16          offset= offSetIn;
    A_UINT16          maxPiers = 0;
    EEPROM_POWER_5112 eepPower;
    HEADER_WMODE      headerMode;
    EEP_HEADER_INFO   *pHeaderInfo;

    pHeaderInfo = pArDev->pHalInfo->pEepData->pEepHeader;

    for (headerMode = headerInfo11A; headerMode <= headerInfo11G; headerMode++) {
        switch (headerMode) {
        case headerInfo11A:
            if (!pHeaderInfo->Amode) {
                continue;
            }
            maxPiers = NUM_11A_EEPROM_CHANNELS;
            break;
        case headerInfo11B:
            if (!pHeaderInfo->Bmode) {
                continue;
            }
            maxPiers = NUM_2_4_EEPROM_CHANNELS;
            break;
        case headerInfo11G:
            if (!pHeaderInfo->Gmode) {
                continue;
            }
            maxPiers = NUM_2_4_EEPROM_CHANNELS;
            break;
        }

        /* Zero the reused basic power structure */
        A_MEM_ZERO(&eepPower, sizeof(EEPROM_POWER_5112));

        /* Copy from the EEPROM locations to the basic power structure */
        ar5212readPowerDataFromEeprom5112(pArDev, &eepPower, offset, maxPiers, pRawEeprom, headerMode);

        /* Allocate the expanded Power structure for easier use by reset */
        status = ar5212AllocExpnPower5112(&pPowerExpn[headerMode], eepPower.numChannels, eepPower.pChannels);
        if (status != A_OK) {
            uiPrintf("ar5212ReadEepromPowerCal5112: Failed to allocate power structs\n");
            return status;
        }

        /* Copy and expand the basic power structure into the expanded structure */
        if(ar5212ExpandPower5112(&eepPower, &pPowerExpn[headerMode]) == FALSE) {
            return A_ERROR;
        }
        offset += (eepPower.numChannels * 5);
        if (headerMode == headerInfo11A) {
            /* Add 5 locations for the 11A frequency piers */
            offset += 5;
        }
    }

    return(status);
}
/************** END EEPROM REV 4 5112 Power Extract *****************/

/**************************************************************
 * ar5212ReadEepromTargetPowerCalInfo
 *
 * Copy EEPROM Target Power Calbration per rate contents
 * into the allocated space
 */
INLINE void
ar5212ReadEepromTargetPowerCalInfo(AR_DEV_INFO *pArDev, A_UINT16 *pRawEeprom, A_UINT16 offsetIn, A_UINT16 *endOffset)
{
    A_UINT16             tempValue;
    A_UINT32             i;
    A_UINT16             offset = 0;
    A_UINT16             enable24;
    HEADER_WMODE         mode;
    A_UINT16             numChannels = 0;
    struct eepMap        *pMap;

    ASSERT(pArDev->pHalInfo && pArDev->pHalInfo->pEepData);

    pMap            = pArDev->pHalInfo->pEepData;
    enable24        = pMap->pEepHeader->Bmode;

    for (mode = headerInfo11A; mode <= headerInfo11G; mode++) {
        TRGT_POWER_INFO *pPowerInfo = NULL;
        A_UINT16        *pNumTrgtChannels = NULL;

        offset = offsetIn;

        switch (mode) {
        case headerInfo11A:
            offset          += GROUP5_OFFSET;
            numChannels      = NUM_TEST_FREQUENCIES;
            pPowerInfo       = pMap->pTrgtPowerInfo->trgtPwr_11a;
            pNumTrgtChannels = &pMap->pTrgtPowerInfo->numTargetPwr_11a;
            break;

        case headerInfo11B:
            offset          += GROUP6_OFFSET;
            if (!enable24) {
                /* Keep the offset consistent */
                offset += NUM_TARGET_POWER_LOCATIONS_11B;
                continue;
            }
            numChannels      = 2;
            pPowerInfo       = pMap->pTrgtPowerInfo->trgtPwr_11b;
            pNumTrgtChannels = &pMap->pTrgtPowerInfo->numTargetPwr_11b;
            break;

        case headerInfo11G:
            offset          += GROUP7_OFFSET;
            if (!enable24) {
                /* Keep the offset consistent */
                offset += NUM_TARGET_POWER_LOCATIONS_11G;
                continue;
            }
            numChannels      = 3;
            pPowerInfo       = pMap->pTrgtPowerInfo->trgtPwr_11g;
            pNumTrgtChannels = &pMap->pTrgtPowerInfo->numTargetPwr_11g;
            break;

        default:
            ASSERT(0);
        } //end mode switch

        *pNumTrgtChannels = 0;
        for (i = 0; i < numChannels; i++) {
            tempValue = pRawEeprom[offset++];
            if (pMap->version >= EEPROM_VER3_3) {
                pPowerInfo->testChannel = (A_UINT16)(( tempValue >> 8 ) & 0xff);
            } else {
                pPowerInfo->testChannel = (A_UINT16)(( tempValue >> 9 ) & 0x7f);
            }

            if (pPowerInfo->testChannel != 0) {
                /* get the channel value and read rest of info */
                if (mode == headerInfo11A) {
                    pPowerInfo->testChannel = fbin2freq(pMap->version, pPowerInfo->testChannel, FALSE);
                } else {
                    pPowerInfo->testChannel = fbin2freq(pMap->version, pPowerInfo->testChannel, TRUE);
                }

                if (pMap->version >= EEPROM_VER3_3) {
                    pPowerInfo->twicePwr6_24 = (A_UINT16)(( tempValue >> 2 ) & POWER_MASK);
                    pPowerInfo->twicePwr36   = (A_UINT16)(( tempValue << 4 ) & POWER_MASK);
                } else {
                    pPowerInfo->twicePwr6_24 = (A_UINT16)(( tempValue >> 3 ) & POWER_MASK);
                    pPowerInfo->twicePwr36   = (A_UINT16)(( tempValue << 3 ) & POWER_MASK);
                }

                tempValue = pRawEeprom[offset++];
                if (pMap->version >= EEPROM_VER3_3) {
                    pPowerInfo->twicePwr36 =
                        (A_UINT16)(( ( tempValue >> 12 ) & 0xf ) | pPowerInfo->twicePwr36);
                    pPowerInfo->twicePwr48 = (A_UINT16)(( tempValue >> 6 ) & POWER_MASK);
                    pPowerInfo->twicePwr54 = (A_UINT16)( tempValue & POWER_MASK);
                } else {
                    pPowerInfo->twicePwr36 =
                        (A_UINT16)(( ( tempValue >> 13 ) & 0x7 ) | pPowerInfo->twicePwr36);
                    pPowerInfo->twicePwr48 = (A_UINT16)(( tempValue >> 7 ) & POWER_MASK);
                    pPowerInfo->twicePwr54 = (A_UINT16)(( tempValue >> 1 ) & POWER_MASK);
                }
                (*pNumTrgtChannels)++;
            } else {
                /* Keep the offset consistent */
                offset++;
            }
            pPowerInfo++;
        }
    }
    *endOffset = offset;
}

/**************************************************************
 * ar5212ReadEepromCTLInfo
 *
 * Now copy EEPROM Conformance Testing Limits contents
 * into the allocated space
 */
INLINE void
ar5212ReadEepromCTLInfo(AR_DEV_INFO *pArDev, A_UINT16 *pRawEeprom, A_UINT16 offsetIn)
{
    A_UINT16             tempValue;
    A_UINT32             i, j;
    A_UINT32             offset = offsetIn;
    RD_EDGES_POWER       *pRdEdgePwrInfo;
    PCDACS_ALL_MODES     *pEepromData;
    struct eepMap        *pMap;

    ASSERT(pArDev->pHalInfo && pArDev->pHalInfo->pEepData);

    pMap           = pArDev->pHalInfo->pEepData;
    pEepromData    = pMap->pPcdacInfo;
    pRdEdgePwrInfo = pMap->pRdEdgesPower;

    for (i = 0; i < pMap->pEepHeader->numCtls; i++) {
        if (pMap->pEepHeader->ctl[i] == 0) {
            /* Move offset and edges */
            offset += ((pMap->version >= EEPROM_VER3_3) ? 8 : 7);
            pRdEdgePwrInfo += NUM_EDGES;
            continue;
        }

        if (pMap->version >= EEPROM_VER3_3) {
            for (j = 0; j < NUM_EDGES; j += 2) {
                tempValue = pRawEeprom[offset++];
                pRdEdgePwrInfo[j].rdEdge = (A_UINT16)((tempValue >> 8) & FREQ_MASK_3_3);
                pRdEdgePwrInfo[j + 1].rdEdge = (A_UINT16)(tempValue & FREQ_MASK_3_3);
            }
            for (j = 0; j < NUM_EDGES; j += 2) {
                tempValue = pRawEeprom[offset++];
                pRdEdgePwrInfo[j].twice_rdEdgePower =
                                (A_UINT16)((tempValue >> 8) & POWER_MASK);
                pRdEdgePwrInfo[j].flag = (A_BOOL)((tempValue >> 14) & 1);
                pRdEdgePwrInfo[j + 1].twice_rdEdgePower =
                                (A_UINT16)(tempValue & POWER_MASK);
                pRdEdgePwrInfo[j + 1].flag = (A_BOOL)((tempValue >> 6) & 1);
            }
        } else {
            tempValue = pRawEeprom[offset++];
            pRdEdgePwrInfo[0].rdEdge = (A_UINT16)(( tempValue >> 9 ) & FREQ_MASK);
            pRdEdgePwrInfo[1].rdEdge = (A_UINT16)(( tempValue >> 2 ) & FREQ_MASK);
            pRdEdgePwrInfo[2].rdEdge = (A_UINT16)(( tempValue << 5 ) & FREQ_MASK);

            tempValue = pRawEeprom[offset++];
            pRdEdgePwrInfo[2].rdEdge = (A_UINT16)((( tempValue >> 11 ) & 0x1f) | pRdEdgePwrInfo[2].rdEdge);
            pRdEdgePwrInfo[3].rdEdge = (A_UINT16)(( tempValue >> 4 ) & FREQ_MASK);
            pRdEdgePwrInfo[4].rdEdge = (A_UINT16)(( tempValue << 3 ) & FREQ_MASK);

            tempValue = pRawEeprom[offset++];
            pRdEdgePwrInfo[4].rdEdge = (A_UINT16)((( tempValue >> 13 ) & 0x7) | pRdEdgePwrInfo[4].rdEdge);
            pRdEdgePwrInfo[5].rdEdge = (A_UINT16)(( tempValue >> 6 ) & FREQ_MASK);
            pRdEdgePwrInfo[6].rdEdge = (A_UINT16)(( tempValue << 1 ) & FREQ_MASK);

            tempValue = pRawEeprom[offset++];
            pRdEdgePwrInfo[6].rdEdge = (A_UINT16)((( tempValue >> 15 ) & 0x1) | pRdEdgePwrInfo[6].rdEdge);
            pRdEdgePwrInfo[7].rdEdge = (A_UINT16)(( tempValue >> 8 ) & FREQ_MASK);

            pRdEdgePwrInfo[0].twice_rdEdgePower = (A_UINT16)(( tempValue >> 2 ) & POWER_MASK);
            pRdEdgePwrInfo[1].twice_rdEdgePower = (A_UINT16)(( tempValue << 4 ) & POWER_MASK);

            tempValue = pRawEeprom[offset++];
            pRdEdgePwrInfo[1].twice_rdEdgePower = (A_UINT16)((( tempValue >> 12 ) & 0xf) | pRdEdgePwrInfo[1].twice_rdEdgePower);
            pRdEdgePwrInfo[2].twice_rdEdgePower = (A_UINT16)(( tempValue >> 6 ) & POWER_MASK);
            pRdEdgePwrInfo[3].twice_rdEdgePower = (A_UINT16)(tempValue & POWER_MASK);

            tempValue = pRawEeprom[offset++];
            pRdEdgePwrInfo[4].twice_rdEdgePower = (A_UINT16)(( tempValue >> 10 ) & POWER_MASK);
            pRdEdgePwrInfo[5].twice_rdEdgePower = (A_UINT16)(( tempValue >> 4 ) & POWER_MASK);
            pRdEdgePwrInfo[6].twice_rdEdgePower = (A_UINT16)(( tempValue << 2 ) & POWER_MASK);

            tempValue = pRawEeprom[offset++];
            pRdEdgePwrInfo[6].twice_rdEdgePower = (A_UINT16)((( tempValue >> 14 ) & 0x3) | pRdEdgePwrInfo[6].twice_rdEdgePower);
            pRdEdgePwrInfo[7].twice_rdEdgePower = (A_UINT16)(( tempValue >> 8 ) & POWER_MASK);
        }

        for (j = 0; j < NUM_EDGES; j++ ) {
            if (pRdEdgePwrInfo[j].rdEdge != 0 || pRdEdgePwrInfo[j].twice_rdEdgePower != 0) {
                if (((pMap->pEepHeader->ctl[i] & CTL_MODE_M) == CTL_11A) ||
                    ((pMap->pEepHeader->ctl[i] & CTL_MODE_M) == CTL_TURBO))
                {
                    pRdEdgePwrInfo[j].rdEdge = fbin2freq(pMap->version, pRdEdgePwrInfo[j].rdEdge, FALSE);
                } else {
                    pRdEdgePwrInfo[j].rdEdge = fbin2freq(pMap->version, pRdEdgePwrInfo[j].rdEdge, TRUE);
                }
            }
        }
        pRdEdgePwrInfo += NUM_EDGES;
    }
}

/**************************************************************
 * ar5212ReadEepromIntoDataset
 *
 * Now verify and copy EEPROM contents into the allocated space
 */
static A_STATUS
ar5212ReadEepromIntoDataset(AR_DEV_INFO *pArDev, A_UINT16 *pRawEeprom)
{
    A_STATUS      status = A_OK;
    struct eepMap *pMap;
    A_UINT16      offset, endOffset;

    ASSERT(pArDev->pHalInfo && pArDev->pHalInfo->pEepData);
    pMap = pArDev->pHalInfo->pEepData;

    /* Read the header information here */
    ar5212ReadHeaderInfo(pArDev, pMap->pEepHeader, pRawEeprom);

    /* Require 5112 devices to have EEPROM 4.0 EEP_MAP set */
    if (IS_5112(pArDev) && !pMap->pEepHeader->eepMap) {
        uiPrintf("ar5212ReadEepromIntoDataset: ERROR - 5112 devices must have an EEPROM 4.0 with the EEP_MAP set\n");
        status = A_HARDWARE;
        return status;
    }

    /*
     * Group 1: frequency pier locations readback
     * check that the structure has been populated with enough space to hold the channels
     * Frequency piers are selected calibrated frequencies... all other frequencies
     * interpolate their values between the nearest piers.
     *
     * NOTE: Group 1 contains the 5 GHz channel numbers that have dBm->pcdac calibrated information
     */
    offset = (A_UINT16)(((pMap->version >= EEPROM_VER3_3) ? EEPROM_GROUPS_OFFSET3_3 : EEPROM_GROUPS_OFFSET3_2) +
        GROUP1_OFFSET - ATHEROS_EEPROM_OFFSET);
    ar5212ReadEepromFreqPierInfo(pArDev, pRawEeprom, offset);

    /*
     * Group 2:  readback data for all frequency piers
     *
     * NOTE: Group 2 contains the raw power calibration information for each of the channels
     * that we recorded above
     */
    offset = (A_UINT16)(((pMap->version >= EEPROM_VER3_3) ?
             EEPROM_GROUPS_OFFSET3_3 : EEPROM_GROUPS_OFFSET3_2) - ATHEROS_EEPROM_OFFSET);
    if ((pMap->version >= EEPROM_VER4_0) && pMap->pEepHeader->eepMap) {
        status = ar5212ReadEepromPowerCal5112(pArDev, pRawEeprom, offset, pMap->modePowerArray5112);
        if (status != A_OK) {
            return status;
        }
    } else {
        ar5212ReadEepromRawPowerCalInfo(pArDev, pRawEeprom, offset);
    }

    /*
     * Group 5: target power values per rate
     *
     * NOTE: Group 5 contains the recorded maximum power in dB that can be attained
     * for the given rate.
     */
    /* Read the power per rate info for test channels */
    if (pMap->version >= EEPROM_VER4_0) {
        offset = (A_UINT16)(pMap->pEepHeader->targetPowersStart - ATHEROS_EEPROM_OFFSET - GROUP5_OFFSET);
    } else if (pMap->version >= EEPROM_VER3_3) {
        offset = EEPROM_GROUPS_OFFSET3_3 - ATHEROS_EEPROM_OFFSET;
    } else {
        offset = EEPROM_GROUPS_OFFSET3_2 - ATHEROS_EEPROM_OFFSET;
    }
    ar5212ReadEepromTargetPowerCalInfo(pArDev, pRawEeprom, offset, &endOffset);

    /*
     * Group 8: Conformance Test Limits information
     *
     * NOTE: Group 8 contains the values to limit the maximum transmit power
     * value based on any band edge violations.
     */
    /* Read the RD edge power limits */
    ar5212ReadEepromCTLInfo(pArDev, pRawEeprom, endOffset);

    return status;
}

/**************************************************************
 * ar5212ReadHeaderInfo
 *
 * Read the individual header fields for a Rev 3 EEPROM
 */
static void
ar5212ReadHeaderInfo(AR_DEV_INFO *pArDev, EEP_HEADER_INFO *pHeaderInfo, A_UINT16* pRawEeprom)
{
    A_UINT16 tempValue;
    A_UINT32 offset;
    A_UINT16 i;
    A_UINT32 *pHeadOff;
    A_UINT16 version = pArDev->pHalInfo->pEepData->version;

    static A_UINT32 headerOffset3_0[] = {
        0x00C2, /* 0 - Mode bits, device type, max turbo power */
        0x00C4, /* 1 - 2.4 and 5 antenna gain */
        0x00C5, /* 2 - Begin 11A modal section */
        0x00D0, /* 3 - Begin 11B modal section */
        0x00DA, /* 4 - Begin 11G modal section */
        0x00E4  /* 5 - Begin CTL section */
    };
    static A_UINT32 headerOffset3_3[] = {
        0x00C2, /* 0 - Mode bits, device type, max turbo power */
        0x00C3, /* 1 - 2.4 and 5 antenna gain */
        0x00D4, /* 2 - Begin 11A modal section */
        0x00F2, /* 3 - Begin 11B modal section */
        0x010D, /* 4 - Begin 11G modal section */
        0x0128  /* 5 - Begin CTL section */
    };

    /* Initialize cckOfdmGainDelta for < 4.2 EEPROM's */
    pHeaderInfo->cckOfdmGainDelta = CCK_OFDM_GAIN_DELTA;
    pHeaderInfo->scaledCh14FilterCckDelta = TENX_CH14_FILTER_CCK_DELTA_INIT;

    if (version >= EEPROM_VER3_3) {
        pHeadOff = headerOffset3_3;
        pHeaderInfo->numCtls = NUM_CTLS_3_3;
    } else {
        pHeadOff = headerOffset3_0;
        pHeaderInfo->numCtls = NUM_CTLS;
    }

    offset = pHeadOff[0] - ATHEROS_EEPROM_OFFSET;
    tempValue = pRawEeprom[offset++];
    pHeaderInfo->rfKill           = (tempValue >> 14) & 0x01;
    pHeaderInfo->deviceType       = (tempValue >> 11) & 0x07;
    pHeaderInfo->turbo2WMaxPower5 = (tempValue >> 4) & 0x7F;
    pHeaderInfo->turbo5Disable    = (tempValue >> 15) & 0x01;
    pHeaderInfo->turbo2Disable    = (tempValue >> 3) & 0x01;
    pHeaderInfo->Gmode            = (tempValue >> 2) & 0x01;
    pHeaderInfo->Bmode            = (tempValue >> 1) & 0x01;
    pHeaderInfo->Amode            = (tempValue & 0x01) | ar5212EnableMKKNew11a(pArDev);

    offset = pHeadOff[1] - ATHEROS_EEPROM_OFFSET;
    tempValue = pRawEeprom[offset++];
    pHeaderInfo->antennaGainMax[0] = (A_INT8)((tempValue >> 8) & 0xFF);
    pHeaderInfo->antennaGainMax[1] = (A_INT8)(tempValue & 0xFF);

    if (version >= EEPROM_VER4_0) {
        tempValue = pRawEeprom[offset++];
        pHeaderInfo->eepMap   = (tempValue >> 14) & 0x03;
#if defined(EAR_SUPPORT)
        pHeaderInfo->earStart = tempValue & 0x0FFF;
#endif /* EAR_SUPPORT */
        tempValue = pRawEeprom[offset++];
        pHeaderInfo->targetPowersStart = tempValue & 0x0FFF;
        pHeaderInfo->exist32kHzCrystal = (A_BOOL)((tempValue >> 14) & 0x01);
    }

    /* Read the moded sections of the EEPROM header in the order A, B, G */
    for (i = headerInfo11A; i <= headerInfo11G; i++) {
        /* Set the offset via the index */
        offset = pHeadOff[2 + i] - ATHEROS_EEPROM_OFFSET;

        tempValue = pRawEeprom[offset++];
        pHeaderInfo->switchSettling[i] = (A_UINT16)((tempValue >> 8) & 0x7f);
        pHeaderInfo->txrxAtten[i] = (A_UINT16)((tempValue >> 2) & 0x3f);
        pHeaderInfo->antennaControl[0][i] = (A_UINT16)((tempValue << 4) & 0x3f);

        tempValue = pRawEeprom[offset++];
        pHeaderInfo->antennaControl[0][i] = (A_UINT16)(((tempValue >> 12) & 0x0f) | pHeaderInfo->antennaControl[0][i]);
        pHeaderInfo->antennaControl[1][i] = (A_UINT16)((tempValue >> 6) & 0x3f);
        pHeaderInfo->antennaControl[2][i] = (A_UINT16)(tempValue  & 0x3f);

        tempValue = pRawEeprom[offset++];
        pHeaderInfo->antennaControl[3][i] = (A_UINT16)((tempValue >> 10)  & 0x3f);
        pHeaderInfo->antennaControl[4][i] = (A_UINT16)((tempValue >> 4)  & 0x3f);
        pHeaderInfo->antennaControl[5][i] = (A_UINT16)((tempValue << 2)  & 0x3f);

        tempValue = pRawEeprom[offset++];
        pHeaderInfo->antennaControl[5][i] = (A_UINT16)(((tempValue >> 14)  & 0x03) | pHeaderInfo->antennaControl[5][i]);
        pHeaderInfo->antennaControl[6][i] = (A_UINT16)((tempValue >> 8)  & 0x3f);
        pHeaderInfo->antennaControl[7][i] = (A_UINT16)((tempValue >> 2)  & 0x3f);
        pHeaderInfo->antennaControl[8][i] = (A_UINT16)((tempValue << 4)  & 0x3f);

        tempValue = pRawEeprom[offset++];
        pHeaderInfo->antennaControl[8][i] = (A_UINT16)(((tempValue >> 12)  & 0x0f) | pHeaderInfo->antennaControl[8][i]);
        pHeaderInfo->antennaControl[9][i] = (A_UINT16)((tempValue >> 6)  & 0x3f);
        pHeaderInfo->antennaControl[10][i] = (A_UINT16)(tempValue & 0x3f);

        tempValue = pRawEeprom[offset++];
        pHeaderInfo->adcDesiredSize[i] = (A_INT8)((tempValue >> 8)  & 0xff);
        switch (i) {
        case headerInfo11A:
            pHeaderInfo->ob4 = (A_UINT16)((tempValue >> 5)  & 0x07);
            pHeaderInfo->db4 = (A_UINT16)((tempValue >> 2)  & 0x07);
            pHeaderInfo->ob3 = (A_UINT16)((tempValue << 1)  & 0x07);
            break;
        case headerInfo11B:
            pHeaderInfo->obFor24 = (A_UINT16)((tempValue >> 4)  & 0x07);
            pHeaderInfo->dbFor24 = (A_UINT16)(tempValue & 0x07);
            break;
        case headerInfo11G:
            pHeaderInfo->obFor24g = (A_UINT16)((tempValue >> 4)  & 0x07);
            pHeaderInfo->dbFor24g = (A_UINT16)(tempValue & 0x07);
            break;
        }

        if (i == headerInfo11A) {
            tempValue = pRawEeprom[offset++];
            pHeaderInfo->ob3 = (A_UINT16)(((tempValue >> 15)  & 0x01) | pHeaderInfo->ob3);
            pHeaderInfo->db3 = (A_UINT16)((tempValue >> 12)  & 0x07);
            pHeaderInfo->ob2 = (A_UINT16)((tempValue >> 9)  & 0x07);
            pHeaderInfo->db2 = (A_UINT16)((tempValue >> 6)  & 0x07);
            pHeaderInfo->ob1 = (A_UINT16)((tempValue >> 3)  & 0x07);
            pHeaderInfo->db1 = (A_UINT16)(tempValue & 0x07);
        }

        tempValue = pRawEeprom[offset++];
        pHeaderInfo->txEndToXLNAOn[i] = (A_UINT16)((tempValue >> 8)  & 0xff);
        pHeaderInfo->thresh62[i] = (A_UINT16)(tempValue  & 0xff);

        tempValue = pRawEeprom[offset++];
        pHeaderInfo->txEndToXPAOff[i] = (A_UINT16)((tempValue >> 8)  & 0xff);
        pHeaderInfo->txFrameToXPAOn[i] = (A_UINT16)(tempValue  & 0xff);

        tempValue = pRawEeprom[offset++];
        pHeaderInfo->pgaDesiredSize[i] = (A_INT8)((tempValue >> 8)  & 0xff);
        pHeaderInfo->noiseFloorThresh[i] = (tempValue  & 0xff);
        if (pHeaderInfo->noiseFloorThresh[i] & 0x80) {
            pHeaderInfo->noiseFloorThresh[i] = 0 - ((pHeaderInfo->noiseFloorThresh[i] ^ 0xff) + 1);
        }

        tempValue = pRawEeprom[offset++];
        pHeaderInfo->xlnaGain[i] = (A_UINT16)((tempValue >> 5)  & 0xff);
        pHeaderInfo->xgain[i] = (A_UINT16)((tempValue >> 1)  & 0x0f);
        pHeaderInfo->xpd[i] = (A_UINT8)(tempValue  & 0x01);
        if (version >= EEPROM_VER4_0) {
            switch (i) {
            case headerInfo11A:
                pHeaderInfo->fixedBias5 = (tempValue >> 13) & 0x1;
                break;
            case headerInfo11G:
                pHeaderInfo->fixedBias2 = (tempValue >> 13) & 0x1;
                break;
            }
        }
        if (version >= EEPROM_VER3_3) {
            tempValue = pRawEeprom[offset++];
            pHeaderInfo->falseDetectBackoff[i] = (tempValue >> 6) & 0x7F;
            switch (i) {
            case headerInfo11B:
                pHeaderInfo->ob2GHz[0] = tempValue & 0x7;
                pHeaderInfo->db2GHz[0] = (tempValue >> 3) & 0x7;
                break;
            case headerInfo11G:
                pHeaderInfo->ob2GHz[1] = tempValue & 0x7;
                pHeaderInfo->db2GHz[1] = (tempValue >> 3) & 0x7;
                break;
            case headerInfo11A:
                pHeaderInfo->xrTargetPower5 = tempValue & 0x3F;
                break;
            }
        }
        if (version >= EEPROM_VER3_4) {
            pHeaderInfo->gainI[i] = (tempValue >> 13) & 0x07;

            tempValue = pRawEeprom[offset++];
            pHeaderInfo->gainI[i] |= (tempValue << 3) & 0x38;
            if (i == headerInfo11G) {
                pHeaderInfo->cckOfdmPwrDelta = (tempValue >> 3) & 0xFF;
    			if(version >= EEPROM_VER4_6) {
	    			pHeaderInfo->scaledCh14FilterCckDelta = (tempValue >> 11)  & 0x1f;
		    	}
            }
            if ((i == headerInfo11A) && (version >= EEPROM_VER4_0)) {
                pHeaderInfo->iqCalI[0] = (tempValue >> 8) & 0x3F;
                pHeaderInfo->iqCalQ[0] = (tempValue >> 3) & 0x1F;
            }
        } else {
            pHeaderInfo->gainI[i] = 10;
            pHeaderInfo->cckOfdmPwrDelta = TENX_OFDM_CCK_DELTA_INIT;
        }
        if (version >= EEPROM_VER4_0) {
            if (i == headerInfo11B) {
                tempValue = pRawEeprom[offset++];
                pHeaderInfo->calPier11b[0] = fbin2freq(version, tempValue & 0xff, TRUE);
                pHeaderInfo->calPier11b[1] = fbin2freq(version, (tempValue >> 8) & 0xff, TRUE);
                tempValue = pRawEeprom[offset++];
                pHeaderInfo->calPier11b[2] = fbin2freq(version, tempValue & 0xff, TRUE);
                if (version >= EEPROM_VER4_1) {
                    pHeaderInfo->rxtxMargin[headerInfo11B] = (tempValue >> 8) & 0x3f;
                }
            } else if (i == headerInfo11G) {
                tempValue = pRawEeprom[offset++];
                pHeaderInfo->calPier11g[0] = fbin2freq(version, tempValue & 0xff, TRUE);
                pHeaderInfo->calPier11g[1] = fbin2freq(version, (tempValue >> 8) & 0xff, TRUE);
                tempValue = pRawEeprom[offset++];
                pHeaderInfo->turbo2WMaxPower2 = tempValue & 0x7F;
                pHeaderInfo->xrTargetPower2 = (tempValue >> 7) & 0x3f;
                tempValue = pRawEeprom[offset++];
                pHeaderInfo->calPier11g[2] = fbin2freq(version, tempValue & 0xff, TRUE);
                if (version >= EEPROM_VER4_1) {
                    pHeaderInfo->rxtxMargin[headerInfo11G] = (tempValue >> 8) & 0x3f;
                }
                tempValue = pRawEeprom[offset++];
                pHeaderInfo->iqCalI[1] = (tempValue >> 5) & 0x3F;
                pHeaderInfo->iqCalQ[1] = tempValue & 0x1F;
                if (version >= EEPROM_VER4_2) {
                    tempValue = pRawEeprom[offset++];
                    pHeaderInfo->cckOfdmGainDelta = (A_INT8)(tempValue & 0xFF);
                }
            } else if ((i == headerInfo11A) && (version >= EEPROM_VER4_1)) {
                tempValue = pRawEeprom[offset++];
                pHeaderInfo->rxtxMargin[headerInfo11A] = tempValue & 0x3f;
            }
        }
    }
    if (version < EEPROM_VER3_3) {
        /* Version 3.1, 3.2 specific parameters */
        tempValue = pRawEeprom[offset++];
        pHeaderInfo->ob2GHz[0] = tempValue & 0x7;
        pHeaderInfo->db2GHz[0] = (tempValue >> 3) & 0x7;

        tempValue = pRawEeprom[offset++];
        pHeaderInfo->ob2GHz[1] = tempValue & 0x7;
        pHeaderInfo->db2GHz[1] = (tempValue >> 3) & 0x7;

    }

    /*
     * Read the conformance test limit identifiers
     * These are used to match regulatory domain testing needs with
     * the RD-specific tests that have been calibrated in the EEPROM.
     */
    offset = pHeadOff[5] - ATHEROS_EEPROM_OFFSET;
    for (i = 0; i < pHeaderInfo->numCtls; i += 2) {
        tempValue = pRawEeprom[offset++];
        pHeaderInfo->ctl[i] = (A_UINT16)((tempValue >> 8) & 0xff);
        pHeaderInfo->ctl[i+1] = (A_UINT16)(tempValue & 0xff);
    }

    /* WAR for recent changes to NF scale */
    if (version <= EEPROM_VER3_2) {
        pHeaderInfo->noiseFloorThresh[headerInfo11A] = -54;
        pHeaderInfo->noiseFloorThresh[headerInfo11B] = -1;
        pHeaderInfo->noiseFloorThresh[headerInfo11G] = -1;
    }

    /* WAR to override thresh62 for better 2.4 and 5 operation */
    if (version <= EEPROM_VER3_2) {
        pHeaderInfo->thresh62[headerInfo11A] = 15; // 11A
        pHeaderInfo->thresh62[headerInfo11B] = 28; // 11B
        pHeaderInfo->thresh62[headerInfo11G] = 28; // 11G
    }
}

/**************************************************************************
 * getPcdacInterceptsFromPcdacMinMax
 *
 * Fills in the pcdac table with the correct intercepts given the min/max pcdac
 */
static void
getPcdacInterceptsFromPcdacMinMax(A_UINT16 version, A_UINT16 pcdacMin,
    A_UINT16 pcdacMax, A_UINT16 *pPcdacValues)
{
    const static A_UINT16 intercepts3[] = {0,5,10,20,30,50,70,85,90,95,100};
    const static A_UINT16 intercepts3_2[] = {0,10,20,30,40,50,60,70,80,90,100};

    A_UINT16 i;
    const A_UINT16 *pIntercepts;

    if (version >= EEPROM_VER3_2) {
        pIntercepts = intercepts3_2;
    } else {
        pIntercepts = intercepts3;
    }

    /* loop for the percentages in steps or 5 */
    for (i = 0; i < NUM_INTERCEPTS; i++ ) {
        *pPcdacValues =  (A_UINT16)((pIntercepts[i] * pcdacMax + (100 - pIntercepts[i]) * pcdacMin) / 100);
        pPcdacValues++;
    }
}

/**************************************************************************
 * fbin2freq - Get channel value from binary representation held in eeprom
 *
 * RETURNS: the frequency in MHz
 */
static A_UINT16
fbin2freq(A_UINT16 version, A_UINT16 fbin, A_BOOL is2GHz)
{
    A_UINT16 returnValue;

    /*
     * Reserved value 0xFF provides an empty definition both as
     * an fbin and as a frequency - do not convert
     */
    if (fbin == CHANNEL_UNUSED) {
        return fbin;
    }

    if (is2GHz) {
        if (version <= EEPROM_VER3_2) {
            returnValue = (A_UINT16)(2400 + fbin);
        } else {
            returnValue = (A_UINT16)(2300 + fbin);
        }
    } else {
        if (version <= EEPROM_VER3_2) {

            returnValue = (fbin>62) ? (A_UINT16)(5100 + 10*62 + 5*(fbin-62)) : (A_UINT16)(5100 + 10*fbin);
        } else {
            returnValue = (A_UINT16)(4800 + 5*fbin);
        }
    }
    return returnValue;
}

/************** EEPROM REV 3/4 DEBUG PRINT FUNCTIONS *****************/
#if defined(DEBUG)
static void
printHeaderInfo(AR_DEV_INFO *pArDev, EEP_MAP *pEepMap, HEADER_WMODE modeArray)
{
    A_UINT16 j;
    EEP_HEADER_INFO *pHeaderInfo = pEepMap->pEepHeader;
    A_UINT16 version = pEepMap->version;
    A_UINT32 regDmn = pArDev->hwInfo.regDomain;

    uiPrintf("\n");
    uiPrintf(" =======================Header Information======================\n");

    uiPrintf(" |  Major Version           %2d  ", version >> 12);
    uiPrintf("|  Minor Version           %2d  |\n", version & 0xFFF);
    uiPrintf(" |-------------------------------------------------------------|\n");

    uiPrintf(" |  A Mode         %1d  ", pHeaderInfo->Amode);
    uiPrintf("|  B Mode         %1d  ", pHeaderInfo->Bmode);
    uiPrintf("|  G Mode        %1d  |\n", pHeaderInfo->Gmode);

    if(regDmn >> 15) {
        uiPrintf(" |  Country Code %03x  ", regDmn >> 15);
    } else {
        uiPrintf(" |  Reg. Domain  %03x  ", regDmn & 0xFFF);
    }

    uiPrintf("|  turbo Disable  %1d  ", pHeaderInfo->turbo5Disable);
    uiPrintf("|  RF Silent     %1d  |\n", pHeaderInfo->rfKill);
    uiPrintf(" | Turbo 2W Maximum dBm %2d | cckOfdmDelta(10x) %2d | GainI %2d   |\n",
        pHeaderInfo->turbo2WMaxPower5, pHeaderInfo->cckOfdmPwrDelta, pHeaderInfo->gainI[modeArray]);
    uiPrintf(" |-------------------------------------------------------------|\n");
    if (version >= EEPROM_VER3_3) {
        uiPrintf(" |  global mode              %1x  ", (regDmn & 0xF0) == 0x60);
        uiPrintf("|  False detect backoff  0x%02x  |\n", pHeaderInfo->falseDetectBackoff[modeArray]);
    }
    uiPrintf(" |  device type              %1x  ", pHeaderInfo->deviceType);

    uiPrintf("|  Switch Settling Time  0x%02x  |\n", pHeaderInfo->switchSettling[modeArray]);
    uiPrintf(" |  ADC Desired size       %2d  ", pHeaderInfo->adcDesiredSize[modeArray]);
    uiPrintf("|  XLNA Gain             0x%02x  |\n", pHeaderInfo->xlnaGain[modeArray]);

    uiPrintf(" |  tx end to XLNA on     0x%02x  ", pHeaderInfo->txEndToXLNAOn[modeArray]);
    uiPrintf("|  Threashold 62         0x%02x  |\n", pHeaderInfo->thresh62[modeArray]);

    uiPrintf(" |  tx end to XPA off     0x%02x  ", pHeaderInfo->txEndToXPAOff[modeArray]);
    uiPrintf("|  tx end to XPA on      0x%02x  |\n", pHeaderInfo->txFrameToXPAOn[modeArray]);

    uiPrintf(" |  PGA Desired size       %2d  ", pHeaderInfo->pgaDesiredSize[modeArray]);
    uiPrintf("|  Noise Threshold        %3d  |\n", pHeaderInfo->noiseFloorThresh[modeArray]);

    uiPrintf(" |  XPD Gain              0x%02x  ", pHeaderInfo->xgain[modeArray]);
    uiPrintf("|  XPD                      %1d  |\n", pHeaderInfo->xpd[modeArray]);

    uiPrintf(" |  txrx Attenuation      0x%02x  ", pHeaderInfo->txrxAtten[modeArray]);


    uiPrintf("|  Antenna control    0  0x%02X  |\n", pHeaderInfo->antennaControl[0][modeArray]);
    for(j = 1; j <= 10; j+=2) {
        uiPrintf(" |  Antenna control   %2d  0x%02X  ", j, pHeaderInfo->antennaControl[j][modeArray]);
        uiPrintf("|  Antenna control   %2d  0x%02X  |\n", j + 1, pHeaderInfo->antennaControl[j + 1][modeArray]);
    }

    uiPrintf(" |-------------------------------------------------------------|\n");
    if (modeArray == headerInfo11A) {
        uiPrintf(" |   OB_1   %1d   ", pHeaderInfo->ob1);
        uiPrintf("|   OB_2    %1d   ", pHeaderInfo->ob2);
        uiPrintf("|   OB_3   %1d  ", pHeaderInfo->ob3);
        uiPrintf("|   OB_4     %1d   |\n", pHeaderInfo->ob4);
        uiPrintf(" |   DB_1   %1d   ", pHeaderInfo->db1);
        uiPrintf("|   DB_2    %1d   ", pHeaderInfo->db2);
        uiPrintf("|   DB_3   %1d  ", pHeaderInfo->db3);
        uiPrintf("|   DB_4     %1d   |\n", pHeaderInfo->db4);
    } else {
        if(version >= EEPROM_VER3_1) {
            if (modeArray == headerInfo11B) {
                uiPrintf(" |   OB_1   %1d   ", pHeaderInfo->obFor24);
                uiPrintf("|   B_OB    %1d   ", pHeaderInfo->ob2GHz[0]);
                uiPrintf("|   DB_1   %1d  ", pHeaderInfo->dbFor24);
                uiPrintf("|   B_DB     %1d   |\n", pHeaderInfo->db2GHz[0]);
            } else {
                uiPrintf(" |   OB_1   %1d   ", pHeaderInfo->obFor24g);
                uiPrintf("|   B_OB    %1d   ", pHeaderInfo->ob2GHz[1]);
                uiPrintf("|   DB_1   %1d  ", pHeaderInfo->dbFor24g);
                uiPrintf("|   B_DB     %1d   |\n", pHeaderInfo->db2GHz[1]);
            }
        } else {
            if (modeArray == headerInfo11B) {
                uiPrintf(" |  OB_1                     %1d  ", pHeaderInfo->obFor24);
                uiPrintf("|  DB_1                     %1d  |\n", pHeaderInfo->dbFor24);
            } else {
                uiPrintf(" |  OB_1                     %1d  ", pHeaderInfo->obFor24g);
                uiPrintf("|  DB_1                     %1d  |\n", pHeaderInfo->dbFor24g);
            }

        }
    }
    uiPrintf(" ===============================================================\n");
    return;
}

static void
printChannelInfo(PCDACS_ALL_MODES *pEepromData, HEADER_WMODE mode)
{
    A_UINT16            i, j, k = 0;
    DATA_PER_CHANNEL    *pDataPerChannel;

    switch (mode) {
    case headerInfo11A:
        pDataPerChannel = pEepromData->DataPerChannel11a;
        break;

    case headerInfo11G:
        pDataPerChannel = pEepromData->DataPerChannel11g;
        break;

    case headerInfo11B:
        pDataPerChannel = pEepromData->DataPerChannel11b;
        break;

    default:
        uiPrintf("Illegal mode passed to printChannelInfo\n");
        return;
    }

    uiPrintf("\n");
    if (mode == headerInfo11A) {
        uiPrintf("=========================Calibration Information============================\n");

        for (k = 0; k < 10; k+=5) {
            for (i = k; i < k + 5; i++) {
                uiPrintf("|     %04d     ",
                    pDataPerChannel[i].channelValue);
            }
            uiPrintf("|\n");

            uiPrintf("|==============|==============|==============|==============|==============|\n");
            for (i = k; i < k + 5; i++) {
                uiPrintf("|pcdac pwr(dBm)");
            }
            uiPrintf("|\n");

            for (j = 0; j < pDataPerChannel[0].numPcdacValues; j++) {
                for (i = k; i < k + 5; i++) {
                    uiPrintf("|  %02d    %2d.%02d ",
                        pDataPerChannel[i].PcdacValues[j],
                        pDataPerChannel[i].PwrValues[j] / EEP_SCALE,
                        pDataPerChannel[i].PwrValues[j] % EEP_SCALE);
                }
                uiPrintf("|\n");
            }

            uiPrintf("|              |              |              |              |              |\n");
            for (i = k; i < k + 5; i++) {
                uiPrintf("| pcdac min %02d ", pDataPerChannel[i].pcdacMin);
            }
            uiPrintf("|\n");
            for (i = k; i < k + 5; i++) {
                uiPrintf("| pcdac max %02d ", pDataPerChannel[i].pcdacMax);
            }
            uiPrintf("|\n");
            uiPrintf("|==============|==============|==============|==============|==============|\n");
        }
    } else {
        uiPrintf("               ==========Calibration Information=============\n");

        for (i = 0; i < 3; i++) {
            if (0 == i) {
                uiPrintf("               ");
            }
            uiPrintf("|     %04d     ",
                pDataPerChannel[i].channelValue);
        }
        uiPrintf("|\n");

        uiPrintf("               |==============|==============|==============|\n");
        for (i = 0; i < 3; i++) {
            if (0 == i) {
                uiPrintf("               ");
            }
            uiPrintf("|pcdac pwr(dBm)");
        }
        uiPrintf("|\n");

        for (j = 0; j < pDataPerChannel[0].numPcdacValues; j++) {
            for (i = 0; i < 3; i++) {
                if (0 == i) {
                    uiPrintf("               ");
                }
                uiPrintf("|  %02d    %2d.%02d ",
                    pDataPerChannel[i].PcdacValues[j],
                    pDataPerChannel[i].PwrValues[j] / EEP_SCALE,
                    pDataPerChannel[i].PwrValues[j] % EEP_SCALE);

            }
            uiPrintf("|\n");
        }

        uiPrintf("               |              |              |              |\n");
        uiPrintf("               ");
        for (i = 0; i < 3; i++) {
            uiPrintf("| pcdac min %02d ", pDataPerChannel[i].pcdacMin);
        }
        uiPrintf("|\n");
        uiPrintf("               ");
        for (i = 0; i < 3; i++) {
            uiPrintf("| pcdac max %02d ", pDataPerChannel[i].pcdacMax);
        }
        uiPrintf("|\n");
        uiPrintf("               |==============|==============|==============|\n");

    }
}

static void
printExpnPower5112(EEPROM_POWER_EXPN_5112 *pExpnPower, HEADER_WMODE mode)
{
    A_UINT16        i, j=0, kk;
    A_UINT16        singleXpd = 0xDEAD;
    A_CHAR          *modeString[3] = {"11A", "11B", "11G"};

    if (pExpnPower->xpdMask != 0x9) {
        for (j = 0; j < NUM_XPD_PER_CHANNEL; j++) {
            if (pExpnPower->xpdMask == (1 << j)) {
                singleXpd = j;
                break;
            }
        }
    }

    uiPrintf("======================5112 Power Calibration Information========================\n");

    uiPrintf("| XPD_Gain_mask = 0x%2x              | Number of channels for mode %s: %2d      |\n",
        pExpnPower->xpdMask, modeString[mode], pExpnPower->numChannels);
    if (pExpnPower->xpdMask != 0x9) {
        uiPrintf("| XPD_GAIN = %d                                                              |\n", j);
    }
    uiPrintf("|==============================================================================|\n");

    //print the frequency values
    uiPrintf("| freq |  pwr1  |  pwr2  |  pwr3  |  pwr4  |");
    if (pExpnPower->xpdMask == 0x9) {
        uiPrintf(" pwr1_x3| pwr2_x3| pwr3_x3| maxPow |\n");
        uiPrintf("|      | [pcd]  | [pcd]  | [pcd]  | [pcd]  | [pcd]  | [pcd]  | [pcd]  |        |\n");
    } else {
        uiPrintf("        [pcd]    [pcd]    [pcd]    [pcd]    \n");
    }

    for (i = 0; i < pExpnPower->numChannels; i++) {
        uiPrintf("|==============================================================================|\n");
        uiPrintf("| %4d |", pExpnPower->pChannels[i]);
        if (pExpnPower->xpdMask != 0x9) {
            j = singleXpd;
        } else {
            j = 0;
        }
        for (kk=0; kk < pExpnPower->pDataPerChannel[i].pDataPerXPD[j].numPcdacs; kk++) {
            uiPrintf(" %2d.%02d  |", (pExpnPower->pDataPerChannel[i].pDataPerXPD[j].pwr_t4[kk] / 4),
                (pExpnPower->pDataPerChannel[i].pDataPerXPD[j].pwr_t4[kk] % 4) * 25);
        }
        if (pExpnPower->xpdMask == 0x9) {
            for (kk=0; kk < pExpnPower->pDataPerChannel[i].pDataPerXPD[3].numPcdacs; kk++) {
                uiPrintf(" %2d.%02d  |", pExpnPower->pDataPerChannel[i].pDataPerXPD[3].pwr_t4[kk] / 4,
                    (pExpnPower->pDataPerChannel[i].pDataPerXPD[3].pwr_t4[kk] % 4) * 25);
            }
        }
        uiPrintf(" %2d.%02d  |\n", pExpnPower->pDataPerChannel[i].maxPower_t4 / 4,
            (pExpnPower->pDataPerChannel[i].maxPower_t4 % 4) * 25);
        uiPrintf("|      |");
        for (kk=0; kk<pExpnPower->pDataPerChannel[i].pDataPerXPD[j].numPcdacs; kk++) {
            uiPrintf("  [%2d]  |", pExpnPower->pDataPerChannel[i].pDataPerXPD[j].pcdac[kk]);
        }
        if (pExpnPower->xpdMask == 0x9) {
            for (kk=0; kk<pExpnPower->pDataPerChannel[i].pDataPerXPD[3].numPcdacs; kk++) {
                uiPrintf("  [%2d]  |", pExpnPower->pDataPerChannel[i].pDataPerXPD[3].pcdac[kk]);
            }
        }
        uiPrintf("        |\n");
    }
    uiPrintf("================================================================================\n");
}

static void
printTargetPowerInfo(TRGT_POWER_ALL_MODES *pPowerInfoAllModes, HEADER_WMODE mode)
{
    A_UINT16 i, k;
    TRGT_POWER_INFO     *pPowerInfo;

    uiPrintf("\n");
    if (mode == headerInfo11A) {
        pPowerInfo = pPowerInfoAllModes->trgtPwr_11a;
        uiPrintf("============================Target Power Info===============================\n");

        for (k = 0; k < 8; k+=4) {
            uiPrintf("|     rate     ");
            for (i = k; i < k + 4; i++) {
                uiPrintf("|     %04d     ", pPowerInfo[i].testChannel);
            }
            uiPrintf("|\n");
            uiPrintf("|==============|==============|==============|==============|==============|\n");

            uiPrintf("|     6-24     ");
            for (i = k; i < k + 4; i++) {
                uiPrintf("|     %2d.%d     ", pPowerInfo[i].twicePwr6_24 / 2,
                    (pPowerInfo[i].twicePwr6_24 % 2) * 5);
            }
            uiPrintf("|\n");

            uiPrintf("|      36      ");
            for (i = k; i < k + 4; i++) {
                uiPrintf("|     %2d.%d     ", pPowerInfo[i].twicePwr36 / 2,
                    (pPowerInfo[i].twicePwr36 % 2) * 5);
            }
            uiPrintf("|\n");

            uiPrintf("|      48      ");
            for (i = k; i < k + 4; i++) {
                uiPrintf("|     %2d.%d     ", pPowerInfo[i].twicePwr48 / 2,
                    (pPowerInfo[i].twicePwr48 % 2) * 5);
            }
            uiPrintf("|\n");

            uiPrintf("|      54      ");
            for (i = k; i < k + 4; i++) {
                uiPrintf("|     %2d.%d     ", pPowerInfo[i].twicePwr54 / 2,
                    (pPowerInfo[i].twicePwr54 % 2) * 5);
            }

            uiPrintf("|\n");
            uiPrintf("|==============|==============|==============|==============|==============|\n");
        }
    } else {
        if (mode == headerInfo11B) {
            pPowerInfo = pPowerInfoAllModes->trgtPwr_11b;
        } else {
            pPowerInfo = pPowerInfoAllModes->trgtPwr_11g;
        }
        uiPrintf("=============Target Power Info================\n");

        uiPrintf("|     rate     ");
        for (i = 0; i < 2; i++) {
            uiPrintf("|     %04d     ",
                pPowerInfo[i].testChannel);
        }
        uiPrintf("|\n");

        uiPrintf("|==============|==============|==============|\n");

        if (mode == headerInfo11B) {
            uiPrintf("|      1       ");
        } else {
            uiPrintf("|     6-24     ");
        }

        for (i = 0; i < 2; i++) {
            uiPrintf("|     %2d.%d     ", pPowerInfo[i].twicePwr6_24 / 2,
                (pPowerInfo[i].twicePwr6_24 % 2) * 5);
        }
        uiPrintf("|\n");

        if (mode == headerInfo11B) {
            uiPrintf("|      2       ");
        } else {
            uiPrintf("|      36      ");
        }
        for (i = 0; i < 2; i++) {
            uiPrintf("|     %2d.%d     ", pPowerInfo[i].twicePwr36 / 2,
                (pPowerInfo[i].twicePwr36 % 2) * 5);
        }
        uiPrintf("|\n");

        if (mode == headerInfo11B) {
            uiPrintf("|      5.5     ");
        } else {
            uiPrintf("|      48      ");
        }
        for (i = 0; i < 2; i++) {
            uiPrintf("|     %2d.%d     ", pPowerInfo[i].twicePwr48 / 2,
                (pPowerInfo[i].twicePwr48 % 2) * 5);
        }
        uiPrintf("|\n");

        if (mode == headerInfo11B) {
            uiPrintf("|      11      ");
        } else {
            uiPrintf("|      54      ");
        }
        for (i = 0; i < 2; i++) {
            uiPrintf("|     %2d.%d     ", pPowerInfo[i].twicePwr54 / 2,
                (pPowerInfo[i].twicePwr54 % 2) * 5);
        }

        uiPrintf("|\n");
        uiPrintf("|==============|==============|==============|\n");

    }
}

static void
printRDEdges(RD_EDGES_POWER *pRdEdgePwrInfo, A_UINT16 *pTestGroups,
             HEADER_WMODE mode, A_UINT16 maxNumCtl, A_UINT16 version)
{
    A_UINT16    i=0, j;
    A_UINT16    ctlMode = 0;
    const static char *ctlType[4] = {
        "11a base mode ] ",
        "11b mode ]      ",
        "11g mode ]      ",
        "11a TURBO mode ]"
    };

    uiPrintf("\n");
    uiPrintf("=======================Test Group Band Edge Power========================\n");
    while ((pTestGroups[i] != 0) && (i < maxNumCtl)) {
        switch (pTestGroups[i] & 0x3) {
        case 0:
        case 3:
            ctlMode = headerInfo11A;
            break;
        case 1:
            ctlMode = headerInfo11B;
            break;
        case 2:
            ctlMode = headerInfo11G;
            break;
        }
        if (mode != ctlMode) {
            i++;
            pRdEdgePwrInfo += NUM_EDGES;
            continue;
        }
        uiPrintf("|                                                                       |\n");
        uiPrintf("| CTL: 0x%02x   [ 0x%x %s", pTestGroups[i] & 0xff, pTestGroups[i], ctlType[pTestGroups[i] & 0x3]);

        uiPrintf("                                   |\n");
        uiPrintf("|=======|=======|=======|=======|=======|=======|=======|=======|=======|\n");

        uiPrintf("| edge  ");
        for (j = 0; j < NUM_EDGES; j++) {
            if (pRdEdgePwrInfo[j].rdEdge == 0) {
                uiPrintf("|  --   ");
            } else {
                uiPrintf("| %04d  ", pRdEdgePwrInfo[j].rdEdge);
            }
        }

        uiPrintf("|\n");
        uiPrintf("|=======|=======|=======|=======|=======|=======|=======|=======|=======|\n");
        uiPrintf("| power ");
        for (j = 0; j < NUM_EDGES; j++) {
            if (pRdEdgePwrInfo[j].rdEdge == 0) {
                uiPrintf("|  --   ");
            } else {
                uiPrintf("| %2d.%d  ", pRdEdgePwrInfo[j].twice_rdEdgePower / 2,
                    (pRdEdgePwrInfo[j].twice_rdEdgePower % 2) * 5);
            }
        }

        uiPrintf("|\n");
        if (version >= EEPROM_VER3_3) {
            uiPrintf("|=======|=======|=======|=======|=======|=======|=======|=======|=======|\n");
            uiPrintf("| flag  ");
            for (j = 0; j < NUM_EDGES; j++) {
                if (pRdEdgePwrInfo[j].rdEdge == 0) {
                    uiPrintf("|  --   ");
                } else {
                    uiPrintf("|   %1d   ", pRdEdgePwrInfo[j].flag);
                }
            }

            uiPrintf("|\n");
        }
        uiPrintf("=========================================================================\n");
        i++;
        pRdEdgePwrInfo += NUM_EDGES;
    }
}

void
printEepromStruct(AR_DEV_INFO *pArDev, EEP_MAP *pEepMap, HEADER_WMODE mode)
{
    printHeaderInfo(pArDev, pEepMap, mode);
    if ((pEepMap->version >= EEPROM_VER4_0) && pEepMap->pEepHeader->eepMap) {
        printExpnPower5112(&(pEepMap->modePowerArray5112[mode]), mode);
    } else {
        printChannelInfo(pEepMap->pPcdacInfo, mode);
    }
    printTargetPowerInfo(pEepMap->pTrgtPowerInfo, mode);
    printRDEdges(pEepMap->pRdEdgesPower, pEepMap->pEepHeader->ctl, mode, pEepMap->pEepHeader->numCtls,
        pEepMap->version);
    return;
}

void
dump5212Eeprom(AR_DEV_INFO *pArDev)
{
#if !defined(ECOS_NOTDONE)
    EEP_MAP *pEepMap;
    ASSERT(pArDev->pHalInfo->pEepData);

    pEepMap = pArDev->pHalInfo->pEepData;
    if (pEepMap->pEepHeader->Amode) {
        printEepromStruct(pArDev, pEepMap, headerInfo11A);
    }
    if (pEepMap->pEepHeader->Bmode) {
        printEepromStruct(pArDev, pEepMap, headerInfo11B);
    }
    if (pEepMap->pEepHeader->Gmode) {
        printEepromStruct(pArDev, pEepMap, headerInfo11G);
    }
#else
    ECOS_NOTDONE_XXX;
#endif
    return;
}

#endif /* #ifdef DEBUG */

#if defined(EAR_SUPPORT)
/************ FORWARD SOFTWARE COMPATIBILITY EAR FUNCTIONS ************/
/* Globals */
int EarDebugLevel = EAR_DEBUG_OFF;

/**************************************************************************
 * ar5212EarPreParse
 *
 * Parse the entire EAR saving away the size required for each EAR register header
 * Allocation is only performed for matching version ID's.
 */
static int
ar5212EarPreParse(A_UINT16 *in, int numLocs, EAR_ALLOC *earAlloc)
{
    A_UINT16      numFound = 0;
    int           curEarLoc = 0;
    A_UINT16      numLocsConsumed;
    A_UINT16      type, regHead, tag, last;
    A_UINT16      verId;
    A_BOOL        verMaskMatch = FALSE;

    /* Record the version Id */
    verId = in[curEarLoc++];

    while (curEarLoc < numLocs) {
        /* Parse Version/Header/End */
        if (IS_VER_MASK(in[curEarLoc])) {
            verMaskMatch = IS_EAR_VMATCH(verId, in[curEarLoc]);
            curEarLoc++;
        } else if (IS_END_HEADER(in[curEarLoc])) {
            break;
        } else {
            /* Must be a register header - find number of locations consumed */
            regHead = in[curEarLoc++];

            if (IS_CM_SET(regHead)) {
                /* Bump location for channel modifier */
                curEarLoc++;
            }

            if (IS_DISABLER_SET(regHead)) {
                if (IS_PLL_SET(in[curEarLoc])) {
                    /* Bump location for PLL */
                    curEarLoc++;
                }
                /* Bump location for disabler */
                curEarLoc++;
            }

            numLocsConsumed = 0;
            /* Now into the actual register writes */
            type = (A_UINT16)((regHead & RH_TYPE_M) >> RH_TYPE_S);
            switch (type) {
            case EAR_TYPE0:
                do {
                    tag = (A_UINT16)(in[curEarLoc] & T0_TAG_M);
                    if ((tag == T0_TAG_32BIT) || (tag == T0_TAG_32BIT_LAST)) {
                        /* Full word writes */
                        numLocsConsumed += 3;
                        curEarLoc += 3;
                    } else {
                        /* Half word writes */
                        numLocsConsumed += 2;
                        curEarLoc += 2;
                    }
                }
                while ((tag != T0_TAG_32BIT_LAST) && (curEarLoc < numLocs));
                break;
            case EAR_TYPE1:
                numLocsConsumed = (A_UINT16)(((in[curEarLoc] & T1_NUM_M) * sizeof(A_UINT16)) + 3);
                curEarLoc += numLocsConsumed;
                break;
            case EAR_TYPE2:
                do {
                    last = IS_TYPE2_LAST(in[curEarLoc]);
                    if(IS_TYPE2_EXTENDED(in[curEarLoc])) {
                        numLocsConsumed += 2 + A_DIV_UP(in[curEarLoc+1], 16);
                        curEarLoc += 2 + A_DIV_UP(in[curEarLoc+1], 16);
                    } else {
                        numLocsConsumed += 2;
                        curEarLoc += 2;
                    }
                } while (!last && (curEarLoc < numLocs));
                break;
            case EAR_TYPE3:
                do {
                    last = IS_TYPE3_LAST(in[curEarLoc]);
                    numLocsConsumed += 2 + A_DIV_UP(in[curEarLoc] & 0x1F, 16);
                    curEarLoc += 2 + A_DIV_UP(in[curEarLoc] & 0x1F, 16);
                } while (!last && (curEarLoc < numLocs));
                break;
            }
            /* Only record allocation information for matching versions */
            if (verMaskMatch) {
                if (numLocsConsumed > EAR_MAX_RH_REGS) {
                    uiPrintf("ar5212EarPreParse: A register header exceeds the max allowable size\n");
                    earAlloc->numRHs = 0;
                    return 0;
                }
                earAlloc->locsPerRH[numFound] = numLocsConsumed;
                numFound++;
            }
        }
    }

    earAlloc->numRHs = numFound;

    if (EarDebugLevel >= 1) {
        uiPrintf("Number of locations parsed in preParse: %d\n", curEarLoc);
    }
    return curEarLoc;
}

/**************************************************************************
 * ar5212EarAllocate
 *
 * Now that the Ear structure size is known, allocate the EAR header and
 * the individual register headers
 */
static A_BOOL
ar5212EarAllocate(EAR_ALLOC *earAlloc, EAR_HEADER **ppEarHead)
{
    int             sizeForRHs;
    int             sizeForRegs = 0;
    int             i;
    A_UINT16        *currentAlloc;
    REGISTER_HEADER *pRH;

    /* Return if no applicable EAR exists */
    if (earAlloc->numRHs == 0) {
        return TRUE;
    }

    /* Allocate the Ear Header */
    *ppEarHead = (EAR_HEADER *)A_DRIVER_MALLOC(sizeof(EAR_HEADER));
    if (*ppEarHead == NULL) {
        uiPrintf("ar5212EarAllocate: Failed to retrieve space for EAR Header\n");
        return FALSE;
    }
    A_MEM_ZERO(*ppEarHead, sizeof(EAR_HEADER));

    /* Save number of register headers */
    (*ppEarHead)->numRHs = earAlloc->numRHs;

    /* Size the malloc for the Ear Register Headers */
    sizeForRHs = earAlloc->numRHs * sizeof(REGISTER_HEADER);
    for (i = 0; i < earAlloc->numRHs; i++) {
        sizeForRegs += earAlloc->locsPerRH[i] * sizeof(A_UINT16);
    }

    if (EarDebugLevel >= EAR_DEBUG_VERBOSE) {
        uiPrintf("ar5212EarAllocate: Size for RH's %d, size for reg data %d\n", sizeForRHs, sizeForRegs);
    }

    /* Malloc and assign the space to the RH's */
    (*ppEarHead)->pRH = (REGISTER_HEADER *) A_DRIVER_MALLOC(sizeForRegs + sizeForRHs);
    if ((*ppEarHead)->pRH == NULL) {
        uiPrintf("ar5212EarAllocate: Failed to retrieve space for EAR individual registers\n");
        return 0;
    }
    A_MEM_ZERO((*ppEarHead)->pRH, sizeForRegs + sizeForRHs);
    (*ppEarHead)->earSize = sizeForRegs + sizeForRHs;

    currentAlloc = (A_UINT16 *)(((A_UINT8 *)(*ppEarHead)->pRH) + sizeForRHs);
    for (i = 0; i < earAlloc->numRHs; i++) {
        if (EarDebugLevel >= EAR_DEBUG_EXTREME) {
             uiPrintf("ar5212EarAllocate: reg data %2d memory location 0x%08X\n", i, (A_UINT32)currentAlloc);
        }
        pRH = &((*ppEarHead)->pRH[i]);
        pRH->regs = currentAlloc;
        currentAlloc += earAlloc->locsPerRH[i];
    }
    return TRUE;
}

/**************************************************************************
 * ar5212EarCheckAndFill
 *
 * Save away EAR contents
 */
static A_BOOL
ar5212EarCheckAndFill(EAR_HEADER *earHead, A_UINT16 *in, int totalLocs, EAR_ALLOC *pEarAlloc)
{
    int             curEarLoc = 0, currentRH = 0, regLoc, i;
    A_UINT16        regHead, tag, last, currentVerMask, addr, num, numBits, startBit;
    REGISTER_HEADER *pRH, tempRH;
    A_UINT16        tempRHregs[EAR_MAX_RH_REGS], *pReg16;
    A_BOOL          verMaskUsed, verMaskMatch;

    /* Setup the temporary register header */

    /* Save the version Id */
    earHead->versionId = in[curEarLoc++];

    /* Save the first version mask */
    verMaskUsed = 0;
    verMaskMatch = IS_EAR_VMATCH(earHead->versionId, in[curEarLoc]);
    currentVerMask = in[curEarLoc++];
    if (!IS_VER_MASK(currentVerMask)) {
        uiPrintf("ar5212EarCheckAndFill: ERROR: First entry after Version ID is not a Version Mask\n");
        return FALSE;
    }

    while (curEarLoc < totalLocs) {
        /* Parse Version/Header/End */
        if (IS_VER_MASK(in[curEarLoc])) {
            if (!verMaskUsed) {
                uiPrintf("ar5212EarCheckAndFill: ERROR: Multiple version masks setup with no register headers location %d\n",
                         curEarLoc);
                return FALSE;
            }
            verMaskUsed = 0;
            verMaskMatch = IS_EAR_VMATCH(earHead->versionId, in[curEarLoc]);
            currentVerMask = in[curEarLoc++];
        } else if (IS_END_HEADER(in[curEarLoc])) {
            uiPrintf("ar5212EarCheckAndFill: ERROR: Somehow we've hit the END location but parse shouldn't let us get here. loc %d\n",
                     curEarLoc);
            return FALSE;
        } else {
            if (currentRH > earHead->numRHs) {
                uiPrintf("ar5212EarCheckAndFill: ERROR: Exceeded number of register headers found in preParse. loc %d\n",
                         curEarLoc);
                return FALSE;
            }
            A_MEM_ZERO(&tempRH, sizeof(REGISTER_HEADER));
            A_MEM_ZERO(&tempRHregs, sizeof(A_UINT16) * EAR_MAX_RH_REGS);
            tempRH.regs = tempRHregs;
            pRH = &tempRH;

            /* Must be a register header - save last version mask */
            pRH->versionMask = currentVerMask;
            regHead = in[curEarLoc++];
            pRH->modes = (A_UINT16)(regHead & RH_MODES_M);
            if ((pRH->modes != RH_ALL_MODES) && (pRH->modes & RH_RESERVED_MODES)) {
                uiPrintf("ar5212EarCheckAndFill: WARNING: Detected that reserved modes have been set. loc %d\n", curEarLoc);
            }
            pRH->type = (A_UINT16)((regHead & RH_TYPE_M) >> RH_TYPE_S);
            pRH->stage = (A_UINT16)((regHead & RH_STAGE_M) >> RH_STAGE_S);
            if (IS_CM_SET(regHead)) {
                pRH->channel = in[curEarLoc++];
                if (IS_CM_SINGLE(pRH->channel) && !IS_5GHZ_CHAN(pRH->channel) &&
                    !IS_2GHZ_CHAN(pRH->channel))
                {
                    uiPrintf("ar5212EarCheckAndFill: WARNING: Specified single channel %d is not within tunable range. loc %d\n",
                             pRH->channel & CM_SINGLE_CHAN_M, curEarLoc);
                }
            }

            if (IS_DISABLER_SET(regHead)) {
                pRH->disabler.valid = TRUE;
                pRH->disabler.disableField = in[curEarLoc++];
                if (IS_PLL_SET(pRH->disabler.disableField)) {
                    pRH->disabler.pllValue = in[curEarLoc++];
                    if (pRH->disabler.pllValue & RH_RESERVED_PLL_BITS) {
                        uiPrintf("ar5212EarCheckAndFill: WARNING: Detected that reserved pll bits have been set. loc %d\n", curEarLoc);
                    }
                }
            }

            /* Now into the actual register writes */
            regLoc = 0;
            switch (pRH->type) {
            case EAR_TYPE0:
                do {
                    pRH->regs[regLoc] = in[curEarLoc++];
                    tag = (A_UINT16)(pRH->regs[regLoc] & T0_TAG_M);
                    if (!IS_VALID_ADDR(pRH->regs[regLoc] & ~T0_TAG_M)) {
                        uiPrintf("ar5212EarCheckAndFill: WARNING: Detected invalid register address 0x%04X at loc %d\n",
                                 pRH->regs[regLoc] & ~T0_TAG_M, curEarLoc);
                    }
                    regLoc++;
                    if ((tag == T0_TAG_32BIT) || (tag == T0_TAG_32BIT_LAST)) {
                        /* Full word writes */
                        pRH->regs[regLoc++] = in[curEarLoc++];
                        pRH->regs[regLoc++] = in[curEarLoc++];
                    } else {
                        /* Half word writes */
                        pRH->regs[regLoc++] = in[curEarLoc++];
                    }
                } while ((tag != T0_TAG_32BIT_LAST) && (curEarLoc < totalLocs));
                break;
            case EAR_TYPE1:
                pRH->regs[regLoc] = in[curEarLoc++];
                addr = (A_UINT16)(pRH->regs[regLoc] & ~T1_NUM_M);
                num = (A_UINT16)(pRH->regs[regLoc] & T1_NUM_M);
                regLoc++;
                for (i = 0; i < num + 1; i++) {
                    /* Full word writes */
                    if (!IS_VALID_ADDR(addr + (sizeof(A_UINT32) * i))) {
                        uiPrintf("ar5212EarCheckAndFill: WARNING: Detected invalid register address 0x%04X at loc %d\n",
                                 addr + (sizeof(A_UINT32) * i), curEarLoc);
                    }
                    pRH->regs[regLoc++] = in[curEarLoc++];
                    pRH->regs[regLoc++] = in[curEarLoc++];
                }
                break;
            case EAR_TYPE2:
                do {
                    pRH->regs[regLoc] = in[curEarLoc++];
                    last = IS_TYPE2_LAST(pRH->regs[regLoc]);
                    if (((pRH->regs[regLoc] & T2_BANK_M) >> T2_BANK_S) == 0) {
                        uiPrintf("ar5212EarCheckAndFill: WARNING: Bank 0 update found in Type2 write at loc %d\n", curEarLoc);
                    }
                    startBit = (A_UINT16)(pRH->regs[regLoc] & T2_START_M);
                    if (IS_TYPE2_EXTENDED(pRH->regs[regLoc])) {
                        regLoc++;
                        pRH->regs[regLoc] = in[curEarLoc++];
                        if (pRH->regs[regLoc] < 12) {
                            uiPrintf("ar5212EarCheckAndFill: WARNING: Type2 Extended Write used when number of bits is under 12 at loc %d\n",
                                     curEarLoc);
                        }
                        num = A_DIV_UP(pRH->regs[regLoc], 16);
                        numBits = pRH->regs[regLoc];
                        if (startBit + numBits > MAX_ANALOG_START) {
                            uiPrintf("ar5212EarCheckAndFill: ERROR: Type2 write will exceed analog buffer limits (at loc %d)\n", curEarLoc);
                            return FALSE;
                        }
                        regLoc++;
                        for (i = 0; i < num; i++) {
                            /* Add check data exceeds num bits check? */
                            pRH->regs[regLoc++] = in[curEarLoc++];
                        }
                        if ((~((1 << (numBits % 16)) - 1) & in[curEarLoc - 1]) && ((numBits % 16) != 0)){
                            uiPrintf("ar5212EarCheckAndFill: WARNING: Type2 extended Write data exceeds number of bits specified at loc %d\n",
                                     curEarLoc);
                        }
                    } else {
                        regLoc++;
                        pRH->regs[regLoc] = in[curEarLoc++];
                        numBits = (A_UINT16)((pRH->regs[regLoc] & T2_NUMB_M) >> T2_NUMB_S);
                        if (startBit + numBits > MAX_ANALOG_START) {
                            uiPrintf("ar5212EarCheckAndFill: ERROR: Type2 write will exceed analog buffer limits (at loc %d)\n", curEarLoc);
                            return FALSE;
                        }
                        if (~((1 << numBits) - 1) &
                            (pRH->regs[regLoc] & T2_DATA_M))
                        {
                            uiPrintf("ar5212EarCheckAndFill: WARNING: Type2 Write data exceeds number of bits specified at loc %d\n",
                                     curEarLoc);
                        }
                        regLoc++;
                    }
                } while (!last && (curEarLoc < totalLocs));
                break;
            case EAR_TYPE3:
                do {
                    pRH->regs[regLoc] = in[curEarLoc++];
                    last = IS_TYPE3_LAST(pRH->regs[regLoc]);
                    num = (A_UINT16)(A_DIV_UP((pRH->regs[regLoc] & T3_NUMB_M), 16));
                    if (((pRH->regs[regLoc] & T3_START_M) >> T3_START_S) +
                        (pRH->regs[regLoc] & T3_NUMB_M) > 31)
                    {
                        uiPrintf("ar5212EarCheckAndFill: WARNING: Type3 StartBit plus Number of Bits > 31 at loc %d\n",
                                 curEarLoc);
                    }
                    if (((pRH->regs[regLoc] & T3_OPCODE_M) >> T3_OPCODE_S) > T3_MAX_OPCODE) {
                        uiPrintf("ar5212EarCheckAndFill: WARNING: Type3 OpCode exceeds largest selectable opcode at loc %d\n",
                                 curEarLoc);
                    }
                    if (pRH->regs[regLoc] & T3_RESERVED_BITS) {
                        uiPrintf("ar5212EarCheckAndFill: WARNING: Type3 Reserved bits used at loc %d\n",
                                 curEarLoc);
                    }
                    numBits = (A_UINT16)(pRH->regs[regLoc] & T3_NUMB_M);
                    regLoc++;
                    /* Grab Address */
                    pRH->regs[regLoc] = in[curEarLoc++];
                    if (!IS_VALID_ADDR(pRH->regs[regLoc])) {
                        uiPrintf("ar5212EarCheckAndFill: WARNING: Detected invalid register address 0x%04X at loc %d\n",
                                 pRH->regs[regLoc], curEarLoc);
                    }
                    regLoc++;
                    for (i = 0; i < num; i++) {
                        pRH->regs[regLoc++] = in[curEarLoc++];
                    }
                    if ((~((1 << (numBits % 16)) - 1) & in[curEarLoc - 1]) && ((numBits % 16) != 0)) {
                        uiPrintf("ar5212EarCheckAndFill: WARNING: Type3 write data exceeds number of bits specified at loc %d\n",
                                 curEarLoc);
                    }
                } while (!last && (curEarLoc < totalLocs));
                break;
            }
            verMaskUsed = 1;
            /* Save any register headers that match the version mask*/
            if (verMaskMatch) {
                if (regLoc != pEarAlloc->locsPerRH[currentRH]) {
                    uiPrintf("ar5212EarCheckAndFill: Ear Allocation did not match Ear Usage\n");
                    return FALSE;
                }
                /*
                 * Copy stack register header to real register header on match
                 * BCOPY the regs, save away the reg16 ptr as the struct copy will overwrite it
                 * finally restore the reg16 ptr.
                 */
                A_DRIVER_BCOPY(pRH->regs, earHead->pRH[currentRH].regs, regLoc * sizeof(A_UINT16));
                pReg16 = earHead->pRH[currentRH].regs;
                earHead->pRH[currentRH] = *pRH;
                earHead->pRH[currentRH].regs = pReg16;

                currentRH++;
            }
        } /* End Register Header Parse */
    } /* End Location Reading */
    return TRUE;
}

#ifdef DEBUG
static void
showEarAlloc(EAR_ALLOC *earAlloc)
{
    int i;

    uiPrintf("Found %d register headers\n", earAlloc->numRHs);
    for (i = 0; i < earAlloc->numRHs; i++) {
        uiPrintf("Register Header %3d requires %2d 16-bit locations\n", i + 1,
                 earAlloc->locsPerRH[i]);
    }
}

static void
printEar(EAR_HEADER *earHead, EAR_ALLOC *earAlloc)
{
    REGISTER_HEADER *pRH;
    int i, j;

    uiPrintf("\n=== EAR structure dump ===\n");
    uiPrintf("Version ID: 0x%04X\n", earHead->versionId);
    uiPrintf("Number of Register Headers: %d\n", earHead->numRHs);

    for (i = 0; i < earHead->numRHs; i++) {
        pRH = &(earHead->pRH[i]);
        uiPrintf("\n=== Register Header %2d ===\n", i + 1);
        uiPrintf("Version Mask   : 0x%04X\n", pRH->versionMask);
        uiPrintf("Register Type  : %d\n", pRH->type);
        uiPrintf("Modality Mask  : ");
        printModeString(pRH->modes);
        uiPrintf("Stage to Use   : %d\n", pRH->stage);
        uiPrintf("Channel Mask   : 0x%04X", pRH->channel);
        if (IS_CM_SINGLE(pRH->channel)) {
            uiPrintf(" -> Single Channel : %d", pRH->channel & CM_SINGLE_CHAN_M);
        }
        uiPrintf("\n");
        uiPrintf("Disabler Mask  : 0x%04X", pRH->disabler.disableField);
        if (IS_PLL_SET(pRH->disabler.disableField)) {
            uiPrintf(" -> PLL Setting : 0x%x", pRH->disabler.pllValue);
        }
        uiPrintf("\n");
        for (j = 0; j < earAlloc->locsPerRH[i]; j++) {
            uiPrintf("= %2d = 0x%04X =\n", j, pRH->regs[j]);
        }
    }
}

static void
printModeString(A_UINT16 modes)
{
    const static A_CHAR *pModesArray[9] = {"11B ", "11G ", "T2 ", "11GXR ",
                                            "11A ", "NA ", "T5 ", "11AXR ", "NA "};
    int i;
    int newModes;

    if (modes == RH_ALL_MODES) {
        newModes = 0xDF;
    } else {
        newModes = modes;
    }

    for(i = 0; i < 9; i++) {
        if (newModes & (1 << i)) {
            uiPrintf("%s", pModesArray[i]);
        }
    }
    uiPrintf("\n");
}
#endif /* DEBUG */
#endif /* EAR_SUPPORT */

/**************************************************************************
 * ar5212EnableMKKNew11a
 *
 * Return 1 if should enable 11a for new Japan regulatory, or 0 otherwise
 */
static A_UINT16
ar5212EnableMKKNew11a(AR_DEV_INFO *pDev)
{
    A_UINT16   data;
    A_STATUS   status;
    A_UINT32   offset;

    status = ar5212EepromRead(pDev, REGULATORY_DOMAIN_OFFSET, &data);
    if (status != A_OK) {
        return 0; /* There was an error say 0 */
    }
#if 0
    if (!wlanIsMKKSku(data)) {
        return 0; /* Is not a MKK SKU return 0 */
    }
#endif
    offset = (pDev->pHalInfo->pEepData->version >= EEPROM_VER4_0) ? 0xca : 0xcf;
    status = ar5212EepromRead(pDev, offset, &data);

    data &= (pDev->pHalInfo->pEepData->version >= EEPROM_VER4_0) ?
        EEPCAP_REG_EN_KK_NEW_11A_M : EEPCAP_REG_EN_KK_NEW_11A_LEGACY_M;

    data >>= (pDev->pHalInfo->pEepData->version >= EEPROM_VER4_0) ?
        EEPCAP_REG_EN_KK_NEW_11A_S : EEPCAP_REG_EN_KK_NEW_11A_LEGACY_S;

    return data;
}

#endif /* #ifdef BUILD_AR5212 */
