/*
 * Copyright (c) 2000-2002 Atheros Communications, Inc., All Rights Reserved
 *
 * Routines used to initialize and generated beacons for the AR5212/AR531X.
 *
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/hal/ar5212/ar5212Beacon.c#3 $
 */

#ifdef BUILD_AR5212

#include "arDev.h"
#include "halApi.h"
#include "hal.h"

/* Headers for HW private items */
#include "ar5212Reg.h"
#include "ar5212Beacon.h"
#include "ar5212Interrupts.h"
#include "ar5212Misc.h"
#include "ar5212.h"


/**************************************************************
 * ar5212SetupBeaconDesc
 *
 * Fills in the control fields of the beacon descriptor
 *
 * Assumes: frameLen does not include the FCS
 */
void
ar5212SetupBeaconDesc(AR_DEV_INFO* pArDev, AR_DESC *pDesc, 
                      A_UINT32 antMode, A_BOOL isAp)
{
    AR5212_TX_CONTROL *pTxControl  = TX_CONTROL(pDesc);
    const RATE_TABLE  *pRateTable  = pDesc->pOpBss->pRateTable;
    A_UINT16           rateIndex   = pDesc->pOpBss->config.defaultRateIndex;

    /* clear the status fields */
    pDesc->hw.word[4] = pDesc->hw.word[5] = 0;

    /* TURBO_PRIME */
    if ((pArDev->config.abolt & ABOLT_TURBO_PRIME) && isAp) {
        pTxControl->interruptReq = 1;
    }

#if !defined(ECOS_NOTDONE)
    if ((pDesc->pOpBss == &pArDev->bssTable[0]) && 
        CHAN_IS_G(&pArDev->config.chanDesc)) {
        rateIndex = pArDev->config.gBeaconRate;
    }
#endif

    pTxControl->TXRate0 = pRateTable->info[rateIndex].rateCode;

    /* exit early if this is a AP beacon desc antenna update */
    if (isAp && antMode != 0) { 
        /* set tx antenna */
        pTxControl->antModeXmit = antMode;

        ASSERT(pTxControl->bufferLength);
        ASSERT(pTxControl->frameLength == 
               (pTxControl->bufferLength + FCS_FIELD_SIZE));

        return;
    } 

#if !defined(ECOS_NOTDONE)
    if (pRateTable->info[rateIndex].phy == WLAN_PHY_XR) {
        pTxControl->CTSEnable   = 1;
        pTxControl->RTSCTSRate  = XR_CTS_RATE(pArDev);
        pTxControl->RTSCTSDur   = PHY_COMPUTE_TX_TIME(
            pRateTable, pTxControl->frameLength, rateIndex, FALSE);
    }
#endif

    pTxControl->TXDataTries0    = 1;
    pTxControl->noAck           = 1;
    pTxControl->transmitPwrCtrl = MAX_RATE_POWER;
    pTxControl->antModeXmit     = antMode;
    pTxControl->PktType         = HAL_DESC_PKT_TYPE_BEACON;
    pTxControl->compProc        = 3;

    ASSERT(pTxControl->bufferLength);
    ASSERT(pTxControl->frameLength == 
           (pTxControl->bufferLength + FCS_FIELD_SIZE));
    pTxControl->more            = 0;

    /* Fill in the link pointers. */
    if (!isAp) {
        /*
         * If we're a client, link the beacon descriptor back onto itself
         * with the VEOL bit set.
         *
         * NOTE WELL: Do NOT link the virtual pointer back to itself as this
         *            will cause a hang when resetting the queue.
         */
        pDesc->nextPhysPtr  = pDesc->thisPhysPtr;
        pTxControl->VEOL    = 1;
        ASSERT(pDesc->pNextVirtPtr != pDesc);
    }
}

/**************************************************************
 * ar5212BeaconInit
 *
 * Initializes all of the hardware registers used to send beacons.
 *
 */
void
ar5212BeaconInit(AR_DEV_INFO *pArDev, WDC_BSS_ATTRIBUTES *pBssConfig)
{
    WLAN_TIMESTAMP  tsf;
    A_UINT32        nextTbtt, value;

    /*
     * Calculate next tbtt as a multiple of beacon interval
     * and sufficiently ahead of the current tsf value
     */
    ar5212GetTsf(pArDev, &tsf);
    nextTbtt = TSF_TO_TU(tsf) + pBssConfig->beaconInterval;
    nextTbtt = A_ROUNDUP(nextTbtt, pBssConfig->beaconInterval);

    /* Set the next beacon time register */
    writePlatformReg(pArDev, MAC_TIMER0, nextTbtt);

    /* Set DMA Beacon Alert (DBA) timer (in 1/8 TU = 128 us) */
    value = (nextTbtt * 8) - (DMA_BEACON_RESPONSE_TIME / 128);
    writePlatformReg(pArDev, MAC_TIMER1, value);

    /* Set Software Beacon Alert (SWBA) timer (in 1/8 TU = 128 us) */
    value = (nextTbtt * 8) - (SW_BEACON_RESPONSE_TIME / 128);
    writePlatformReg(pArDev, MAC_TIMER2, value);

    if (pBssConfig->bssType == INFRASTRUCTURE_BSS) {
        /* Enable SWBA interrupt. */
        ar5212EnableInterrupts(pArDev, HAL_INT_SWBA);
    } else {
        /* Set the ATIM window 
         * Our hardware does not support an ATIM window of 0 (beacons will not work).
         * If the ATIM windows is 0, force it to 1.
         */
        value = nextTbtt + (pBssConfig->atimWindow ? pBssConfig->atimWindow : 1);
        writePlatformReg(pArDev, MAC_TIMER3, value);
    }

    /*
     * Set the Beacon Control register
     * AP always starts with tsf = 0
     */
    value = pBssConfig->beaconInterval | MAC_BEACON_EN;
    if (pBssConfig->bssType == INFRASTRUCTURE_BSS) {
        value |= MAC_BEACON_RESET_TSF;
        /*
         * workaround for hw bug! when resetting the TSF, write twice to the
         * corresponding register; each write to the RESET_TSF bit toggles
         * the internal signal to cause a reset of the TSF - but if the signal
         * is left high, it will reset the TSF on the next chip reset also!
         * writing the bit an even number of times fixes this issue
         */
        writePlatformReg(pArDev, MAC_BEACON, MAC_BEACON_RESET_TSF);
    }
    writePlatformReg(pArDev, MAC_BEACON, value);

    /*
     * Quiet Time configuration for TGh
     * For now: hw test mode only
     */
    if (pBssConfig->quietDuration) {
#ifdef DEBUG
        uiPrintf("Quiet Period start @ %d for %d (every %d)\n",
                 nextTbtt + pBssConfig->quietOffset,
                 pBssConfig->quietDuration,
                 pBssConfig->beaconInterval);
#endif
        A_REG_WR_FIELD(pArDev, MAC_QUIET2, QUIET_PERIOD, pBssConfig->beaconInterval);
        A_REG_RMW_FIELD(pArDev, MAC_QUIET2, QUIET_DURATION, pBssConfig->quietDuration);

        A_REG_WR_FIELD(pArDev, MAC_QUIET1, NEXT_QUIET, nextTbtt + pBssConfig->quietOffset);
        if (pBssConfig->quietAckCtsAllow) {
            A_REG_SET_BIT(pArDev, MAC_QUIET1, QUIET_ACK_CTS_ENABLE);
        }
        A_REG_SET_BIT(pArDev, MAC_QUIET1, QUIET_ENABLE);
    }
}

/**************************************************************
 * ar5212SetupGrpPollChain
 *
 * Fills in the control fields of the group poll desc chain
 *
 * Assumes: frameLen does not include the FCS
 */
void
ar5212SetupGrpPollChain(AR_DEV_INFO *pArDev, AR_DESC *pHead, 
                        AR_DESC *pTail, A_UINT32 hwIndex)
{

/* xxxxPGSxxxx TODO - beacon partition */
#if !defined(ECOS_NOTDONE)
    OP_BSS *pOpBss = &pDev->xrBss;
    AR5212_TX_CONTROL *pTxControl;
    ATHEROS_DESC *pDesc;
   
    for (pDesc = pHead; pDesc != pTail; pDesc = pDesc->pNextVirtPtr)  {
        pTxControl = TX_CONTROL(pDesc);
        pTxControl->transmitPwrCtrl = MAX_RATE_POWER;
        pTxControl->clearDestMask   = (pDesc == pHead);
        pTxControl->destIdxValid    = (hwIndex != HWINDEX_INVALID);
        pTxControl->destIdx         = hwIndex;
        pTxControl->antModeXmit     = 0;
        pTxControl->TXRate0         = pOpBss->pRateTable->info[pOpBss->defaultRateIndex].rateCode;
        pTxControl->TXDataTries0    = 1;
        pTxControl->PktType         = HAL_DESC_PKT_TYPE_GRP_POLL;
        pTxControl->CTSEnable       = 1;
        pTxControl->RTSCTSRate      = XR_CTS_RATE(pDev);
        pTxControl->RTSCTSDur       = XR_UPLINK_TRANSACTION_TIME(pDev);
    }

    pTxControl = TX_CONTROL(pTail);
    pTxControl->transmitPwrCtrl = MAX_RATE_POWER;
    pTxControl->clearDestMask   = 1;
    pTxControl->destIdxValid    = (hwIndex != HWINDEX_INVALID);
    pTxControl->destIdx         = hwIndex;
    pTxControl->antModeXmit     = 0;
    pTxControl->PktType         = HAL_DESC_PKT_TYPE_NORMAL;
    pTxControl->TXRate0         = XR_CTS_RATE(pDev);
    pTxControl->TXDataTries0    = 1;
    pTxControl->noAck           = 1;
    pTxControl->VEOL            = 1;

    pTail->nextPhysPtr = pHead->thisPhysPtr;
#else
    ECOS_NOTDONE_XXX;
#endif
}

#endif /* BUILD_AR5212 */
