/*
 * Copyright (c) 2000-2002 Atheros Communications, Inc., All Rights Reserved
 *
 *  Chips-specific key cache routines.
 *
 */

#ifdef BUILD_AR5212

#include "arDev.h"
#include "arTransmit.h"

#if !defined(SPLIT_ENFORCE)
/* Standard HAL Headers */
#include "wlantype.h"
#include "wlandrv.h"
#include "wlanext.h"
#include "ui.h"
#endif

#include "halApi.h"
#include "hal.h"
#include "halUtil.h"
#include "wdcTypes.h"

/* Headers for HW private items */
#include "ar5212Reg.h"
#include "ar5212KeyCache.h"
#include "ar5212Misc.h"
#include "ar5212.h"

#ifdef DEBUG
extern int keyDebugLevel;
#endif /* DEBUG */

/**************************************************************
 * ar5212GetKeyCacheSize
 *
 * Returns size of hardware key cache.
 */
A_UINT32
ar5212GetKeyCacheSize(AR_DEV_INFO *pArDev)
{
    return MAC_KEY_CACHE_SIZE;
}

/**************************************************************
 * ar5212ResetKeyCacheEntry
 *
 * Clears the specified key cache entry
 */
void
ar5212ResetKeyCacheEntry(AR_DEV_INFO *pArDev, A_UINT16 keyCacheIndex)
{
    A_UINT32 keyCacheOffset;

    keyCacheOffset = MAC_KEY_CACHE +
        (keyCacheIndex * sizeof(AR5212_KEY_CACHE_ENTRY));

    writePlatformReg(pArDev, keyCacheOffset +
                     FIELD_OFFSET(AR5212_KEY_CACHE_ENTRY, keyVal0), 0);
    writePlatformReg(pArDev, keyCacheOffset +
                     FIELD_OFFSET(AR5212_KEY_CACHE_ENTRY, keyVal1), 0);
    writePlatformReg(pArDev, keyCacheOffset +
                     FIELD_OFFSET(AR5212_KEY_CACHE_ENTRY, keyVal2), 0);
    writePlatformReg(pArDev, keyCacheOffset +
                     FIELD_OFFSET(AR5212_KEY_CACHE_ENTRY, keyVal3), 0);
    writePlatformReg(pArDev, keyCacheOffset +
                     FIELD_OFFSET(AR5212_KEY_CACHE_ENTRY, keyVal4), 0);
    writePlatformReg(pArDev, keyCacheOffset +
                     FIELD_OFFSET(AR5212_KEY_CACHE_ENTRY, keyType),
                     MAC_KEY_TYPE_CLEAR);
    writePlatformReg(pArDev, keyCacheOffset +
                     FIELD_OFFSET(AR5212_KEY_CACHE_ENTRY, macAddrLo), 0);
    writePlatformReg(pArDev, keyCacheOffset +
                     FIELD_OFFSET(AR5212_KEY_CACHE_ENTRY, macAddrHi), 0);
}


// Base addresses of key cache in h/w
#define KEYREGS_INDEX(field) \
    (FIELD_OFFSET(AR5212_KEY_CACHE_ENTRY, field) / sizeof (A_UINT32))

// Set up key values
#define bytesToInt(str,idx) \
    ((str[idx]) | (str[idx+1] << 8) | (str[idx+2] << 16) | (str[idx+3] << 24))

/**************************************************************
 * ar5212SetKeyCacheEntry
 *
 * Sets the specified key cache entry
 *
 * Note: WEP key cache hardware requires that each double-word pair be
 * written in even/odd order (since the destination is a 64-bit register).
 * So we compile all the values for an entry into an array, then write the
 * array.  Reads can be done in any order, but the CACHE_PROTECT bit in
 * the eeprom must allow it for STA configs.
 *
 * AR5212 Supports WEP, TKIP and AES-OCB encryption in all macRevs.
 *        Supports AES-CCM only in macRev > 1.3
 *        Supports TKIP-Michael only macRev > 1.2
 */

void
ar5212SetKeyCacheEntry(AR_DEV_INFO *pArDev,
                       A_UINT32 keyCacheIndex,
                       WLAN_MACADDR *pBssId,
                       WLAN_PRIV_RECORD *pKey)
{
    A_UINT32            keyField, savedKey[2];
    A_UINT16            tkipIndex;
    A_UINT32            macHi = 0, macLo = 0, savedMacLo = 0, savedMacHi = 0;
    int                 i;
    A_UINT8             *pKeyVal = NULL;
    A_UINT16            keyLength;
    REGISTER_VAL        keyRegs[8];

    /* Treat key destruction first */
    if (pKey->keyFlags & PRIV_KEY_DESTROY) {
        ar5212ResetKeyCacheEntry(pArDev, keyCacheIndex);

        if (pKey->keyType == PRIV_KEY_TYPE_WEP_COMP) {
            ar5212ResetKeyCacheEntry(pArDev, keyCacheIndex + 32);
        }
        if (pKey->keyType == PRIV_KEY_TYPE_TKIP) {
            ar5212ResetKeyCacheEntry(pArDev, keyCacheIndex + 32);
            ar5212ResetKeyCacheEntry(pArDev, keyCacheIndex + 64);
            ar5212ResetKeyCacheEntry(pArDev, keyCacheIndex + 96);
        }
        return;
    }

    /* Receive key cache line for compression and WEP shared key */
    if (pKey->keyType == PRIV_KEY_TYPE_WEP_COMP) {
        keyCacheIndex += 32;
    }

    /*
     *  We have to sequence the writes to prevent MIC errors of pending rx frames.
     *  When we install either the encryption or decryption keys, we install
     *  the ones complement of the real key so that the encrypt/decrypt data path
     *  will fail with decryption errors whilst we are mucking with the MIC
     *  keys. After all of the potentially 4 key entries are set up,
     *  we must then go back and correct the corrupted key info.
     */

    /*
     * starting with clear key type, compression needs the keyIndex
     * associated with mac address
     */

    A_MEM_ZERO(keyRegs, sizeof(keyRegs));
    keyRegs[KEYREGS_INDEX(keyType)].Value = MAC_KEY_TYPE_CLEAR;
    savedKey[0] = savedKey[1] = 0;

    keyLength = pKey->keyLength;
    pKeyVal = pKey->keyVal;

    switch (pKey->keyType) {
    case PRIV_KEY_TYPE_AES_OCB:
        keyRegs[KEYREGS_INDEX(keyType)].Value = MAC_KEY_TYPE_AES;
        break;

    case PRIV_KEY_TYPE_AES_CCM:
        keyRegs[KEYREGS_INDEX(keyType)].Value
            = pArDev->hwInfo.supportCipherAES_CCM
            ? MAC_KEY_TYPE_CCM
            : MAC_KEY_TYPE_CLEAR;
        break;

    case PRIV_KEY_TYPE_TKIP:
        keyRegs[KEYREGS_INDEX(keyType)].Value = MAC_KEY_TYPE_TKIP;
        break;

    case PRIV_KEY_TYPE_CKIP:
    case PRIV_KEY_TYPE_TKIP_SW:
    case PRIV_KEY_TYPE_NULL:
        keyRegs[KEYREGS_INDEX(keyType)].Value = MAC_KEY_TYPE_CLEAR;
        break;

    case PRIV_KEY_TYPE_WEP:
    case PRIV_KEY_TYPE_WEP_COMP:
        switch (keyLength) {
        case 128:
            keyRegs[KEYREGS_INDEX(keyType)].Value = MAC_KEY_TYPE_WEP_128;
            break;

        case 104:
            keyRegs[KEYREGS_INDEX(keyType)].Value = MAC_KEY_TYPE_WEP_104;
            break;

        case 40:
            keyRegs[KEYREGS_INDEX(keyType)].Value = MAC_KEY_TYPE_WEP_40;
            break;
        }
        break;
    default:
        ASSERT(0);
        return;
    }

    switch (keyLength) {
    case 128:
    case 104:
        if (keyLength == 104) {
            // Only one byte (13th) in this register
            keyRegs[KEYREGS_INDEX(keyVal4)].Value =
                bytesToInt(pKeyVal,12)  & 0xff;
        } else {
            keyRegs[KEYREGS_INDEX(keyVal4)].Value = bytesToInt(pKeyVal,12);
        }
        // fall-through

        keyRegs[KEYREGS_INDEX(keyVal3)].Value =
            bytesToInt(pKeyVal,10) & 0xffff;

        keyRegs[KEYREGS_INDEX(keyVal2)].Value = bytesToInt(pKeyVal,6);

        // fall-through

    case 40:

        /* intentionally corrupt the first 2 words at keyIndex */

        savedKey[1] =  bytesToInt(pKeyVal,4) & 0xffff;
        keyRegs[KEYREGS_INDEX(keyVal1)].Value = (~savedKey[1] & 0xffff);

        savedKey[0] = bytesToInt(pKeyVal,0);
        keyRegs[KEYREGS_INDEX(keyVal0)].Value = ~savedKey[0];

        break;

    default:
        if (pKey->keyType != PRIV_KEY_TYPE_NULL) {
            return;     /* EINVAL */
        }
    }

    /* Set MAC address -- shifted right by 1.  MacLo is
     * the 4 MSBs, and MacHi is the 2 LSBs.
     */
    if (pBssId != NULL) {
        macHi = (pBssId->octets[5] << 8) | pBssId->octets[4];
        macLo = (pBssId->octets[3] << 24) | (pBssId->octets[2] << 16) |
                (pBssId->octets[1] << 8)  | pBssId->octets[0];

        macLo >>= 1;
        macLo |= (macHi & 1) << 31; /* carry */
        macHi >>= 1;
    }
    /*
     * Don't put the macAddrs into this entry;
     * save them for the decrypt entry
    */
    savedMacLo = macLo;
    savedMacHi = macHi;
    /* Set valid bit in savedMacHi */
    ((MAC_ADDR_HI_KEY_VALID *)&savedMacHi)->keyValid = TRUE;

    keyRegs[KEYREGS_INDEX(macAddrLo)].Value = 0;
    keyRegs[KEYREGS_INDEX(macAddrHi)].Value = 0;

    // Set "key is valid" bit
    keyField = keyRegs[KEYREGS_INDEX(macAddrHi)].Value;
    ((MAC_ADDR_HI_KEY_VALID *)&keyField)->keyValid = TRUE;
    keyRegs[KEYREGS_INDEX(macAddrHi)].Value = keyField;

    if (pKey->keyType != PRIV_KEY_TYPE_TKIP ||
        !(pArDev->hwInfo.supportMicTKIP)) {
        // Write them to hardware from the array, thereby ensuring that
        // they are written in order (even/odd).
        for (i = 0; i < (sizeof (keyRegs) / sizeof (REGISTER_VAL)); i++) {
            keyRegs[i].Offset = MAC_KEY_CACHE + (i*4) +
                (keyCacheIndex * sizeof(AR5212_KEY_CACHE_ENTRY));
            A_REG_WR(pArDev, keyRegs[i].Offset, keyRegs[i].Value);
        }

        /* Write the correct keyval0 now */
        A_REG_WR(pArDev, MAC_KEY_CACHE +
                 (keyCacheIndex * sizeof(AR5212_KEY_CACHE_ENTRY)),
                 savedKey[0]);
        A_REG_WR(pArDev, MAC_KEY_CACHE + 4 +
                 (keyCacheIndex * sizeof(AR5212_KEY_CACHE_ENTRY)),
                 savedKey[1]);

        A_REG_WR(pArDev, MAC_KEY_CACHE + (24) +
                 (keyCacheIndex * sizeof(AR5212_KEY_CACHE_ENTRY)),
                 savedMacLo);
        A_REG_WR(pArDev, MAC_KEY_CACHE + (28) +
                 (keyCacheIndex * sizeof(AR5212_KEY_CACHE_ENTRY)),
                 savedMacHi);

        return;
    }

    /*
     *  Encrypt key = i (already done above)
     *  Enmic key   = i + 64
     *  Decrypt key = i + 32
     *  Demic key   = i + 32 + 64
     */

    /* Now do the TKIP sub-keys */

    /* STA */
    if (pBssId != NULL) {

        tkipIndex = keyCacheIndex; // encrypt key
        pKeyVal = pKey->keyVal;
        /* Do not clear this entry! */

        A_MEM_ZERO(keyRegs, sizeof(keyRegs));

        keyRegs[KEYREGS_INDEX(keyType)].Value = MAC_KEY_TYPE_TKIP;
        keyRegs[KEYREGS_INDEX(keyVal4)].Value = bytesToInt(pKeyVal,12);
        keyRegs[KEYREGS_INDEX(keyVal3)].Value =
            bytesToInt(pKeyVal,10) & 0xffff;
        keyRegs[KEYREGS_INDEX(keyVal2)].Value = bytesToInt(pKeyVal,6);
        keyRegs[KEYREGS_INDEX(keyVal1)].Value = (~savedKey[1] & 0xffff);
        keyRegs[KEYREGS_INDEX(keyVal0)].Value = ~savedKey[0];

        keyRegs[KEYREGS_INDEX(macAddrLo)].Value = 0;
        keyRegs[KEYREGS_INDEX(macAddrHi)].Value = 0;

        // Set "key is valid" bit
        keyField = keyRegs[KEYREGS_INDEX(macAddrHi)].Value;
        ((MAC_ADDR_HI_KEY_VALID *)&keyField)->keyValid = TRUE;
        keyRegs[KEYREGS_INDEX(macAddrHi)].Value = keyField;

        for (i = 0; i < (sizeof (keyRegs) / sizeof (REGISTER_VAL)); i++) {
            keyRegs[i].Offset = MAC_KEY_CACHE + (i*4) +
                (tkipIndex * sizeof(AR5212_KEY_CACHE_ENTRY));
            A_REG_WR(pArDev, keyRegs[i].Offset, keyRegs[i].Value);
        }

        tkipIndex = keyCacheIndex + 64; // enmic key

        pKeyVal = pKey->micTxKeyVal;
        A_MEM_ZERO(keyRegs, sizeof(keyRegs));

        keyRegs[0].Value = bytesToInt(pKeyVal, 0);
        keyRegs[2].Value = bytesToInt(pKeyVal, 4);
        /* Valid bit is zero */

        for (i = 0; i < (sizeof (keyRegs) / sizeof (REGISTER_VAL)); i++) {
            keyRegs[i].Offset = MAC_KEY_CACHE + (i*4) +
                (tkipIndex * sizeof(AR5212_KEY_CACHE_ENTRY));
            A_REG_WR(pArDev, keyRegs[i].Offset, keyRegs[i].Value);
        }

        /* patch up the encrypt key */
        A_REG_WR(pArDev, MAC_KEY_CACHE +
                 (keyCacheIndex * sizeof(AR5212_KEY_CACHE_ENTRY)),
                 savedKey[0]);
        A_REG_WR(pArDev, MAC_KEY_CACHE + 4 +
                 (keyCacheIndex * sizeof(AR5212_KEY_CACHE_ENTRY)),
                 savedKey[1]);

        tkipIndex = keyCacheIndex + 32; // decrypt key
        pKeyVal = pKey->keyVal;

        A_MEM_ZERO(keyRegs, sizeof(keyRegs));

        keyRegs[KEYREGS_INDEX(keyType)].Value = MAC_KEY_TYPE_TKIP;
        keyRegs[KEYREGS_INDEX(keyVal4)].Value = bytesToInt(pKeyVal,12);
        keyRegs[KEYREGS_INDEX(keyVal3)].Value =
            bytesToInt(pKeyVal,10) & 0xffff;
        keyRegs[KEYREGS_INDEX(keyVal2)].Value = bytesToInt(pKeyVal,6);
        keyRegs[KEYREGS_INDEX(keyVal1)].Value = (~savedKey[1] & 0xffff);
        keyRegs[KEYREGS_INDEX(keyVal0)].Value = ~savedKey[0];

        keyRegs[KEYREGS_INDEX(macAddrLo)].Value = savedMacLo;
        keyRegs[KEYREGS_INDEX(macAddrHi)].Value = savedMacHi;

        for (i = 0; i < (sizeof (keyRegs) / sizeof (REGISTER_VAL)); i++) {
            keyRegs[i].Offset = MAC_KEY_CACHE + (i*4) +
                (tkipIndex * sizeof(AR5212_KEY_CACHE_ENTRY));
            A_REG_WR(pArDev, keyRegs[i].Offset,
                     keyRegs[i].Value);
        }

        tkipIndex = keyCacheIndex + 64 + 32; // demic key
        pKeyVal = pKey->micRxKeyVal;

        A_MEM_ZERO(keyRegs, sizeof(keyRegs));

        keyRegs[0].Value = bytesToInt(pKeyVal, 0);
        keyRegs[2].Value = bytesToInt(pKeyVal, 4);
        /* Valid bit is zero */

        for (i = 0; i < (sizeof (keyRegs) / sizeof (REGISTER_VAL)); i++) {
            keyRegs[i].Offset = MAC_KEY_CACHE + (i*4) +
                (tkipIndex * sizeof(AR5212_KEY_CACHE_ENTRY));
            A_REG_WR(pArDev, keyRegs[i].Offset,
                     keyRegs[i].Value);
        }

        /* patch up the decrypt key */
        A_REG_WR(pArDev,MAC_KEY_CACHE +
                 ((keyCacheIndex + 32) *
                 sizeof(AR5212_KEY_CACHE_ENTRY)),
                 savedKey[0]);
        A_REG_WR(pArDev, MAC_KEY_CACHE + 4 +
                 ((keyCacheIndex + 32) *
                 sizeof(AR5212_KEY_CACHE_ENTRY)),
                 savedKey[1]);
    } else {

        /* Group Key installation */

        tkipIndex = keyCacheIndex; // decrypt key
        pKeyVal = pKey->keyVal;

        A_MEM_ZERO(keyRegs, sizeof(keyRegs));

        keyRegs[KEYREGS_INDEX(keyType)].Value = MAC_KEY_TYPE_TKIP;
        keyRegs[KEYREGS_INDEX(keyVal4)].Value = bytesToInt(pKeyVal,12);
        keyRegs[KEYREGS_INDEX(keyVal3)].Value =
            bytesToInt(pKeyVal,10) & 0xffff;
        keyRegs[KEYREGS_INDEX(keyVal2)].Value = bytesToInt(pKeyVal,6);
        keyRegs[KEYREGS_INDEX(keyVal1)].Value = (~savedKey[1] & 0xffff);
        keyRegs[KEYREGS_INDEX(keyVal0)].Value = ~savedKey[0];

        keyRegs[KEYREGS_INDEX(macAddrLo)].Value = savedMacLo;
        keyRegs[KEYREGS_INDEX(macAddrHi)].Value = savedMacHi;

        for (i = 0; i < (sizeof (keyRegs) / sizeof (REGISTER_VAL)); i++) {
            keyRegs[i].Offset = MAC_KEY_CACHE + (i*4) +
                (tkipIndex * sizeof(AR5212_KEY_CACHE_ENTRY));
            A_REG_WR(pArDev, keyRegs[i].Offset, keyRegs[i].Value);
        }

        tkipIndex = keyCacheIndex + 64; // demic key
        pKeyVal = pKey->micRxKeyVal;

        A_MEM_ZERO(keyRegs, sizeof(keyRegs));

        keyRegs[0].Value = bytesToInt(pKeyVal, 0);
        keyRegs[2].Value = bytesToInt(pKeyVal, 4);
        /* Valid bit is zero */

        for (i = 0; i < (sizeof (keyRegs) / sizeof (REGISTER_VAL)); i++) {
            keyRegs[i].Offset = MAC_KEY_CACHE + (i*4) +
                (tkipIndex * sizeof(AR5212_KEY_CACHE_ENTRY));
            A_REG_WR(pArDev, keyRegs[i].Offset, keyRegs[i].Value);
        }

        /* patch up the decrypt key */
        A_REG_WR(pArDev, MAC_KEY_CACHE +
                 (keyCacheIndex * sizeof(AR5212_KEY_CACHE_ENTRY)),
                 savedKey[0]);
        A_REG_WR(pArDev, MAC_KEY_CACHE + 4 +
                 (keyCacheIndex * sizeof(AR5212_KEY_CACHE_ENTRY)),
                 savedKey[1]);
    }
    return;
}

/**************************************************************
 * ar5212SetDecompMask
 *
 * update decompression mask based on keyCache index
 */
A_UINT16
ar5212SetDecompMask(AR_DEV_INFO *pArDev, A_UINT32 connIdx,
                    A_UINT32 keyType, A_BOOL en)
{
    if (pArDev->hwInfo.compressSupport)
    {
        A_UINT16 decompMaskIdx = connIdx + MAX_SHARED_KEYS;

        /* TKIP decrypt index */
        if ((keyType == PRIV_KEY_TYPE_TKIP) ||
            (keyType == PRIV_KEY_TYPE_WEP_COMP)) {
            decompMaskIdx += 32;
        }
        ASSERT(decompMaskIdx < DECOMP_MASK_SIZE);
        A_REG_WR(pArDev, MAC_DCM_A, decompMaskIdx);
        A_REG_WR(pArDev, MAC_DCM_D, en ? MAC_DCM_D_EN : 0);
        pArDev->config.decompMaskArray[decompMaskIdx] = en;

        return (en ? decompMaskIdx : DECOMPINDEX_INVALID);
    }

    return DECOMPINDEX_INVALID;
}

#if defined(DEBUG) || defined(_DEBUG)

const char *
ar5212DebugGetKeyType(AR_DEV_INFO *pArDev, A_UINT32 idx)
{
    static const char *keyTypeToName[] = {
        "wep40",
        "wep104",
        "invalid",
        "wep128",
        "tkip",
        "ocb",
        "ccm",
        "clear"
    };
    A_UINT32 offset;
    A_UINT32 keyType;

    offset = MAC_KEY_CACHE + idx * sizeof(AR5212_KEY_CACHE_ENTRY);
    keyType = readPlatformReg(pArDev, offset +
                              FIELD_OFFSET(AR5212_KEY_CACHE_ENTRY, keyType));
    return keyTypeToName[keyType & 0x07];
}

#endif
#endif // #ifdef BUILD_AR5212

