/*
 *  Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 *  Chips-specific device miscellaneous functions including hardware queries,
 *  EEPROM routines, gpio funcs, beacon creation, ...
 */

#ifdef BUILD_AR5212

#include "arDev.h"

/* Standard HAL Headers */
#include "wlantype.h"
#if !defined(SPLIT_ENFORCE)
#include "wlandrv.h"
#endif
#include "halApi.h"
#include "hal.h"
#if !defined(SPLIT_ENFORCE)
#include "ui.h"
#endif
#include "halUtil.h"
#include "halDevId.h"
#if !defined(SPLIT_ENFORCE)
#include "wlanchannel.h"
#endif

/* Headers for HW private items */
#include "ar5212Reg.h"
#include "ar5212Misc.h"
#include "ar5212Reset.h"
#include "ar5212Power.h"
#include "ar5212Interrupts.h"
#include "ar5212.h"

#if defined(AR531X)
#include "ar531xreg.h"
#endif

#if defined(AR5523)
#include "cyg/hal/plf_bcfg.h"
#endif

//#if defined(NOT_USED)
static A_STATUS
ar5212EepromWrite(AR_DEV_INFO *pArDev, A_UINT32 offset, A_UINT16 data);
//#endif /* NOT_USED */

#define SLOT_TIME_20 880
#define SLOT_TIME_09 396

/**************************************************************
 * ar5212UpdateEepromChecksum
 *
 * Update the EEPROM checksum
 */
A_STATUS
ar5212UpdateEepromChecksum(WLAN_DEV_INFO *pDev)
{
    A_STATUS status = A_OK;
    A_UINT32 eepEndLoc;
    A_UINT32 location;
    A_UINT16 data;
    A_UINT16 chkSum;


    /* Read sizing information */
    status = ar5212EepromRead(pDev, EEPROM_SIZE_UPPER, &data);
    if (status != A_OK) {
        uiPrintf("ar5212UpdateEepromChecksum: Could not access EEPROM\n");
        return A_HARDWARE;
    }

    if (data == 0) {
        eepEndLoc = ATHEROS_EEPROM_END_DEFAULT;
    } else {
        eepEndLoc = (data & EEPROM_SIZE_UPPER_M) << 12;
        status = ar5212EepromRead(pDev, EEPROM_SIZE_LOWER, &data);
        if (status != A_OK) {
            uiPrintf("ar5212UpdateEepromChecksum: Could not access EEPROM\n");
            return A_HARDWARE;
        }
        eepEndLoc |= data;
    }
    ASSERT(eepEndLoc > ATHEROS_EEPROM_OFFSET);

    chkSum = 0;
    /* EEPROM checksum calculation */
    for (location = ATHEROS_EEPROM_OFFSET + 1; location < eepEndLoc; location++) {
        status = ar5212EepromRead(pDev, location, &data);
        if (status != A_OK) {
            uiPrintf("ar5212UpdateEepromChecksum: Failed to read EEPROM\n");
            return status;
        }
        chkSum ^= data;
    }

    chkSum = ~chkSum & 0xffff;

    status = ar5212EepromWrite(pDev, ATHEROS_EEPROM_OFFSET, chkSum);
    if (status != A_OK) {
        uiPrintf("ar5212UpdateEepromChecksum: Failed to write EEPROM checksum\n");
    }
    uiPrintf("ar5212UpdateEepromChecksum: Succeeded to write EEPROM checksum\n");

    return status;
}

/**************************************************************
 * ar5212GetSerialNumber
 *
 * Copy Hal Serial Number into provided string.
 * Returns TRUE if the value was copied.
 */
A_BOOL
ar5212GetSerialNumber(AR_DEV_INFO *pArDev, A_CHAR *pSerialNum, A_UINT16 strLen)
{
    if (strLen < sizeof(pArDev->pHalInfo->serialNumber)) {
        return FALSE;
    }
    A_DRIVER_BCOPY(pArDev->pHalInfo->serialNumber, pSerialNum,
                   sizeof(pArDev->pHalInfo->serialNumber));
    return TRUE;
}

/**************************************************************
 * ar5212SetRegulatoryDomain
 *
 * Attempt to change the cards operating regulatory domain to the given value
 * Returns: A_EINVAL for an unsupported regulatory domain.
 *          A_HARDWARE for an unwritable EEPROM or bad EEPROM version
 */
A_STATUS
ar5212SetRegulatoryDomain(AR_DEV_INFO *pArDev, A_UINT16 regDomain)
{
    EEP_MAP *pEep;

    ASSERT(pArDev->pHalInfo && pArDev->pHalInfo->pEepData);

    pEep = pArDev->pHalInfo->pEepData;
    if (pArDev->hwInfo.regDomain != regDomain) {

        /* TODO: validity check regDomain */

#if defined(NOT_USED)
        if (!(pEep->protect & EEPROM_PROTECT_WP_128_191))
        {

            /* Try to update EEPROM. */
            ar5212EepromWrite(pArDev, REGULATORY_DOMAIN_OFFSET, regDomain);
        } 
#endif /* NOT_USED */

        pArDev->hwInfo.regDomain = regDomain;
    }

    return A_OK;
}

/**************************************************************
 * ar5212GetRfKill
 *
 * Accessor to get rfkill info from private EEPROM structures
 */
A_BOOL
ar5212GetRfKill(AR_DEV_INFO *pArDev)
{
#if defined(TBD) /* xxxxLMWxxxx */
    A_UINT16 rfsilent;

    ASSERT(pArDev->pHalInfo && pArDev->pHalInfo->pEepData);
    if (pArDev->pHalInfo->pEepData->pEepHeader->rfKill) {
        ar5212EepromRead(pArDev, EEPROM_RFSILNT_CLKEN_OFFSET, &rfsilent);
        pArDev->rfSilent.gpioSelect = (rfsilent & EEP_RFSILENT_GPIO_SEL_MASK) >> EEP_RFSILENT_GPIO_SEL_SHIFT;
        pArDev->rfSilent.polarity = (rfsilent & EEP_RFSILENT_POLARITY_MASK) >> EEP_RFSILENT_POLARITY_SHIFT;
        pArDev->rfSilent.eepEnabled = TRUE;
    } else {
        pArDev->rfSilent.gpioSelect = pArDev->rfSilent.polarity = 0;
        pArDev->rfSilent.eepEnabled = FALSE;
    }
    return (A_BOOL)pArDev->pHalInfo->pEepData->pEepHeader->rfKill;
#else
    return FALSE; /* RfKill is unsupported */
#endif /* TBD */
}

/**************************************************************
 * ar5212GetMacAddr
 *
 * Attempt to get the MAC address from the wireless EEPROM
 * Returns: A_HARDWARE for an unreadable EEPROM
 */
A_STATUS
ar5212GetMacAddr(AR_DEV_INFO *pArDev, WLAN_MACADDR *mac)
{
#if (defined(PCI_INTERFACE) || defined(AR5523))
    A_UINT32 total = 0;
    A_UINT16 half;

    if (ar5212EepromRead(pArDev, EEPROM_MAC_MSW_OFFSET, &half) != A_OK) {
        goto ee_error;
    }
    total += half;
    mac->octets[1] = half & 0xff;
    mac->octets[0] = half >> 8;

    if (ar5212EepromRead(pArDev, EEPROM_MAC_MID_OFFSET, &half) != A_OK) {
        goto ee_error;
    }
    total += half;
    mac->octets[3] = half & 0xff;
    mac->octets[2] = half >> 8;
    if (ar5212EepromRead(pArDev, EEPROM_MAC_LSW_OFFSET, &half) != A_OK) {
        goto ee_error;
    }
    total += half;
    mac->octets[5] = half & 0xff;
    mac->octets[4] = half >> 8;

    if ((total == 0) || total == (3 * 0xffff)) {
        uiPrintf("ar5212GetMacAddr: EEPROM MAC is all 0's or all F's\n");
        return A_HARDWARE;
    }
    return A_OK;

ee_error:
    uiPrintf("ar5212GetMacAddr: EEPROM read failed\n");
    return A_HARDWARE;
#elif defined(AR531X)
    /* The ar5312 may store the WLAN mac addr in the system board config */
    if (pArDev->devno == 0) {
        A_DRIVER_BCOPY(sysBoardData.wlan0Mac, mac->octets, WLAN_MAC_ADDR_SIZE);
    } else {
        A_DRIVER_BCOPY(sysBoardData.wlan1Mac, mac->octets, WLAN_MAC_ADDR_SIZE);
    }
#else
#error "Need to implement"
#endif
    return A_OK;
}

/**************************************************************
 * ar5212EepromRead
 *
 * Read 16 bits of data from offset into *data
 */
A_STATUS
ar5212EepromRead(AR_DEV_INFO *pArDev, A_UINT32 offset, A_UINT16 *data)
{
#if defined(PCI_INTERFACE)
    A_UINT32 status;
    int to = 10000;     /* 10ms timeout */

    A_REG_WR(pArDev, MAC_EEPROM_ADDR, offset);
    A_REG_WR(pArDev, MAC_EEPROM_CMD, MAC_EEPROM_CMD_READ);

    while (to > 0) {
        udelay(1);
        status = A_REG_RD(pArDev, MAC_EEPROM_STS);
        if (status & MAC_EEPROM_STS_READ_COMPLETE) {
            status = A_REG_RD(pArDev, MAC_EEPROM_DATA);
            *data = (A_UINT16)(status & 0xffff);
            return A_OK;
        }

        if (status & MAC_EEPROM_STS_READ_ERROR) {
            uiPrintf("ar5212EepromRead: eeprom read error at offset %d.\n", offset);
            return A_HARDWARE;
        }
        to--;
    }
    uiPrintf("ar5212EepromRead: eeprom read timeout at offset %d.\n", offset);
    return A_HARDWARE;
#elif defined(AR531X)
    /* radio configuration data is stored in the system flash */
    *data = sysFlashConfigRead(FLC_RADIOCFG, (offset * 2) + 1) |
            (sysFlashConfigRead(FLC_RADIOCFG, offset * 2) << 8);
#elif defined(AR5523)
    hal_ar5523_bcfgdata_read(offset, 1, (unsigned short *)data);
#else
#error "Need to implement"
#endif /* PLATFORM */
    return A_OK;
}

//#if defined(NOT_USED)
/**************************************************************
 * ar5212EepromWrite
 *
 * Write 16 bits of data from data into offset
 */
static A_STATUS
ar5212EepromWrite(AR_DEV_INFO *pArDev, A_UINT32 offset, A_UINT16 data)
{
#if defined(PCI_INTERFACE)
    A_UINT32 status;
    int to = 15000;     /* 15ms timeout */

    /* Send write data */
    A_REG_WR(pArDev, MAC_EEPROM_ADDR, offset);
    A_REG_WR(pArDev, MAC_EEPROM_DATA, data);
    A_REG_WR(pArDev, MAC_EEPROM_CMD, MAC_EEPROM_CMD_WRITE);

    while (to > 0) {
        udelay(1);
        status = A_REG_RD(pArDev, MAC_EEPROM_STS);
        if (status & MAC_EEPROM_STS_WRITE_COMPLETE) {
            return A_OK;
        }

        if (status & MAC_EEPROM_STS_WRITE_ERROR)    {
            uiPrintf("ar5212EepromWrite: eeprom write error at offset %d.\n", offset);
            return A_HARDWARE;
        }
        to--;
    }

    uiPrintf("ar5212EepromWrite: eeprom write timeout at offset %d.\n", offset);
    return A_HARDWARE;
#elif defined(AR531X)
    char str[2];

    /* Radio configuration data is stored in system flash (in reverse endian) */
    str[0] = (data >> 8) & 0xff;
    str[1] = data & 0xff;
    sysFlashConfigWrite(FLC_RADIOCFG, offset<<1, str, 2);

    return A_OK;

#elif defined(AR5523) || defined(EEPROM_WRITE_ENABLED)



	char str[4];



	str[0] = (data << 8) & 0xff;

	str[1] = (data >> 8) & 0xff;

	str[2] = (data >> 16) & 0xff;

	str[3] = data & 0xff;

		

    hal_ar5523_bcfgdata_write(offset, 2, (unsigned short *)&str);



    return A_OK;

/* BKUO */

#else
#error "Need to implement"
#endif /* PLATFORM */
}
//#endif /* NOT_USED */

/**************************************************************************
 * ar5212EnableRfKill
 *
 * Called if RfKill is supported (according to EEPROM).  Set the interrupt and
 * GPIO values so the ISR can disable RF on a switch signal.  Assumes GPIO port
 * and interrupt polarity are set prior to call.
 */
void
ar5212EnableRfKill(AR_DEV_INFO *pArDev)
{
#if defined(TBD) /* xxxxLMWxxxx */
    /* TODO - can this really be above the hal on the GPIO interface for
     * TODO - the client only?
     */
#if !defined(BUILD_AP)  /* AP uses GPIOs for LEDs and buttons */

    /* Configure the desired GPIO port for input and enable baseband rf silence */
    ar5212GpioCfgInput(pDev, pDev->rfSilent.gpioSelect);
    writePlatformReg(pDev, PHY_BASE, readPlatformReg(pDev, PHY_BASE) | 0x00002000);
    /*
     * If radio disable switch connection to GPIO bit x is enabled
     * program GPIO interrupt.
     * If rfkill bit on eeprom is 1, setupeeprommap routine has already
     * verified that it is a later version of eeprom, it has a place for
     * rfkill bit and it is set to 1, indicating that GPIO bit x hardware
     * connection is present.
     */

    if (pDev->rfSilent.polarity == ar5212GpioGet(pDev, pDev->rfSilent.gpioSelect)) {
        /* switch already closed, set to interrupt upon open */
        ar5212GpioSetIntr(pDev, pDev->rfSilent.gpioSelect, !pDev->rfSilent.polarity);
    } else {
        ar5212GpioSetIntr(pDev, pDev->rfSilent.gpioSelect, pDev->rfSilent.polarity);
    }
#endif /* !BUILD_AP */
#endif /* TBD */
}

/**************************************************************
 * ar5212GpioCfgOutput
 *
 * Configure GPIO Output lines
 */
void
ar5212GpioCfgOutput(AR_DEV_INFO *pDev, A_UINT32 gpio)
{
#if defined(PCI_INTERFACE)
    A_UINT32 reg;

    ASSERT(gpio < MAC_NUM_GPIO);

    reg =  readPlatformReg(pDev, MAC_GPIOCR);
//    reg &= ~(MAC_GPIOCR_0_CR_A << (gpio * MAC_GPIOCR_CR_SHIFT));
    reg |= MAC_GPIOCR_0_CR_A << (gpio * MAC_GPIOCR_CR_SHIFT);

    writePlatformReg(pDev, MAC_GPIOCR, reg);
#endif /* PCI_INTERFACE */
}

/**************************************************************
 * ar5212GpioCfgInput
 *
 * Configure GPIO Input lines
 */
void
ar5212GpioCfgInput(AR_DEV_INFO *pDev, A_UINT32 gpio)
{
#if defined(PCI_INTERFACE)
    A_UINT32 reg;

    ASSERT(gpio < MAC_NUM_GPIO);

    reg =  readPlatformReg(pDev, MAC_GPIOCR);
    reg &= ~(MAC_GPIOCR_0_CR_A << (gpio * MAC_GPIOCR_CR_SHIFT));
    reg |= MAC_GPIOCR_0_CR_N << (gpio * MAC_GPIOCR_CR_SHIFT);

    writePlatformReg(pDev, MAC_GPIOCR, reg);
#endif /* PCI_INTERFACE */
}

/**************************************************************
 * ar5212GpioSet
 *
 * Once configured for I/O - set output lines
 */
void
ar5212GpioSet(AR_DEV_INFO *pDev, A_UINT32 gpio, A_UINT32 val)
{
#if defined(PCI_INTERFACE)
    A_UINT32 reg;

    ASSERT(gpio < MAC_NUM_GPIO);

    reg =  readPlatformReg(pDev, MAC_GPIODO);
    reg &= ~(1 << gpio);
    reg |= (val&1) << gpio;

    writePlatformReg(pDev, MAC_GPIODO, reg);
#endif /* PCI_INTERFACE */
}

/**************************************************************
 * ar5212GpioGet
 *
 * Once configured for I/O - get input lines
 */
A_UINT32
ar5212GpioGet(AR_DEV_INFO *pDev, A_UINT32 gpio)
{
#if defined(PCI_INTERFACE)
    A_UINT32 reg;

    ASSERT(gpio < MAC_NUM_GPIO);

    reg =  readPlatformReg(pDev, MAC_GPIODI);
    reg = ((reg & MAC_GPIOD_MASK) >> gpio) & 0x1;

    return reg;
#else /* !PCI_INTERFACE */
    return 0;
#endif /* PCI_INTERFACE */
}

/**************************************************************
 * ar5212GpioSetIntr
 *
 * Set the desired GPIO Interrupt
 */
void
ar5212GpioSetIntr(AR_DEV_INFO *pDev, A_UINT32 gpio, A_UINT32 ilevel)
{
#if defined(PCI_INTERFACE)
    A_UINT32    reg;

    ASSERT(gpio < MAC_NUM_GPIO);

    reg = readPlatformReg(pDev, MAC_GPIOCR);
    /* clear the bits that we will modify */
    reg &= ~((MAC_GPIOCR_0_CR_A << (gpio * MAC_GPIOCR_CR_SHIFT)) | MAC_GPIOCR_INT_MASK | MAC_GPIOCR_INT_EN | MAC_GPIOCR_INT_SELH);
    reg |= MAC_GPIOCR_INT_EN | (gpio << MAC_GPIOCR_INT_SHIFT) | (MAC_GPIOCR_0_CR_N << (gpio * MAC_GPIOCR_CR_SHIFT));
    if (ilevel) {
        reg |= MAC_GPIOCR_INT_SELH;
    }
    /* don't need to change anything for low level interrupt. */
    writePlatformReg(pDev, MAC_GPIOCR, reg);
    /* change the interrupt mask */
    writePlatformReg(pDev, MAC_IMR, readPlatformReg(pDev, MAC_IMR) | MAC_IMR_GPIO);
    HACK_TO_PARDEV(pDev)->MaskReg = readPlatformReg(pDev, MAC_IMR);
#endif /* PCI_INTERFACE */
}

A_UINT32 Current_Led_Mode= MAC_PCICFG_LED_MODE_0;
A_UINT32 Current_Led_Blink_Threshold= MAC_PCICFG_LED_BLINK_THRESHOLD_0;
A_UINT32 Current_Led_Slow_Mode= MAC_PCICFG_LED_SLOW_BLINK_MODE_0;

/******************************************************************
 * ar5212SetLedState
 *
 * Change the LED blinking pattern to correspond to the connectivity.
 *   Normal blink when connected, alternate blink when not.
 */
void
ar5212SetLedState(AR_DEV_INFO *pDev, A_BOOL bConnected)
{
#if defined(PCI_INTERFACE) || defined(AR5523)
    A_UINT32 val;

    val = bConnected ?
          MAC_PCICFG_ASSOC_STATUS_ASSOCIATED : MAC_PCICFG_ASSOC_STATUS_PENDING;

    A_REG_RMW_FIELD(pDev, MAC_PCICFG, ASSOC_STATUS, val);

    A_REG_RMW_FIELD(pDev, MAC_PCICFG, LED_MODE, Current_Led_Mode);

    A_REG_RMW_FIELD(pDev, MAC_PCICFG, LED_BLINK_THRESHOLD, Current_Led_Blink_Threshold);
  
    A_REG_RMW_FIELD(pDev, MAC_PCICFG, LED_SLOW_BLINK_MODE, Current_Led_Slow_Mode);

#elif defined(AR531X)
    A_UINT32 reg = sysRegRead(AR531X_PCICFG) & ~ASSOC_STATUS_M;

    if (bConnected) {
        reg |= ASSOC_STATUS_ASSOCIATED;
    } else {
        reg |= ASSOC_STATUS_PENDING;
    }
    sysRegWrite(AR531X_PCICFG, reg);
#endif /* PLATFORM */
}

/**************************************************************************
 * ar5212WriteAssocid - Change association related fields programmed into the hardware.
 *
 * Writing a valid BSSID to the hardware effectively enables the hardware
 * to synchronize its TSF to the correct beacons and receive frames coming
 * from that BSSID. It is called by the SME JOIN operation.
 */
void
ar5212WriteAssocid(AR_DEV_INFO *pDev, WLAN_MACADDR *bssid, A_UINT16 assocId)
{
    A_UINT32    ulAddressLow, ulAddressHigh;

    if (bssid != NULL) {
        ulAddressLow  = cpu2le32(bssid->st.word);
        ulAddressHigh = cpu2le16(bssid->st.half);
    } else {
        ulAddressLow = ulAddressHigh = 0;       /* default for no paramter */
    }

    A_REG_WR(pDev, MAC_BSS_ID0, ulAddressLow);
    A_REG_WR(pDev, MAC_BSS_ID1, ulAddressHigh);

    /*
     * Workaround for a hardware bug in Oahu. Write a 0 associd to prevent
     * hardware from parsing beyond end of TIM element. It will avoid
     * sending spurious PS-polls. Beacon/TIM processing done in software
     */
    A_REG_RMW_FIELD(pDev, MAC_BSS_ID1, AID, 0);
}


/**************************************************************
 * ar5212SetStaBeaconTimers
 *
 *  Sets all the beacon related bits on the h/w for stations
 *  i.e. initializes the corresponding h/w timers;
 *  also tells the h/w whether to anticipate PCF beacons
 */
void
ar5212SetStaBeaconTimers(AR_DEV_INFO *pArDev, WDC_BSS_ATTRIBUTES *pBssConfig)
{
    AR_DEV_INFO   *pDev = HACK_TO_PDEV(pArDev);
    A_UINT32        nextTbtt;
    A_UINT32        sleepDuration;
    WLAN_TIMESTAMP  tsf;

    /*
     * The following assumes that we've already latched onto
     * a beacon/probe-response for this BSS - ideally, one
     * would assert atleast a single beacon recd at this time
     */
    ar5212GetTsf(pArDev, &tsf);
    nextTbtt = TSF_TO_TU(tsf);
    nextTbtt = A_ROUNDUP(nextTbtt, pBssConfig->beaconInterval);

    /* add a beaconPeriod for safety, lest we're on the edge */
    nextTbtt += pBssConfig->beaconInterval;

    /*
     * Handle PCF support if present
     */
    if (pBssConfig->cfpMaxDuration) {
        /* tell the h/w that the associated AP is PCF capable */
        A_REG_SET_BIT(pDev, MAC_STA_ID1, PCF);

        /* set CFP_PERIOD(1.024ms) register */
        A_REG_WR(pDev, MAC_CFP_PERIOD, pBssConfig->cfpInterval);

        /* set CFP_DUR(1.024ms) register to max cfp duration */
        A_REG_WR(pDev, MAC_CFP_DUR, pBssConfig->cfpMaxDuration);

        /* set TIMER2(128us) to anticipated time of next CFP */
        A_REG_WR(pDev, MAC_TIMER2, A_ROUNDUP(nextTbtt, pBssConfig->cfpInterval) * 8);

    } else {
        /* tell the h/w that the associated AP is not PCF capable */
        A_REG_CLR_BIT(pDev, MAC_STA_ID1, PCF);
    }


    /*
     * Set the TIMER0(1.024ms) register to anticipated time of
     * the next beacon. Then start the beacon timers by setting
     * the BEACON register to the beacon interval
     */
    A_REG_WR(pDev, MAC_TIMER0, nextTbtt);
    A_REG_RMW_FIELD(pDev, MAC_BEACON, PERIOD, pBssConfig->beaconInterval);


    /*
     * Configure the BMISS interrupt. Need to disable/re-enable
     * interrupts, as well as clear any pending bmiss interrupts
     * for the same
     */
    ASSERT(pBssConfig->bmissThreshold <= (MAC_RSSI_THR_BM_THR_M >> MAC_RSSI_THR_BM_THR_S));

    ar5212DisableInterrupts(pArDev, HAL_INT_GLOBAL);
    A_REG_WR(pDev, MAC_IMR, pArDev->MaskReg & ~MAC_IMR_BMISS);
    pDev->globISRReg &= ~MAC_IMR_BMISS;
    A_REG_RMW_FIELD(pDev, MAC_RSSI_THR, BM_THR, pBssConfig->bmissThreshold);
    pArDev->MaskReg |= MAC_IMR_BMISS;
    A_REG_WR(pDev, MAC_IMR, pArDev->MaskReg);
    ar5212EnableInterrupts(pArDev, HAL_INT_GLOBAL);

#if !defined(ECOS_NOTDONE)

    /*
     * Quiet Time configuration for TGh
     * For now: hw test mode only
     */
    if (pBssConfig->quietDuration) {
        uiPrintf("Quiet Period start @ %d for %d (every %d)\n",
                 nextTbtt + pBssConfig->quietOffset,
                 pBssConfig->quietDuration,
                 pBssConfig->beaconInterval);
        A_REG_WR_FIELD(pDev, MAC_QUIET2, QUIET_PERIOD, pBssConfig->beaconInterval);
        A_REG_RMW_FIELD(pDev, MAC_QUIET2, QUIET_DURATION, pBssConfig->quietDuration);

        A_REG_WR_FIELD(pDev, MAC_QUIET1, NEXT_QUIET, nextTbtt + pBssConfig->quietOffset);
        if (pBssConfig->quietAckCtsAllow) {
            A_REG_SET_BIT(pDev, MAC_QUIET1, QUIET_ACK_CTS_ENABLE);
        }
        A_REG_SET_BIT(pDev, MAC_QUIET1, QUIET_ENABLE);
    }
#endif

    /*
     * Oahu beacons timers on the station were used for power
     * save operation (waking up in anticipation of a beacon)
     * and any CFP function; Venice does sleep/power-save timers
     * differently - so this is the right place to set them up;
     * don't think the beacon timers are used by venice sta hw
     * for any useful purpose anymore
     * Setup venice's sleep related timers
     * Current implementation assumes sw processing of beacons -
     *   assuming an interrupt is generated every beacon which
     *   causes the hardware to become awake until the sw tells
     *   it to go to sleep again; beacon timeout is to allow for
     *   beacon jitter; cab timeout is max time to wait for cab
     *   after seeing the last DTIM or MORE CAB bit
     */
#define CAB_TIMEOUT_VAL     10 /* in TU */
#define BEACON_TIMEOUT_VAL  10 /* in TU */
#define SLEEP_SLOP          3  /* in TU */

    /* Make sure to sleep atleast for a beacon interval */
    ASSERT(pBssConfig->sleepDuration);
    sleepDuration = A_ROUNDUP(pBssConfig->sleepDuration, pBssConfig->beaconInterval);

    /*
     * If we sleep for more that a beacon interval - make sure that its
     * alteast for a dtim interval
     */
    ASSERT(A_ROUNDUP(pBssConfig->dtimInterval, pBssConfig->beaconInterval) == pBssConfig->dtimInterval);
    if (sleepDuration > pBssConfig->beaconInterval) {
        sleepDuration = A_ROUNDUP(sleepDuration, pBssConfig->dtimInterval);
    }

    /*
     * Now sleep duration is either beacon interval, or dtim interval,
     * or multiple of dtim interval. The first case will wake up for
     * every beacon. The second case, we wake up only for DTIMs as the
     * spec wants us to inorder to receive CAB traffic. The last case
     * is our own max-powersave mode where we may even disregarding CAB
     * traffic for a while
     */
    nextTbtt = A_ROUNDUP(nextTbtt, sleepDuration);
    A_REG_WR_FIELD(pDev, MAC_SLEEP1, NEXT_DTIM,
        (A_ROUNDUP(nextTbtt, pBssConfig->dtimInterval) - SLEEP_SLOP) * 8);
    A_REG_RMW_FIELD(pDev, MAC_SLEEP1, CAB_TIMEOUT, CAB_TIMEOUT_VAL);
    A_REG_SET_BIT2(pDev, MAC_SLEEP1, ASSUME_DTIM, ENH_SLEEP_ENABLE);
    A_REG_WR_FIELD(pDev, MAC_SLEEP2, NEXT_TIM, (nextTbtt - SLEEP_SLOP) * 8);
    A_REG_RMW_FIELD(pDev, MAC_SLEEP2, BEACON_TIMEOUT, BEACON_TIMEOUT_VAL);
    A_REG_WR_FIELD(pDev, MAC_SLEEP3, TIM_PERIOD, sleepDuration);
    A_REG_RMW_FIELD(pDev, MAC_SLEEP3, DTIM_PERIOD,
                    A_MAX(sleepDuration, pBssConfig->dtimInterval));
}


/******************************************************************
 * ar5212GetTsf
 *
 * Get the current hardware tsf for stamlme
 */
void
ar5212GetTsf(AR_DEV_INFO *pArDev, WLAN_TIMESTAMP *tsf)
{
    A_UINT32    high;           /* Timestamp high order 32 bits */

    tsf->low  = readPlatformReg(pArDev, MAC_TSF_L32);
    high      = readPlatformReg(pArDev, MAC_TSF_U32);    
    tsf->high = readPlatformReg(pArDev, MAC_TSF_U32);
    if (tsf->high != high) {    /* Counter rollover? */
        tsf->low = readPlatformReg(pArDev, MAC_TSF_L32);        
    }
}

/******************************************************************
 * ar5212ResetTsf
 *
 * Reset the current hardware tsf for stamlme
 */
void
ar5212ResetTsf(AR_DEV_INFO *pDev)
{
    A_REG_SET_BIT(pDev, MAC_BEACON, RESET_TSF);
    /*
     * workaround for hw bug! when resetting the TSF, write twice to the
     * corresponding register; each write to the RESET_TSF bit toggles
     * the internal signal to cause a reset of the TSF - but if the signal
     * is left high, it will reset the TSF on the next chip reset also!
     * writing the bit an even number of times fixes this issue
     */
    A_REG_SET_BIT(pDev, MAC_BEACON, RESET_TSF);
}

/******************************************************************
 * ar5212SetAdhocMode
 *
 * Set adhoc mode for stamlme
 */
void
ar5212SetAdhocMode(AR_DEV_INFO *pArDev)
{
    A_UINT32 id1 = readPlatformReg(pArDev, MAC_STA_ID1);

    writePlatformReg(pArDev, MAC_CFG,
                     readPlatformReg(pArDev, MAC_CFG) | MAC_CFG_AP_ADHOC_INDICATION);

    /* Update antenna mode */
    id1 &= ~(MAC_STA_ID1_USE_DEFANT | MAC_STA_ID1_DEFANT_UPDATE);
    writePlatformReg(pArDev, MAC_STA_ID1, id1 |
                     MAC_STA_ID1_AD_HOC |
                     MAC_STA_ID1_RTS_USE_DEF);

}

/**************************************************************
 * ar5212SetBasicRate
 *
 * Set or clear hardware basic rate bit
 * Set harfware basic rate set if basic rate is found
 * and basic rate is equal or less than 2Mbps
 */
void
ar5212SetBasicRate(AR_DEV_INFO *pArDev, CONN_RATE_SET *pSet)
{
    int     i;
    A_UINT8 rset, xset = 0;


    if (CHAN_IS_CCK(&pArDev->config.chanDesc) == FALSE) {
        return;
    }

    for (i = 0; i < pSet->length; i++) {
        rset = pSet->rates[i];
        if (rset & 0x80) {              /* Basic rate defined? */
            rset = rset & 0x7f;
            xset = (rset >= xset) ? rset : xset;
        }
    }

    /* If basic rate is found to be equal or less than 2Mbps, then set the hw bit */
    if (xset && xset/2 <= 2) {
        /* tell the h/w that the associated AP is running basic rate */
        A_REG_SET_BIT(pArDev, MAC_STA_ID1, BASE_RATE_11B);
        return;
    }

    /* tell the h/w that the associated AP is not running basic rate */
    A_REG_CLR_BIT(pArDev, MAC_STA_ID1, BASE_RATE_11B);
}

/**************************************************************
 * ar5212GetRandomSeed
 *
 * Grab a semi-random value from hardware registers - may not
 * change often
 */

A_UINT32
ar5212GetRandomSeed(AR_DEV_INFO *pDev)
{
    A_UINT32 seed, nf;

    nf = (readPlatformReg(pDev, PHY_BASE+(25<<2)) >> 19) & 0x1ff;
    if (nf & 0x100) {
        nf = 0 - ((nf ^ 0x1ff) + 1);
    }

    seed = readPlatformReg(pDev, MAC_TSF_U32) ^ readPlatformReg(pDev, MAC_TSF_L32) ^
        nf;

    return seed;
}

/**************************************************************
 * ar5212DetectCardPresent
 *
 * Detect if our card is present
 */

A_BOOL
ar5212DetectCardPresent(AR_DEV_INFO *pArDev)
{
#if defined(PCI_INTERFACE)
    A_UINT32 value;
    A_UINT16  macVersion, macRev;

    /* Read the SREV register and compare to what we read at initialization */
    value     = readPlatformReg(pArDev, MAC_SREV) & MAC_SREV_ID_M;
    macVersion= (A_UINT16) (value >> MAC_SREV_ID_S);
    macRev    = (A_UINT16) (value & MAC_SREV_REVISION_M);

    if ( (pArDev->hwInfo.macVersion == macVersion) && (pArDev->hwInfo.macRevision == macRev) ) {
        return TRUE;
    } else {
        return FALSE;
    }
#else
    return TRUE;                        /* the spirit mac is not going anywhere */
#endif
}

/**************************************************************
 * ar5212MibControl
 *
 * Control of MIB Counters
 */
A_UINT32
ar5212MibControl(AR_DEV_INFO *pDev, HAL_MIB_CMD cmd)
{
    switch (cmd) {
    case UPDATE_SW_ALL:
    case UPDATE_SW_COMMON: {
        AR_STATS     *pDevStats  = &pDev->devStats;

        AR_UPDATE_DEV_STATS(pDevStats, FcsFailCnt,
                        readPlatformReg(pDev, MAC_FCS_FAIL));
        AR_UPDATE_DEV_STATS(pDevStats, AckRcvFailures,
                        readPlatformReg(pDev, MAC_ACK_FAIL));
        AR_UPDATE_DEV_STATS(pDevStats, RtsSuccessCnt,
                        readPlatformReg(pDev, MAC_RTS_OK));
        AR_UPDATE_DEV_STATS(pDevStats, RtsFailCnt,
                        readPlatformReg(pDev, MAC_RTS_FAIL));

        AR_UPDATE_DEV_STATS(pDevStats, CompCPC0Cnt,
                        readPlatformReg(pDev, MAC_CPC_0));
        AR_UPDATE_DEV_STATS(pDevStats, CompCPC1Cnt,
                        readPlatformReg(pDev, MAC_CPC_1));
        AR_UPDATE_DEV_STATS(pDevStats, CompCPC2Cnt,
                        readPlatformReg(pDev, MAC_CPC_2));
        AR_UPDATE_DEV_STATS(pDevStats, CompCPC3Cnt,
                        readPlatformReg(pDev, MAC_CPC_3));

        if (UPDATE_SW_ALL == cmd) {
            /* Read MIB variables that change less frequently. */
            AR_UPDATE_DEV_STATS(pDevStats, RxBeacons,
                            readPlatformReg(pDev, MAC_BEACON_CNT));
        }
        break;
    }

    case GET_CCA_PERCENTAGE: {
        A_UINT32 cycles, rxClr;

        cycles = readPlatformReg(pDev, MAC_CCCNT);
        rxClr  = readPlatformReg(pDev, MAC_RCCNT);

        if (cycles < 100) {
            return 0;
        }

        cycles /= 100;

        /* the value of the rx_clear counter is inverted */
        return 100 - rxClr / cycles;

        break;
    }

    case CLEAR_ALL_COUNTERS: {
        A_UINT32 regVal;

        regVal = readPlatformReg(pDev, MAC_MIBC);
        writePlatformReg(pDev, MAC_MIBC, regVal | MAC_MIBC_CMC);
        writePlatformReg(pDev, MAC_MIBC, regVal & ~MAC_MIBC_CMC);
        break;
    }

    default:
        break;
    }
    return 0;
}

/**************************************************************
 * ar5212IsHwCipherSupported
 *
 * Detect if Cipher requested is implemented in HW.
 */
A_BOOL
ar5212IsHwCipherSupported(AR_DEV_INFO *pDev, A_UINT32 keyType)
{
#if !defined(ECOS_NOTDONE)
    switch (keyType) {
    case PRIV_KEY_TYPE_WEP:
    case PRIV_KEY_TYPE_TKIP:
    case PRIV_KEY_TYPE_NULL:
        return TRUE;
    case PRIV_KEY_TYPE_AES_CCM:
        /* CCM non-compliant up to rev 3 */
        return (pDev->macRev > 3);
    default:
        return FALSE;
    }
#else
    ECOS_NOTDONE_XXX;
    return FALSE;
#endif
}

/**************************************************************
 * ar5212EnableRadarDetection
 *
 * Enable radar detection in the hardware - may later be changed
 * to modify radar "sensitivity"
 */
void
ar5212EnableRadarDetection(AR_DEV_INFO *pDev, HAL_RADAR_PHY_PARAMS *pPhyParams)
{
    A_UINT32 val;
    if (pPhyParams) {
        /* New params supplied by caller */
        val = A_FIELD_VALUE(PHY_RADAR_0, FIRPWR, pPhyParams->firpwr) |
              A_FIELD_VALUE(PHY_RADAR_0, RRSSI, pPhyParams->radarRssi) |
              A_FIELD_VALUE(PHY_RADAR_0, HEIGHT, pPhyParams->height) |
              A_FIELD_VALUE(PHY_RADAR_0, PRSSI, pPhyParams->pulseRssi) |
              A_FIELD_VALUE(PHY_RADAR_0, INBAND, pPhyParams->inband);
        writePlatformReg(pDev, PHY_RADAR_0, val);
    }

    /* Now enable pulse detection */
    A_REG_SET_BIT(pDev, PHY_RADAR_0, EN);

    /*
     * Fast antenna diversity for strong signal disturbs radar detection of 1-2 us pulses.
     * Enable antenna diversity but disable strong signal aspect of it.
     */

    /* enable_ant_fast_div = 1 */
    A_REG_SET_BIT(pDev, PHY_CCK_DETECT, BB_ENABLE_ANT_FAST_DIV);
    /* ant_fast_div_gc_limit = 1 */
    A_REG_WR_FIELD(pDev, PHY_RESTART, FAST_DIV_GC, 1);
}


/*
 * Anti noise immunity support.  We track phy errors and react
 * to excessive errors by adjusting the noise immunity parameters.
 */

#define EP_MUL(x, mul)   ((x) * (mul))
#define EP_RND(x, mul)   ((((x)%(mul)) >= ((mul)/2)) ? \
                          ((x) + ((mul) - 1)) / (mul) : (x)/(mul))
#define	MSEC_TO_USEC(_ms)	((_ms) * 1000)
#define	USEC_TO_MSEC(_us)	((_us) / 1000)
#define	SEC_TO_MSEC(_sec)	((_sec) * 1000)
#define	SEC_TO_USEC(_sec)	(SEC_TO_MSEC(_sec) * 1000)

#define ANI_DEBUG        uiPrintf
#define ANI_MIB_UPDATE   // Update MIB statistics during ANI

/******************************************************************************
 *
 * New Ani Algorithm for Station side only
 *
 * Ported from Linux MadWiFi which is based on Mainline
 * 08/14/04
 *
 * XXX: This code assumes (does not even check) that the hardware
 * as hardware phy counters to read OFDM and CCK errors.
 *
 *****************************************************************************/


A_INT16
ar5212GetAniChannelIndex(AR_DEV_INFO *pArDev, CHANNEL_FREQ freq) {
    int iTmp;

    for (iTmp = 0; 
         iTmp < (sizeof(pArDev->arAniState)/sizeof(AR_ANI)); 
         iTmp++) {

        if (pArDev->arAniState[iTmp].channelFreq == freq)
            return(iTmp);

        if (pArDev->arAniState[iTmp].channelFreq == 0) {
            return(iTmp);
        }
    }

    ANI_DEBUG("Unable to find ANI index %d [%d]\n", 
              freq, __LINE__);
    
    /* XXX: Must return something valid */
    /* XXX: On Disable/Enable we should clear the aniState
     * should we suspend/resume and re-emerge in a new country */

    return 0;               
}

void
ar5212EnableMIBCounters(AR_DEV_INFO *pArDev) 
{

    /* XXX: This code is part of the ar5212_ani.c in MadWifi.
     * It really should be under ar5212MibControl() 
     */

    ANI_DEBUG("%s\n", __FUNCTION__);

#if defined(ANI_MIB_UPDATE)
    ar5212MibControl(pArDev, UPDATE_SW_ALL);
#endif

	A_REG_WR(pArDev,  MAC_PHYCNT_FILTOFDM, 0);
	A_REG_WR(pArDev,  MAC_PHYCNT_FILTCCK,  0);
	A_REG_WR(pArDev,  MAC_MIBC, 
		     ~(MAC_MIBC_COW | MAC_MIBC_FMC | MAC_MIBC_CMC | MAC_MIBC_MCS)
		     & 0x0f);
    A_REG_WR(pArDev, MAC_PHYCNT1MASK, MAC_PHY_ERR_OFDM_TIMING);
    A_REG_WR(pArDev, MAC_PHYCNT2MASK, MAC_PHY_ERR_CCK_TIMING);
}


void 
ar5212DisableMIBCounters(AR_DEV_INFO *pArDev) 
{

    /* XXX: This code is part of the ar5212_ani.c in MadWifi.
     * It really should be under ar5212MibControl() 
     */

    ANI_DEBUG("%s\n", __FUNCTION__);

	A_REG_WR(pArDev, MAC_MIBC,  MAC_MIBC_FMC | MAC_MIBC_CMC);

#if defined(ANI_MIB_UPDATE)
    ar5212MibControl(pArDev, UPDATE_SW_ALL);
#endif

	A_REG_WR(pArDev, MAC_PHYCNT_FILTOFDM, 0);
	A_REG_WR(pArDev, MAC_PHYCNT_FILTCCK,  0);
}


/*
 * Setup ANI handling.  Sets all thresholds and levels to default level AND
 * resets the channel statistics
 */

LOCAL void
ar5212AniStateInit(AR_ANI *aniState) {
    A_MEM_ZERO(aniState, sizeof(AR_ANI));

    aniState->ofdmTrigHigh         = HAL_ANI_OFDM_TRIG_HIGH;
    aniState->ofdmTrigLow          = HAL_ANI_OFDM_TRIG_LOW;
    aniState->cckTrigHigh          = HAL_ANI_CCK_TRIG_HIGH;
    aniState->cckTrigLow           = HAL_ANI_CCK_TRIG_LOW;
    aniState->rssiThrHigh          = HAL_ANI_RSSI_THR_HIGH;
    aniState->rssiThrLow           = HAL_ANI_RSSI_THR_LOW;
    aniState->ofdmWeakSigDetectOff = (HAL_ANI_USE_OFDM_WEAK_SIG ?  0 : 1);
    aniState->cckWeakSigThreshold  = HAL_ANI_CCK_WEAK_SIG_THR;
    aniState->spurImmunityLevel    = HAL_ANI_SPUR_IMMUNE_LVL;
    aniState->firstepLevel         = HAL_ANI_FIRSTEP_LVL;
    aniState->phyErrStatsDisabled  = HAL_ANI_PHYERRSTATS_DIS;
    aniState->ofdmPhyErrBase       = (MAC_PHYCNT_CNTMAX - 
                                      HAL_ANI_OFDM_TRIG_HIGH);
    aniState->cckPhyErrBase        = (MAC_PHYCNT_CNTMAX -  
                                      HAL_ANI_CCK_TRIG_HIGH);

    return;
}


A_BOOL
ar5212AniAttach(AR_DEV_INFO *pArDev) 
{
	int iTmp;

    ANI_DEBUG("%s\n", __FUNCTION__);
    
	for (iTmp=0; 
         iTmp < (sizeof(pArDev->arAniState)/sizeof(AR_ANI));
         iTmp++) {
        ar5212AniStateInit(&pArDev->arAniState[iTmp]);
    }

    A_REG_WR(pArDev, MAC_PHYCNT1,
             pArDev->arAniState[0].ofdmPhyErrBase);
    A_REG_WR(pArDev, MAC_PHYCNT2,
             pArDev->arAniState[0].cckPhyErrBase);
    ar5212EnableMIBCounters(pArDev);
    pArDev->arAniPeriod = HAL_ANI_PERIOD;

#if defined(DO_ANI)
    return(TRUE);
#else
    return(FALSE);
#endif
}



/*
 * Cleanup any ANI state setup.
 */
void
ar5212AniDetach(AR_DEV_INFO *pArDev) 
{
    ANI_DEBUG("%s\n", __FUNCTION__);

    ar5212DisableMIBCounters(pArDev);
    A_REG_WR(pArDev, MAC_PHYCNT1, 0);
    A_REG_WR(pArDev, MAC_PHYCNT2, 0);
    ar5212DisableInterrupts(pArDev, HAL_INT_MIB);
	return;
}


/**************************************************************
 * ar5212AniControl
 *
 * Control Adaptive Noise Immunity Parameters
 */
void
ar5212AniControl(AR_DEV_INFO *pArDev, HAL_ANI_CMD cmd, int param)
{
#define TABLE_SIZE(_table) (sizeof(_table)/sizeof((_table)[0]))
    typedef int TABLE[];

    switch (cmd) {
    case SET_NOISE_IMMUNITY_LEVEL: {
#ifdef AR5312
        const TABLE totalSizeDesired = { -41, -41, -48, -48, -48 };
        const TABLE coarseHigh       = { -18, -18, -16, -14, -12 };
        const TABLE coarseLow        = { -56, -56, -60, -60, -60 };
        const TABLE firpwr           = { -72, -72, -75, -78, -80 };
#else
        const TABLE totalSizeDesired = { -55, -55, -55, -55, -62 };
        const TABLE coarseHigh       = { -14, -14, -14, -14, -12 };
        const TABLE coarseLow        = { -64, -64, -64, -64, -70 };
        const TABLE firpwr           = { -78, -78, -78, -78, -80 };
#endif

        unsigned int level = param;

        ANI_DEBUG("%s:%d SET_NOISE_IMMUNITY_LEVEL %d \n", 
                  __FUNCTION__, pArDev->config.chanDesc.channelFreq,
                  level);

        ASSERT(level < TABLE_SIZE(totalSizeDesired));

        A_REG_RMW_FIELD(pArDev, PHY_DESIRED_SZ, TOT_DES, totalSizeDesired[level]);
        A_REG_RMW_FIELD(pArDev, PHY_AGC_CTL1, COARSE_HIGH, coarseHigh[level]);
        A_REG_RMW_FIELD(pArDev, PHY_AGC_CTL1, COARSE_LOW, coarseLow[level]);
        A_REG_RMW_FIELD(pArDev, PHY_FIND_SIG, FIRPWR, firpwr[level]);

        break;
    }

    case SET_OFDM_WEAK_SIGNAL_DETECTION: {
        const TABLE m1ThreshLow   = { 127,   50 };
        const TABLE m2ThreshLow   = { 127,   40 };
        const TABLE m1Thresh      = { 127, 0x4d };
        const TABLE m2Thresh      = { 127, 0x40 };
        const TABLE m2CountThr    = {  31,   16 };
        const TABLE m2CountThrLow = {  63,   48 };

        unsigned int on = param ? 1 : 0;

        ANI_DEBUG("%s:%d OFDM_WEAK_SIG_DETECT %d \n", 
                  __FUNCTION__, pArDev->config.chanDesc.channelFreq, on);

        A_REG_RMW_FIELD(pArDev, PHY_SFCORR_LOW, M1_THRESH_LOW, m1ThreshLow[on]);
        A_REG_RMW_FIELD(pArDev, PHY_SFCORR_LOW, M2_THRESH_LOW, m2ThreshLow[on]);
        A_REG_RMW_FIELD(pArDev, PHY_SFCORR, M1_THRESH, m1Thresh[on]);
        A_REG_RMW_FIELD(pArDev, PHY_SFCORR, M2_THRESH, m2Thresh[on]);
        A_REG_RMW_FIELD(pArDev, PHY_SFCORR, M2COUNT_THR, m2CountThr[on]);
        A_REG_RMW_FIELD(pArDev, PHY_SFCORR_LOW, M2COUNT_THR_LOW, m2CountThrLow[on]);

        if (on) {
            A_REG_SET_BIT(pArDev, PHY_SFCORR_LOW, USE_SELF_CORR_LOW);
        } else {
            A_REG_CLR_BIT(pArDev, PHY_SFCORR_LOW, USE_SELF_CORR_LOW);
        }

        break;
    }

    case SET_CCK_WEAK_SIGNAL_THR: {
        const TABLE weakSigThrCck = { 8, 6 };

        unsigned int high = param ? 1 : 0;

        ANI_DEBUG("%s:%d CCK_WEAK_SIGNAL_THR %d \n", 
                  __FUNCTION__, pArDev->config.chanDesc.channelFreq, high);

        A_REG_RMW_FIELD(pArDev, PHY_CCK_DETECT, WEAK_SIG_THR_CCK, weakSigThrCck[high]);

        break;
    }

    case SET_FIRSTEP_LEVEL: {
        const TABLE firstep = { 0, 4, 8 };

        unsigned int level = param;

        ANI_DEBUG("%s:%d FIRSTEP_LEVEL %d \n", 
                  __FUNCTION__, pArDev->config.chanDesc.channelFreq, level);

        ASSERT(level < TABLE_SIZE(firstep));

        A_REG_RMW_FIELD(pArDev, PHY_FIND_SIG, FIRSTEP, firstep[level]);

        break;
    }

    case SET_SPUR_IMMUNITY_LEVEL: {
        const TABLE cycpwrThr1 = { 2, 4, 6, 8, 10, 12, 14, 16 };

        unsigned int level = param;

        ANI_DEBUG("%s:%d SPUR_IMMUNITY %d %d\n", 
                  __FUNCTION__, pArDev->config.chanDesc.channelFreq, 
                  level, cycpwrThr1[level]);

        ASSERT(level < TABLE_SIZE(cycpwrThr1));

        A_REG_RMW_FIELD(pArDev, PHY_TIMING5, CYCPWR_THR1, cycpwrThr1[level]);

        break;
    }

    default:
        ASSERT(0);
    }

    return;
}


void
ar5212AniRestart(AR_DEV_INFO *pArDev) 
{
	AR_ANI  *aniState = pArDev->arCurrentAni;

	ASSERT(aniState);

	aniState->listenTime = 0;

    if (aniState->ofdmTrigHigh > MAC_PHYCNT_CNTMAX) {
        aniState->ofdmPhyErrBase = 0;
    } else {
        aniState->ofdmPhyErrBase = (MAC_PHYCNT_CNTMAX - 
                                    aniState->ofdmTrigHigh);
    }

    if (aniState->cckTrigHigh > MAC_PHYCNT_CNTMAX) {
        aniState->cckPhyErrBase = 0;
    } else {
        aniState->cckPhyErrBase = (MAC_PHYCNT_CNTMAX - 
                                   aniState->cckTrigHigh);
    }


    A_REG_WR(pArDev, MAC_PHYCNT1, aniState->ofdmPhyErrBase);
    A_REG_WR(pArDev, MAC_PHYCNT2, aniState->cckPhyErrBase);
    A_REG_WR(pArDev, MAC_PHYCNT1MASK, MAC_PHY_ERR_OFDM_TIMING);
    A_REG_WR(pArDev, MAC_PHYCNT2MASK, MAC_PHY_ERR_CCK_TIMING);

#if defined(ANI_MIB_UPDATE) 
    ar5212MibControl(pArDev, UPDATE_SW_ALL);
#endif

    aniState->ofdmPhyErrCount = 0;
    aniState->cckPhyErrCount  = 0;
}


void
ar5212AniOfdmErrTrigger(AR_DEV_INFO *pArDev)
{
	AR_ANI *aniState = pArDev->arCurrentAni;
	A_INT32 rssi     = EP_RND(pArDev->arAniLastBeaconRssi, RSSI_EP_MULTIPLIER);

    ASSERT(aniState);
    
	uiPrintf("%s\n", __FUNCTION__);

    if (!pArDev->arDoAni)
        return;

	/* First, raise nosie immunity level, up to max */
	if (aniState->noiseImmunityLevel < HAL_NOISE_IMMUNE_MAX) {
		ar5212AniControl(pArDev, SET_NOISE_IMMUNITY_LEVEL, 
                         ++aniState->noiseImmunityLevel);
		return;
	}
	/* then, raise spur immunity level, up to max */
	if (aniState->spurImmunityLevel < HAL_SPUR_IMMUNE_MAX) {
		ar5212AniControl(pArDev, SET_SPUR_IMMUNITY_LEVEL,
                         ++aniState->spurImmunityLevel);
		return;
	}

	/* If beacon rssi is high, can turn off ofdm weak sig detect */
	if (rssi > aniState->rssiThrHigh) {
		if (!aniState->ofdmWeakSigDetectOff) {
			aniState->ofdmWeakSigDetectOff = TRUE;
			ar5212AniControl(pArDev, SET_OFDM_WEAK_SIGNAL_DETECTION,
                             FALSE);
			aniState->spurImmunityLevel = 0;
			ar5212AniControl(pArDev, SET_SPUR_IMMUNITY_LEVEL,
                             aniState->spurImmunityLevel);
			return;
		}
		/* 
		 * If weak sig detect is already off, as last resort, raise
		 * first step level 
		 */
		if (aniState->firstepLevel < HAL_FIRST_STEP_MAX) {
			ar5212AniControl(pArDev, SET_FIRSTEP_LEVEL,
                             ++aniState->firstepLevel);
			return;
		}
	} else if (rssi > aniState->rssiThrLow) {
		/* 
		 * If beacon rssi in mid range, need ofdm weak signal detect,
		 * but we can raise firststepLevel 
		 */
			if (aniState->ofdmWeakSigDetectOff) {
				aniState->ofdmWeakSigDetectOff = FALSE;
				ar5212AniControl(pArDev, SET_OFDM_WEAK_SIGNAL_DETECTION,
                                 TRUE);
			}
			if (aniState->firstepLevel < HAL_FIRST_STEP_MAX) {
				ar5212AniControl(pArDev, SET_FIRSTEP_LEVEL,
                                 ++aniState->firstepLevel);
			}
			return;
    } else {
        /* 
         * Beacon rssi is low, if in 11b/g mode, turn off ofdm
         * weak sign detction and zero firstepLevel to maximize
         * CCK sensitivity 
         */
        if (pArDev->config.chanDesc.channelBand == CHANNEL_BAND_2GHz) {
            if (!aniState->ofdmWeakSigDetectOff) {
                aniState->ofdmWeakSigDetectOff = TRUE;
                ar5212AniControl(pArDev, SET_OFDM_WEAK_SIGNAL_DETECTION,
                                 FALSE);
            }
            if (aniState->firstepLevel > 0) {
                aniState->firstepLevel = 0;
                ar5212AniControl(pArDev, SET_FIRSTEP_LEVEL,
                                 aniState->firstepLevel);
            }
            return;
        }
    }
}


void
ar5212AniCckErrTrigger(AR_DEV_INFO *pArDev) 
{
	AR_ANI *aniState = pArDev->arCurrentAni;
	A_INT32 rssi     = EP_RND(pArDev->arAniLastBeaconRssi, RSSI_EP_MULTIPLIER);

    ASSERT(aniState);

    if (!pArDev->arDoAni)
        return;

	/* first, raise noise immunity level, up to max */
	if (aniState->noiseImmunityLevel < HAL_NOISE_IMMUNE_MAX) {
		ar5212AniControl(pArDev, SET_NOISE_IMMUNITY_LEVEL,
                         ++aniState->noiseImmunityLevel);
		return;
	}

	/* beacon signal in mid and high range, can raise firsteplevel */
	if (rssi >  aniState->rssiThrLow) {
		if (aniState->firstepLevel < HAL_FIRST_STEP_MAX) {
			ar5212AniControl(pArDev, SET_FIRSTEP_LEVEL,
                             ++aniState->firstepLevel);
		}
	} else {
		/* beacon rssi is low, zero firstepLevel to maximize CCK sensitivity */
        if (pArDev->config.chanDesc.channelBand == CHANNEL_BAND_2GHz) {
			if (aniState->firstepLevel > 0) {
				aniState->firstepLevel = 0;
				ar5212AniControl(pArDev, SET_FIRSTEP_LEVEL,
                                 aniState->firstepLevel);
            }
        }
    }
}

/*
 * Restore the ANI parameters in the HAL and reset the
 * statistics.  This routine should be called for every
 * hardware reset and for every channel change.
 */

void
ar5212AniReset(AR_DEV_INFO *pArDev)
{
	AR_ANI   *aniState;
    A_INT32   index;

    index = ar5212GetAniChannelIndex(pArDev, 
                                     pArDev->config.chanDesc.channelFreq);

    aniState             = &pArDev->arAniState[index];
    pArDev->arCurrentAni = aniState;

    ASSERT(aniState);

    if ((aniState->channelFreq    != pArDev->config.chanDesc.channelFreq) ||
        (aniState->phyModulations != pArDev->config.chanDesc.phyModulations)) {
        ar5212AniStateInit(aniState);
        aniState->channelFreq     = pArDev->config.chanDesc.channelFreq;
        aniState->phyModulations  = pArDev->config.chanDesc.phyModulations;
    }
	
    ar5212AniControl(pArDev, SET_NOISE_IMMUNITY_LEVEL,
                     aniState->noiseImmunityLevel);

    ar5212AniControl(pArDev, SET_SPUR_IMMUNITY_LEVEL,
                     aniState->spurImmunityLevel);

    ar5212AniControl(pArDev, SET_OFDM_WEAK_SIGNAL_DETECTION,
                     !aniState->ofdmWeakSigDetectOff);

    ar5212AniControl(pArDev, SET_CCK_WEAK_SIGNAL_THR,
                     aniState->cckWeakSigThreshold);

    ar5212AniControl(pArDev, SET_FIRSTEP_LEVEL, aniState->firstepLevel);

    ar5212AniRestart(pArDev);
    A_REG_WR(pArDev, MAC_PHYCNT1MASK, MAC_PHY_ERR_OFDM_TIMING);
    A_REG_WR(pArDev, MAC_PHYCNT2MASK, MAC_PHY_ERR_CCK_TIMING);
}


/*
 * 08/14/04: MadWifi changed this function name
 * from AniReceiveCheck -> PhyErrReport
 */
void 
ar5212AniReceiveCheck(AR_DEV_INFO *pArDev) 
{
	AR_ANI *aniState = pArDev->arCurrentAni;
    
    ASSERT(aniState);

    uiPrintf("%s\n", __FUNCTION__);
	
    aniState->ofdmPhyErrCount = (A_REG_RD(pArDev, MAC_PHYCNT1) - 
                                 aniState->ofdmPhyErrBase);
    aniState->cckPhyErrCount  = (A_REG_RD(pArDev, MAC_PHYCNT2) - 
                                 aniState->cckPhyErrBase);
    
	if (aniState->ofdmPhyErrCount > aniState->ofdmTrigHigh) {
		ar5212AniOfdmErrTrigger(pArDev);
        ar5212AniRestart(pArDev);
	}
	if (aniState->cckPhyErrCount > aniState->cckTrigHigh) {
		ar5212AniCckErrTrigger(pArDev);
        ar5212AniRestart(pArDev);
	}
}


void ar5212ProcessMibIntr(AR_DEV_INFO *pArDev)
{
    AR_ANI   *aniState = pArDev->arCurrentAni;
    A_UINT32 phyCnt1;
    A_UINT32 phyCnt2;

    ANI_DEBUG("%s\n", __FUNCTION__);

	/* Reset these counters regardless */
	A_REG_WR(pArDev, MAC_PHYCNT_FILTOFDM, 0);
	A_REG_WR(pArDev, MAC_PHYCNT_FILTCCK,  0);

#if defined(ANI_MIB_UPDATE) 
    ar5212MibControl(pArDev, UPDATE_SW_ALL);
#endif

    phyCnt1 = A_REG_RD(pArDev, MAC_PHYCNT1);
    phyCnt2 = A_REG_RD(pArDev, MAC_PHYCNT2);

	if (((phyCnt1 & MAC_PHYCNT_INTRMASK) == MAC_PHYCNT_INTRMASK) ||
	    ((phyCnt2 & MAC_PHYCNT_INTRMASK) == MAC_PHYCNT_INTRMASK)) {
        
        A_INT32 ofdmPhyErrCnt;
        A_INT32 cckPhyErrCnt;

        ofdmPhyErrCnt = phyCnt1 - aniState->ofdmPhyErrBase;
        aniState->ofdmPhyErrCount = ofdmPhyErrCnt;

        cckPhyErrCnt  = phyCnt2 - aniState->cckPhyErrBase;
        aniState->cckPhyErrCount = cckPhyErrCnt;

        if (aniState->ofdmPhyErrCount > aniState->ofdmTrigHigh)
            ar5212AniOfdmErrTrigger(pArDev);
        if (aniState->cckPhyErrCount > aniState->cckTrigHigh)
            ar5212AniCckErrTrigger(pArDev);
        ar5212AniRestart(pArDev);
    }
}

void
ar5212AniLowerImmunity(AR_DEV_INFO *pArDev) 
{
	AR_ANI  *aniState = pArDev->arCurrentAni;
	A_INT32  rssi     = EP_RND(pArDev->arAniLastBeaconRssi, 
                               RSSI_EP_MULTIPLIER);

	ASSERT(aniState);

	if (rssi > aniState->rssiThrHigh) {
		/* 
		 * beacon signal is high, leave ofdm weak signal detection off
		 * or it may oscillate. Let it fall through.
		 */
	} else if (rssi > aniState->rssiThrLow) {
		/* beacon rssi in mid range, turn on ofdm weak signal
		   detection or lower first step level */
		if (aniState->ofdmWeakSigDetectOff) {
			aniState->ofdmWeakSigDetectOff = FALSE;
			ar5212AniControl(pArDev, SET_OFDM_WEAK_SIGNAL_DETECTION,
                             TRUE);
			return;
		}
		if (aniState->firstepLevel > 0) {
			ar5212AniControl(pArDev, SET_FIRSTEP_LEVEL,
                             --aniState->firstepLevel);
			return;
		}
	} else {
		/* beacon rssi is low, reduce first step level */
		if (aniState->firstepLevel > 0) {
			ar5212AniControl(pArDev, SET_FIRSTEP_LEVEL,
                             --aniState->firstepLevel);
			return;
		}
	}
	/* then lower spur immunity level, down to zero */
	if (aniState->spurImmunityLevel > 0) {
		ar5212AniControl(pArDev, SET_SPUR_IMMUNITY_LEVEL,
                         --aniState->spurImmunityLevel);
		return;
	}
	/* 
	 * if all else fails, lower noise immunity level down to a min value
	 * zero for now
	 */
	if (aniState->noiseImmunityLevel > 0) {
		ar5212AniControl(pArDev, SET_NOISE_IMMUNITY_LEVEL,
                         --aniState->noiseImmunityLevel);
		return;
	}
}


/* 
 * A return value of -1 indicates an invalid time
 */

#define CLOCK_RATE 44000
/* convert HW counter values to ms using 11g clock rate, goo9d enough
   for 11a and Turbo */
A_INT32
ar5212AniGetListenTime(AR_DEV_INFO *pArDev) 
{
	AR_ANI   *aniState     = pArDev->arCurrentAni;
	A_UINT32  txFrameCount = 0;
    A_UINT32  rxFrameCount = 0;
    A_UINT32  cycleCount   = 0;
	A_INT32   listenTime   = 0;

    ASSERT(aniState);

	txFrameCount = A_REG_RD(pArDev, MAC_TFCNT);
	rxFrameCount = A_REG_RD(pArDev, MAC_RFCNT);
	cycleCount   = A_REG_RD(pArDev, MAC_CCCNT);

	if ((aniState->cycleCount == 0) || aniState->cycleCount > cycleCount) {
		listenTime = 0;
	} else {
        A_INT32 ccdelta = cycleCount - aniState->cycleCount;
        A_INT32 rfdelta = rxFrameCount - aniState->rxFrameCount;
        A_INT32 tfdelta = txFrameCount - aniState->txFrameCount;

        listenTime = (ccdelta - rfdelta - tfdelta) / CLOCK_RATE;
 	}

	aniState->cycleCount = cycleCount;
	aniState->txFrameCount = txFrameCount;
	aniState->rxFrameCount = rxFrameCount;

	return(listenTime);
}



void
ar5212AniPeriodicCheck(AR_DEV_INFO *pArDev) 
{
	AR_ANI   *aniState = pArDev->arCurrentAni;
	A_UINT32  listenTime;
    A_UINT32  phyCnt1;
    A_UINT32  phyCnt2;

    ASSERT(aniState);
	
    if (!pArDev->arDoAni)
        return;

	listenTime = ar5212AniGetListenTime(pArDev);
	
	if (listenTime < 0) {
		/* restart ANI period if listenTime is invalid */
        ANI_DEBUG("ListenTime < 0 [%d]\n", listenTime);
		ar5212AniRestart(pArDev);
        return;
	} 

    /* XXX: beware of overflow */
    aniState->listenTime += listenTime;

#if defined(ANI_MIB_UPDATE) 
    ar5212MibControl(pArDev, UPDATE_SW_ALL);
#endif

    phyCnt1 = A_REG_RD(pArDev, MAC_PHYCNT1);
    phyCnt2 = A_REG_RD(pArDev, MAC_PHYCNT2);

    /* XXX: sometimes zero.  Why? */
    if ((phyCnt1 < aniState->ofdmPhyErrBase) ||
        (phyCnt2 < aniState->cckPhyErrBase)) {
        
        if (phyCnt1 < aniState->ofdmPhyErrBase) {
            A_REG_WR(pArDev, MAC_PHYCNT1,     aniState->ofdmPhyErrBase);
            A_REG_WR(pArDev, MAC_PHYCNT1MASK, MAC_PHY_ERR_OFDM_TIMING);
        }

        if (phyCnt2 < aniState->cckPhyErrBase) {
            A_REG_WR(pArDev,  MAC_PHYCNT2,     aniState->cckPhyErrBase);
            A_REG_WR(pArDev,  MAC_PHYCNT2MASK, MAC_PHY_ERR_CCK_TIMING);
        }

        ANI_DEBUG("CCK/OFDM Phy Counter < 0\n");
        return; /* XXX */
    }
    
    aniState->ofdmPhyErrCount = (phyCnt1 - aniState->ofdmPhyErrBase);
    aniState->cckPhyErrCount  = (phyCnt2 - aniState->cckPhyErrBase);

#if 0
    if (aniState->listenTime > (5 * pArDev->arAniPeriod)) 
        ANI_DEBUG("%s LT: %d P: %d OfdmErr: %d [%d - %d] "
                  "cckErr: %d [%d - %d]\n",
                  __FUNCTION__,
                  aniState->listenTime,
                  (5 * pArDev->arAniPeriod),
                  aniState->ofdmPhyErrCount,
                  phyCnt1, aniState->ofdmPhyErrBase,
                  aniState->cckPhyErrCount,
                  phyCnt2, aniState->cckPhyErrBase);
#endif
    
    if (aniState->listenTime > (5 * pArDev->arAniPeriod)) {
        /* 
         * Check to see if need to lower immunity if 5 aniPeriods
         * have passed
         */
        if ((aniState->ofdmPhyErrCount <= (aniState->listenTime *
                                           aniState->ofdmTrigLow/1000)) &&
            (aniState->cckPhyErrCount <= (aniState->listenTime *
                                          aniState->cckTrigLow/1000))) {
            ar5212AniLowerImmunity(pArDev);
        }
        ar5212AniRestart(pArDev);

    } else if (aniState->listenTime > pArDev->arAniPeriod) {
        /* check to see if need to raise immunity */
        if (aniState->ofdmPhyErrCount > (aniState->listenTime * 
                                         aniState->ofdmTrigHigh/1000)) {
            ar5212AniOfdmErrTrigger(pArDev);
            ar5212AniRestart(pArDev);            
        } else if (aniState->cckPhyErrCount > (aniState->listenTime *
                                               aniState->cckTrigHigh/1000)) {
            ar5212AniCckErrTrigger(pArDev);
            ar5212AniRestart(pArDev);            
        }
    }
}





static INLINE void
ar5212DumpRegSet(AR_DEV_INFO* pDev, int first, int last)
{
    int i;

    for (i = first; i <= last; i += 4) {
        uiPrintf("=== 0x%04X: 0x%08lX\n", i, readPlatformReg(pDev, i));
    }
}

/**************************************************************
 * ar5212DumpRegisters
 *
 * Print out a bunch of HW registers.  DO NOT CALL FROM ISR ON AP.
 */
void
ar5212DumpRegisters(AR_DEV_INFO *pDev)
{
    uiPrintf("MAC Registers\n");
    ar5212DumpRegSet(pDev, 0x0008, 0x00b4);
    uiPrintf("\nQCU Registers\n");
    ar5212DumpRegSet(pDev, 0x0800, 0x0a40);
    uiPrintf("\nDCU Registers\n");
    ar5212DumpRegSet(pDev, 0x1000, 0x10F0);
    ar5212DumpRegSet(pDev, 0x1230, 0x1230);
    uiPrintf("\nPCI Registers\n");
    ar5212DumpRegSet(pDev, 0x4000, 0x4030);
    uiPrintf("\nEeprom Registers\n");
    ar5212DumpRegSet(pDev, 0x6000, 0x6010);
    uiPrintf("\nPCU Registers\n");
    ar5212DumpRegSet(pDev, 0x8000, 0x8058);
    uiPrintf("\nBB Registers\n");
    ar5212DumpRegSet(pDev, 0x9800, 0x9878);
    ar5212DumpRegSet(pDev, 0x9900, 0x995C);
    ar5212DumpRegSet(pDev, 0x9C00, 0x9C1C);
}

/******************************************************************
 * ar5212GetCurRssi
 *
 * Get the rssi of frame curently being received.
 */
A_UINT32
ar5212GetCurRssi(AR_DEV_INFO *pDev)
{
    return (readPlatformReg(pDev, PHY_CURRENT_RSSI) & 0xff);
}

A_UINT32
ar5212GetDefAntenna(AR_DEV_INFO *pDev)
{
    return (readPlatformReg(pDev, MAC_DEF_ANTENNA) & 0x7);
}

void
ar5212SetDefAntenna(AR_DEV_INFO *pArDev, A_UINT32 antenna)
{
    writePlatformReg(pArDev, MAC_DEF_ANTENNA, (antenna & 0x7));
    /*
     * This section only needed by fast diversity code.
     * Assumed to be protected by the caller of ar5212SetDefAntenna()
     * if necessary.
     */
    pArDev->cachedDefAnt    = (A_UINT8)(antenna & 0x7);
    pArDev->countOtherRxAnt = 0;
}


void
ar5212SetAntennaSwitch(AR_DEV_INFO *pArDev,
                       ANTENNA_CONTROL settings,
                       CHAN_DESC *pChDesc)
{
    A_UINT32        antSwitchA,
                    antSwitchB;
    EEP_HEADER_INFO *pHeaderInfo    = pArDev->pHalInfo->pEepData->pEepHeader;
    AR_DEV_CONFIG   *pConfig        = &pArDev->config;
    int             arrayMode       = 0;

    switch (pChDesc->wlanMode) {
    case WLAN_MODE_11a:
    case WLAN_MODE_11a_TURBO:
    case WLAN_MODE_11a_TURBO_PRIME:
    case WLAN_MODE_11a_XR:
        arrayMode = 0;
        break;
    case WLAN_MODE_11b:
        arrayMode = 1;
        break;
    case WLAN_MODE_11g:
    case WLAN_MODE_11g_TURBO:
    case WLAN_MODE_11g_TURBO_PRIME:
    case WLAN_MODE_11g_XR:
        arrayMode = 2;
        break;
    default:
        ASSERT(0);
        break;
    }

    antSwitchA =  pHeaderInfo->antennaControl[1][arrayMode]        |
                 (pHeaderInfo->antennaControl[2][arrayMode] << 6)  |
                 (pHeaderInfo->antennaControl[3][arrayMode] << 12) |
                 (pHeaderInfo->antennaControl[4][arrayMode] << 18) |
                 (pHeaderInfo->antennaControl[5][arrayMode] << 24);
    antSwitchB =  pHeaderInfo->antennaControl[6][arrayMode]        |
                 (pHeaderInfo->antennaControl[7][arrayMode] << 6)  |
                 (pHeaderInfo->antennaControl[8][arrayMode] << 12) |
                 (pHeaderInfo->antennaControl[9][arrayMode] << 18) |
                 (pHeaderInfo->antennaControl[10][arrayMode] << 24);

    /*
     * For fixed antenna, give the same setting for both switch banks
     */
    if (ANTENNA_FIXED_A == settings) {
        antSwitchB = antSwitchA;
    } else if (ANTENNA_FIXED_B == settings) {
        antSwitchA = antSwitchB;
    } else {
        ASSERT(ANTENNA_CONTROLLABLE == settings);
    }
    pConfig->diversityControl = settings;

    A_REG_WR(pArDev, ANT_SWITCH_TABLE1, antSwitchA);
    A_REG_WR(pArDev, ANT_SWITCH_TABLE2, antSwitchB);

    if (antSwitchA != antSwitchB) {
        /* Enable Fast Receive Antenna Diversity. */
        pArDev->useFastDiversity = TRUE;
        A_REG_SET_BIT(pArDev, PHY_CCK_DETECT, BB_ENABLE_ANT_FAST_DIV);
        pArDev->cachedDefAnt = (A_UINT8)A_REG_RD(pArDev, MAC_DEF_ANTENNA);
        pArDev->countOtherRxAnt = 0;
    } else {
        pArDev->useFastDiversity = FALSE;
        A_REG_CLR_BIT(pArDev, PHY_CCK_DETECT, BB_ENABLE_ANT_FAST_DIV);
    }
}

void
ar5212UpdateAntenna(AR_DEV_INFO *pArDev, AR_CONN_INFO *pSib, int retries,
                    A_UINT8 rssiAck, A_UINT8 curTxAnt)
{
    struct TxRateCtrl_s *pRc;

    pRc = &pSib->txRateCtrl;

#if !defined(NO_CONN_STATS)
    /* curTxAnt is determined from tx descriptor on Venice */
    pSib->stats.AntCnt[curTxAnt]++;
#endif

    if (pSib->txAnt != curTxAnt) {
        /*
         * Hw does AABBAA on transmit attempts, and has flipped on this transmit.
         */
        pSib->txAnt = curTxAnt; /* 0 or 1 */
        AR_UPDATE_STATS(&pArDev->devStats, &pSib->stats, AntSwCnt, 1);
        pRc->antFlipCnt = 1;

#ifndef BUILD_AP
        /*
         * Update rx ant (default) to this transmit antenna if:
         *   1. The very first try on the other antenna succeeded and
         *      with a very good ack rssi.
         *   2. Or if we find ourselves succeeding for RX_FLIP_THRESHOLD
         *      consecutive transmits on the other antenna;
         * NOTE that the default antenna is preserved across a chip reset
         * by the hal software
         */
        if (!pArDev->useFastDiversity &&
            retries == 2
            && rssiAck >= pRc->rssiLast + 2)
        {
            ar5212SetDefAntenna(pArDev, curTxAnt ? 2 : 1);
        }
    } else {
        if (!pArDev->useFastDiversity &&
            pRc->antFlipCnt < RX_FLIP_THRESHOLD)
        {
            pRc->antFlipCnt++;
            if (pRc->antFlipCnt == RX_FLIP_THRESHOLD) {
                ar5212SetDefAntenna(pArDev, curTxAnt ? 2 : 1);
            }
        }
#endif
    }
}

/*
 * ar5212UseShortSlotTime - set short time
 *                          9 us if 'en' == TRUE else 20 us.
 */
void
ar5212UseShortSlotTime(AR_DEV_INFO *pArDev, A_BOOL en)
{
    if (en) {
        A_REG_WR(pArDev, MAC_D_GBL_IFS_SLOT, SLOT_TIME_09);
    } else {
        A_REG_WR(pArDev, MAC_D_GBL_IFS_SLOT, SLOT_TIME_20);
    }
}



void
ar5212DmaDebugDump(AR_DEV_INFO *pDev)
{
#ifdef BUILD_AP
    isrPrintf("Dma Debug Reg 0 0x%08x    Reg 1 0x%08x\n", 
              A_REG_RD(pDev, MAC_DMADBG_0),A_REG_RD(pDev, MAC_DMADBG_1));
    isrPrintf("Dma Debug Reg 2 0x%08x    Reg 3 0x%08x\n", 
              A_REG_RD(pDev, MAC_DMADBG_2),A_REG_RD(pDev, MAC_DMADBG_3));
    isrPrintf("Dma Debug Reg 4 0x%08x    Reg 5 0x%08x\n", 
              A_REG_RD(pDev, MAC_DMADBG_4),A_REG_RD(pDev, MAC_DMADBG_5));
    isrPrintf("Dma Debug Reg 6 0x%08x    Reg 7 0x%08x\n", 
              A_REG_RD(pDev, MAC_DMADBG_6),A_REG_RD(pDev, MAC_DMADBG_7));
#else
    uiPrintf("Dma Debug Reg 0 0x%08x    Reg 1 0x%08x\n", 
             A_REG_RD(pDev, MAC_DMADBG_0),A_REG_RD(pDev, MAC_DMADBG_1));
    uiPrintf("Dma Debug Reg 2 0x%08x    Reg 3 0x%08x\n", 
             A_REG_RD(pDev, MAC_DMADBG_2),A_REG_RD(pDev, MAC_DMADBG_3));
    uiPrintf("Dma Debug Reg 4 0x%08x    Reg 5 0x%08x\n", 
             A_REG_RD(pDev, MAC_DMADBG_4),A_REG_RD(pDev, MAC_DMADBG_5));
    uiPrintf("Dma Debug Reg 6 0x%08x    Reg 7 0x%08x\n", 
             A_REG_RD(pDev, MAC_DMADBG_6),A_REG_RD(pDev, MAC_DMADBG_7));
#endif
}


/*
 * Routine to control the Sleep Registers more in refined way. This routine has 
 * been adopted from the main line code. Some part of this routine has been 
 * commented out, but will be brought back in after the power mgmt rotuines has 
 * been enabled. This routine will eventually replace the SetupClk and RestoreClk
 * functions 
 */

void
ar5212SetSleepRegisters(AR_DEV_INFO *pArDev, A_BOOL enSlowClockSleep, A_BOOL enPllBandGap)
{
    if (enSlowClockSleep) {
        
        if (enPllBandGap) {
            A_REG_WR(pArDev, PHY_SLEEP_CTR_CONTROL, 0x1f); //reg 28 =  9870
        } else {
            A_REG_WR(pArDev, PHY_SLEEP_CTR_CONTROL, 0x0f); //reg 28 =  9870
        }


#if !defined(AR5312) && defined(MAC_1MHZ_SUSPEND)

        A_REG_WR(pArDev, PHY_M_SLEEP,           0x03); //reg 124 = 99f0
        /* # Set sleep clock rate to 1 Mhz. */
        A_REG_RMW_FIELD(pArDev, MAC_PCICFG, SLEEP_CLK_SEL, 1);
        A_REG_RMW_FIELD(pArDev, MAC_PCICFG, SLEEP_CLK_RATE_IND, 2);
#endif

        /* Set USEC32 to 1 */
        A_REG_RMW_FIELD(pArDev, MAC_USEC, 32, 1);
        /* Set TSF Increment for 32 KHz */
        A_REG_WR_FIELD(pArDev, MAC_TSF_PARM, INCREMENT, 61);


        A_REG_WR(pArDev, PHY_SLEEP_CTR_LIMIT,   0x34);
        A_REG_WR(pArDev, PHY_SLEEP_SCAL,        0x0d);
#if 0
        A_REG_WR(pArDev, PHY_REFCLKDLY,         0xA0);
#else
        /* Per bug 11345 */
        A_REG_WR(pArDev, PHY_REFCLKDLY,         0x3F);
#endif
    
    } else {
        /* Set sleep clock rate back to refclk. */
#if !defined(AR5312) && defined(MAC_1MHZ_SUSPEND)

        A_REG_RMW_FIELD(pArDev, MAC_PCICFG, SLEEP_CLK_RATE_IND, 0);
        A_REG_RMW_FIELD(pArDev, MAC_PCICFG, SLEEP_CLK_SEL, 0);
        A_REG_WR(pArDev, PHY_M_SLEEP,       0x0c);
#endif

        /* Set TSF Increment for refclk */
        A_REG_WR_FIELD(pArDev, MAC_TSF_PARM, INCREMENT, 1);
#if !defined(BUILD_AP)
        A_REG_RMW_FIELD(pArDev, MAC_USEC, 32, 39);
#else
        A_REG_RMW_FIELD(pArDev, MAC_USEC, 32, 31);
#endif   /* BUILD_AP*/
        A_REG_WR(pArDev, PHY_SLEEP_CTR_LIMIT,   0x7f);

#if defined(AR5312)
        /* Set ADC/DAC select values */
        A_REG_WR(pArDev, PHY_SLEEP_SCAL, 0x00000004);
#else
        A_REG_WR(pArDev, PHY_SLEEP_SCAL,    0x0e);
        A_REG_WR(pArDev, PHY_REFCLKDLY,     0xff);
#endif /* AR5312 */
        /*
         * Restore BB registers to power-on defaults
         */
        if (enPllBandGap) {
            A_REG_WR(pArDev, PHY_SLEEP_CTR_CONTROL, 0x1f); //reg 28 =  9870
        } else {
            A_REG_WR(pArDev, PHY_SLEEP_CTR_CONTROL, 0x0f); //reg 28 =  9870
        }
    }
}

/**************************************************************
 * ar5212SetupClock
 *
 * If 32KHz clock exists, use it to lower power consumption during sleep
 *
 * Note: If clock is set to 32 KHz, delays on accessing certain
 *       baseband registers (27-31, 124-127) are required.
 *
 */
void
ar5212SetupClock(AR_DEV_INFO *pArDev)
{
#if !defined(BUILD_AP)
    struct eepMap   *pEepData = pArDev->pHalInfo->pEepData;

    if ( (pArDev->config.enable32KHzClock == USE_32KHZ) ||
         ((pArDev->config.enable32KHzClock == AUTO_32KHZ) &&
         pEepData->pEepHeader->exist32kHzCrystal) )
    {
        /*
         * If this card has an external 32 KHz crystal,
         * enable clocks to be turned OFF in BB during sleep
         * and also enable turning OFF 32MHz/40MHz Refclk
         * from A2.
         */
        A_REG_WR(pArDev, PHY_SLEEP_CTR_CONTROL, 0x0f); //reg 28 =  9870
        A_REG_WR(pArDev, PHY_SLEEP_CTR_LIMIT,   0x0d); //reg 29 =  9874
        A_REG_WR(pArDev, PHY_SLEEP_SCAL,        0x0c); //reg 30 =  9878
        A_REG_WR(pArDev, PHY_M_SLEEP,           0x01); //reg 124 = 99f0
        A_REG_WR(pArDev, PHY_REFCLKDLY,         0x05); //reg 125 = 99f4
        if (IS_5112(pArDev)) {
            A_REG_WR(pArDev, PHY_REFCLKPD,      0x14); //reg 126 = 99f8
        } else {
            A_REG_WR(pArDev, PHY_REFCLKPD,      0x18); //reg 126 = 99f8
        }

        /* Set USEC32 to 1 */
        A_REG_RMW_FIELD(pArDev, MAC_USEC, 32, 1);
        /* Set TSF Increment for 32 KHz */
        A_REG_WR(pArDev, MAC_TSF_PARM, 61);

        /* # Set sleep clock rate to 32 KHz. */
        A_REG_RMW_FIELD(pArDev, MAC_PCICFG, SLEEP_CLK_SEL, 1);
        A_REG_RMW_FIELD(pArDev, MAC_PCICFG, SLEEP_CLK_RATE_IND, 0x0);
    } else {
#endif
        A_REG_WR(pArDev, PHY_SLEEP_CTR_CONTROL, 0x0f);
        A_REG_WR(pArDev, PHY_SLEEP_CTR_LIMIT,   0x7f);

#if defined(AR5312)
        /* Set ADC/DAC select values */
        A_REG_WR(pArDev, PHY_SLEEP_SCAL, 0x00000004);
#else
        A_REG_WR(pArDev, PHY_SLEEP_SCAL,        0x0e);
        A_REG_WR(pArDev, PHY_M_SLEEP,           0x0c);
        A_REG_WR(pArDev, PHY_REFCLKDLY,         0xff);
        if (IS_5112(pArDev)) {
            A_REG_WR(pArDev, PHY_REFCLKPD,      0x14); //reg 126 = 99f8
        } else {
            A_REG_WR(pArDev, PHY_REFCLKPD,      0x18); //reg 126 = 99f8
        }
#endif

#ifndef BUILD_AP
    }
#endif
}


/**************************************************************
 * ar5212RestoreClock
 *
 * If 32KHz clock exists, turn it off and turn back on the 32Mhz
 */
void
ar5212RestoreClock(AR_DEV_INFO *pArDev)
{
#ifndef BUILD_AP
    struct eepMap   *pEepData = pArDev->pHalInfo->pEepData;

    if ( (pArDev->config.enable32KHzClock == USE_32KHZ) ||
         ((pArDev->config.enable32KHzClock == AUTO_32KHZ) &&
         pEepData->pEepHeader->exist32kHzCrystal) )
    {
        /* # Set sleep clock rate back to 32 MHz. */

        A_REG_RMW_FIELD(pArDev, MAC_PCICFG, SLEEP_CLK_RATE_IND, 0);
        A_REG_RMW_FIELD(pArDev, MAC_PCICFG, SLEEP_CLK_SEL, 0);

        /* # Set TSF Increment for 32 MHz */
        A_REG_WR(pArDev, MAC_TSF_PARM, 1);
         /* # Set USEC32 to 1 */
        A_REG_RMW_FIELD(pArDev, MAC_USEC, 32, 31);

        /*
         * Restore BB registers to power-on defaults
         */
        A_REG_WR(pArDev, PHY_SLEEP_CTR_CONTROL, 0x0f);
        A_REG_WR(pArDev, PHY_SLEEP_CTR_LIMIT,   0x7f);
#if defined(AR5312)
        /* Set ADC/DAC select values */
        A_REG_WR(pArDev, PHY_SLEEP_SCAL, 0x00000004);
#else
        A_REG_WR(pArDev, PHY_SLEEP_SCAL,        0x0e);
        A_REG_WR(pArDev, PHY_M_SLEEP,           0x0c);
        A_REG_WR(pArDev, PHY_REFCLKDLY,         0xff);
        if (IS_5112(pArDev)) {
            A_REG_WR(pArDev, PHY_REFCLKPD,      0x14); //reg 126 = 99f8
        } else {
            A_REG_WR(pArDev, PHY_REFCLKPD,      0x18); //reg 126 = 99f8
        }
#endif
    }
#endif
}

/**************************************************************
 * ar5212GetApSwitchHelper
 *
 * Read some Beacon relevant registers in anticipation of saving
 * them for future restoration.
 */
void
ar5212GetApSwitchHelper(AR_DEV_INFO *pDev, SAVE_SIX_REG *pRegs)
{
    if (pRegs) {
        pRegs->r1 = readPlatformReg(pDev, MAC_TIMER0);
        pRegs->r2 = readPlatformReg(pDev, MAC_TIMER1);
        pRegs->r3 = readPlatformReg(pDev, MAC_TIMER2);
        pRegs->r4 = readPlatformReg(pDev, MAC_QUIET1);
        pRegs->r5 = readPlatformReg(pDev, MAC_QUIET2);
    }

    /* TODO: temp for debug */
    ar5212MacStop(HACK_TO_PARDEV(pDev));
}

/**************************************************************
 * ar5212SetApSwitchHelper
 *
 * Restore some Beacon relevant registers.
 */
void
ar5212SetApSwitchHelper(AR_DEV_INFO *pDev, SAVE_SIX_REG *pRegs)
{
#if !defined(ECOS_NOTDONE)
    ASSERT(pRegs);

    writePlatformReg(pDev, MAC_TIMER0, pRegs->r1);
    writePlatformReg(pDev, MAC_TIMER1, pRegs->r2);
    writePlatformReg(pDev, MAC_TIMER2, pRegs->r3);
    writePlatformReg(pDev, MAC_QUIET1, pRegs->r4);
    writePlatformReg(pDev, MAC_QUIET2, pRegs->r5);

    /* Enable SWBA interrupt. */
    ar5212EnableInterrupts(HACK_TO_PARDEV(pDev), HAL_INT_SWBA);
    ar5212EnableInterrupts(HACK_TO_PARDEV(pDev), HAL_INT_TXDESC);

    /* Set the Beacon Control register */
    writePlatformReg(pDev, MAC_BEACON, pDev->bssDescr->beaconInterval | MAC_BEACON_EN);
#else
    ECOS_NOTDONE_XXX;
#endif
}

/**************************************************************
 * ar5212ApplyMKKIrreversibleUpdate
 *
 */
A_STATUS
ar5212ApplyMKKIrreversibleUpdate(AR_DEV_INFO *pDev, A_UINT32 enOddU1, A_UINT32 undoUpdate)
{
    A_UINT16  data;
    A_UINT32  offset;
    A_STATUS  status = A_OK;
    
	status = ar5212EepromRead(pDev, 0xc2, &data);
    if (status != A_OK) {
        return status;
    }

#ifdef DEBUG
    if (undoUpdate) {
        if (!(data & 0x1)) {  
            A_UINT16  mkkNew11a, mask;
            /* 
             * If we are here we have a MKK SKU with the Amode bit not set.  We will restore
             * it if the Japan New 11a bit is set 
             */
            offset = (pDev->pHalInfo->pEepData->version >= EEPROM_VER4_0) ? 0xca : 0xcf;
            status = ar5212EepromRead(pDev, offset, &mkkNew11a);
            if (status != A_OK) {
                return status;
            }
            mask = (pDev->pHalInfo->pEepData->version >= EEPROM_VER4_0) ? EEPCAP_REG_EN_KK_NEW_11A_M :
                    EEPCAP_REG_EN_KK_NEW_11A_LEGACY_M;
            if (mkkNew11a & mask) {
                mkkNew11a &= ~mask; /* Clear the new 11a field and write back */
            }
            status = ar5212EepromWrite(pDev, offset, mkkNew11a);
            if (status != A_OK) {
                return status;
            }      
            status = ar5212EepromWrite(pDev, 0xc2, (data | 0x1));
            if (status != A_OK) {
                return status;
            }      
            ar5212UpdateEepromChecksum(pDev);
        }
        return A_OK;
    }
#endif

    if (!(data & 0x1)) {  
        return A_OK; /* The device has Amode clear already bail out A_OK */
    }

    /* If we get here we are going to apply the irreversible update that is
       mandated by the MKK government.  First clear the 11a mode field */
    data &= ~0x1;
    status = ar5212EepromWrite(pDev, 0xc2, data);
    if (status != A_OK) {
        return status;
    }

    offset = (pDev->pHalInfo->pEepData->version >= EEPROM_VER4_0) ? 0xca : 0xcf;
    status = ar5212EepromRead(pDev, offset, &data);
    if (status != A_OK) {
        return status;
    }

    if (pDev->pHalInfo->pEepData->version >= EEPROM_VER5_3) {
        /* For EEPROM v5.3 or greater we will flag UNI-1 Even, legacy EEPROMs it is assumed */
        data |= EEPCAP_REG_EN_KK_U1_EVEN_M;
    }

    data |= (pDev->pHalInfo->pEepData->version >= EEPROM_VER4_0) ?
        EEPCAP_REG_EN_KK_NEW_11A_M : EEPCAP_REG_EN_KK_NEW_11A_LEGACY_M;

    data &= (pDev->pHalInfo->pEepData->version >= EEPROM_VER4_0) ?  /* Disable odd UNI-1 here */
        ~EEPCAP_REG_EN_KK_U1_ODD_M : ~EEPCAP_REG_EN_KK_U1_ODD_LEGACY_M;

    if (enOddU1) {
        data |= (pDev->pHalInfo->pEepData->version >= EEPROM_VER4_0) ?
            EEPCAP_REG_EN_KK_U1_ODD_M : EEPCAP_REG_EN_KK_U1_ODD_LEGACY_M;
    }

    status = ar5212EepromWrite(pDev, offset, data);
    if (status != A_OK) {
        return status;
    }

    ar5212UpdateEepromChecksum(pDev);

    return A_OK;
}

/**************************************************************
 * ar5212MkkLockEeprom
 *
 */
A_STATUS
ar5212MkkLockEeprom(AR_DEV_INFO *pDev, A_BOOL readOnly)
{
    A_STATUS  status;
    A_UINT16  data;

    status = ar5212EepromRead(pDev, EEPROM_PROTECT_OFFSET, &data);
    if (status != A_OK) {
        return status;
    }

    /* Clear bits to be read/write capable */
    data &= ~EEPROM_PROTECT_MASK_BITS98;
    if (readOnly) {
        data |= EEPROM_PROTECT_READONLY_BITS98;
    }

    status = ar5212EepromWrite(pDev, EEPROM_PROTECT_OFFSET, data);
    /* Do not need to update checksum since not in checksum region */

    return status;
}

/**************************************************************
 * ar5212RawEepromRead
 *
 */
A_STATUS
ar5212RawEepromRead(AR_DEV_INFO *pDev, A_UINT16 offset, A_UINT16 *data)
{
    return ar5212EepromRead(pDev, offset, data);
}

#endif /* #ifdef BUILD_AR5212 */
