/*
 *  Copyright (c) 2000-2002 Atheros Communications, Inc., All Rights Reserved
 *
 *  Chips-specific power management routines.
 */

#ifdef BUILD_AR5212

#include "arDev.h"

/* Standard HAL Headers */
#include "wlantype.h"
#if !defined(SPLIT_ENFORCE)
#include "wlandrv.h"
#include "wlanPhy.h"
#endif
#include "halApi.h"
#include "hal.h"
#if !defined(SPLIT_ENFORCE)
#include "ui.h"
#endif
#include "halUtil.h"

/* Headers for HW private items */
#include "ar5212Reg.h"
#include "ar5212Power.h"
#include "ar5212Misc.h"
#include "ar5212.h"

/**************************************************************
 * ar5212SetPowerMode
 *
 * Set power mgt to the requested mode, and conditionally set
 * the chip as well
 */
A_STATUS
ar5212SetPowerMode(AR_DEV_INFO *pArDev, A_UINT32 powerRequest, A_BOOL setChip)
{
    A_STATUS status = A_HARDWARE;

    switch (powerRequest) {

    case NETWORK_SLEEP:
        /*
         * Notify Power Mgt is enabled in self-generated frames.
         * If requested, set Power Mode of chip to auto/normal.
         * Duration in units of 128us (1/8 TU)
         */
        A_REG_SET_BIT(pArDev, MAC_STA_ID1, PWR_SAV);

#if defined(PCI_INTERFACE) || defined(AR5523)
        if (setChip) {
            A_REG_CLR_BIT(pArDev, MAC_PCICFG, SL_INPEN);
            A_REG_RMW_FIELD(pArDev, MAC_SCR, SLMODE, MAC_SLMODE_NORMAL);
        }
        status = A_OK;
#endif
        break;

    case AWAKE:
        /*
         * Notify Power Mgt is disabled in self-generated frames.
         * If requested, force chip awake.
         *
         * Returns A_OK if chip is awake or successfully forced awake.
         *
         * WARNING WARNING WARNING
         * There is a problem with the chip where sometimes it will not wake up.
         */
/* usecs to wait for chip to come out of sleep state */
#define POWER_UP_TIME               4000

#if defined(PCI_INTERFACE) || defined(AR5523)

        if (setChip) {
            int i;

            A_REG_RMW_FIELD(pArDev, MAC_SCR, SLMODE, MAC_SLMODE_FWAKE);

            for (i = 0; i < POWER_UP_TIME / 200; i++) {
                if ((A_REG_RD(pArDev, MAC_PCICFG) & MAC_PCICFG_SPWR_DN) == 0) {
                    status = A_OK;
                    break;
                }
                udelay(200);
                A_REG_RMW_FIELD(pArDev, MAC_SCR, SLMODE, MAC_SLMODE_FWAKE);
            }
            ASSERT(status == A_OK);
            if (status != A_OK) {
                uiPrintf("ar5212SetPowerModeAwake: Failed to leave sleep\n");
            }
        } else {
            status = A_OK;
        }

        if (status == A_OK) {
            A_REG_CLR_BIT(pArDev, MAC_STA_ID1, PWR_SAV);
        }
#else
        status = A_OK;
#endif
        break;

    case FULL_SLEEP:
        /*
         * Notify Power Mgt is enabled in self-generated frames.
         * If requested, force chip to sleep.
         */
        A_REG_SET_BIT(pArDev, MAC_STA_ID1, PWR_SAV);

#if defined(PCI_INTERFACE) || defined(AR5523)
        if (setChip) {
            A_REG_SET_BIT(pArDev, MAC_PCICFG, SL_INPEN);
            A_REG_RMW_FIELD(pArDev, MAC_SCR, SLMODE, MAC_SLMODE_FSLEEP);
        }
#endif
        status = A_OK;
        break;

    case DEVICE_SLEEP:
        if (setChip) {
            int i;
                        
            A_REG_CLR_BIT(pArDev, MAC_SFR, WAKE);
            A_REG_SET_BIT(pArDev, MAC_PCICFG, SL_INPEN);
            
            ar5212SetSleepRegisters(pArDev, TRUE, TRUE);
            A_REG_RMW_FIELD(pArDev, MAC_SCR, SLMODE, MAC_SLMODE_FSLEEP);

            for (i = 0; i < POWER_UP_TIME / 10; i++) {
                if (A_REG_RD(pArDev, MAC_PCICFG) & MAC_PCICFG_SPWR_DN) {
                    status = A_OK;
                    break;
                }
                udelay(10);
                A_REG_RMW_FIELD(pArDev, MAC_SCR, SLMODE, MAC_SLMODE_FSLEEP);
                A_REG_SET_BIT(pArDev, MAC_SFR, SLEEP);
            }
            ASSERT(status == A_OK);
        }
        break;
    
    case DEVICE_WAKE:
        if (setChip) {
            int i;
            
            A_REG_CLR_BIT(pArDev, MAC_SFR, SLEEP);
            A_REG_RMW_FIELD(pArDev, MAC_SCR, SLMODE, MAC_SLMODE_FWAKE);

            for (i = 0; i < POWER_UP_TIME / 10; i++) {
                if (!(A_REG_RD(pArDev, MAC_PCICFG) & MAC_PCICFG_SPWR_DN)) {
                    status = A_OK;
                    break;
                }
                udelay(10);
                A_REG_RMW_FIELD(pArDev, MAC_SCR, SLMODE, MAC_SLMODE_FWAKE);
                A_REG_SET_BIT(pArDev, MAC_SFR, WAKE);
            }
            A_REG_RMW_FIELD(pArDev, MAC_SCR, SLMODE, MAC_SLMODE_FWAKE);
            ASSERT(status == A_OK);
            ar5212SetSleepRegisters(pArDev, FALSE, FALSE);



        }
        break;

    default:
        status = A_ENOTSUP;
        break;
    }

    return status;
}



/**************************************************************
 * ar5212GetPowerMode
 *
 * Return the current sleep mode of the chip
 */
A_UINT32
ar5212GetPowerMode(AR_DEV_INFO *pArDev)
{
#if defined(PCI_INTERFACE)  || defined(AR5523)
    // Just so happens the h/w maps directly to the abstracted value
    return A_REG_RD_FIELD(pArDev, MAC_SCR, SLMODE);
#else
    return AWAKE;
#endif
}

/**************************************************************
 * ar5212GetPowerStatus
 *
 * Return the current sleep state of the chip
 * TRUE = sleeping
 */
A_BOOL
ar5212GetPowerStatus(AR_DEV_INFO *pArDev)
{
#if defined(PCI_INTERFACE)  || defined(AR5523)
    return A_REG_IS_BIT_SET(pArDev, MAC_PCICFG, SPWR_DN);
#else
    return FALSE;
#endif
}

/**************************************************************
 * ar5212SetupPSPollDesc
 *
 * Initialize for PS-Polls
 */
void
ar5212SetupPsPollDesc(AR_DEV_INFO *pArDev, AR_DESC *pDesc)
{

/* xxxxPGSxxxx - TODO - power management partition */
#if 0
    AR5212_TX_CONTROL *pTxControl = TX_CONTROL(pDesc);

    // Send PS-Polls at 6mbps.
    pTxControl->TXRate0 =
        pArDev->bssTable[0].pRateTable->info[PSPOLL_RATE_INDEX].rateCode;
    pTxControl->TXDataTries0 = 1 + pArDev->config.hwTxRetries;
    pTxControl->antModeXmit  = 0;

    // PS-Poll descriptor points to itself with the VEOL bit set.
    pTxControl->noAck         = 0;
    pTxControl->clearDestMask = 1;
    pTxControl->PktType       = HAL_DESC_PKT_TYPE_PSPOLL;
    pTxControl->VEOL          = 1;
    pDesc->nextPhysPtr        = pDesc->thisPhysPtr;
#else
#if defined(ECOS_NOTDONE)
    ECOS_NOTDONE_XXX;
#endif
#endif
}

#endif // #ifdef BUILD_AR5212
