/*
 *  Copyright (c) 2000-2002 Atheros Communications, Inc., All Rights Reserved
 *
 *  Transmit HAL functions
 */

#ifdef BUILD_AR5212

#include "arDev.h"

#if !defined(SPLIT_ENFORCE)
/* Standard HAL Headers */
#include "wlantype.h"
#include "wlandrv.h"
#include "wlanext.h"
#include "ui.h"
#endif

#include "halApi.h"
#include "hal.h"


/* Headers for HW private items */
#include "ar5212Reg.h"
#include "ar5212Transmit.h"
#include "ar5212Misc.h"
#include "ar5212.h"

/**************************************************************
 * ar5212GetRxDP
 *
 * Get the RXDP.
 */
A_UINT32
ar5212GetRxDP(AR_DEV_INFO *pArDev)
{
    return readPlatformReg(pArDev, MAC_RXDP);
}

/**************************************************************
 * ar5212SetRxDP
 *
 * Set the RxDP.
 */
void
ar5212SetRxDP(AR_DEV_INFO *pArDev, A_UINT32 rxdp)
{
    ASSERT(!A_REG_IS_BIT_SET(pArDev, MAC_CR, RXE));
    writePlatformReg(pArDev, MAC_RXDP, rxdp);
    ASSERT(readPlatformReg(pArDev, MAC_RXDP) == rxdp);
}

/**************************************************************
 * ar5212EnableReceive
 *
 * Set Receive Enable bits.
 */
void
ar5212EnableReceive(AR_DEV_INFO *pArDev)
{
    writePlatformReg(pArDev, MAC_CR, MAC_CR_RXE);
    pArDev->bRxEnabled = TRUE;
}

/**************************************************************
 * ar5212StopDmaReceive
 *
 * Stop Receive at the DMA engine
 */
A_STATUS
ar5212StopDmaReceive(AR_DEV_INFO *pArDev)
{
    A_UINT32    regVal, i;

    /* Set receive disable bit */
    writePlatformReg(pArDev, MAC_CR, MAC_CR_RXD);

    /* Wait up to 10 ms. Was 100 ms for FPGA. */
    for (i=0; i<1000; i++) {

        regVal = readPlatformReg(pArDev, MAC_CR);

        if (!(regVal & MAC_CR_RXE)) {
            pArDev->bRxEnabled = FALSE;
            return A_OK;
        }

        udelay(10);
    }

    ASSERT(0);
    return A_HARDWARE;
}

/**************************************************************
 * ar5212StartPcuReceive
 *
 * Start Transmit at the PCU engine (unpause receive)
 */
void
ar5212StartPcuReceive(AR_DEV_INFO *pArDev)
{
    if (pArDev->arDoAni)
        ar5212AniReset(pArDev);

    writePlatformReg(pArDev, MAC_DIAG_SW,
        readPlatformReg(pArDev, MAC_DIAG_SW) & ~(MAC_DIAG_RX_DIS));
}

/**************************************************************
 * ar5212StopPcuReceive
 *
 * Stop Transmit at the PCU engine (pause receive)
 */
void
ar5212StopPcuReceive(AR_DEV_INFO *pArDev)
{
    writePlatformReg(pArDev, MAC_DIAG_SW, readPlatformReg(pArDev, MAC_DIAG_SW) | MAC_DIAG_RX_DIS);
}

/**************************************************************
 * ar5212SetMulticastFilter
 *
 * Set multicast filter 0 (lower 32-bits)
 *               filter 1 (upper 32-bits)
 */
void
ar5212SetMulticastFilter(AR_DEV_INFO *pArDev, A_UINT32 filter0, A_UINT32 filter1)
{
    writePlatformReg(pArDev, MAC_MCAST_FIL0, filter0);
    writePlatformReg(pArDev, MAC_MCAST_FIL1, filter1);
}

/**************************************************************
 * ar5212MulticastFilterIndex
 *
 * Clear/Set multicast filter by index
 */
void
ar5212MulticastFilterIndex(AR_DEV_INFO *pArDev, A_UINT32 index, A_BOOL bSet)
{
    A_UINT32 filterReg;
    A_UINT32 filterIndex;
    A_UINT32 value;

    ASSERT(index < 64);

    if (index < 32) {
        filterReg   = MAC_MCAST_FIL0;
        filterIndex = index;
    } else {
        filterReg   = MAC_MCAST_FIL1;
        filterIndex = index - 32;
    }

    value = readPlatformReg(pArDev, filterReg);

    if (bSet) {
        value |= (1 << filterIndex);
    } else {
        value &= ~(1 << filterIndex);
    }

    writePlatformReg(pArDev, filterReg, value);
}

/**************************************************************************
 * ar5212SetRxFilter
 *
 * Turn off/on bits in the receive packet filter.  Save it for resets, etc.
 *
 * Bits can be MAC_RX_UCAST, MAC_RX_MCAST, MAC_RX_BCAST, MAC_RX_BEACON,
 * MAC_RX_PROM, MAC_RX_CONTROL
 *
 * oper can be HAL_RX_FILTER_INIT, HAL_RX_FILTER_SET, HAL_RX_FILTER_CLEAR,
 * HAL_RX_FILTER_TEMP, HAL_RX_FILTER_RESTORE.
 *
 *
 */
void
ar5212SetRxFilter(AR_DEV_INFO *pArDev, A_UINT32 bits)
{

/*
 * Helpful macros used in the context of this routine only!
 */
#define MAC_BITS(_bits) (                                                           \
    ((_bits) & RX_COMMON) |                                                         \
    (((_bits) & RX_XR_POLL) ? MAC_RX_XR_POLL : 0) |                                     \
    (((_bits) & RX_PROBE_REQ) ? MAC_RX_PROBE_REQ : 0)                                   \
)
#define PHY_BITS(_bits) (                                                               \
    (((_bits) & RX_PHY_RADAR) ? MAC_PHY_ERR_RADAR : 0) |                                \
    (((_bits) & RX_PHY_ERR) ? (MAC_PHY_ERR_OFDM_TIMING | MAC_PHY_ERR_CCK_TIMING) : 0)   \
)
#define SET_BITS(_pArDev, _bits) {                                                            \
    writePlatformReg(_pArDev, MAC_RX_FILTER, MAC_BITS(_bits));                                \
    writePlatformReg(_pArDev, MAC_PHY_ERR, PHY_BITS(_bits));                                  \
    if (PHY_BITS(_bits)) {                                                                  \
        A_REG_SET_BIT(_pArDev, MAC_RXCFG, ZLFDMA);                                            \
    } else {                                                                                \
        A_REG_CLR_BIT(_pArDev, MAC_RXCFG, ZLFDMA);                                            \
    }                                                                                       \
}
    /* Set frame type bits */
    pArDev->rxFilterReg = bits;
    SET_BITS(pArDev, pArDev->rxFilterReg);
}

/**************************************************************
 * ar5212ProcessRxDesc
 *
 * Process an RX descriptor, and return the status to the caller.
 * Copy some hardware specific items into the software portion
 * of the descriptor.
 */
A_STATUS
ar5212ProcessRxDesc(AR_DEV_INFO *pArDev, AR_DESC *pDesc)
{
    AR5212_RX_STATUS *pRxStatus;
    A_UINT8          rssi;

    A_DESC_CACHE_INVAL(pDesc);

    pRxStatus = RX_STATUS(pDesc);

    if (!pRxStatus->done) {
        return A_EBUSY;
    }

#if defined(RX_SELF_LINKED_QUEUE)
    {
        /*
         * Given the use of a self-linked tail be very sure that the hw is
         * done with this descriptor; the hw may have done this descriptor
         * once and picked it up again...make sure the hw has moved on.
         */
        AR5212_RX_STATUS *pNextStatus;
    
        A_DESC_CACHE_INVAL(pDesc->pNextVirtPtr);
        pNextStatus = RX_STATUS(pDesc->pNextVirtPtr);
        if ((!pNextStatus->done) &&
            (readPlatformReg(pArDev, MAC_RXDP) == pDesc->thisPhysPtr))
        {
            return A_EBUSY;
        }
    }
#endif /* RX_SELF_LINKED_QUEUE */

    pDesc->status.rx.decryptError = FALSE;      /* assume frame is ok */

    /*
     * Drop multiple-descriptor frames.  This is not WECA-compliant if our
     * buffer size (sans wlan header size) is larger than 1514 bytes.
     */
    if (pRxStatus->more) {
        /* capture if the next one has to be discarded */
        return A_ERROR;
    }


    pDesc->status.rx.timestamp = (A_UINT16)pRxStatus->rxTimestamp;

    /*
     * RSSI is an 8-bit signed value.  We only care about it if it's positive.
     * So, if it's negative, cap it at zero.
     */
    rssi = ((A_INT8)pRxStatus->rxSigStrength > 0) ?
           (A_UINT8)pRxStatus->rxSigStrength : 0;

    /*
     * Fill in software versions of information that rest of RX processing
     * requires. Some are valid even for errored frames.
     */
    pDesc->status.rx.dataLength = (A_UINT16)pRxStatus->dataLength;
    pDesc->status.rx.hwIndex    = (A_UINT8)pRxStatus->keyIndex;
    pDesc->status.rx.rssi       = rssi;
    pDesc->status.rx.antenna    = (A_UINT8)pRxStatus->rxAntenna;

    /* Report key cache misses to the host and let it decide how to proceed */
    pDesc->status.rx.keyCacheMiss = (A_UINT8)pRxStatus->keyCacheMiss;

#ifdef DEBUG_PKT_LOG
{
    WLAN_FRAME_HEADER *pHeader = pDesc->pBufferVirtPtr.header;
    /*
     * Log receive packet info
     */
    logEventRxPkt(pRxStatus->dataLength, pRxStatus->rxRate,
            rssi,
            *(A_UINT16*)&pHeader->frameControl,
            *(A_UINT16*)&pHeader->seqControl,
            PHY_ERROR_CODE(pRxStatus)  << 4 |
            pRxStatus->decryptCRCError << 3 |
            pRxStatus->CRCError        << 1 |
            pRxStatus->pktReceivedOK   << 0,
            pRxStatus->rxAntenna,
            (A_UINT16)pRxStatus->rxTimestamp);
}
#endif

    if (!pRxStatus->pktReceivedOK) {

        /*
         * which of all these counters do we ever use? these
         * and other stats counters should be macroized
         */
        if (pRxStatus->CRCError) {
            return A_CRC_ERROR;
        }
        if (pRxStatus->phyErrorOccured) {
            pDesc->status.rx.phyError = (A_UINT8)PHY_ERROR_CODE(pRxStatus);
            return A_PHY_ERROR;
        }
    }

#if 1
    /* xxxxPGSxxxx - no XR support yet */
    pDesc->pOpBss = &pArDev->bssTable[0];
#else
    /* demux the recd frame to the right vdev */
    pDesc->pOpBss = isXrBssRxFrame(pArDev, pDesc->pBufferVirtPtr.header)
                  ? &pArDev->xrBss
                  : &pArDev->baseBss;
#endif

    if (!pRxStatus->pktReceivedOK) {
        if (pRxStatus->decryptCRCError) {
            ASSERT(!pRxStatus->michaelError);

            pDesc->status.rx.decryptError |= DECRYPTERROR_CRC; /* CRC Error */
        }
        if (pRxStatus->michaelError) {
            pDesc->status.rx.decryptError |= DECRYPTERROR_MIC; /* MIC Error */
        }
        if (pRxStatus->decompCRCError) {
            /*
             * We shouldn't get any compression errors unless a
             * valid key cache entry was found.
             */
            ASSERT(pRxStatus->keyCacheMiss == 0);
            return A_DECOMP_ERROR;
        }
    }

    pDesc->status.rx.rate = (pDesc->pOpBss->pRateTable) ?
        (A_UINT8)pDesc->pOpBss->pRateTable->rateCodeToIndex[pRxStatus->rxRate] :
        0;

    /*
     * If using fast diversity; change default antenna if fast diversity
     * chooses other antenna 3 times in a row.
     */
    if (pArDev->useFastDiversity) {
#ifdef BUILD_AP
        /* AP needs protection from beacon interrupts. */
        int lock = intLock();
#endif
        if(pArDev->cachedDefAnt != pDesc->status.rx.antenna) {
            if (++pArDev->countOtherRxAnt >= RX_FLIP_THRESHOLD) {
                ar5212SetDefAntenna(pArDev, pDesc->status.rx.antenna);
            }
        } else {
            pArDev->countOtherRxAnt = 0;
        }
#ifdef BUILD_AP
        /* AP needs protection from beacon interrupts. */
        intUnlock(lock);
#endif
    }   /* end if fast diversity */
    return A_OK;
}

/**************************************************************
 * ar5212SetupRxDesc
 *
 * Initialize RX descriptor, by clearing the status and clearing
 * the size.  This is not strictly HW dependent, but we want the
 * control and status words to be opaque above the hal.
 */
void
ar5212SetupRxDesc(AR_DESC *pDesc, A_UINT32 size)
{
    AR5212_RX_CONTROL *pRxControl = RX_CONTROL(pDesc);

    pDesc->hw.word[0] = 0;
    pDesc->hw.word[1] = 0;
    pDesc->hw.word[2] = 0;
    pDesc->hw.word[3] = 0;
    pRxControl->bufferLength = size;
}

A_BOOL
ar5212GetPcuState(AR_DEV_INFO *pArDev)
{
    A_UINT32    regVal;

    regVal = readPlatformReg(pArDev, MAC_CR);

    if (regVal & MAC_CR_RXE) {
        return TRUE;
    } else {
        return FALSE;
    }
}


#if (defined(UPSD) && defined(BUILD_AP))
/*
 * Upsd Receive
 *  traverse completed receive descriptors.
 *  If we've received a uplink DATA frame and the pSib has anything
 *  in the downlink upsd queue, then move everything in the pSib holding queue
 *  to the hw upsd queue for transmission.  Receive descriptors and their payloads
 *  are untouched and travel up the normal Rx processing chain.
 *  This hw ISR routine is the only code that appends to the hw upsdq, thus
 *  the intLock() in queueSwTxDescriptor() in wlanSend.c is the only synchronization
 *  needed.  Also, the processTxDesc codes can safely reclaim Tx descriptors
 *  for the same reason.
 */
void
ar5212UpsdResponse(AR_DEV_INFO *pArDev)
{
    ATHEROS_DESC *prx, *prxnext;
    QUEUE_DETAILS *sibq, *upsdq;
    AR5212_RX_STATUS *pRxStatus, *pNextStatus;
    SIB_ENTRY *pSib;
    unsigned char ftype;
    unsigned char fsubtype;

    /* The hw upsd downlink queue */
    upsdq = &pArDev->baseBss.wmequeues[TXQ_ID_FOR_UPSD];
    pSib = NULL;

    /* For each FRAME_DATA receive desc */
    for(prx=pArDev->rxQueue.pDescQueueHead; prx; prx=prx->pNextVirtPtr) {

        /* exit if descriptor is not done */
        A_DESC_CACHE_INVAL(prx);
        pRxStatus = RX_STATUS(prx);
        if (!pRxStatus->done) {
            return;
        }

        /* exit if current descriptor is self-linked */
        prxnext = prx->pNextVirtPtr;
        if (prxnext == prx) {
            return;
        }
        /* exit if next descriptor is not done and hw RXDP is still on this desc */
        A_DESC_CACHE_INVAL(prx->pNextVirtPtr);
        pNextStatus = RX_STATUS(prxnext);
        if ((!pNextStatus->done) &&
            (readPlatformReg(pArDev, MAC_RXDP) == prx->thisPhysPtr)) {
            return;
        }

        /* Skip if status is not OK */
        if (!pRxStatus->pktReceivedOK) {
            continue;
        }

        ftype = prx->pBufferVirtPtr.header->frameControl.fType;
        fsubtype = prx->pBufferVirtPtr.header->frameControl.fSubtype;

        pSib = sibEntryFindLocked(pArDev, &prx->pBufferVirtPtr.header->address2, prx->pOpBss);

        /* If Sib has a non-empty upsdq, move it to hw TXQ_ID_FOR_UPSD and transmit */
        if (pSib==NULL) {
            continue;
        }
        sibq = &pSib->upsdQueue;
#ifdef DEBUGUPSDISR
        if (A_SEM_VALID(sibq->qSem))
            isrPrintf("%x.%x:%d\n", ftype, fsubtype, sibq->qFrameCount);
#endif
        if (A_SEM_VALID(sibq->qSem) && sibq->qFrameCount && ftype==FRAME_DATA) {
            ATHEROS_DESC *pHead, *pTail;

            pHead = sibq->pDescQueueHead;
            pTail = sibq->pDescQueueTail;

            sibq->pDescQueueHead = sibq->pDescQueueTail = NULL;

            if (upsdq->pDescQueueTail == NULL) {
                upsdq->pDescQueueHead = pHead;
                ar5212SetTxDP(pArDev, TXQ_ID_FOR_UPSD, pHead->thisPhysPtr);
            } else {
                upsdq->pDescQueueTail->pNextVirtPtr = pHead;
                upsdq->pDescQueueTail->nextPhysPtr = pHead->thisPhysPtr;
                A_DESC_CACHE_FLUSH(upsdq->pDescQueueTail);
            }
            upsdq->pDescQueueTail = pTail;
            A_PIPEFLUSH();

            writePlatformReg(pArDev, MAC_Q_TXE, (1 << TXQ_ID_FOR_UPSD));

            upsdq->qFrameCount += sibq->qFrameCount;
            pSib->numTxPending += sibq->qFrameCount;
            sibq->qFrameCount = 0;
        }

    }
}
#endif

#endif // #ifdef BUILD_AR5212
