/*
 *  Copyright (c) 2000-2002 Atheros Communications, Inc., All Rights Reserved
 *
 *  Connects reset Reg Vectors, EEPROM Data, and device Functions to pDev
 */

#include "wlantype.h"
#if !defined(SPLIT_ENFORCE)
#include "wlandrv.h"
#endif
#include "wdcApi.h"
#include "wdcMsg.h"
#include "arMsgApi.h"
#include "arDev.h"
#include "arReceive.h"
#include "arTransmit.h"
#include "halApi.h"
#include "hal.h"
#if !defined(SPLIT_ENFORCE)
#include "ui.h"
#endif

#include "halDevId.h"

#if defined(BUILD_AR5212)
#include "ar5212/ar5212Attach.h"
#endif

#if defined(AR5523)
#include "cyg/hal/ar5523reg.h"
#endif

typedef struct DeviceProbeData {
    A_UINT16 deviceID;
    A_STATUS (*hwProbe)(AR_DEV_INFO *);
} DEVICE_PROBE_DATA;

static DEVICE_PROBE_DATA ar5kProbeData[] = {
#ifdef BUILD_AR5212
    {AR5212_DEVID,      ar5212Probe},
    {AR5212_DEFAULT,    ar5212Probe},
    {AR5312_DEVID,      ar5212Probe},
    {AR5212_DEVID_0014, ar5212Probe},
    {AR5212_DEVID_0015, ar5212Probe},
    {AR5212_DEVID_0016, ar5212Probe},
    {AR5212_DEVID_0017, ar5212Probe},
    {AR5212_DEVID_0018, ar5212Probe},
    {AR5212_DEVID_0019, ar5212Probe},
#if defined(PREDATOR_FB24)
    {0, ar5212Probe}, /* TEST BOARD shows 0 at 0xa0904020 */
#endif
#endif
};

const static A_UINT32 NumDevids = (sizeof(ar5kProbeData) / sizeof(DEVICE_PROBE_DATA));

typedef struct macRevData {
    A_UINT16 macRev;
    A_UINT16 deviceID;
} MAC_REV_DATA;


static MAC_REV_DATA ar5kMacRevData[] = {
#ifdef BUILD_AR5212
    {AR5212_SREV_1_0, AR5212_DEVID},
    {AR5212_SREV_1_1, AR5212_DEVID},
    {AR5212_SREV_1_3, AR5212_DEVID},
    {AR5212_SREV_1_4, AR5212_DEVID},
    {AR5213_SREV_1_0, AR5212_DEVID}, 
    {AR5213_SREV_1_0, AR5212_DEVID},
    {AR5212_SREV_AR5523_0, AR5212_DEVID},
    {AR5212_SREV_AR5523_1, AR5212_DEVID},
#endif
};

const static A_UINT32 NumMacRev = (sizeof(ar5kMacRevData) / sizeof(MAC_REV_DATA));

#define MAC_SREV_MASK 0xff
#define MAC_SREV_REG  0x4020

/**************************************************************
 * halGetDeviceId
 *
 * halGetDeviceId is a private function in the HAL layer.
 * It's used to get the DeviceID for the WLAN.
 */
A_UINT16
halGetDeviceId(AR_DEV_INFO *pArDev)
{
    A_UINT32 macRev;
    A_UINT32 count;

    ASSERT(pArDev);

    macRev = A_REG_RD(pArDev, MAC_SREV_REG) & MAC_SREV_MASK;
    for (count=0; count < NumMacRev; count++) {
        if (macRev == ar5kMacRevData[count].macRev) {
            return ar5kMacRevData[count].deviceID;
        }
    }

#if defined(AR5523) && ( !defined(PREDATOR_FB24) && !defined(PREDATOR_TB84) )
#error "No platform defined?!"
#endif

    uiPrintf("%s: Warning: DeviceID == 0.\n", __FUNCTION__);

    return 0; /* Unknown Device ID */
}

/*
 * Establish a default Target configuration.
 * Before starting the Target, the Host may query and change
 * the current configuration.
 */
static void
arDevConfigInit(AR_DEV_CONFIG *cfg)
{
    cfg->id = 0;
    cfg->serviceType = WLAN_STA_SERVICE;
    cfg->chanDesc.wlanMode = WLAN_MODE_11g;
    cfg->chanDesc.channelBand = CHANNEL_BAND_2GHz;
    cfg->chanDesc.channelFreq = 2462;
    cfg->chanDesc.phyModulations = PHY_MOD_CCK | PHY_MOD_OFDM;
    cfg->chanDesc.maxRDPower = 5;    
    cfg->chanDesc.cfgCtl = CTL_11G;
    cfg->chanDesc.twiceAntennaReduction = 0;
    cfg->rateControlEnable = 1;
    cfg->hwTxRetries = 4;
    cfg->swTxRetries = 3;
    cfg->calibrationInterval = 30;
    cfg->slowClockEnable = 0;
    cfg->compProc = 0;
    cfg->userRTSThreshold = 2346;
    cfg->protectionType = PROT_TYPE_CTSONLY;
    cfg->burstSeqThreshold = 3;
    cfg->abolt = 0;
    cfg->modeCTS = PROT_MODE_AUTO;
    cfg->WmeEnabled = FALSE;
    cfg->compWinSize = 4096;
    cfg->diversityControl = ANTENNA_CONTROLLABLE;
    cfg->tpScale = 0;
    cfg->tpcHalfDbm5 = 60;
    cfg->tpcHalfDbm2 = 60;
    cfg->overRideTxPower = 0;
    cfg->enable32KHzClock = 0;
}

/**************************************************************
 * halProbe
 * 
 * Resets the device and reads version and revision information.
 * Runs a brief device sanity test.
 * Reads EEPROM information in order to determine device capabilities.
 * Establishes the hardware-specific call table for the device.
 *
 * This MUST be the first hal function called.
 */
A_STATUS
halProbe(DEVICE_DATA devData, DEVICE_HANDLE *pDeviceHandle)
{
    A_STATUS    status;
    A_UINT16    deviceID;
    int         i;
    int         numDeviceIDs;
    AR_DEV_INFO *pArDev;

    pArDev = (AR_DEV_INFO *)A_DRIVER_MALLOC(sizeof(AR_DEV_INFO));
    if (!pArDev) {
        status = A_NO_MEMORY;
        goto probeError;
    }

    A_MEM_ZERO(pArDev, sizeof(AR_DEV_INFO));
#if defined(PCI_INTERFACE)
    pArDev->magicNumber = ARDEV_MAGICNUM; /* temp porting hack */

    {
        A_UINT32 baseAddress = (A_UINT32)devData;

        status = arPciAttach(&pArDev->pciInfo, baseAddress);
        if (status != A_OK) {
            status = A_DEVICE_NOT_FOUND;
            goto probeError;
        }
    }
#elif defined(AR5523)
    {
        A_UINT32 ar5523Revision = A_CFG_RD(pArDev, AR5523_REVISION_REG);

        pArDev->hwInfo.targetVersion =
            (ar5523Revision & AR5523_VER_M) >> AR5523_VER_S;

#if !defined(PREDATOR_FB24)
	// XXX: TBD until real chip gets here
	ASSERT(pArDev->hwInfo.targetVersion == AR5523_VERSION_NUMBER);
#endif

        pArDev->hwInfo.targetRevision =
            (ar5523Revision & AR5523_REV_M) >> AR5523_REV_S;
        ASSERT((A_UINT32)devData == AR5523_WLAN);
        pArDev->WLANBaseAddr = (A_UINT32)devData;
    }
#else
#error "NEED TO IMPLEMENT"
#endif

    deviceID = halGetDeviceId(pArDev);
    uiPrintf("Probe 0x%x 0x%p\n", deviceID, (void *)pArDev);

    /* Find a matching DeviceID probe routine */
    numDeviceIDs = (sizeof(ar5kProbeData)/sizeof(DEVICE_PROBE_DATA));
    for (i = 0; i < numDeviceIDs; i++) {
        if (deviceID == ar5kProbeData[i].deviceID) {
            break;
        }
    }
    if (i == numDeviceIDs) {
        uiPrintf("halProbe: Failed to find a driver for devid 0x%04X\n", deviceID);
        return A_DEVICE_NOT_FOUND;
    }

    pArDev->pHalInfo = (HAL_INFO *) A_DRIVER_MALLOC(sizeof(*pArDev->pHalInfo));
    if (pArDev->pHalInfo == NULL) {
        uiPrintf("halProbe: Error allocating memory for info struct\n");
        return A_ERROR;
    }
    A_MEM_ZERO(pArDev->pHalInfo, sizeof(*pArDev->pHalInfo));

    arDevConfigInit(&pArDev->config);

    /* Call the device-specific probe function */
    status = ar5kProbeData[i].hwProbe(pArDev);

    /* If unsuccessful, free any allocated memory */
    if (status != A_OK) {
        A_DRIVER_FREE(pArDev->pHalInfo, sizeof(*pArDev->pHalInfo));
        pArDev->pHalInfo = NULL;
    }

    ASSERT(pDeviceHandle);
    *pDeviceHandle = (DEVICE_HANDLE)pArDev;

    goto probeExit;

probeError:
    if (pArDev) {
        A_DRIVER_FREE(pArDev, sizeof(AR_DEV_INFO));
    }

probeExit:
    return status;
}

#if !defined(HAL_CODE_REDUCTION)
/**************************************************************
 * halDevInit
 *
 * halDevInit performs one-time initialization for the device.
 *
 * Any failures will return an error that should cause the hardware
 * to be disabled.
 */

A_STATUS
halDevInit(AR_DEV_INFO *pArDev)
{
    A_STATUS      status;

    ASSERT(pArDev);

    /* Call the device-specific initialization function */
    ASSERT(pArDev->pHwFunc);
    ASSERT(pArDev->pHwFunc->hwDevInit);
    status = pArDev->pHwFunc->hwDevInit(pArDev);
    if (status != A_OK) {
        ASSERT(0);
        goto DevInitExit;
    }

DevInitExit:
    return status;
}

/**************************************************************
 * halDevFini
 *
 * Remove HAL layer allocated structures
 */
A_STATUS
halDevFini(AR_DEV_INFO *pArDev)
{
    ASSERT(pArDev);

    if (pArDev->pHalInfo == NULL) {
        return A_ERROR;
    }

    ASSERT(pArDev->pHwFunc);
    ASSERT(pArDev->pHwFunc->hwDevFini);
    return pArDev->pHwFunc->hwDevFini(pArDev);
}

/**************************************************************
 * halFillCapabilityInfo
 *
 * Request the HAL to recache the static capability info (which
 * is changing).
 */
A_STATUS
halFillCapabilityInfo(AR_DEV_INFO *pArDev)
{
    ASSERT(pArDev->pHwFunc);
    ASSERT(pArDev->pHwFunc->hwFillCapabilityInfo);
    return pArDev->pHwFunc->hwFillCapabilityInfo(pArDev);
}
#endif
/**************************************************************
 * halGetNumDevid
 *
 * Returns the number of devids in the table
 */
A_UINT32
halGetNumDevid(void)
{
    return NumDevids;
}

/**************************************************************
 * halGetDevids
 *
 * Return the devid at the given index
 */
A_UINT16
halGetDevids(A_UINT32 index)
{
    ASSERT(index < NumDevids);

    return ar5kProbeData[index].deviceID;
}
