/*
 * Copyright (c) 2000-2002 Atheros Communications, Inc., All Rights Reserved
 *
 * Hardware-abstracted functions used to deal with interrupts.  These functions
 * are primarily called in the driver code residing above the HAL layer.
 */

#ident "ACI $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/hal/halInterrupts.c#3 $"

#include "wlantype.h"   /* A_UINTxx, etc. */
#include "arDev.h"      /* AR_DEV_INFO struct */
#include "halApi.h"     /* fn prototypes */
#include "hal.h"        /* HW_FUNCS struct */


/**************************************************************
 * halIsInteruptPending
 *
 * Checks to see if an interrupt is pending on our NIC
 *
 * Returns: TRUE    if an interrupt is pending
 *          FALSE   if not
 */
A_BOOL
halIsInterruptPending(AR_DEV_INFO *pArDev)
{
    ASSERT(pArDev && pArDev->pHwFunc && pArDev->pHwFunc->hwIsInterruptPending);

    return pArDev->pHwFunc->hwIsInterruptPending(pArDev);
}

/**************************************************************
 * halGetInterrupts
 *
 * Reads the Interrupt Status Register value from the NIC, thus deasserting
 * the interrupt line, and returns both the masked and unmasked mapped ISR
 * values.  The value returned is mapped to abstract the hw-specific bit
 * locations in the Interrupt Status Register.
 *
 * Returns: A hardware-abstracted bitmap of all non-masked-out
 *          interrupts pending, as well as an unmasked value
 */
HAL_INT_TYPE
halGetInterrupts(AR_DEV_INFO *pArDev, HAL_INT_TYPE *pUnmaskedValue, A_UINT32 *pDescQueueBitMask)
{
    ASSERT(pArDev && pArDev->pHwFunc && pArDev->pHwFunc->hwGetInterrupts);

    return pArDev->pHwFunc->hwGetInterrupts(pArDev, pUnmaskedValue, pDescQueueBitMask);
}

/**************************************************************
 * halEnableInterrupts
 *
 * Atomically enables NIC interrupts.  Interrupts are passed in
 * via the enumerated bitmask in "ints".
 */
void
halEnableInterrupts(AR_DEV_INFO *pArDev, HAL_INT_TYPE ints)
{
    ASSERT(pArDev && pArDev->pHwFunc && pArDev->pHwFunc->hwEnableInterrupts);
    ASSERT((ints & HAL_INT_INVALID) == 0);

    pArDev->pHwFunc->hwEnableInterrupts(pArDev, ints);
}

/**************************************************************
 * halDisableInterrupts
 *
 * Atomically disables NIC interrupts.  Interrupts are passed in
 * via the enumerated bitmask in "ints".
 */
void
halDisableInterrupts(AR_DEV_INFO *pArDev, HAL_INT_TYPE ints)
{

    ASSERT(pArDev && pArDev->pHwFunc &&
           pArDev->pHwFunc->hwDisableInterrupts);
    ASSERT((ints & HAL_INT_INVALID) == 0);

    pArDev->pHwFunc->hwDisableInterrupts(pArDev, ints);
}

