/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/arBss.c#3 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 */

#include "wlantype.h"
#include "queue.h"
#include "arDev.h"
#include "arBss.h"
#include "halApi.h"
#include "hal.h"

#include "targWdc.h"

#define DMA_STOP_WAIT_TIME 4

/*
 * TODO xxxamanxxx: these should be promoted to some
 * well known protocol place
 */
const struct {
    A_UINT32                          AIFS, LOG_CW_MIN, LOG_CW_MAX ;
} WLAN_MODE_PARAMETER[] = {
    { /* WLAN_MODE_NONE            */    0,          0,          0 },
    { /* WLAN_MODE_11b             */    2,          5,         10 },
    { /* WLAN_MODE_11a             */    2,          4,         10 },
    { /* WLAN_MODE_11g             */    2,          4,         10 },
    { /* WLAN_MODE_11a_TURBO       */    1,          3,         10 },
    { /* WLAN_MODE_11g_TURBO       */    1,          3,         10 },
    { /* WLAN_MODE_11a_TURBO_PRIME */    2,          3,         10 },
    { /* WLAN_MODE_11g_TURBO_PRIME */    2,          3,         10 },
    { /* WLAN_MODE_11a_XR          */    0,          2,          3 },
    { /* WLAN_MODE_11g_XR          */    0,          2,          3 },
};


/*******************************************************************************
 *
 * Utility routines local to this file.
 *
 */

/*
 * Setting up HW related BSS configuration. It is used
 * when new BSS parameters are available - e.g. wdcStaJoinBss,
 * or following a hw reset operation - e.g. through
 * wdcStaAssocBss
 */
static void
setHwBssConfig(AR_DEV_INFO *pArDev, AR_BSS_INFO *pBssInfo)
{
    halWriteAssocid(pArDev, &pBssInfo->bssId, pBssInfo->assocId);
    if (pBssInfo->config.bssType == INDEPENDENT_BSS) {
        halSetAdhocMode(pArDev);
    }
    if (pBssInfo->config.wlanMode == WLAN_MODE_11g) {

        ASSERT(!((pBssInfo->config.bssType == INDEPENDENT_BSS) &&
              (pBssInfo->config.shortSlotTime11g)));

        halUseShortSlotTime(pArDev, (A_BOOL)pBssInfo->config.shortSlotTime11g);
    }
    halSetTxPowerLimit(pArDev, 2 * pBssInfo->config.tpcPowerLimit);

    return;
}

static void
clrPSPollGen(AR_DEV_INFO *pArDev, AR_BSS_INFO *pBssInfo)
{
    // halStopTxDma(pArDev, pPSPollQueue->halTxQueueNum, 100);
    return;
}

static void
setPSPollGen(AR_DEV_INFO *pArDev, AR_BSS_INFO *pBssInfo)
{
    /*
     * TODO: i.e. once we start using that hardware
     * capability; remember to take out the Oahu WAR
     * in the halWriteAssocid call in that case!!
     */

    return;
}

static void
clrBeaconGen(AR_DEV_INFO *pArDev, AR_BSS_INFO *pBssInfo)
{
    if (pBssInfo->pBeaconDesc &&
        (pBssInfo->config.bssType != INFRASTRUCTURE_BSS))
    {
        halStopTxDma(pArDev, pBssInfo->halBeaconQueue, DMA_STOP_WAIT_TIME);
        halReleaseTxQueue(pArDev, pBssInfo->halBeaconQueue);

        A_DRIVER_FREE(pBssInfo->pBeaconDesc, sizeof(AR_DESC) + AR_MAX_BEACON_SIZE);
        pBssInfo->pBeaconDesc = NULL;
    }
}

static void
setBeaconGen(AR_DEV_INFO *pArDev, AR_BSS_INFO *pBssInfo, WDC_BUFFER_DESC *pBeaconData)
{
    HAL_TX_QUEUE_INFO halQueue;
    AR_DESC           *pDesc;

    if (!(pDesc = pBssInfo->pBeaconDesc)) {
        if (!pBeaconData) {
            return;
        }
        pDesc = (AR_DESC *)A_DRIVER_MALLOC(sizeof(AR_DESC) +
                                           AR_MAX_BEACON_SIZE);
        ASSERT(pDesc);

        A_MEM_ZERO(pDesc, sizeof(AR_DESC));

        pBssInfo->pBeaconDesc = pDesc;

        pDesc->pNextVirtPtr         = NULL;
        pDesc->pBufferVirtPtr.ptr   = pDesc + 1;
        pDesc->thisPhysPtr          = A_DATA_V2P(pDesc);
        pDesc->nextPhysPtr          = A_DATA_V2P(pDesc->pNextVirtPtr);
        pDesc->bufferPhysPtr        = A_DATA_V2P(pDesc->pBufferVirtPtr.ptr);
    }

    if (pBeaconData) {
        AR_DESC_INIT_HW_AREA(pDesc);
        while (pBeaconData) {
            A_MEM_CPY(pDesc->pBufferVirtPtr.byte +
                      pDesc->hw.txControl.bufferLength,
                      pBeaconData->pBuffer,
                      pBeaconData->bufferLength);
            pDesc->hw.txControl.bufferLength += pBeaconData->bufferLength;
            ASSERT(pDesc->hw.txControl.bufferLength <= AR_MAX_BEACON_SIZE);
            pBeaconData = pBeaconData->pNext;
        }
        pDesc->hw.txControl.frameLength = (pDesc->hw.txControl.bufferLength +
                                           FCS_FIELD_SIZE);
        pDesc->pOpBss = pBssInfo;

        halSetupBeaconDesc(pArDev, pDesc, 0,
                           (pBssInfo->config.bssType == INFRASTRUCTURE_BSS));

    }

    A_MEM_ZERO(&halQueue, sizeof(HAL_TX_QUEUE_INFO));
    halQueue.mode     = TXQ_MODE_BEACON;
    halQueue.priority = AR_BEACON_QUEUE;
    halQueue.aifs     = WLAN_MODE_PARAMETER[pBssInfo->config.wlanMode].AIFS;
    halQueue.logCwMin = WLAN_MODE_PARAMETER[pBssInfo->config.wlanMode].LOG_CW_MIN;
    halQueue.logCwMax = WLAN_MODE_PARAMETER[pBssInfo->config.wlanMode].LOG_CW_MAX;
    if (pBssInfo->config.bssType == INFRASTRUCTURE_BSS) {
        if (pArDev->config.abolt & ABOLT_WME_ELE) {
            /* AP can burst out beacons if ABOLT_WME is enabled */
            halQueue.aifs     = 1;
            halQueue.logCwMin = 0;
            halQueue.logCwMax = 0;
        }
    } else {
        halQueue.logCwMin += 1;
    }

    /* Release/Stop the beacon queue if already allocated.
     *
     * The wdcStartBss API is a bit confusing in that it can be called after hardware
     * reset, or during regular operation.  It is difficult to tell
     * if the beacon queue must be stopped/released.
     *
     * As a quick/dirty workaround, peek into the HAL to see if the
     * queue is already allocated.
     */
    if (pArDev->pHalInfo->txQueueAllocMask & (1<<halQueue.priority)) {
        halStopTxDma(pArDev, halQueue.priority, DMA_STOP_WAIT_TIME);
        halReleaseTxQueue(pArDev, halQueue.priority);
    }

    pBssInfo->halBeaconQueue = halSetupTxQueue(pArDev, &halQueue);

    if (pBssInfo->config.bssType != INFRASTRUCTURE_BSS) {
        halSetTxDP(pArDev, pBssInfo->halBeaconQueue, pDesc->thisPhysPtr);
        halStartTxDma(pArDev, pBssInfo->halBeaconQueue);
    }

    halBeaconInit(pArDev, &pBssInfo->config);

    return;
}


/*******************************************************************************
 *
 * Exported WDC API.
 *
 */

/*
 * Join just writes the BSS ID - so that we can start receiving the
 * appropriate beacons and/or probe responses - thereby latching on
 * the network TSF. We also reset the TSF so that its value is lower
 * than the beacon/probe-response that we're hoping to catch after
 * the Join - only in that case would the HW update the TSF register
 * with the new beacon! HW related bss configuration is deployed
 * here.
 * Once we join - we can start using the rate tables based on the
 * WLAN mode - i.e. we are now ready for data frames! In reality we
 * won't be sending any data frames until wdcStaAssocBss for infra
 * station and/or wdcStartBss for adhoc station or AP - nevertheless
 * we do have the wlan mode info at join time!
 */
void
TARG_wdcStaJoinBss(
    DEVICE_HANDLE       deviceHandle,
    A_UINT32            wdcBssId,
    WLAN_MACADDR        *pBssId,
    WDC_BSS_ATTRIBUTES  *pBssConfig)
{
    AR_DEV_INFO *pArDev   = (AR_DEV_INFO *)deviceHandle;
    AR_BSS_INFO *pBssInfo = &pArDev->bssTable[wdcBssId];

    ASSERT(wdcBssId < AR_MAX_BSS);
    ASSERT(pBssId && pBssConfig);
    ASSERT(pBssConfig->wlanMode != WLAN_MODE_NONE);

    /* The caller may not do a good job of cleaning up */
    if (pBssInfo->config.wlanMode != WLAN_MODE_NONE) {
        TARG_wdcDetachBss(pArDev, wdcBssId);
    }

    A_MACADDR_COPY(pBssId, &pBssInfo->bssId);

    pBssInfo->config     = *pBssConfig;
    pBssInfo->assocId    = 0;

    pArDev->primeInUse = FALSE;
    switch(pBssConfig->wlanMode) {
    case WLAN_MODE_NONE:
        pBssInfo->pRateTable = pArDev->hwRateTable[WIRELESS_MODE_11b];
        break;
    case WLAN_MODE_11b:
        pBssInfo->pRateTable = pArDev->hwRateTable[WIRELESS_MODE_11b];
        break;
    case WLAN_MODE_11g_TURBO_PRIME:
        pBssInfo->pRateTable = pArDev->hwRateTable[WIRELESS_MODE_11g];
        pArDev->primeInUse = TRUE;
        break;
    case WLAN_MODE_11a_TURBO_PRIME:
        pBssInfo->pRateTable = pArDev->hwRateTable[WIRELESS_MODE_11a];
        pArDev->primeInUse = TRUE;
        break;
    case WLAN_MODE_11g:
        pBssInfo->pRateTable = pArDev->hwRateTable[WIRELESS_MODE_11g];
        break;
    case WLAN_MODE_11a:
        pBssInfo->pRateTable = pArDev->hwRateTable[WIRELESS_MODE_11a];
        break;
    case WLAN_MODE_11a_TURBO:
        pBssInfo->pRateTable = pArDev->hwRateTable[WIRELESS_MODE_TURBO];
        break;
    case WLAN_MODE_11g_TURBO:
        pBssInfo->pRateTable = pArDev->hwRateTable[WIRELESS_MODE_108g];
        break;

    default:
        ASSERT(0);
        break;
    }

    halResetTsf(pArDev);
    setHwBssConfig(pArDev, pBssInfo);

    return;
}


/*
 * Association implies that we have and AID - which implies start
 * watching for TIM. It also implies the need to start tracking
 * beacons - waking up in their anticipation etc. - i.e. setting
 * up the HW related beacon timers. Consequently, automated PS-Poll
 * generation is also fired off here!
 * The HW related bss config parameters are also updated here for
 * cases where the assoc operation is being used after a hw reset.
 */
void
TARG_wdcStaAssocBss(
    DEVICE_HANDLE  targetHandle,
    A_UINT32       wdcBssId,
    A_UINT16       assocId)
{
    AR_DEV_INFO *pArDev   = (AR_DEV_INFO *) targetHandle;
    AR_BSS_INFO *pBssInfo = &pArDev->bssTable[wdcBssId];

    ASSERT(wdcBssId < AR_MAX_BSS);
    ASSERT(pBssInfo->config.wlanMode != WLAN_MODE_NONE);

    pBssInfo->assocId = assocId;

    setHwBssConfig(pArDev, pBssInfo);
    halSetStaBeaconTimers(pArDev, &pBssInfo->config);
    setPSPollGen(pArDev, pBssInfo);

    return;
}


void
TARG_wdcStartBss(
    DEVICE_HANDLE   targetHandle,
    A_UINT32        wdcBssId,
    WDC_BUFFER_DESC *pBeaconData)
{
    AR_DEV_INFO *pArDev   = (AR_DEV_INFO *) targetHandle;
    AR_BSS_INFO *pBssInfo = &pArDev->bssTable[wdcBssId];

    ASSERT(wdcBssId < AR_MAX_BSS);
    ASSERT(pBssInfo->config.wlanMode != WLAN_MODE_NONE);

    setHwBssConfig(pArDev, pBssInfo);
    setBeaconGen(pArDev, pBssInfo, pBeaconData);

    return;
}


/*
 * Update and possibly apply changes to BSS parameters. Implemented
 * only for parameters expected to change.
 */
void
TARG_wdcUpdateBssAttribute(
    DEVICE_HANDLE           targetHandle,
    A_UINT32                wdcBssId,
    WDC_BSS_ATTRIBUTE       attribute,
    IN  A_UINT32            cfgSz,
    IN  TARGET_CONFIG_VAL   cfgVal)
{
    AR_DEV_INFO *pArDev   = (AR_DEV_INFO *) targetHandle;
    AR_BSS_INFO *pBssInfo = &pArDev->bssTable[wdcBssId];

    ASSERT(wdcBssId < AR_MAX_BSS);
    ASSERT(pBssInfo->config.wlanMode != WLAN_MODE_NONE);

    switch (attribute) {

    case DTIM_INTERVAL:
        /* the host will still have to apply by calling wdcStaAssocBss */
        pBssInfo->config.dtimInterval = (A_UINT32)cfgVal;
        break;

    case ATIM_WINDOW:
        /* the host will still have to apply by calling wdcStartBss */
        pBssInfo->config.atimWindow = (A_UINT32)cfgVal;
        break;

    case SHORT_SLOT_TIME_11g:
        pBssInfo->config.shortSlotTime11g = (A_UINT32)cfgVal;
        if (pBssInfo->config.wlanMode == WLAN_MODE_11g) {
            halUseShortSlotTime(pArDev, (A_BOOL)pBssInfo->config.shortSlotTime11g);
        }
        break;

    case TPC_POWER_LIMIT:
        pBssInfo->config.tpcPowerLimit = (A_UINT32)cfgVal;
        halSetTxPowerLimit(pArDev, 2 * pBssInfo->config.tpcPowerLimit);
        break;

    case BEACON_INTERVAL:
        ASSERT(0);
        break;

    case CFP_INTERVAL:
        ASSERT(0);
        break;

    case CFP_MAX_DURATION:
        ASSERT(0);
        break;

    case DEFAULT_RATE_INDEX:
        pBssInfo->config.defaultRateIndex = (A_UINT32)cfgVal;
        break;

    case SLEEP_DURATION:
        ASSERT(0);
        break;

    case BMISS_THRESHOLD:
        ASSERT(0);
        break;

    case BSS_KEY_UPDATE: {
        BSS_KEY_RECORD      *pBssRecord;
        WLAN_PRIV_RECORD    *pKey;
        A_UINT32            keyCacheIndex;

        ASSERT(cfgSz == sizeof(BSS_KEY_RECORD));

        pBssRecord = (BSS_KEY_RECORD *)cfgVal;
        pKey = (WLAN_PRIV_RECORD *)&pBssRecord->bssKey;
        keyCacheIndex = pBssRecord->bssKeyIdx;
        
        if (pBssRecord->isDefaultBssKey) {
            pBssInfo->config.bssDefaultKey = keyCacheIndex;
        }

        halSetKeyCacheEntry(pArDev, keyCacheIndex, NULL, pKey);
        break;
    }

    default:
        ASSERT(0);
        break;
    }

    return;
}


/*
 * Clean up the AssocId so that we can stop responding to PS-Polls
 * etc. and do the corresponding clean-up.
 * Note: doesn't make sense to write BSSID, except for not messing
 * with history.
 */
void
TARG_wdcDetachBss(
    TARGET_HANDLE  targetHandle,
    A_UINT32       wdcBssId)
{
    AR_DEV_INFO *pArDev   = (AR_DEV_INFO *) targetHandle;
    AR_BSS_INFO *pBssInfo = &pArDev->bssTable[wdcBssId];

    ASSERT(wdcBssId < AR_MAX_BSS);

    pArDev->primeInUse = FALSE;
    pArDev->devStats.txRateKb = 0;
    pBssInfo->config.wlanMode = WLAN_MODE_NONE;
    halWriteAssocid(pArDev, &pBssInfo->bssId, 0);
    halDisableInterrupts(pArDev, HAL_INT_BMISS);
    clrPSPollGen(pArDev, pBssInfo);
    clrBeaconGen(pArDev, pBssInfo);

    return;
}
