/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/arConfig.c#3 $
 *
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 */

#include "arDev.h"
#include "halApi.h"
#include "hal.h"

void
TARG_wdcTargetSetConfiguration(
    DEVICE_HANDLE       deviceHandle,
    TARGET_CONFIG_ID    attrId,
    A_UINT32            cfgSz,
    TARGET_CONFIG_VAL   cfgVal,
    A_STATUS            *pStatus
)
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *)deviceHandle;
    A_STATUS        status = A_OK;

    ASSERT(pArDev);

    switch(attrId) {
    case CFG_REG_DOMAIN: {
#if defined(RD_OVERRIDE) || defined(DEBUG)
        /*
         * To facilitate testing, DEBUG kernels permit the Regulatory
         * Domain to be overriden by the Host driver.  Production
         * kernels disallow this override in order to insure regulatory
         * compliance.
         */
        A_UINT32 regDomain = (A_UINT32)cfgVal;
        status = halSetRegulatoryDomain(pArDev, (A_UINT16)regDomain);
#else
        status = A_ENOTSUP;
#endif /* RD_OVERRIDE || DEBUG */
        break;
    }

    case CFG_RATE_CONTROL_ENABLE:
        pArDev->config.rateControlEnable = (A_UINT32)cfgVal;
        break;

    case CFG_HW_TX_RETRIES: {
        A_UINT32 hwTxRetries = (A_UINT32)cfgVal;

        ASSERT(hwTxRetries >= HW_TX_RETRY_MIN);
        ASSERT(hwTxRetries <= HW_TX_RETRY_MAX);
        pArDev->config.hwTxRetries = hwTxRetries;
        break;
    }

    case CFG_SW_TX_RETRIES: {
        A_UINT32 swTxRetries = (A_UINT32)cfgVal;

        ASSERT(swTxRetries <= SW_TX_RETRY_MAX);
        pArDev->config.swTxRetries = swTxRetries;
        break;
    }

    case CFG_CALIBRATION_INTERVAL: {
        A_UINT32 calibrationInterval = (A_UINT32)cfgVal;

        pArDev->config.calibrationInterval = calibrationInterval;

        A_UNTIMEOUT(&pArDev->periodicCalTimer);

        if (calibrationInterval != 0) {
            A_TIMEOUT(&pArDev->periodicCalTimer, 
                      100 * calibrationInterval, TRUE);
        }

        break;
    }
    case CFG_SLOW_CLOCK_ENABLE:
        ASSERT(0); /* Not yet supported */
        break;

    case CFG_COMP_PROC:
        ASSERT(0); /* Not yet supported */
        break;

    case CFG_USER_RTS_THRESHOLD: {
        A_UINT32 rtsThreshold = (A_UINT32)cfgVal;
        ASSERT(rtsThreshold <= MAX_RTS_THRESHOLD);
        pArDev->config.userRTSThreshold = rtsThreshold;
        break;
    }

    case CFG_PROTECTION_TYPE:
        pArDev->config.protectionType = (A_UINT32)cfgVal;
        break;

    case CFG_BURST_SEQ_THRESHOLD:
        ASSERT(0); /* Not yet supported */
        break;

    case CFG_ABOLT: {
        A_UINT32 abolt = (A_UINT32)cfgVal;

        pArDev->config.abolt = abolt;
        break;
    }

    case CFG_MODE_CTS:
        pArDev->config.modeCTS = (A_UINT32)cfgVal;
        break;

    case CFG_WME_ENABLED:
        ASSERT(0); /* Not yet supported */
        break;

    case CFG_GPRS_CBR_PERIOD:
        ASSERT(0); /* Not yet supported */
        break;

    case CFG_SERVICE_TYPE: {
        A_UINT32 serviceType = (A_UINT32)cfgVal;

        CURRENT_SERVICE_TYPE(pArDev) = serviceType;
        break;
    }

    case CFG_MAC_ADDR: {
        /* Takes effect next time the Target is started. */
        ASSERT(cfgSz == WLAN_MAC_ADDR_SIZE);
        memcpy(pArDev->config.macAddr.octets, (char *)cfgVal, cfgSz);
        break;
    }

#if defined(EAR_SUPPORT)
    case CFG_DEBUG_EAR:
        ASSERT(0); /* Not yet supported */
        break;
#endif

    case CFG_INIT_REGS: {
        /* Takes effect at next Target Reset */
        static REGISTER_VAL *pInitRegs;
        static A_UINT32      cfgSzSave;

        if ((pInitRegs == NULL) || (cfgSzSave != cfgSz)) {
            if (pInitRegs) {
                A_DRIVER_FREE(pInitRegs, cfgSzSave);
            }
            pInitRegs = (REGISTER_VAL *)A_DRIVER_MALLOC(cfgSz);
            cfgSzSave = cfgSz;
        }

        if (!pInitRegs) {
            status = A_NO_MEMORY;
            break;
        } else {
            ASSERT(pArDev->config.pInitRegs == NULL);
            ASSERT((cfgSz % sizeof(REGISTER_VAL)) == 0);
            memcpy(pInitRegs, cfgVal, cfgSz);
            ASSERT(pInitRegs[(cfgSz/sizeof(REGISTER_VAL)) - 1].Offset == 0);
            pArDev->config.pInitRegs = pInitRegs;
        }

        break;
    }

    case CFG_DEBUG_ID: {
        pArDev->config.id = (A_UINT32)cfgVal;
        break;
    }

    case CFG_COMP_WIN_SZ: {
        A_UINT32 compWinSize = (A_UINT32)cfgVal;
        
        ASSERT(compWinSize >= COMPRESSION_WINSIZE_MIN);
        ASSERT(compWinSize <= COMPRESSION_WINSIZE_MAX);
        ASSERT((compWinSize & (compWinSize-1)) == 0); /* is power of 2 */
        pArDev->config.compWinSize = compWinSize;
        break;
    }

    case CFG_DIVERSITY_CTL:  {
        /* Restore power to Awake state if sleeping */
        status = arMacSetAwakeMode(pArDev, TRUE);
        ASSERT(status == A_OK);

        halSetAntennaSwitch(pArDev,
                            (ANTENNA_CONTROL)cfgVal,
                            CURRENT_PCHDESC(pArDev));

        /* Restore power to former or desired state */
        status = arMacSetAwakeMode(pArDev, FALSE);
        ASSERT(status == A_OK);
        break;
    }

    case CFG_TP_SCALE: {
        A_UINT16 tpScale = (A_UINT16)(A_UINT32)cfgVal;

        /* ASSERT(tpScale >= TP_SCALE_LOWEST); */
        ASSERT(tpScale <= TP_SCALE_HIGHEST);
        pArDev->config.tpScale = tpScale;
        break;
    }

    case CFG_TPC_HALF_DBM5: { /* need this? */
        A_INT16 tpcHalfDbm5 = (A_INT16)(A_INT32)cfgVal;

        ASSERT(tpcHalfDbm5 >= 0);
        ASSERT(tpcHalfDbm5 <= 60);
        pArDev->config.tpcHalfDbm5 = tpcHalfDbm5;
        break;
    }

    case CFG_TPC_HALF_DBM2: { /* need this? */
        A_INT16 tpcHalfDbm2 = (A_INT16)(A_UINT32)cfgVal;

        ASSERT(tpcHalfDbm2 >= 0);
        ASSERT(tpcHalfDbm2 <= 60);
        pArDev->config.tpcHalfDbm2 = tpcHalfDbm2;
        break;
    }

    case CFG_OVERRD_TX_POWER: {
        A_UINT16 overRideTxPower = (A_UINT16)(A_UINT32)cfgVal;

        /* ASSERT(overRideTxPower >= 0); */
        ASSERT(overRideTxPower <= 30);
        pArDev->config.overRideTxPower = overRideTxPower;
        break;
    }
    
    case CFG_GMODE_PROTECTION: {
        A_UINT8 protectVal = (A_UINT8)(A_UINT32)cfgVal;

        pArDev->protectOn = protectVal;
        break;
    }
    
    case CFG_GMODE_PROTECT_RATE_INDEX:{
        A_UINT8 protectRate = (A_UINT8)(A_UINT32)cfgVal;

        pArDev->protectRateIdx = protectRate;
        break;
    }
    
    case CFG_GMODE_NON_ERP_PREAMBLE: {
        A_BOOL nonErpPreambleMode= (A_BOOL)(A_UINT32)cfgVal;

        pArDev->nonErpPreamble = nonErpPreambleMode;
        break;
    }

    case CFG_USE_32KHZ_CLOCK:
        ASSERT(0); /* Not yet supported */
        break;


    default:
        ASSERT(0); /* unknown configuration ID */
        break;
    }

    if (pStatus) {
        *pStatus = status;
    }

}

void
TARG_wdcSetLedState(
      IN  DEVICE_HANDLE     deviceHandle, 
      IN  A_UINT32          ledState)
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *)deviceHandle;
    A_STATUS        status;

    ASSERT(pArDev);

    pArDev->connectStatus = (A_BOOL)ledState;
    /* Restore power to Awake state is sleeping */
    status = arMacSetAwakeMode(pArDev, TRUE);
    ASSERT(status == A_OK);
    halSetLedState(pArDev, pArDev->connectStatus);
    /* Restore power to former or desired state */
    status = arMacSetAwakeMode(pArDev, FALSE);
    ASSERT(status == A_OK);
}
