/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/arConn.c#3 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 */

#include "wdcApi.h"
#include "arDev.h"
#include "arBss.h"
#include "arConn.h"
#include "arRateCtrl.h"
#include "halApi.h"
#include "hal.h"


/* Local Functions */

LOCAL A_UINT16
findIcvLength(A_UINT32 keyType)
{
    if (keyType == PRIV_KEY_TYPE_AES_CCM ||
        keyType == PRIV_KEY_TYPE_AES_OCB)
    {
        return AES_ICV_FIELD_SIZE;
    }

    return WEP_ICV_FIELD_SIZE;
}

LOCAL A_UINT16
findIvLength(A_UINT32 keyType)
{
    A_UINT16 ivLen=0;

    if (keyType != PRIV_KEY_TYPE_NULL) {
        ivLen = WEP_IV_FIELD_SIZE;

        if (keyType == PRIV_KEY_TYPE_AES_CCM ||
            keyType == PRIV_KEY_TYPE_TKIP ||
            keyType == PRIV_KEY_TYPE_TKIP_SW)
        {
            ivLen += EXT_IV_FIELD_SIZE;
        }
    }
    return ivLen;
}


/*******************************************************************************
 *
 * Exported WDC API.
 *
 */

/*
 * Used by the host to create a connection in the target
 */
void
TARG_wdcCreateConnection(
    IN  DEVICE_HANDLE       targetHandle,
    IN  WLAN_MACADDR       *pMacAddr,
    IN  A_UINT32            wdcConnId,
    IN  A_UINT32            wdcBssId,
    IN  CONN_ATTRIBUTES    *pOptions)
{
    AR_DEV_INFO  *pArDev = (AR_DEV_INFO *) targetHandle;
    AR_CONN_INFO *pArConn;
    AR_BSS_INFO  *pArBss;


    ASSERT(pMacAddr);
    ASSERT(wdcConnId < AR_MAX_CONN);
    ASSERT(wdcBssId < AR_MAX_BSS);

    pArConn = &pArDev->connTable[wdcConnId];
    ASSERT(IS_FREE_AR_CONN(pArConn));

    if (!IS_FREE_AR_CONN(pArConn)) {
        /* Host must delete connection first before reuse of the connection ID */
        return;
    }

    A_MEM_ZERO(pArConn, sizeof(*pArConn));

    pArBss = &pArDev->bssTable[wdcBssId];
    ASSERT(IS_VALID_AR_BSS(pArBss));

    if (!IS_VALID_AR_BSS(pArBss)) {
        /* Host must setup BSS before creating connection */
        return;
    }

    /* Connection in use */
    pArConn->bInUse = TRUE;
    pArConn->needClearDest = TRUE;

    A_MACADDR_COPY(pMacAddr, &pArConn->macAddr);
    pArConn->options = *pOptions;

    /* Bind connection to BSS */
    pArConn->pOpBss = pArBss;

    /*
     * Initialize rate control processing for traffic
     * to this destination
     */
    rcSibInit(pArConn);
    rcSibUpdate(pArConn, FALSE, pArDev);

    A_SEM_INIT(pArConn->swRetryQueue.qSem, 0, CREATE_UNLOCKED);
}


/*
 * Is used by the host to set/clear/modify an attribute related to a
 * particular connection. Whether it makes sense to modify an attribute
 * is attribute specific. The target is not required (but it could)
 * perform any validation of whether an attribute should be changed
 * or not (eg. it currently doesn't make sense to change compression after
 * associating but the target might not reject such request).
 */
void
TARG_wdcUpdateConnectionAttribute(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcConnId,
    IN  CONN_ATTRIBUTE      attribute,
    IN  A_UINT32            cfgSz,
    IN  TARGET_CONFIG_VAL   cfgVal
)
{
    AR_DEV_INFO *pArDev   = (AR_DEV_INFO *) targetHandle;
    AR_CONN_INFO *pArConn;

    ASSERT(wdcConnId < AR_MAX_CONN);

    pArConn = &pArDev->connTable[wdcConnId];
    ASSERT(!IS_FREE_AR_CONN(pArConn));

    switch (attribute) {

    case CONN_KEYTYPE:
        ASSERT(0);
        break;

    case CONN_COMPRESSION: {
#if defined(COMPRESSION_ENABLED)
        A_UINT32 compression = (A_UINT32)cfgVal;
        /* TBD, from pArConn->options.keytype */
        A_UINT32 keyType = pArConn->options.keytype;
        pArConn->options.compression = compression;
        halSetDecompMask(pArDev, wdcConnId, keyType, compression);
#endif /* COMPRESSION_ENABLED */
        break;
    }

    case CONN_LONG_PREAMBLE_ONLY: {
        A_UINT32 longPreambleOnly = (A_UINT32)cfgVal;
        pArConn->options.longPreambleOnly = longPreambleOnly;
        break;
    }

    case CONN_CURRENT_TX_RATE_ANT: {
        RATE_ANT_CODE currentTxRateAnt = (RATE_ANT_CODE)cfgVal;
        pArConn->options.currentTxRateAnt = currentTxRateAnt;
        break;
    }

    case CONN_SERVICE_TYPE: {
        CONN_TYPE serviceType = (CONN_TYPE)cfgVal;
        pArConn->options.serviceType = serviceType;
        break;
    }
    case CONN_WLAN_MODE: {
        WLAN_MODE wlanMode = (A_UINT32)cfgVal;
        pArConn->options.wlanMode = wlanMode;
        // xxx: this appears to not work.
        break;
    }
    case CONN_WORKING_RATE_SET: {
        CONN_RATE_SET *pConnRateSet = (CONN_RATE_SET *)cfgVal;

        ASSERT(cfgSz == sizeof(*pConnRateSet));
        pArConn->options.workingRateSet = *pConnRateSet;

        /* Update connection related modules */
        rcSibUpdate(pArConn, FALSE, pArDev);
        halSetBasicRate(pArDev, &pArConn->options.workingRateSet);

        break;
    }
    case CONN_KEY_UPDATE: {
        CONNECTION_KEY_RECORD   *pConnRecord;
        WLAN_MACADDR            *pBssId;
        WLAN_PRIV_RECORD        *pKey;
        A_UINT32                keyCacheIndex;

        ASSERT(cfgSz == sizeof(CONNECTION_KEY_RECORD));

        pConnRecord = (CONNECTION_KEY_RECORD *)cfgVal;
        pBssId = (WLAN_MACADDR *)&pConnRecord->bssId;
        pKey = (WLAN_PRIV_RECORD *)&pConnRecord->connKey;

        /* Update connection key type */
        pArConn->options.keytype = pKey->keyType;

        /* Find the iv and icv length for this keytype */
        pArConn->ivLength = findIvLength(pKey->keyType);
        pArConn->icvLength = findIcvLength(pKey->keyType);

        /*
         * Map ConnId to key cache index
         * XXX Divy. should be abstracted
         */
        keyCacheIndex = wdcConnId + MAX_SHARED_KEYS;

        halSetKeyCacheEntry(pArDev, keyCacheIndex, pBssId, pKey);
        break;
    }

    case CONN_WORKING_RATE_SET_KEEPRC: {
        CONN_RATE_SET *pConnRateSet = (CONN_RATE_SET *)cfgVal;

        ASSERT(cfgSz == sizeof(*pConnRateSet));
        pArConn->options.workingRateSet = *pConnRateSet;

        /* Update connection related modules */
        rcSibUpdate(pArConn, TRUE, pArDev);

        break;
    }

    default:
        ASSERT(0);
        break;
    }
}

/*
 * Used by the host to add/modify/clear a key for a particular connection
 */
void
TARG_wdcSetConnectionKey(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcConnId,
    IN  A_UINT32            keyType,
    IN  A_UINT32            keySize,            /* in bytes */
    IN  A_UCHAR             *pKeyMaterial
)
{
    ASSERT(0);
}

void
TARG_wdcGetConnectionStats(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcConnId,
    OUT CONN_STATS          *pConnStats
)
{
    ASSERT(0);
}

void
TARG_wdcDeleteConnection(
    IN  TARGET_HANDLE       targetHandle,
    IN  A_UINT32            wdcConnId
)
{
    AR_DEV_INFO *pArDev   = (AR_DEV_INFO *) targetHandle;
    AR_CONN_INFO *pArConn;

    ASSERT(wdcConnId < AR_MAX_CONN);

    pArConn = &pArDev->connTable[wdcConnId];
    ASSERT(!IS_FREE_AR_CONN(pArConn));

    /*
     * Synchronization of data pipe and delete connection (control pipe)
     *
     * - Host will send a WDC Flush prior to sending wdcDeleteConnection
     * - Host will also guarantee not to issue transmits to the
     *   connection ID that is in the process of being deleted by the target.
     */

    if (pArConn->swRetryQueue.qFrameCount) {
        extern void resetTxQueue(AR_DEV_INFO *, AR_QUEUE_INFO *);
        isrPrintf("%s:%d Resetting Retry Queue!\n", __FILE__, __LINE__);
        resetTxQueue(pArDev, &pArConn->swRetryQueue);
    }

    /* Free connection resources */
    A_SEM_DELETE(pArConn->swRetryQueue.qSem);

    /* Connection is available for re-use */
    A_MEM_ZERO(pArConn, sizeof(*pArConn));
    pArConn->bInUse = FALSE;
}


void
arDeleteAllConnections(TARGET_HANDLE targetHandle)
{
    AR_DEV_INFO *pArDev   = (AR_DEV_INFO *) targetHandle;
    int iTmp;

    for (iTmp = 0; iTmp < AR_MAX_CONN; iTmp++) {
        if (!IS_FREE_AR_CONN(&pArDev->connTable[iTmp]))
            TARG_wdcDeleteConnection(targetHandle, iTmp);
    }

    return;
}


/*******************************************************************************
 *
 *  Functions local to this module.
 *
 */
