/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/arDesc.c#3 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 * Functions to operate on target descriptor queues (AR_QUEUE_INFO/AR_DESC)
 */

#include "arDev.h"
#include "arDesc.h"

#if ECOS
#define _LOCK(s,t) A_TASK_LOCK()
#define _UNLOCK(s) A_TASK_UNLOCK()
#else
#define _LOCK(s,t) A_SEM_LOCK((s), (t))
#define _UNLOCK(s) A_SEM_UNLOCK(s);
#endif


/**************************************************************************
 * descQInit - initialize the data structure corresponding to a descriptor
 *  queue
 *
 * RETURNS: success or failure
 */
A_STATUS
arDescQInit(AR_QUEUE_INFO *pQueue)
{
    A_MEM_ZERO(pQueue, sizeof(AR_QUEUE_INFO));   
    A_SEM_INIT(pQueue->qSem, 0, CREATE_UNLOCKED);

    return A_SEM_VALID(pQueue->qSem) ? A_OK : A_ERROR;
}

/**************************************************************************
 * descQGetHead - get pointer to first descriptor on the queue,
 *                don't remove it
 *
 * RETURNS: AR_DESC *
 */
AR_DESC *
arDescQGetHead(AR_QUEUE_INFO *pQueue)
{
    return pQueue->pDescQueueHead;
}

/**************************************************************************
 * descQGetTail - Get pointer to last descriptor on the queue,
 *                don't remove it
 *
 * RETURNS: AR_DESC *
 */
AR_DESC *
arDescQGetTail(AR_QUEUE_INFO *pQueue)
{
    return pQueue->pDescQueueTail;
}

/**************************************************************************
* descQPopHead - get a descriptor from the given queue
*
* RETURNS: AR_DESC *
*/
AR_DESC *
arDescQPopHead(AR_QUEUE_INFO *pQueue)
{
    AR_DESC *pDesc;

    _LOCK(pQueue->qSem, WAIT_FOREVER);
    pDesc = pQueue->pDescQueueHead;
    if (pDesc) {
        pQueue->pDescQueueHead = pDesc->pNextVirtPtr;

        /*
         * See if the new head is the same as the one we just popped;
         * this should only occur if we just popped the tail desc of
         * a self-linked queue.
         */
        if (pQueue->pDescQueueHead == pDesc) {
            ASSERT(pQueue->isTailSelfLinked);
            pQueue->pDescQueueHead = NULL;
        }

        if (pQueue->pDescQueueHead == NULL) {
            pQueue->pDescQueueTail = NULL;
        }

        pDesc->pNextVirtPtr = NULL;
        pDesc->nextPhysPtr = 0;
    }
    _UNLOCK(pQueue->qSem);

    return pDesc;
}

/**************************************************************************
* descQPushTail - add the descriptor to the tail of queue struct
*
* RETURNS: N/A
*/
void
arDescQPushTail(AR_QUEUE_INFO *pQueue, AR_DESC *pDesc, AR_DESC *pTail)
{
    ASSERT(pDesc);
    ASSERT(pQueue->pDescQueueTail != pDesc);

    _LOCK(pQueue->qSem, WAIT_FOREVER);
    if (pQueue->isTailSelfLinked) {
        pTail->nextPhysPtr  = pTail->thisPhysPtr;
        pTail->pNextVirtPtr = pTail;
    } else {
        pTail->nextPhysPtr  = 0;
        pTail->pNextVirtPtr = NULL;
    }

    /* check to see if anything is in the queue */
    if (pQueue->pDescQueueTail == NULL) {
        ASSERT(pQueue->pDescQueueHead == NULL);
        pQueue->pDescQueueHead = pDesc;
    } else {
        pQueue->pDescQueueTail->pNextVirtPtr = pDesc;
        pQueue->pDescQueueTail->nextPhysPtr = pDesc->thisPhysPtr;
    }
    pQueue->pDescQueueTail = pTail;
    _UNLOCK(pQueue->qSem);

    return;
}

/**************************************************************************
* descQPushHead - add the descriptor to the head of queue struct
*                 (non conventional!)
*
* RETURNS: N/A
*/
void
arDescQPushHead(AR_QUEUE_INFO *pQueue, AR_DESC *pDesc)
{
    ASSERT(pDesc);
    ASSERT(pQueue->pDescQueueHead != pDesc);
    ASSERT(!pQueue->isTailSelfLinked);

    /* check to see if anything is in the queue */
    _LOCK(pQueue->qSem, WAIT_FOREVER);
    if (pQueue->pDescQueueHead == NULL) {
        ASSERT(pQueue->pDescQueueTail == NULL);
        pQueue->pDescQueueTail = pDesc;
    } else {
        pDesc->pNextVirtPtr = pQueue->pDescQueueHead;
        pDesc->nextPhysPtr  = pQueue->pDescQueueHead->thisPhysPtr;
    }
    pQueue->pDescQueueHead = pDesc;
    _UNLOCK(pQueue->qSem);

    return;
}


/**************************************************************************
 * arMemAllocateDescriptor - allocate a single descriptor from the
 *  given descriptor queue
 *
 * RETURNS: AR_DESC *
 */
AR_DESC *
arMemAllocateDescriptor(AR_QUEUE_INFO *pQueue)
{
    AR_DESC *pDesc;

    pDesc = arDescQPopHead(pQueue);

    if (pDesc) {
        /* clean up the descriptor */
        AR_DESC_INIT_SW_AREA(pDesc);
        AR_DESC_INIT_HW_AREA(pDesc);
    }

    return pDesc;
}

/**************************************************************************
* arMemFreeDescriptor - free a descriptor
*
* This routine will free a descriptor that was already created by
* arMemAllocateDescriptor().
*
* RETURNS: N/A
*/
void
arMemFreeDescriptor(AR_QUEUE_INFO *pQueue, AR_DESC *pDesc)
{
    pDesc->nextPhysPtr = 0;
    pDesc->pNextVirtPtr = NULL;

    arDescQPushTail(pQueue, pDesc, pDesc);

    return;
}
