/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/arDev.c#3 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 */

#include "wlantype.h"
#include "wdcApi.h"
#include "wdcMsg.h"
#include "arMsgApi.h"
#include "arDev.h"
#include "arReceive.h"
#include "arTransmit.h"
#include "arPower.h"
#include "halApi.h"
#include "hal.h"
#include "queue.h"
#include "targWdc.h"
#include "arEvent.h"
#include "arConn.h"
#include "cyg/hal/plf_misc.h"

#include "targTransport.h"

extern A_STATUS halProbe(DEVICE_DATA, DEVICE_HANDLE *);
extern A_STATUS TARG_wdcCtrlMsgDispatchInit(MASTER_HANDLE masterHandle);

extern A_STATUS TARG_deviceInit(DEVICE_HANDLE  deviceHandle);

LOCAL void
arConfigGpio(AR_DEV_INFO     *pArDev);

LOCAL void
arDeconfigGpio(AR_DEV_INFO     *pArDev);

/* We only support one master per target. */
MASTER_HANDLE theMasterHandle;

/*
 * This is essentially the initialization routine for the target WDC.
 * It is called when the target-side software first discovers a device
 * that can serve as a target.
 */
void
TARG_wdcDeviceDiscovered(DEVICE_DATA devData, A_STATUS *pStatus)
{
    A_STATUS          status;
    PER_MASTER_INFO   *masterHandle;
    MASTER_MSG_HANDLE masterMsgHandle;
    DEVICE_HANDLE     deviceHandle;
    PLATFORM_HANDLE   plfHandle;
    /*
     * Allocate a structure that we'll use to identify the "master",
     * which is the host that controls us.
     */
    masterHandle = (PER_MASTER_INFO *)A_DRIVER_MALLOC(sizeof(PER_MASTER_INFO));
    if (!masterHandle) {
        status = A_NO_MEMORY;
        goto discError;
    }
    A_MEM_ZERO(masterHandle, sizeof(*masterHandle));

    /*
     * Verify that this is a device we understand, and collect
     * some basic information about the device and its capabilities.
     * The drive returns a DEVICE_HANDLE, which we pass back to it
     * whenever it needs to do anything.
     */
    status = halProbe(devData, &deviceHandle);
    if (status != A_OK) {
        goto discError;
    }
    masterHandle->deviceHandle = deviceHandle;
                                                                       
    /* Initialize the discovered device */
    status = TARG_deviceInit(deviceHandle);
    if (status != A_OK) {
        goto discError;
    }
    
    /* Initialize the Control Message Dispatcher. */
    status = TARG_wdcCtrlMsgDispatchInit(masterHandle);
    if (status != A_OK) {
        goto discError;
    }

    /* Initialize the Async Event Dispatcher. */
    status = TARG_wdcAsyncEvtDispatchInit(masterHandle);
    if (status != A_OK) {
        goto discError;
    }

    /* Initialize messaging to the master. */
    masterMsgHandle = TARG_wdcMsgInit(masterHandle);
    if (!masterMsgHandle) {
        status = A_NO_MEMORY;
        goto discError;
    }

    masterHandle->masterMsgHandle = masterMsgHandle;
    plfHandle = TARG_platformInit(masterHandle, masterHandle->masterMsgHandle);
    if (!masterMsgHandle) {
        status = A_NO_MEMORY;
        goto discError;
    }
    
    masterHandle->platformHandle = plfHandle;

    theMasterHandle = masterHandle;

    goto discExit;

discError:
    ASSERT(0);

discExit:

    if (pStatus) {
        *pStatus = status;
    }
}


A_STATUS 
TARG_deviceInit(
    DEVICE_HANDLE     deviceHandle
    )
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *)deviceHandle;
    A_STATUS        status;

    ASSERT(pArDev);

    arMacInitPsMode(pArDev);
   
    if (arMacSetPowerMode(pArDev, AR_MAC_AWAKE_MODE) != A_OK) {
        status = A_HARDWARE;
        goto TargetInitExit;
    }
    status = halDevInit(pArDev);
    if (status != A_OK) {
        goto TargetInitExit;
    }
    status = halReset(pArDev, CURRENT_SERVICE_TYPE(pArDev), CURRENT_PCHDESC(pArDev), TRUE);
    if (status != A_OK) {
        goto TargetInitExit;
    }

    /* 
     * In order to save power, force the chip to sleep until the
     * master expresses its interest via a wdcTargetStart.
     */
    if (arMacSetPowerMode(pArDev, AR_MAC_SLEEP_MODE) != A_OK) {
        status = A_HARDWARE;
        goto TargetInitExit;
    }

    
TargetInitExit:
    return status;
}



void
TARG_wdcDeviceRemoved(
    DEVICE_DATA      devData,
    A_STATUS         *pStatus
)
{
    MASTER_MSG_HANDLE masterMsgHandle;

    /*
     * Determine which device has been removed.
     * It's easy when there's only one.
     */
    masterMsgHandle = theMasterHandle->masterMsgHandle;

    /*
     * Send a WDCMSG_DEVICE_UNAVAILABLE message to the Host.
     */
    arNotifyDeviceUnavailable(theMasterHandle);
}


/*
 * When a host-side driver decides to manage this target,
 * it binds to the target.
 */
void
TARG_wdcBind(
    MASTER_HANDLE       masterHandle,
    DEVICE_HANDLE       deviceHandle,
    A_UINT32            hostApiVersion,
    A_STATUS            *pStatus
)
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *)deviceHandle;
    A_STATUS        status;

    ASSERT(pArDev);
    ASSERT(masterHandle == theMasterHandle);

    /* Check host API version and verify that it's OK to bind */
    /* Perhaps we want to save the host API version in pArDev, too. */

    pArDev->masterHandle = masterHandle;

    status = A_OK;

    *pStatus = status;
}

void 
TARG_wdcUnbind(
    MASTER_HANDLE masterHandle,
    DEVICE_HANDLE deviceHandle
)
{
    ASSERT(masterHandle == theMasterHandle);
    return;
}

/*
 * After the host understand capabilities of this target and
 * after the host has configured this target as desired, it
 * starts the device.
 */
void
TARG_wdcTargetStart(
    DEVICE_HANDLE       deviceHandle,
    A_STATUS            *pStatus
)
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *)deviceHandle;
    A_STATUS        status;

    ASSERT(pArDev);

    status = arMacSetPowerMode(pArDev, AR_MAC_AWAKE_MODE);
    if (status != A_OK) {
        goto TargetStartExit;
    }

    /*
     * If this is the first time that this device has been started,
     * perform one-time initialization.  (Otherwise, the device
     * must have been stopped and we're now restarting it.)
     */
    if (!pArDev->oneTimeInitDone) {
        pArDev->oneTimeInitDone = TRUE;
        /* One-time Receive path initializations */

        status = arRxInit(pArDev);
        if (status != A_OK) {
            ASSERT(0);
            goto TargetStartExit;
        }

        /* One-time Transmit path initializations */
        status = arTxInit(pArDev);
        if (status != A_OK) {
            ASSERT(0);
            goto TargetStartExit;
        }
        pArDev->oneTimeInitDone = TRUE;
    }

    TARG_wdcTargetEnable(deviceHandle, &status);
    if (status != A_OK) {
        goto TargetStartExit;
    }
    
    /* Device is started.  Do not allow low power Board suspend. */
    pArDev->bUSBPowerSave = FALSE;

TargetStartExit:
    if (pStatus) {
        *pStatus = status;
    }
}


void
TARG_wdcTargetStop(
    DEVICE_HANDLE       deviceHandle,
    A_STATUS            *pStatus
)
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *)deviceHandle;
    A_STATUS        status = A_OK;

    ASSERT(pArDev);

    if (pArDev->bTargetStarted == TRUE) {

        TARG_wdcTargetDisable(deviceHandle, &status);
        if (status != A_OK) {
            goto TargetStopExit;
        }

        arDeleteAllConnections((TARGET_HANDLE) pArDev);

        if (arMacSetPowerMode(pArDev, AR_MAC_SLEEP_MODE) != A_OK) {
            ASSERT(0);
        }
    }

TargetStopExit:
    /* Windows Device Disable.  Do not allow Low Power Board Suspend. */
    pArDev->bUSBPowerSave = FALSE;

    if (!pStatus) {
        *pStatus = status;
    }
}

void
TARG_wdcTargetEnable(
    DEVICE_HANDLE       deviceHandle,
    A_STATUS            *pStatus
)
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *)deviceHandle;
    A_STATUS        status;

    ASSERT(pArDev);

    status = arMacSetPowerMode(pArDev, AR_MAC_AWAKE_MODE);
    if (status != A_OK) {
        goto TargetEnableExit;
    }

    arStartReceive(pArDev);
    arStartTransmit(pArDev);

    A_TIMEOUT(&pArDev->autoNoiseImmunityTimer,
              100 * 2,  // Every 2 seconds
              TRUE);

    if (pArDev->config.calibrationInterval != 0) {
        A_TIMEOUT(&pArDev->periodicCalTimer, 
                  100 * pArDev->config.calibrationInterval, TRUE);
    }

#if defined(TRANSPORT_POLL)
    TARG_txportPollEnable(pArDev->masterHandle->masterMsgHandle);
#endif

    halEnableInterrupts(pArDev, HAL_INT_GLOBAL);
    arConfigGpio(pArDev);
    pArDev->bTargetStarted = TRUE;
    pArDev->powerMgmt.powerState    = TARGET_DEVICE_AWAKE;

TargetEnableExit:

    /* Device is enabled.  Do not allow low power Board suspend. */
    pArDev->bUSBPowerSave = FALSE;

    if (pStatus) {
        *pStatus = status;
    }
}

void
TARG_wdcTargetDisable(
    DEVICE_HANDLE       deviceHandle,
    A_STATUS            *pStatus
)
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *)deviceHandle;

    ASSERT(pArDev);

    if (pArDev->bTargetStarted == TRUE) {
        if (arMacSetPowerMode(pArDev, AR_MAC_AWAKE_MODE) != A_OK) {
            ASSERT(0);
        }

        halDisableInterrupts(pArDev, HAL_INT_GLOBAL);
        
        A_UNTIMEOUT(&pArDev->periodicCalTimer);
        A_UNTIMEOUT(&pArDev->autoNoiseImmunityTimer);
        TARG_wdcTxMsgFlushqueue(deviceHandle);
        TARG_wdcAsyncEvtFlushqueue(deviceHandle);

#if !defined(AR_POLL)
        arFlushTxCompleteMsg(pArDev);
#endif

        arStopReceive(pArDev);
        arStopTransmit(pArDev, DO_WAIT);
        arDeconfigGpio(pArDev);

#if defined(TRANSPORT_POLL)
        TARG_txportPollDisable(pArDev->masterHandle->masterMsgHandle);
#endif
        
        pArDev->bTargetStarted = FALSE;
    }
    /* Host standby. Allow Low Power Board Suspend. */
    pArDev->bUSBPowerSave = TRUE;
    if (pStatus) {
        *pStatus = A_OK;
    }
}

void
TARG_wdcHostAvailable(
    MASTER_HANDLE        masterHandle,
    DEVICE_HANDLE        deviceHandle,
    A_STATUS            *pStatus
) 
{

    /* Ensure the target is stopped and in
     * an available state prior to sending 
     * DEVICE_AVAILABLE */

    TARG_wdcTargetStop(deviceHandle, pStatus);
    ASSERT(*pStatus == A_OK);


    /* We delete connections here again - because the target
     * will not see a TargetStop() -- but rather a TargetDisable()
     * during a reboot.
     */
    arDeleteAllConnections((TARGET_HANDLE) deviceHandle);
        
    arNotifyDeviceAvailable(masterHandle);

    if (pStatus)
        *pStatus = A_OK;

    return;
}

void
TARG_wdcBmissAck(
      IN  DEVICE_HANDLE     deviceHandle, 
      IN  A_BOOL            bEnable)
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *)deviceHandle;
    A_STATUS        status;

    ASSERT(pArDev);

    /* Restore power to Awake state is sleeping */
    status = arMacSetAwakeMode(pArDev, TRUE);
    ASSERT(status == A_OK);

    if (bEnable) {
        halEnableInterrupts(pArDev, HAL_INT_BMISS);
    } else {
        halDisableInterrupts(pArDev, HAL_INT_BMISS);
    }

    /* Restore power to former or desired state */
    status = arMacSetAwakeMode(pArDev, FALSE);
    ASSERT(status == A_OK);
}

#define LED_GOTO_GPIO1   17

LOCAL void
arConfigGpio(AR_DEV_INFO     *pArDev)
{
    pArDev->gpioFunc[0].enabled = TRUE;
    pArDev->gpioFunc[0].pin = 1;
    
    hal_ar5523_set_gpio_config(LED_GOTO_GPIO1, FALSE);
    hal_ar5523_set_gpio_config(pArDev->gpioFunc[0].pin, TRUE);
}

LOCAL void
arDeconfigGpio(AR_DEV_INFO     *pArDev)
{
    pArDev->gpioFunc[0].enabled = FALSE; 

    hal_ar5523_set_gpio_config(LED_GOTO_GPIO1, TRUE);
    hal_ar5523_set_gpio_config(pArDev->gpioFunc[0].pin, FALSE);
}


void
TARG_wdcSetLedSteadyMode(
    IN  DEVICE_HANDLE   deviceHandle, 
    A_UINT32            ledNumber,
    A_UINT32            mode)
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *)deviceHandle;
    A_STATUS        status;

    ASSERT(ledNumber < 4);
    ASSERT(mode < 2);

    /* Restore power to Awake state is sleeping */
    status = arMacSetAwakeMode(pArDev, TRUE);
    ASSERT(status == A_OK);

    /* use gpio_out to control led */
    hal_ar5523_set_gpio_config( (ledNumber + 16), TRUE); 
    hal_ar5523_set_gpio_config( ledNumber, TRUE);

    /* mode=1 to turn on Led, mode=0 to turn off Led */
    hal_ar5523_set_gpio( ledNumber, mode);

    /* Restore power to former or desired state */
    status = arMacSetAwakeMode(pArDev, FALSE);
    ASSERT(status == A_OK);
}

extern A_UINT32 Current_Led_Mode;
extern A_UINT32 Current_Led_Blink_Threshold;
extern A_UINT32 Current_Led_Slow_Mode;


void
TARG_wdcSetLedBlinkMode(
    IN  DEVICE_HANDLE   deviceHandle, 
    A_UINT32            ledNumber,
    A_UINT32            ledMode,
    A_UINT32            blinkRate,
    A_UINT32            slowMode)
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *)deviceHandle;
    A_STATUS        status;

    /* Only power LED(gpio_0) and network LED(gpio_1) can use h/w blink mode */
    ASSERT(ledNumber < 2);

    /* Restore power to Awake state if sleeping */
    status = arMacSetAwakeMode(pArDev, TRUE);
    ASSERT(status == A_OK);

    Current_Led_Mode= ledMode; 
    Current_Led_Blink_Threshold= blinkRate;
    Current_Led_Slow_Mode= slowMode;

    /* don't use gpio_out to control led */
    hal_ar5523_set_gpio_config( (ledNumber + 16), FALSE); 
    hal_ar5523_set_gpio_config( ledNumber, TRUE);

    A_REG_RMW_FIELD(pArDev, MAC_PCICFG, LED_MODE, Current_Led_Mode);
    A_REG_RMW_FIELD(pArDev, MAC_PCICFG, LED_BLINK_THRESHOLD, Current_Led_Blink_Threshold);
    A_REG_RMW_FIELD(pArDev, MAC_PCICFG, LED_SLOW_BLINK_MODE, Current_Led_Slow_Mode);

    /* Restore power to former or desired state */
    status = arMacSetAwakeMode(pArDev, FALSE);
    ASSERT(status == A_OK);
}

