/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/arEvent.c#3 $
 *
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 */

#include "wlantype.h"
#include "wdcApi.h"
#include "wdcMsg.h"
#include "queue.h"
#include "arMsgApi.h"
#include "arDev.h"
#include "arMsgApi.h"
#include "targWdc.h"
#include "masterInfo.h"
#include "arReceive.h"
#include "arTransmit.h"
#include "arEvent.h"
#include "ecosdrv.h"

#include "athusbdrv.h"
#include "targTransport.h"
#include "targMsg.h"

/* Standard HAL Headers */
#include "halApi.h"
#include "hal.h"


#define ASYNC_OUTSTANDING_COUNT  2


LOCAL void arCalibration(AR_DEV_INFO *pArDev);
LOCAL void arAutoNoiseImmunity(AR_DEV_INFO *pArDev);

A_STATUS
TARG_wdcAsyncEvtEnqueue(
    IN MASTER_HANDLE  masterHandle,
    IN A_UINT32       event,
    IN void          *pOpaque,
    IN A_BOOL         bWait
)
{
    struct AsyncEvtQ *msgQItem;

    if (bWait) {
        A_CNTSEM_WAIT(masterHandle->AsyncEvtQFreeSem);
    } else {
        if (A_CNTSEM_TRYWAIT(masterHandle->AsyncEvtQFreeSem) != TRUE)
            return(A_NO_MEMORY);
    }

	A_TASK_LOCK();

    msgQItem = TAILQ_FIRST(&masterHandle->AsyncEvtQFree);
    ASSERT(msgQItem);

    TAILQ_REMOVE(&masterHandle->AsyncEvtQFree, msgQItem, Qlinkage);
    ASSERT(msgQItem);
    ASSERT(msgQItem->event    == (A_UINT32) -1);
    ASSERT(msgQItem->opaque   == NULL);

    msgQItem->event    = event;
    msgQItem->opaque   = pOpaque;    // TBD

    TAILQ_INSERT_TAIL(&masterHandle->AsyncEvtQHead, msgQItem, Qlinkage);

	A_TASK_UNLOCK();

    A_CNTSEM_POST(masterHandle->AsyncEvtQSem);

    return A_OK;
}

LOCAL WDC_EVENT
TARG_wdcAsyncEvtDequeue(
    IN  MASTER_HANDLE   masterHandle,
    OUT void          **ppOpaque
)
{
    struct AsyncEvtQ   *msgQItem;
    A_UINT32            event;

    A_CNTSEM_WAIT(masterHandle->AsyncEvtQSem);

	A_TASK_LOCK();

    msgQItem = TAILQ_FIRST(&masterHandle->AsyncEvtQHead);
    ASSERT(msgQItem);
    TAILQ_REMOVE(&masterHandle->AsyncEvtQHead, msgQItem, Qlinkage);

    event = msgQItem->event;
    if (ppOpaque) {
        *ppOpaque = msgQItem->opaque;
    }

    msgQItem->event    = (A_UINT32) -1;
    msgQItem->opaque   = NULL;

    TAILQ_INSERT_TAIL(&masterHandle->AsyncEvtQFree, msgQItem, Qlinkage);
    A_CNTSEM_POST(masterHandle->AsyncEvtQFreeSem);

	A_TASK_UNLOCK();

    return event;
}

void
TARG_wdcAsyncEvtFlushqueue(
    AR_DEV_INFO     *pArDev
    )
{
    int             semcount   = 0;
    void            *pOpaque;
    WDC_EVENT       wdcEvent;
    MASTER_HANDLE   masterHandle;

    masterHandle = pArDev->masterHandle;
    do {
        A_CNTSEM_PEEK(masterHandle->AsyncEvtQSem, &semcount);
        if (semcount) {
            wdcEvent = TARG_wdcAsyncEvtDequeue(masterHandle, &pOpaque);
        }
    } while (semcount);
}

LOCAL void
TARG_wdcAsyncEventLoop(MASTER_HANDLE masterHandle, 
                       A_UINT32  event,
                       void      *pOpaque) 
{
    WDC_MSG     *wdcMsg = NULL;
    AR_DEV_INFO *pArDev = masterHandle->deviceHandle;

    switch(event) {

    case WDCEVT_WLAN_STATS_UPDATE: {

        wdcMsg = TARG_wdcCtrlMsgCreate(masterHandle->masterMsgHandle, 
                                       WDCMSG_TARGET_STATS_UPDATE);
        ASSERT(wdcMsg);

        TARG_wdcMsgAddData(wdcMsg, sizeof(pArDev->devStats),
                            &pArDev->devStats);

        TARG_wdcCtrlMsgSend(masterHandle->masterMsgHandle, wdcMsg);
        ASSERT(pOpaque == NULL);

        break;
    }

    case WDCEVT_CONNECTION_STATS_UPDATE:
        ASSERT(0);
        break;

    case WDCEVT_BEACON_MISS: {
        halMibControl(pArDev, UPDATE_SW_ALL);

        wdcMsg = TARG_wdcCtrlMsgCreate(masterHandle->masterMsgHandle, 
                                       WDCMSG_BEACON_MISS_INTERRUPT);

        TARG_wdcMsgAddData(wdcMsg, sizeof(A_LONGSTATS),
                            &pArDev->devStats.RxBeacons);

        ASSERT(wdcMsg);
        TARG_wdcCtrlMsgSend(masterHandle->masterMsgHandle, wdcMsg);
        ASSERT(pOpaque == NULL);
        break;
    }
    case WDCEVT_STORE_INFO:
    case WDCEVT_RETRIEVE_INFO:
    case WDCEVT_WLAN_BEACON_UPDATE:
    case WDCEVT_ADHOC_MERGE_UPDATE:
        ASSERT(0);
        break;

    case WDCEVT_DEVICE_AVAILABLE: {
        wdcMsg = TARG_wdcCtrlMsgCreate(masterHandle->masterMsgHandle, 
                                       WDCMSG_DEVICE_AVAILABLE);
        ASSERT(wdcMsg);
        TARG_wdcCtrlMsgSend(masterHandle->masterMsgHandle, wdcMsg);
        ASSERT(pOpaque == NULL);
        break;
    }
    case WDCEVT_DEVICE_UNAVAILABLE:
        wdcMsg = TARG_wdcCtrlMsgCreate(masterHandle->masterMsgHandle, 
                                       WDCMSG_DEVICE_UNAVAILABLE);
        ASSERT(wdcMsg);
        TARG_wdcCtrlMsgSend(masterHandle->masterMsgHandle, wdcMsg);
        ASSERT(pOpaque == NULL);
        break;
    case WDCEVT_DEBUG_MESSAGE_NOTIFY:
    case WDCEVT_DATA_INDICATION:
    case WDCEVT_SEND_COMPLETE:
    case WDCEVT_STATUS_NOTIFY:
    case WDCEVT_RX_IN_SYNC:
    case WDCEVT_FLUSH_COMPLETE:
        ASSERT(0);
        break;
    case AREVT_PERIODIC_CAL:
        arCalibration(masterHandle->deviceHandle);
        break;
    case AREVT_PERIODIC_ANI:
        arAutoNoiseImmunity(masterHandle->deviceHandle);
        break;
        
    default:
        ASSERT(0);
        break;
    }
    return;
}

LOCAL void
TARG_wdcAsyncEvtDispatchLoop(MASTER_HANDLE masterHandle)
{
    WDC_EVENT    wdcEvent;
    AR_DEV_INFO *pArDev = masterHandle->deviceHandle;
    void        *pOpaque;
    A_STATUS     status;

    for(;;) {
        wdcEvent = TARG_wdcAsyncEvtDequeue(masterHandle, &pOpaque);
        A_SEM_LOCK(pArDev->asyncEvtLock, WAIT_FOREVER);

        /* Restore power to Awake state if sleeping */
        status = arMacSetAwakeMode(pArDev, TRUE);
        ASSERT(status == A_OK);
        
        TARG_wdcAsyncEventLoop(masterHandle, wdcEvent, pOpaque);
        
        /* Restore power to former or desired state */
        status = arMacSetAwakeMode(pArDev, FALSE);
        ASSERT(status == A_OK);
        A_SEM_UNLOCK(pArDev->asyncEvtLock);
        
    }
}




#if defined(AR_POLL)
LOCAL void
arCompleteLoop(MASTER_HANDLE masterHandle) {
    AR_DEV_INFO *pArDev = masterHandle->deviceHandle;
    A_STATUS     status;
    int          iTmp;
    int          iRxLock = 0;
    int          iTxLock = 0;

    for (;;) {

        if (pArDev->bTargetStarted == FALSE) {
            /* If we've been suspended/disabled, wait until
             * we're started again.  In this instance,
             * we use the rxCompleteLock */
            do {
                A_SEM_LOCK(pArDev->rxCompleteLock, WAIT_FOREVER);
                A_SEM_UNLOCK(pArDev->rxCompleteLock);
            } while (pArDev->bTargetStarted == FALSE);
       }

        if (pArDev->oneTimeInitDone) {
            
            A_SEM_LOCK(pArDev->pwrLock, WAIT_FOREVER);
            if (pArDev->powerMgmt.powerState == TARGET_DEVICE_PWRSAVE) {
                udelay(50000);
            }
            /* Restore power to Awake state if sleeping */
            status = arMacSetAwakeMode(pArDev, TRUE);
            ASSERT(status == A_OK);

            if (A_SEM_TRYLOCK(pArDev->rxCompleteLock)) {
                iRxLock = 1;
            }

            if (A_SEM_TRYLOCK(pArDev->txCompleteLock))  {
                iTxLock = 1;
            }

#define AR_COMPLETE_LOOP_POLL_COUNT 100
            if (iRxLock && iTxLock) {

                for (iTmp = 0; iTmp < AR_COMPLETE_LOOP_POLL_COUNT; iTmp++) {

                    ASSERT(pArDev->powerMgmt.powerState == TARGET_DEVICE_AWAKE);
                    arRxCompleteHandler(pArDev);
#if defined(TRANSPORT_POLL)
                    /* XXX: This needs to move under it's own transportLock */
                    TARG_txportPoll(pArDev->masterHandle->masterMsgHandle);
#endif
                    arTxCompleteHandler(pArDev);
#if defined(TRANSPORT_POLL)
                    /* XXX: This needs to move under it's own transportLock */
                    TARG_txportPoll(pArDev->masterHandle->masterMsgHandle);
#endif
                }
            }

            if (iRxLock) {
                A_SEM_UNLOCK(pArDev->rxCompleteLock);
                iRxLock = 0;
            }

            if (iTxLock) {
                A_SEM_UNLOCK(pArDev->txCompleteLock);
                iTxLock = 0;
            }

            /* Restore power to former or desired state */
            status = arMacSetAwakeMode(pArDev, FALSE);
            ASSERT(status == A_OK);
            A_SEM_UNLOCK(pArDev->pwrLock);
        }
        
        A_TASK_YIELD();
    }
}
#endif


#if defined(ECOS)
LOCAL void
arPeriodicCalTimerHandler(cyg_handle_t   alarm_handle,
                          cyg_addrword_t data) {
    MASTER_HANDLE masterHandle = (MASTER_HANDLE) data;
    arNotifyPeriodicCal(masterHandle);
    return;
}

LOCAL void
arAutoNoiseImmunityTimerHandler(cyg_handle_t   alarm_handle,
                                cyg_addrword_t data) {
    MASTER_HANDLE masterHandle = (MASTER_HANDLE) data;

    if (masterHandle->deviceHandle->arDoAni) {
        arNotifyPeriodicAni(masterHandle);
    }
    return;
}

LOCAL void
arTimerInit(MASTER_HANDLE masterHandle) {
    AR_DEV_INFO *pArDev = (AR_DEV_INFO *) masterHandle->deviceHandle;
    
    cyg_clock_to_counter(cyg_real_time_clock(),
                         &pArDev->rtcCounterHandle);

    A_INIT_TIMER(pArDev, &pArDev->periodicCalTimer,
                 arPeriodicCalTimerHandler,
                 masterHandle);

    A_INIT_TIMER(pArDev, &pArDev->autoNoiseImmunityTimer,
                 arAutoNoiseImmunityTimerHandler,
                 masterHandle);


    A_TIMEOUT(&pArDev->autoNoiseImmunityTimer,
              100 * 2,  // Every 2 seconds
              TRUE);

#if 0
    A_TIMEOUT(&pArDev->periodicCalTimer,
              100 * 30,
              TRUE);
#endif

    return;
}
#endif


A_STATUS
TARG_wdcAsyncEvtDispatchInit(MASTER_HANDLE masterHandle) 
{
    A_STATUS     status = A_OK;
    AR_DEV_INFO *pArDev = masterHandle->deviceHandle;
    int          iTmp; 

    TAILQ_INIT(&masterHandle->AsyncEvtQHead);
    TAILQ_INIT(&masterHandle->AsyncEvtQFree);
    A_CNTSEM_INIT(masterHandle->AsyncEvtQSem, 0);
    A_CNTSEM_INIT(masterHandle->AsyncEvtQFreeSem, 0);

    for (iTmp = 0; iTmp < ASYNC_OUTSTANDING_COUNT; iTmp++) {
        struct AsyncEvtQ   *msgQItem;

        msgQItem = A_DRIVER_MALLOC(sizeof(*msgQItem));
        if (!msgQItem) {
            ASSERT(0);
            status = A_NO_MEMORY;
            goto msgQInitExit;
        }
        A_MEM_ZERO(msgQItem, sizeof(*msgQItem));

        msgQItem->event    = (A_UINT32) -1;
        msgQItem->opaque   = NULL;

        TAILQ_INSERT_TAIL(&masterHandle->AsyncEvtQFree, msgQItem, Qlinkage);
        A_CNTSEM_POST(masterHandle->AsyncEvtQFreeSem);
    }
    
    A_SEM_INIT(pArDev->asyncEvtLock, 0, CREATE_UNLOCKED);
    
#if defined(AR_POLL)
    A_TASK_CREATE_STACKSZ(arCompleteLoop, masterHandle, 2048);
#endif

#if defined(AGGRESSIVE_TASK_STACKSZ)
    // Measured usage on 05/24/2004: 704
    A_TASK_CREATE_STACKSZ(TARG_wdcAsyncEvtDispatchLoop, masterHandle, 1024);
#else
    A_TASK_CREATE(TARG_wdcAsyncEvtDispatchLoop, masterHandle);
#endif

    arTimerInit(masterHandle);

 msgQInitExit:
    return(status);
}






#if defined(ECOS)

A_UINT32
ar_wmac_isr(A_UINT32 IntVector, A_UINT32 intrHandle) {
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *) intrHandle;
    A_UINT32         retcode = CYG_ISR_HANDLED;
    HAL_INT_TYPE     isrValue;
    HAL_INT_TYPE     unmaskedIsrValue;
    A_UINT32         descQueueBitMask = 0;


    A_DISABLE_INTERRUPT(IntVector);

    if (!halIsInterruptPending(pArDev))
        goto done;
    
    isrValue = halGetInterrupts(pArDev, &unmaskedIsrValue, 
                                   &descQueueBitMask);

#if !defined(ECOS_NOTDONE)
    /*
     * Do this after the call to halGetInterrupts() so that we clear any
     * interrupt condition, even in a zombie state.
     */
    if (!pDev->pOSHandle->openForBusiness) {
        return;
    }

    /* Check for suprise removal of our card */
    if (isrValue == HAL_INT_NOCARD) {
        uiPrintf("%s: Card Removed!\n", __FUNCTION__);
        *pIsRecognized = FALSE;
        return;
    }

    if (isrValue & HAL_INT_GPIO) {
        /* rfkill interrupt from GPIO: flip the polarity of the GPIO interrupt */
        if (halGpioGet(pDev, pDev->rfSilent.gpioSelect) == pDev->rfSilent.polarity) {
            halGpioSetIntr(pDev, pDev->rfSilent.gpioSelect, !pDev->rfSilent.polarity);
        } else {
            halGpioSetIntr(pDev, pDev->rfSilent.gpioSelect, pDev->rfSilent.polarity);
        }
    }
#endif
    if (isrValue & HAL_INT_RX) {
        halDisableInterrupts(pArDev, HAL_INT_RX);
    }

    if (isrValue & HAL_INT_RXORN) {
        /*
         * On some HW, RXORN will continue to assert on each new arriving frame.
         * This would keep us from ever entering our DPC, so we must disable
         * the interrupt here in the ISR to avoid this condition.
         */
        pArDev->devStats.RcvDmaOverrunErrors++;
        halDisableInterrupts(pArDev, HAL_INT_RXORN);
    }


    if (isrValue & HAL_INT_TXURN) {
        /*
         * After a frame started transmitting, the NIC was unable to get the rest of the
         * data from host memory in time before the transmitter needed it.  Adjust the tx
         * trigger up by half the difference between maximum and the current value.
         */
        uiPrintf("%s: Transmit Underrun!\n", __FUNCTION__);
        halUpdateTxTrigLevel(pArDev, TRUE);
        isrValue &= ~HAL_INT_TXURN;
    }

    if (isrValue & HAL_INT_TX) {
        halDisableInterrupts(pArDev, HAL_INT_TX);
    }


    if (isrValue & HAL_INT_BMISS) {
        halDisableInterrupts(pArDev, HAL_INT_BMISS);
    }

    if (isrValue & HAL_INT_TXDESC) {

        /* 
         * TODO :Queue Num is used to just pick the burst queue in the 
         * baseBSS eventually it has to reverse looked up to indentify 
         * the BSS it belongs to and then the burst queue of the BSS 
         * should be used here
         */

        if (descQueueBitMask & (1 << TXQ_ID_FOR_GBURST))
        {
            AR_QUEUE_INFO *pQueue;

            ASSERT(pArDev->config.abolt & ABOLT_BURST);
            pQueue = &pArDev->txHwQueues[TXQ_ID_FOR_GBURST];

            /* Check to see if we have to trigger another burst, which is 
             * decided based on pending burst counts and if there is no 
             * pending operations on the Queue 
             */
            if (!pQueue->queuingBurst) {
                if (pQueue->qBurstCount > 0) {
                    pQueue->qBurstCount -= 1;
                    ASSERT(pQueue->halTxQueueNum == TXQ_ID_FOR_GBURST);
                    halStartTxDma(pArDev, pQueue->halTxQueueNum);
                }
            }
            pQueue->queuingBurst = FALSE;
            isrValue &= ~HAL_INT_TXDESC;
        }
    }

    /*
     * We check to see if any RX or TX ints are present in the ISR but not
     * actually asserting because they are masked.  This keeps us from dropping
     * interrupts and leaving completed frames on the RX or TX queues.
     *
     * If any masked or unmasked interrupts have not been handled in the ISR
     * we instruct the OS to queue up our DPC to process the rest of the
     * interrupts at a lower IRQL.
     */
#if !defined(AR_POLL)
    unmaskedIsrValue &= HAL_INT_RX | HAL_INT_TX;
#else 
    unmaskedIsrValue = 0;
#endif
    if (isrValue || unmaskedIsrValue) {
        pArDev->globISRReg |= isrValue | unmaskedIsrValue;
        retcode = CYG_ISR_CALL_DSR | CYG_ISR_HANDLED;
    }
    
 done:
    A_ACK_INTERRUPT(IntVector);
    if (retcode == CYG_ISR_HANDLED) {
        A_ENABLE_INTERRUPT(IntVector);
    }
    return(retcode);
}

void
ar_wmac_dsr(A_UINT32 IntVector, A_UINT32 IntCount, A_UINT32 intrHandle) {
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *) intrHandle;
    HAL_INT_TYPE    isrValue, enableMask = 0;
    int             i;
    A_STATUS        status;


    /* After entering hibernate, ignore subsequent DPCs */
    if (pArDev->powerMgmt.powerState == TARGET_DEVICE_PWRDN) {
        goto intr_done;
    }
    status = arMacSetAwakeMode(pArDev, TRUE);
    ASSERT(status == A_OK);

#if !defined(ECOS_NOTDONE)
    if (!pDev->pOSHandle->openForBusiness) {
        return;
    }

#endif

    /*
     * We loop through NUM_DPC_LOOPS times to avoid some of the ISR->DPC
     * latency during heavy CPU load.
     */
#define NUM_DPC_LOOPS 1
    for (i = 0; i < NUM_DPC_LOOPS; i++) {
        /* Read and then clear the ISR value that the ISR might modify. */
        halDisableInterrupts(pArDev, HAL_INT_GLOBAL);
        isrValue           = pArDev->globISRReg;
        pArDev->globISRReg = 0;
        halEnableInterrupts(pArDev, HAL_INT_GLOBAL);

        if (!isrValue) {
            break;
        }

#undef NUM_DPC_LOOPS

#if !defined(ECOS_NOTDONE)
        if (isrValue & HAL_INT_GPIO) {
            /* GPIO interrupt received for switching RF on or off. */
            if (pArDev->rfSilent.polarity == 
                halGpioGet(pArDev, pArDev->rfSilent.gpioSelect))
            {
                /* Switch closed, turn off RF */
                uiPrintf("RF disable switch closed\n");
                athDrvRadioDisable(pArDev, &pArDev->rfSilent.hwRadioDisable);
                /* Radio is off and chips put to sleep */
            } else {
                /* switch opened, activate now. */
                uiPrintf("RF disable switch opened\n");
                athDrvRadioEnable(pArDev, &pArDev->rfSilent.hwRadioDisable);
            }
            /*
             * Either no further h/w access allowed (disable) or chip is now
             * awake to receive subsequent interrupts; time to leave
             */
            powerSet(pArDev, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
            return;
        }
#endif

        /*
         * RXORN is fatal for some HW, and merely unfortunate on others.  If it
         * is fatal, the HAL_INT_FATAL bit will be set in the value returned
         * from halGetInterrupts().
         */
        if (isrValue & HAL_INT_RXORN) {
            /*
             * RXORNs are not fatal anymore as updated in bug 9208, and can 
             * happen quite frequently on Venice/Hainan based MAC with 
             * Compression and Fast Frames Enabled 
             */ 
#if !defined(AR_POLL)
            enableMask |= HAL_INT_RXORN;
#endif
        }

#if !defined(ECOS_NOTDONE)
        /* Check for fatal interrupts */
        if (isrValue & HAL_INT_FATAL) {
            uiPrintf("athHandleInterrupt: Fatal error!  Resetting...\n");
            ResetOnError(pArDev, DONT_CLEAR_TX_Q, RESET_INC_CTR, DONT_WAIT);

            /* 
             * If master cycle abort went off, maybe returning cache line size
             * to its' original value will prevent recurrence 
             */
            uiPrintf("athHandleInterrupt: Reverting to previous cache line size\n");
            NdisWritePciSlotInformation(pArDev->pOSHandle->NicAdapterHandle,
                                        0, PCI_CACHE_LINE_REGISTER,
                                        &pArDev->staConfig.cacheLineSize, 0x1);
            powerSet(pArDev, WAKE_UP, REDUCE_PWR_FROM_3, FALSE, 0);
            return;
        }
#endif

        if (isrValue & HAL_INT_BMISS)
        {
            /*
             * We missed some beacons. We assume the AP is either down or we
             * have moved out of range. We inform connection services to take
             * appropriate action.
             */
            if (halGetPcuState(pArDev)) {
                if (arNotifyBeaconMiss(pArDev->masterHandle) != A_OK) {
                    enableMask |= HAL_INT_BMISS;
                }
            } else {
                enableMask |= HAL_INT_BMISS;
            }
        }

        if (isrValue & HAL_INT_TX) {
#if !defined(AR_POLL)
            arNotifyTxComplete(pArDev);
            /* enableMask |= HAL_INT_TX; */
#endif
        }

        if (isrValue & HAL_INT_RX) {
#if !defined(AR_POLL)
            arNotifyRxComplete(pArDev);
            /*
             * Since we use a self-linked tail on the RX queue,
             * handling for RX interrupt is a little tricky.  If we
             * were to enable it here, we'd just get another interrupt
             * right away.  So instead, allow arNotifyRxComplete to
             * enable HAL_INT_RX as appropriate.
             */
            /* enableMask |= HAL_INT_RX; */
#endif
        }


#if !defined(ECOS_NOTDONE)
        if (isrValue & HAL_INT_DTIM) {
// RE-INSERT WHEN H/W PROCESSING OF BEACONS IS INTEGRATED
//            powerTimIndication(pDev, FALSE, TRUE);
        }
#endif

    } /* for (i < NUM_DPC_LOOPS) */

    

    /* If we disabled some interrupts in the ISR, re-enable them here. */
    halEnableInterrupts(pArDev, enableMask);

    status = arMacSetAwakeMode(pArDev, FALSE);
    ASSERT(status == A_OK);

intr_done:
    
    A_ACK_INTERRUPT(IntVector);
    A_ENABLE_INTERRUPT(IntVector);
    return;
}

LOCAL void
arCalibration(AR_DEV_INFO *pArDev) {
    A_STATUS status;

    if (!pArDev->bTargetStarted ||
        !pArDev->oneTimeInitDone || 
        (pArDev->powerMgmt.powerState == TARGET_DEVICE_PWRDN) ||
        (pArDev->powerMgmt.powerState == TARGET_DEVICE_SLEEP)) 
    {
        return;
    }

    A_TASK_LOCK();

#if 0
    if (halGetRfgain(pArDev) == RFGAIN_NEED_CHANGE) {
        // TBD xxx
    }
#endif

    status = halPerCalibration(pArDev, &pArDev->config.chanDesc);
    
    if (status == A_EBADCHANNEL) {
        // TBD xxx
        goto arCalExit;
    }

 arCalExit:

    A_TASK_UNLOCK();

    return;
}


LOCAL void
arAutoNoiseImmunity(AR_DEV_INFO *pArDev) 
{

    if (!pArDev->bTargetStarted ||
        !pArDev->oneTimeInitDone || 
        (pArDev->powerMgmt.powerState == TARGET_DEVICE_PWRDN) ||
        (pArDev->powerMgmt.powerState == TARGET_DEVICE_SLEEP)) 
    {
        return;
    }
    A_TASK_LOCK();

    halAniPeriodicCheck(pArDev);

    A_TASK_UNLOCK();

    return;
}


#endif

