/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/arPower.c#3 $
 *
 * Copyright (c) 2000-2003 Atheros Communications, Inc., All Rights Reserved
 *
 */


#include "wlantype.h"
#include "queue.h"
#include "wdcApi.h"
#include "arDev.h"
#include "arReceive.h"
#include "arTransmit.h"
#include "arPower.h"
#include "halApi.h"
#include "hal.h"
#include "targWdc.h"
#include "targTransport.h"

#define powerPrintf  uiPrintf


/**************************************************************************
 * TARG_wdcTargetSetPsMode - Sets the Target PS MODE
 *
 * RETURNS: none
 */
void
TARG_wdcTargetSetPsMode(TARGET_HANDLE targetHandle, TARGET_POWER_MODE wdcPsMode, A_STATUS *pStatus)
{
    A_STATUS        status  = A_OK;
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *)targetHandle;
    
    switch (wdcPsMode) {
        case TARGET_DEVICE_AWAKE:
            powerPrintf("awake\n");
            if ((pArDev->powerMgmt.powerState == TARGET_DEVICE_SLEEP)) {
                if (pArDev->bTargetStarted) {
#if defined(TRANSPORT_POLL)
                    TARG_txportPollEnable(pArDev->masterHandle->masterMsgHandle);
#endif
                    A_SEM_UNLOCK(pArDev->pwrLock);
                }
            }
            /* Set the MAC to awake Mode */
            if (arMacSetPowerMode(targetHandle, AR_MAC_AWAKE_MODE) != A_OK) {
                status = A_HARDWARE;
            }
            break;

        case TARGET_DEVICE_SLEEP:
            powerPrintf("sleep\n");

            if (pArDev->bTargetStarted) {
#if defined(TRANSPORT_POLL) 
                TARG_txportPollDisable(pArDev->masterHandle->masterMsgHandle);
#endif
                A_SEM_LOCK(pArDev->pwrLock, WAIT_FOREVER);
            }
            /* Set the MAC to Sleep Mode */
            if (arMacSetPowerMode(targetHandle, AR_MAC_SLEEP_MODE) != A_OK) {
                status = A_HARDWARE;
            }
            break;
        
        case TARGET_DEVICE_PWRDN:
            powerPrintf("pwrdn\n");
            /* Disable Global Interrupts while we channel change. */
            halDisableInterrupts(pArDev, HAL_INT_GLOBAL);

            if (arMacSetPowerMode(targetHandle, AR_MAC_AWAKE_MODE) != A_OK) {
                status = A_HARDWARE;
            }

            arStopReceive(pArDev);
            arStopTransmit(pArDev, DO_WAIT);

#if defined(TRANSPORT_POLL)
            TARG_txportPollDisable(pArDev->masterHandle->masterMsgHandle);
#endif

            arDeleteAllConnections((TARGET_HANDLE) pArDev);
            pArDev->bTargetStarted = FALSE;

            /* Renable Global Interrupts */
            halEnableInterrupts(pArDev, HAL_INT_GLOBAL);
            if (arMacSetPowerMode(pArDev, AR_MAC_SLEEP_MODE) != A_OK) {
                ASSERT(0);
            }
            break;

        case TARGET_DEVICE_PWRSAVE:
            powerPrintf("pssave\n");

            if (arMacSetPowerMode(targetHandle, AR_MAC_PWRSAVE_MODE) != A_OK) {
                status = A_HARDWARE;
            }
            break;

        default:
            ASSERT(FALSE);
            break;
    }
    if (status == A_OK) {
        pArDev->powerMgmt.powerState = wdcPsMode;
    }
    if (pStatus) {
        *pStatus = status;
    }
}

A_STATUS
arMacSetAwakeMode(TARGET_HANDLE targetHandle, A_UINT8 setAwake)
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *) targetHandle;
    A_STATUS        status = A_OK;

    A_TASK_LOCK();
    A_DISABLE_ALL_INTERRUPTS();

    if (setAwake) {
        if (!pArDev->powerMgmt.awakeCount && 
            (pArDev->powerMgmt.halPwrState != AWAKE)) {
            if (halSetPowerMode(pArDev, AWAKE, TRUE) != A_OK) {
                status = A_HARDWARE;
            }
        }
        pArDev->powerMgmt.awakeCount++;
    } else {
        pArDev->powerMgmt.awakeCount--;
        if (!pArDev->powerMgmt.awakeCount && 
            (pArDev->powerMgmt.halPwrState != AWAKE)) {
            if (halSetPowerMode(pArDev, pArDev->powerMgmt.halPwrState, TRUE) != A_OK) {
                status = A_HARDWARE;
            }
        }    
    }
    A_ENABLE_ALL_INTERRUPTS();
    A_TASK_UNLOCK();
    return status;
}

void
arMacInitPsMode (TARGET_HANDLE targetHandle) 
{
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *) targetHandle;

    A_SEM_INIT(pArDev->pwrLock, 0, CREATE_UNLOCKED);
    
    pArDev->powerMgmt.awakeCount    = 0;
    pArDev->powerMgmt.halPwrState   = AWAKE;
    pArDev->powerMgmt.powerState    = TARGET_DEVICE_AWAKE;
}


A_STATUS
arMacSetPowerMode(TARGET_HANDLE targetHandle, AR_MAC_PSMODE macPwrMode)
{
    A_STATUS        status  = A_OK;
    AR_DEV_INFO     *pArDev = (AR_DEV_INFO *) targetHandle;
    
    A_TASK_LOCK();

    switch(macPwrMode) {
        case AR_MAC_AWAKE_MODE:
            
            if (!pArDev->powerMgmt.awakeCount) {
                if (halSetPowerMode(pArDev, AWAKE, TRUE) != A_OK) {
                    status = A_HARDWARE;
                }
            }
            pArDev->powerMgmt.halPwrState = AWAKE;
            break;
        
        case AR_MAC_SLEEP_MODE:
            if (!pArDev->powerMgmt.awakeCount) {
                if (halSetPowerMode(pArDev, FULL_SLEEP, TRUE) != A_OK) {
                    status = A_HARDWARE;
                }
            }
            pArDev->powerMgmt.halPwrState = FULL_SLEEP;
            break;

        case AR_MAC_PWRSAVE_MODE:
            if (!pArDev->powerMgmt.awakeCount) {
                if (halSetPowerMode(pArDev, NETWORK_SLEEP, TRUE) != A_OK) {
                    status = A_HARDWARE;
                }
            }
            pArDev->powerMgmt.halPwrState = NETWORK_SLEEP;
            break;
    }
    A_TASK_UNLOCK();
    return status;
}

A_STATUS
arDevSetPowerMode(TARGET_HANDLE targetHandle, AR_DEVICE_PSMODE devicePwrState)
{
    AR_DEV_INFO     *pArDev   = (AR_DEV_INFO *) targetHandle;
    A_STATUS        status = A_OK;
    
    switch(devicePwrState) {
        case AR_DEVICE_SLEEP_MODE:
            /* Set the MAC to awake Mode */
            if (pArDev->powerMgmt.halPwrState != AWAKE) { 
                if (halSetPowerMode(pArDev, AWAKE, TRUE) != A_OK) {
                    status = A_HARDWARE;
                }
            }
            if (pArDev->oneTimeInitDone && pArDev->bTargetStarted == TRUE) {
                
                A_TASK_LOCK();
                /* Disable Global Interrupts while we channel change. */
                halDisableInterrupts(pArDev, HAL_INT_GLOBAL);
                /* Disable PCU RX engine. */
                halStopPcuReceive(pArDev);

                /* Disable DMA RX engine. */
                halStopDmaReceive(pArDev);
            } else {            
                status = halReset(pArDev, CURRENT_SERVICE_TYPE(pArDev), CURRENT_PCHDESC(pArDev), TRUE);
                if (status != A_OK) {
                    status = A_HARDWARE;
                }
            }
            if (status == A_OK) {
                /* Set the MAC to Sleep Mode */
                if (halSetPowerMode(pArDev, DEVICE_SLEEP, TRUE) != A_OK) {
                    status = A_HARDWARE;
                }
            }
            break;
        
        case AR_DEVICE_NORMAL_MODE:
           /* Set the MAC to Sleep Mode */
            if (halSetPowerMode(pArDev, DEVICE_WAKE, TRUE) != A_OK) {
                status = A_HARDWARE;
            }
            if (pArDev->oneTimeInitDone && pArDev->bTargetStarted == TRUE) {
            
                /* Renable Global Interrupts */
                halEnableInterrupts(pArDev, HAL_INT_GLOBAL);
            
                /* Enable DMA RX engine. */
                halEnableReceive(pArDev);

                /* Enable PCU RX engine. */
                halStartPcuReceive(pArDev);

                A_TASK_UNLOCK();
            }
            if (status == A_OK) {
                if ((pArDev->powerMgmt.halPwrState == FULL_SLEEP) &&
                    (!pArDev->powerMgmt.awakeCount)) 
                { 
                    if (halSetPowerMode(pArDev, FULL_SLEEP, TRUE) != A_OK) {
                        status = A_HARDWARE;
                    }   
                } else if ((pArDev->powerMgmt.halPwrState == NETWORK_SLEEP) &&
                    (!pArDev->powerMgmt.awakeCount))
                {
                    if (halSetPowerMode(pArDev, NETWORK_SLEEP, TRUE) != A_OK) {
                        status = A_HARDWARE;
                    }   
                }
            }
            break;

        default:
            ASSERT(FALSE);
            status = A_ERROR;
            break;
    }
    return  status;
}

