
/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/arRatectrl.c#3 $
 *
 * Copyright (c) 2001-2004 Atheros Communications, Inc., All Rights Reserved
 *
 * DESCRIPTION
 *
 * This file contains the data structures and routines for transmit rate
 * control.
 *
 * Releases prior to 2.0.1 used a different algorithm that was based on
 * an active of unicast packtets at different rate.  Starting in 2.0.1
 * a new algorithm was introduced.
 *
 * Basic algorithm:
 *   - Keep track of the ackRssi from each successful transmit.
 *   - For each user (SIB entry) maintain a table of rssi thresholds, one
 *     for each rate (eg: 8 or 4). This is the minimum rssi required for
 *     each rate for this user.
 *   - Pick the rate by lookup in the rssi table.
 *   - Update the rssi thresholds based on number of retries and Xretries.
 *     Also, slowly reduce the rssi thresholds over time towards their
 *     theoretical minimums.  In this manner the thresholds will adapt
 *     to the channel characteristics, like delay spread etc, while not
 *     staying abnormally high.
 *
 * More specific details:
 *  - The lkupRssi we use to lookup is computed as follows:
 *  - Use the median of the last three ackRssis.
 *  - Reduce the lkupRssi if the last ackRssi is older than 32msec.
 *    Specifically, reduce it linearly by up to 10dB for ages between
 *    25msec and 185msec, and a fixed 10dB for ages more than 185msec.
 *  - Reduce the lkupRssi by 1 or 2 dB if the packet is long or very long
 *    (this tweaks the rate for longer versus short packets).
 *  - When we get Xretries, reduce lkupRssi by 10dB, unless this was
 *    a probe (see below).
 *  - Maintain a maxRate (really an index, not an actual rate).  We don't
 *    pick rates above the maxRate.  The rssi lookup is free to choose any
 *    rate between [0,maxRate] on each packet.
 *
 *  The maxRate is increased by periodically probing:
 *    - No more than once every 100msec, and only if lkupRssi is high
 *      enough, we try a rate one higher than the current max rate.  If
 *      it succeeds with no retries then we increase the maxRate by one.
 *
 *  The rssi lkup is then free to use this new rate whenever the
 *  rssi warrants.
 *
 *  The maxRate is decreased if a rate looks poor (way too many
 *  retries or a couple of Xretries without many good packets in
 *  between):
 *    - We maintain a packet error rate estimate (per) for each rate.
 *      Each time we get retries or Xretries we increase the PER.
 *      Each time we get no retries we decrease the PER.  If the
 *      PER ever exceeds X (eg: an Xretry will increase the PER by Y)
 *      we set the maxRate to one below this rate.
 */

#include "arDev.h"
#include "arRateCtrl.h"
#include "halApi.h"

#if !defined(ECOS_NOTDONE)
/* JK - remove later */
#include "wlanframe.h"
#endif

/* Access functions for validTxRateMask */

static void
rcInitValidTxMask(struct TxRateCtrl_s *pRc)
{
    pRc->validTxRateMask = 0;
}

static void
rcSetValidTxMask(struct TxRateCtrl_s *pRc, A_UINT8 index, A_BOOL validTxRate)
{
    ASSERT(index < pRc->rateTableSize);

    if (validTxRate) {
        pRc->validTxRateMask |= (1 << index);
    } else {
        pRc->validTxRateMask &= ~(1 << index);
    }
}

/* Iterators for validTxRateMask */

INLINE A_BOOL
rcGetFirstValidTxRate(struct TxRateCtrl_s *pRc, A_UINT8 *pIndex, const RATE_TABLE *pRateTable, A_UINT8 turboPhyOnly, WLAN_PHY turboPhy)
{
    A_UINT32    mask = pRc->validTxRateMask;
    A_UINT8     i;

    for (i = 0; i < pRc->rateTableSize; i++) {
        if (mask & (1 << i)) {
            if ((!turboPhyOnly) || (pRateTable->info[i].phy == turboPhy)) {
                *pIndex = i;
                return TRUE;
            }
        }
    }

    /*
     * No valid rates - this should not happen.
     *
     * Note - Removed an assert here because the dummy sib
     * was causing the assert to trigger.
     */
    *pIndex = 0;

    return FALSE;
}

INLINE A_BOOL
rcGetNextValidTxRate(struct TxRateCtrl_s *pRc, A_UINT8 curValidTxRate, A_UINT8 *pNextIndex, const RATE_TABLE *pRateTable, A_UINT8 excludeTurboPhy, WLAN_PHY turboPhy)
{
    A_UINT32    mask = pRc->validTxRateMask;
    A_UINT8     i;

    for (i = curValidTxRate + 1; i < pRc->rateTableSize; i++) {
        if (mask & (1 << i)) {
            if ((!excludeTurboPhy) || (pRateTable->info[i].phy != turboPhy)) {
                *pNextIndex = i;
                return TRUE;
            }
        }
    }

    /* No more valid rates */
    *pNextIndex = 0;

    return FALSE;
}

static A_BOOL
rcGetPrevValidTxRate(struct TxRateCtrl_s *pRc, A_UINT8 curValidTxRate, A_UINT8 *pPrevIndex )
{
    A_UINT32    mask = pRc->validTxRateMask;
    int         i;

    if (curValidTxRate == 0) {
        *pPrevIndex = 0;
        return FALSE;
    }

    for (i = curValidTxRate - 1; i >= 0; i--) {
        if (mask & (1 << i)) {
            *pPrevIndex = (A_UINT8) i;
            return TRUE;
        }
    }

    /* No more valid rates */
    *pPrevIndex = 0;

    return FALSE;
}

static A_BOOL
rcGetNextLowerValidTxRate(const RATE_TABLE *pRateTable, struct TxRateCtrl_s *pRc, A_UINT8 curValidTxRate, A_UINT8 *pNextIndex )
{
    A_UINT32    mask = pRc->validTxRateMask;
    A_UINT8     i;

    *pNextIndex = 0;
    for (i = 1; i < pRc->rateTableSize; i++) {
        if ((mask & (1 << i)) && (pRateTable->info[i].rateKbps < pRateTable->info[curValidTxRate].rateKbps)) {
            if (*pNextIndex) {
                if (pRateTable->info[i].rateKbps > pRateTable->info[*pNextIndex].rateKbps) {
                    *pNextIndex = i;
                }
            } else {
                *pNextIndex = i;
            }
        }
    }

    if (*pNextIndex) {
        return TRUE;
    } else {
        *pNextIndex = curValidTxRate;
        return FALSE;
    }
}

/*
 *  Update the SIB's rate control information
 *
 *  This should be called when the supported rates change
 *  (e.g. SME operation, wireless mode change)
 *
 *  It will determine which rates are valid for use.
 */
void
rcSibUpdate(AR_CONN_INFO *pSib, A_BOOL keepState, AR_DEV_INFO  *pArDev)
{
    const RATE_TABLE    *pRateTable = pSib->pOpBss->pRateTable;
    CONN_RATE_SET       *pRateSet   = &pSib->options.workingRateSet;
    struct TxRateCtrl_s *pRc        = &pSib->txRateCtrl;
    A_UINT8             i, j, hi = 0, hiTurbo = 0, hiNonTurbo = 0,count;
    int                 rateCount;
    A_UINT8             primeInUse        = pArDev->primeInUse;
    A_UINT8             curPrimeState = pArDev->curPrimeState;
    /* Initial rate table size. Will change depending on the working rate set */
    pRc->rateTableSize = MAX_TX_RATE_TBL;

    /* Initialize thresholds according to the global rate table */
    for (i = 0 ; (i < pRc->rateTableSize) && (!keepState); i++) {
        pRc->state[i].rssiThres = pRateTable->info[i].rssiAckValidMin;
        pRc->state[i].per       = 0;
    }

    /* Determine the valid rates */
    rcInitValidTxMask(pRc);
    rateCount = pRateTable->rateCount;

#if !defined(ECOS_NOTDONE)
    if (wlanIs5211Channel14(pSib->hostConnHandle)) {
        rateCount = 2;
    }
#else
//    ECOS_NOTDONE_XXX_C("wlanIs5211Channel14 XXX?");
#endif

    count = 0;
    if (!pRateSet->length) {
        /* No working rate, use valid rates */
        for (i = 0; i < rateCount; i++) {
            if (pRateTable->info[i].valid == TRUE) {
                pRc->validRateIndex[count] = i;
                count ++;
                rcSetValidTxMask(pRc, i, TRUE);
				if (pArDev->primeInUse) {
				    if (pRateTable->info[i].phy == WLAN_PHY_TURBO) {
                        hiTurbo = A_MAX(hiTurbo, i);
					} else {
                        hiNonTurbo = A_MAX(hiNonTurbo, i);
					}
				}
                hi = A_MAX(hi, i);
            }
        }

        pRc->maxValidRate = count;
        pRc->maxValidTurboRate = pRateTable->numTurboRates;

    } else {
        A_UINT8  turboCount;
        A_UINT32 mask;

        /* Use intersection of working rates and valid rates */
        turboCount = 0;
        for (i = 0; i < pRateSet->length; i++) {
            for (j = 0; j < rateCount; j++) {
                if (((pRateSet->rates[i] & 0x7F) == (pRateTable->info[j].dot11Rate & 0x7F)) &&
                    (pRateTable->info[j].valid == TRUE))
                {
                    rcSetValidTxMask(pRc, j, TRUE);
				    if (pRateTable->info[j].phy == WLAN_PHY_TURBO) {
                        hiTurbo = A_MAX(hiTurbo, j);
					} else {
                        hiNonTurbo = A_MAX(hiNonTurbo, j);
					}
	                hi = A_MAX(hi, j);
                }
            }
        }

        /* Get actually valid rate index, previous we get it from rate table,
         * now get rate table which include all working rate, so we need make
         * sure our valid rate rable align with working rate */
        mask = pRc->validTxRateMask;
        for (i = 0; i < pRc->rateTableSize; i ++) {
            if (mask & (1 << i)) {
                pRc->validRateIndex[count] = i;
                count ++;
                if(pRateTable->info[i].phy == WLAN_PHY_TURBO) {
                    turboCount ++;
                }
            }
        }

        pRc->maxValidRate = count;
        pRc->maxValidTurboRate = turboCount;
    }

    pRc->rateTableSize = hi + 1;
	if (primeInUse) {
	    if (curPrimeState == TURBO_MODE) {
	     	pRc->rateMax = A_MIN(hiTurbo, pRateTable->initialRateMax);
		} else {
			pRc->rateMax = A_MIN(hiNonTurbo, pRateTable->initialRateMax);
		}
	} else{
        pRc->rateMax = A_MIN(hi, pRateTable->initialRateMax);
	}


    ASSERT(pRc->rateTableSize <= MAX_TX_RATE_TBL);
}

/*
 *  This routine is called to initialize the rate control parameters
 *  in the SIB. It is called initially during system initialization
 *  or when a station is associated with the AP.
 */
void
rcSibInit(AR_CONN_INFO *pSib)
{
    struct TxRateCtrl_s *pRc = &pSib->txRateCtrl;

    A_MEM_ZERO((char *)pRc, sizeof(*pRc));
    pRc->rssiDownTime = A_MS_TICKGET();
}

/************************************************************************/

#ifdef DEBUG_PKT_LOG
/*
 * Logging code
 */

/* Ver 10001 - Logs (rssiLast,rssiLastPrev) */
/*     10002 - Logs (rssiLast,rssiLastPrev,rssiLastPrev2) */
#define CURRENT_PKT_LOG_VER     10002 /* 10001 */
#define LOG_TYPE_RX             1
#define LOG_TYPE_TX             2
#define LOG_TYPE_RATE_GET       3
#define LOG_TYPE_RATE_UPDATE    4

/*
 * NOTE: the Perl script that processes the packet log data
 * structures has hardcoded structure offsets.
 *
 * When adding new fields it is important to:
 * - assign a newer verison number (CURRENT_VER_PKT_LOG)
 * - Add fields to the end of the structure.
 * - Observe proper memory alignment of fields. This
 *   eases the process of adding the structure offsets
 *   to the Perl script.
 * - note the new size of the pkt log data structure
 * - Update the Perl script to reflect the new structure and version/size changes
 */
typedef struct TxRateCtrlLog_s {
    A_UINT8 logType;
    struct TxRateCtrl_s rc;
    union {
        struct {
            A_UINT16    frameLen;
            A_UINT8     rate;
            A_UINT8     defAnt;
            A_UINT8     sibAnt;
        } g;
        struct {
            A_UINT8     rate;
            A_UINT8     xRetry;
            A_UINT8     retries;
            A_UINT16    frameLen;
            A_UINT8     rssiAck;
            A_UINT8     defAnt;
            A_UINT8     sibAnt;
        } t;
    } u;
} TxRateCtrlLog;

typedef struct {
    A_UINT32 timeStamp;
    union {
        struct {
            A_UINT8         logType;
            A_UINT8         rate;
            A_UINT8         rssi;
            A_UINT8         status;
            A_UINT16        length;
            A_UINT16        seqCtrl;
            A_UINT16        frameCtrl;
            A_UINT8         antRx;
            A_UINT8         dummy;
            A_UINT16        timeStamp;
        } rx;
        struct {
            A_UINT8         logType;
            A_UINT8         rate;
            A_UINT8         retries;
            A_UINT8         status;
            A_UINT16        length;
            A_UINT16        seqCtrl;
            A_UINT16        frameCtrl;
            A_UINT8         ackRssi;
            A_UINT8         antTx;
            A_UINT16        timeStamp;
        } tx;
    TxRateCtrlLog rate;
    } d;
} LogInfo;

#define LOG_MAX   10000

static A_INT32 LogIdx   = -1;
static LogInfo *LogData = NULL;

void
logEventDelete(void)
{
    if (LogData) {
        A_DRIVER_FREE(LogData, sizeof(LogInfo) * LOG_MAX);
        LogData = NULL;
    }
    LogIdx = -1;
}

void
logEventClear(void)
{
    if (!LogData) {
        LogData = (LogInfo*) A_DRIVER_MALLOC(sizeof(LogInfo) * LOG_MAX);
        if (!LogData) {
            LogIdx = -1;
            return;
        }
    }
    memset(LogData, 0, sizeof(LogInfo) * LOG_MAX);
    LogIdx = 0;
}

void
logEventRxPkt(A_UINT32 length, A_UINT32 rate, A_UINT32 rssi,
              A_UINT32 frameCtrl, A_UINT32 seqCtrl, A_UINT32 status,
              A_UINT32 antRx, A_UINT16 timeStamp)
{
    LogInfo log;

    if (LogIdx < 0) {
        logEventClear();
        if (LogIdx < 0) {
            return;
        }
    }

    log.timeStamp       = A_MS_TICKGET();
    log.d.rx.logType    = LOG_TYPE_RX;
    log.d.rx.rate       = (A_UINT8)rate;
    log.d.rx.rssi       = (A_UINT8)rssi;
    log.d.rx.frameCtrl  = (A_UINT16)frameCtrl;
    log.d.rx.seqCtrl    = (A_UINT16)seqCtrl;
    log.d.rx.status     = (A_UINT8)status;
    log.d.rx.length     = (A_UINT16)length;
    log.d.rx.antRx      = (A_UINT8)antRx;
    log.d.rx.timeStamp  = timeStamp;
    LogData[LogIdx++]   = log;
    if (LogIdx >= LOG_MAX) {
        LogIdx = 0;
    }
}

void
logEventTxPkt(A_UINT32 length, A_UINT32 rate, A_UINT32 ackRssi,
              A_UINT32 retries, A_UINT32 status, A_UINT32 frameCtrl,
              A_UINT32 seqCtrl, A_UINT32 antTx, A_UINT16 timeStamp)
{
    LogInfo log;

    if (LogIdx < 0) {
        logEventClear();
        if (LogIdx < 0) {
            return;
        }
    }
    log.timeStamp       = A_MS_TICKGET();
    log.d.tx.logType    = LOG_TYPE_TX;
    log.d.tx.rate       = (A_UINT8)rate;
    log.d.tx.length     = (A_UINT16)length;
    log.d.tx.retries    = (A_UINT8)retries;
    log.d.tx.status     = (A_UINT8)status;
    log.d.tx.ackRssi    = (A_UINT8)ackRssi;
    log.d.tx.antTx      = (A_UINT8)antTx;
    log.d.tx.frameCtrl  = (A_UINT16)frameCtrl;
    log.d.tx.seqCtrl    = (A_UINT16)seqCtrl;
    log.d.tx.timeStamp  = timeStamp;
    LogData[LogIdx++]   = log;
    if (LogIdx >= LOG_MAX) {
        LogIdx = 0;
    }
}

void
logEventRateCtrl(TxRateCtrlLog *rateLog)
{
    LogInfo log;

    if (LogIdx < 0) {
        logEventClear();
        if (LogIdx < 0) {
            return;
        }
    }

    log.timeStamp     = A_MS_TICKGET();
    log.d.rate        = *rateLog;
    LogData[LogIdx++] = log;

    if (LogIdx >= LOG_MAX) {
        LogIdx = 0;
    }
}

void
logEventGet(A_UINT32 *len, A_UINT32 *maxCnt, A_UINT32 *currIdx,
            A_UINT32 getIdx, void *data)
{
    if (LogIdx < 0) {
        *len     = sizeof(LogInfo);
        *maxCnt  = 0;
        *currIdx = 0;
        return;
    }
    *len     = sizeof(LogInfo);
    *maxCnt  = LOG_MAX;
    *currIdx = LogIdx;
    if (getIdx >= LOG_MAX) {
        getIdx = LOG_MAX - 1;
    }
    if (LogIdx >= 0) {
        *(LogInfo*)data = LogData[getIdx];
    }
}

void
logEventGetHeader(A_UINT32 *len, A_UINT32 *logVer)
{
    *len     = sizeof(LogInfo);
    *logVer  = CURRENT_PKT_LOG_VER;
}

#endif /* #ifdef DEBUG_PKT_LOG */

/************************************************************************/

/*
 * Return the median of three numbers
 */
INLINE
A_UINT8 median(A_UINT8 a, A_UINT8 b, A_UINT8 c)
{
    if (a >= b) {
        if (b >= c) {
            return b;
        } else if (a > c) {
            return c;
        } else {
            return a;
        }
    } else {
        if (a >= c) {
            return a;
        } else if (b >= c) {
            return c;
        } else {
            return b;
        }
    }
}

/*
 * Determines and returns the new Tx rate index.
 */
A_UINT16
rcRateFind(AR_DEV_INFO *pArDev, AR_DESC *pDesc)
{
    AR_CONN_INFO         *pSib       = pDesc->pArConnInfo;
    AR_BSS_INFO          *pBss       = pDesc->pOpBss;
    WLAN_DATA_MAC_HEADER *pHdr       = pDesc->pBufferVirtPtr.header;
    A_UINT32             frameLen    = pDesc->hw.txControl.frameLength;
    const RATE_TABLE     *pRateTable = NULL;
    struct TxRateCtrl_s  *pRc;
    A_UINT32             dt;
    A_UINT32             bestThruput,thisThruput;
    A_UINT32             nowMsec;
    A_UINT8              rate, nextRate, bestRate;
    A_UINT8              rssiLast, rssiReduce;

    /* TURBO_PRIME */
    A_UINT8              primeInUse        = pArDev->primeInUse;
    A_UINT8              currentPrimeState = pArDev->curPrimeState;

#ifdef DEBUG_PKT_LOG
    TxRateCtrlLog        log;
#endif

    A_UINT8              maxIndex, minIndex;
    A_INT32              index;


    if (pSib == NULL)  {
        /* No connection/BSS information - use lowest rate */
        ASSERT(pBss == NULL);
        return 0;
    }

    ASSERT(pBss != NULL);
    pRateTable = pBss->pRateTable;

    /*
     * Matt says - this is supported only for test modes! there
     * is no point of a control/basic rate if all your data frames
     * are required to be this fixed rate! So just live with this
     * fixed rate for everything!
     */

    if (!pArDev->config.rateControlEnable) {
        return (A_UINT16)pBss->config.defaultRateIndex;
    }

    /*
     * The spec and WECA definition is currently to use 6, 12 & 24 as
     * the base modes. I think all multicast, management & control frames
     * have to go at those rates. That may change after future debate, some
     * think that 6 should be the base rate to maximize the range of control
     * packets.                                         - AndyD
     * These can really be sent at the lowest rate without worrying about
     * performance too much - it is more robust! We may revisit this in
     * the future
     */
    if (isGrp(&pHdr->address1) || (pHdr->frameControl.fType != FRAME_DATA)) {
        return (A_UINT16)pBss->config.defaultRateIndex;
    }

    /* have the real rate control logic kick in */
    pRc = &pSib->txRateCtrl;

    /* make sure that rateMax is correct when in TURBO_PRIME mode */
    if (primeInUse) {
        if ((currentPrimeState == TURBO_MODE) &&
            (pRateTable->info[pRc->rateMax].phy != WLAN_PHY_TURBO))
        {
            pRc->rateMax = pRateTable->rateCount - 1;
        } else if ((currentPrimeState == BASE_MODE) &&
            (pRateTable->info[pRc->rateMax].phy == WLAN_PHY_TURBO))
        {
            pRc->rateMax = pRateTable->rateCount - 1 - pRateTable->numTurboRates;
        }
    }

    rssiLast   = pRc->rssiMedian;
    rssiReduce = 0;

    /*
     * Age (reduce) last ack rssi based on how old it is.
     * The bizarre numbers are so the delta is 160msec,
     * meaning we divide by 16.
     *   0msec   <= dt <= 25msec:   don't derate
     *   25msec  <= dt <= 185msec:  derate linearly by 10dB
     *   185msec <= dt:             derate by 10dB
     */
    nowMsec = A_MS_TICKGET() * 10;
    dt = nowMsec - pRc->rssiTime;

    if (dt >= 185) {
        rssiReduce = 10;
    } else if (dt >= 25) {
        rssiReduce = (A_UINT8)((dt - 25) >> 4);
    }

    /* Reduce rssi for long packets */
    if (frameLen > 800) {
         rssiReduce += 1;    /* need 1 more dB for packets > 800 bytes */
    }

    /* Now reduce rssiLast by rssiReduce */
    if (rssiLast < rssiReduce) {
        rssiLast = 0;
    } else {
        rssiLast -= rssiReduce;
    }

    /*
     * Now look up the rate in the rssi table and return it.
     * If no rates match then we return 0 (lowest rate)
     */

    bestThruput = 0;
    bestRate    = 0; 
    maxIndex    = pRc->maxValidRate - 1;
    minIndex    = 0;

    /*If we use TURBO_PRIME table, but we aren't in turbo mode, we need reduce maxIndex to actually size*/
    if (pRc->maxValidTurboRate) {
        if (!CHAN_IS_TURBO(&pArDev->config.chanDesc)) {
            maxIndex -= pRc->maxValidTurboRate;
            /*This shouldn't happne unless we use wrong rate table*/
            if (maxIndex == 0) {
                return 0;
            }
        } else {
            if (pRc->maxValidRate > pRc->maxValidTurboRate) {
                minIndex = pRc->maxValidRate - pRc->maxValidTurboRate;
            }
        }
    }
    bestRate    = pRc->validRateIndex[minIndex];

    /*
     * Try the higher rate first. It will reduce memory moving time
     * if we have very good channel characteristics.
     */
    for (index = maxIndex; index >= minIndex ; index --) {
        rate = pRc->validRateIndex[index];

        if (rssiLast < pRc->state[rate].rssiThres) {
            continue;
        }

        thisThruput = pRateTable->info[rate].userRateKbps *
                      (100 - pRc->state[rate].per);

        if (bestThruput <= thisThruput) {
            bestThruput = thisThruput;
            bestRate    = rate;
        }
    }

    rate = bestRate;

#ifdef DEBUG_PKT_LOG
    pRc->misc[4] = primeInUse;
    pRc->misc[5] = currentPrimeState;
    /* boost */
    pRc->misc[9] = pRc->rateTableSize;

    pRc->misc[8] = rcGetNextValidTxRate(pRc, rate, &pRc->misc[7], pRateTable,
                                        !IS_CHAN_TURBO(pArDev->staConfig.pChannel->channelFlags),
                                        WLAN_PHY_TURBO);
    pRc->misc[10] = rate;
    /* useTurboPrime */
#endif

    pRc->rssiLastLkup = rssiLast;

    /*
     * Must check the actual rate (rateKbps) to account for non-monoticity of
     * 11g's rate table
     */

    if (rate > pRc->rateMax) {
        rate = pRc->rateMax;

        /*
         * Always probe the next rate in the rate Table (ignoring monotonicity).
         * Reason:  If OFDM is broken, when rateMax = 5.5, it will probe
         *          11 Mbps first.
         */
        if (rcGetNextValidTxRate(pRc,
                                 rate,
                                 &nextRate,
                                 pRateTable,
                                 !CHAN_IS_TURBO(&pArDev->config.chanDesc),
                                 WLAN_PHY_TURBO) &&
            (nowMsec - pRc->probeTime > pRateTable->probeInterval) &&
            (pRc->hwMaxRetryPktCnt >= 4))
        {
            rate = nextRate;
            pRc->probeRate        = (A_UINT8)rate;
            pRc->probeTime        = nowMsec;
            pRc->hwMaxRetryPktCnt = 0;
        }
       
    }

    /*
     * Make sure rate is not higher than the allowed maximum.
     * We should also enforce the min, but I suspect the min is
     * normally 1 rather than 0 because of the rate 9 vs 6 issue
     * in the old code.
     */
    if (rate > (pSib->txRateCtrl.rateTableSize - 1)) {
        rate = pSib->txRateCtrl.rateTableSize - 1;
    }

#ifdef DEBUG_PKT_LOG
#ifndef BUILD_AP
    /*
     * for the adhoc case the sib state may go out of sync! so
     * resync - if someone is looking at the stats
     */
    pSib->antTx = (halGetDefAntenna(pArDev) == 1) ? 0 : 1;
#endif
    log.rc.misc[4]      = pRc->misc[4];
    log.rc.misc[5]      = pRc->misc[5];
    log.rc.misc[6]      = pRc->misc[6];
    log.rc.misc[7]      = pRc->misc[7];
    log.rc.misc[8]      = pRc->misc[8];	
    log.rc.misc[9]      = pRc->misc[9];
    log.rc.misc[10]     = pRc->misc[10];
    log.rc.misc[11]     = pRc->misc[11];
    log.rc.misc[12]     = pRc->misc[12];
    log.rc              = *pRc;
    log.logType         = LOG_TYPE_RATE_GET;
    log.u.g.frameLen    = (A_UINT16)frameLen;
    log.u.g.rate        = (A_UINT8)rate;
    log.u.g.defAnt      = (A_UINT8)0;
    log.u.g.sibAnt      = (A_UINT8)pSib->antTx;
    logEventRateCtrl(&log);
#endif
    return rate;
}

/*
 * This routine is called by the Tx interrupt service routine to give
 * the status of previous frames.
 */
void
rcUpdate(AR_DEV_INFO *pArDev, AR_CONN_INFO *pSib, int txRate, int frameLen,
         A_BOOL Xretries, int retries, A_UINT8 rssiAck, A_UINT8 curTxAnt)
{
    const RATE_TABLE    *pRateTable;
    struct TxRateCtrl_s *pRc;
    A_UINT32            nowMsec     = A_MS_TICKGET();
    A_UINT8             lastPer;
    A_BOOL              stateChange = FALSE;
    int                 rate, i, count;
    A_UINT8             primeInUse        = pArDev->primeInUse;
    A_UINT8             currentPrimeState = pArDev->curPrimeState;

    static A_UINT32 nRetry2PerLookup[10] = {
        100 * 0 / 1,
        100 * 1 / 4,
        100 * 1 / 2,
        100 * 3 / 4,
        100 * 4 / 5,
        100 * 5 / 6,
        100 * 6 / 7,
        100 * 7 / 8,
        100 * 8 / 9,
        100 * 9 / 10
    };

#ifdef DEBUG_PKT_LOG
    TxRateCtrlLog log;
#endif

    if (!pSib) {
        return;
    }

    ASSERT(pSib->pOpBss);

    pRateTable = pSib->pOpBss->pRateTable;
    pRc        = &pSib->txRateCtrl;

    /* TURBO_PRIME - choose the correct rate index when in turbo mode */
    if (primeInUse && (currentPrimeState == TURBO_MODE)) {
        if(txRate < pRateTable->initialRateMax + 1) {
            rate = txRate + pRateTable->numTurboRates;

            if (pRateTable->info[txRate].rateCode != pRateTable->info[rate].rateCode) {
                rate++;

                for (i = rate; i < pRateTable->rateCount; i ++) {
                    if (pRateTable->info[txRate].rateCode == pRateTable->info[i].rateCode) {
                        txRate = i;
                        break;
                    }
                }
            } else {
                txRate = rate;
            }
        }
    }


    lastPer = pRc->state[txRate].per;

    ASSERT(retries >= 0 && retries < MAX_TX_RETRIES);
    ASSERT(txRate >= 0 && txRate < pRc->rateTableSize);
    ASSERT(rssiAck < 128);       /* rssi cannot be -ve */

    if (Xretries) {
        /* Update the PER. */
        pRc->state[txRate].per += 35;
        if (pRc->state[txRate].per > 100) {
            pRc->state[txRate].per = 100;
        }

        for (rate = txRate+1; rate < pRc->rateTableSize; rate++) {
            if (pRateTable->info[rate].phy != pRateTable->info[txRate].phy) {
                break;
            }

            if (pRc->state[rate].per < pRc->state[txRate].per) {
                pRc->state[rate].per = pRc->state[txRate].per;
            }
        }

        /* Update the RSSI threshold*/
        /*
         * Oops, it didn't work at all.  Set the required rssi
         * to the rssiAck we used to lookup the rate, plus 4dB.
         * The immediate effect is we won't try this rate again
         * until we get an rssi at least 4dB higher.
         */
        if (txRate > 0) {
            int rssi = A_MAX(pRc->rssiLastLkup, pRc->rssiLast - 2);

            if (pRc->state[txRate].rssiThres + 2 < rssi) {
                pRc->state[txRate].rssiThres += 4;
            } else if (pRc->state[txRate].rssiThres < rssi + 2) {
                pRc->state[txRate].rssiThres = rssi + 2;
            } else {
                pRc->state[txRate].rssiThres += 2;
            }

            pRc->hwMaxRetryRate = (A_UINT8)txRate;
            stateChange         = TRUE;
        }

        /*
         * Also, if we are not doing a probe, we force a significant
         * backoff by reducing rssiLast.  This doesn't have a big
         * impact since we will bump the rate back up as soon as we
         * get any good ACK.  RateMax will also be set to the current
         * txRate if the failed frame is not a probe.
         */
        if (pRc->probeRate == 0 || pRc->probeRate != txRate) {
            pRc->rssiLast      = 10 * pRc->state[txRate].rssiThres / 16;
            pRc->rssiLastPrev  = pRc->rssiLast;
            pRc->rssiLastPrev2 = pRc->rssiLast;
        }

        pRc->probeRate = 0;
    } else {
        /* Update the PER. */
        /* Make sure it doesn't index out of array's bounds. */
        count = sizeof(nRetry2PerLookup) / sizeof(nRetry2PerLookup[0]);
        if (retries >= count) {
            retries = count - 1;
        }

        pRc->state[txRate].per = (A_UINT8)(pRc->state[txRate].per - (pRc->state[txRate].per >> 3) +
                                 (nRetry2PerLookup[retries] >> 3));

        /* Update the RSSI threshold*/
        halUpdateAntenna(pArDev, pSib, retries, rssiAck, curTxAnt);

        pRc->rssiLastPrev2 = pRc->rssiLastPrev;
        pRc->rssiLastPrev  = pRc->rssiLast;
        pRc->rssiLast      = rssiAck;
        pRc->rssiTime      = nowMsec;

        /*
         * If we got at most one retry then increase the max rate if
         * this was a probe.  Otherwise, ignore the probe.
         */

        if (pRc->probeRate && pRc->probeRate == txRate) {
            if (retries > 1) {
                pRc->probeRate = 0;
            } else {
                pRc->rateMax = pRc->probeRate;

                if (pRc->state[pRc->probeRate].per > 45) {
                    pRc->state[pRc->probeRate].per = 20;
                }

                pRc->probeRate = 0;

                /*
                 * Since this probe succeeded, we allow the next probe
                 * twice as soon.  This allows the maxRate to move up
                 * faster if the probes are succesful.
                 */
                pRc->probeTime = nowMsec - pRateTable->probeInterval / 2;
            }
        }

        if (retries > 0) {
            /*
             * Don't update anything.  We don't know if this was because
             * of collisions or poor signal.
             *
             * Later: if rssiAck is close to pRc->state[txRate].rssiThres
             * and we see lots of retries, then we could increase
             * pRc->state[txRate].rssiThres.
             */
            pRc->hwMaxRetryPktCnt = 0;

            if (retries >= 3 && rssiAck >= pRc->state[txRate].rssiThres &&
                rssiAck < pRc->state[txRate].rssiThres)
            {
                pRc->state[txRate].rssiThres += 1;
                stateChange = TRUE;
            }
        } else {
            /*
             * It worked with no retries.  First ignore bogus (small)
             * rssiAck values.
             */
            if (pRc->hwMaxRetryPktCnt < 255) {
                pRc->hwMaxRetryPktCnt++;
            }

            if (rssiAck >= pRateTable->info[txRate].rssiAckValidMin) {
                /* Average the rssi */
                if (txRate != pRc->rssiSumRate) {
                    pRc->rssiSumRate = (A_UINT8)txRate;
                    pRc->rssiSum     = pRc->rssiSumCnt = 0;
                }

                pRc->rssiSum += rssiAck;
                pRc->rssiSumCnt++;

                if (pRc->rssiSumCnt >= 4) {
                    A_UINT8 rssiAckAvg = (pRc->rssiSum + 2) / 4;

                    pRc->rssiSum = pRc->rssiSumCnt = 0;

                    /* Now reduce the current rssi threshold. */
                    if ((rssiAckAvg < pRc->state[txRate].rssiThres + 2) &&
                        (pRc->state[txRate].rssiThres > pRateTable->info[txRate].rssiAckValidMin))
                    {
                        pRc->state[txRate].rssiThres--;
                    }

                    stateChange = TRUE;
                }
            }
        }
    }
    pRc->rssiMedian = median(pRc->rssiLast, pRc->rssiLastPrev, pRc->rssiLastPrev2);

    /*
     * If this rate looks bad (high PER) then stop using it for
     * a while (except if we are probing).
     */
    if (pRc->state[txRate].per > 60 && txRate > 0 &&
        pRateTable->info[txRate].rateKbps <= pRateTable->info[pRc->rateMax].rateKbps)
    {
        rcGetNextLowerValidTxRate(pRateTable, pRc, (A_UINT8) txRate, &pRc->rateMax);

        /* Don't probe for a little while. */
        pRc->probeTime = nowMsec;
    }

    if (stateChange) {
        /*
         * Make sure the rates above this have higher rssi thresholds.
         * (Note:  Monotonicity is kept within the OFDM rates and within the CCK rates.
         *         However, no adjustment is made to keep the rssi thresholds monotonically
         *         increasing between the CCK and OFDM rates.)
         */
        for (rate = txRate ; rate < pRc->rateTableSize - 1; rate++) {
            if (pRateTable->info[rate+1].phy != pRateTable->info[txRate].phy) {
                break;
            }

            if (pRc->state[rate].rssiThres + pRateTable->info[rate].rssiAckDeltaMin >
                pRc->state[rate+1].rssiThres)
            {
                pRc->state[rate+1].rssiThres =
                    pRc->state[rate].rssiThres + pRateTable->info[rate].rssiAckDeltaMin;
            }
        }

        /* Make sure the rates below this have lower rssi thresholds. */
        for (rate = txRate - 1 ; rate >= 0 ; rate--) {
            if (pRateTable->info[rate].phy != pRateTable->info[txRate].phy) {
                break;
            }

            if (pRc->state[rate].rssiThres + pRateTable->info[rate].rssiAckDeltaMin >
                pRc->state[rate+1].rssiThres)
            {
                if (pRc->state[rate+1].rssiThres < pRateTable->info[rate].rssiAckDeltaMin) {
                    pRc->state[rate].rssiThres = 0;
                } else {
                    pRc->state[rate].rssiThres =
                        pRc->state[rate+1].rssiThres - pRateTable->info[rate].rssiAckDeltaMin;
                }

                if (pRc->state[rate].rssiThres < pRateTable->info[rate].rssiAckValidMin) {
                    pRc->state[rate].rssiThres = pRateTable->info[rate].rssiAckValidMin;
                }
            }
        }
    }

    /* Make sure the rates below this have lower PER */
    /* Monotonicity is kept only for rates below the current rate. */
    if (pRc->state[txRate].per < lastPer) {
        for (rate = txRate-1; rate >=0; rate--) {
            if (pRateTable->info[rate].phy != pRateTable->info[txRate].phy) {
                break;
            }

            if (pRc->state[rate].per > pRc->state[rate+1].per) {
                pRc->state[rate].per = pRc->state[rate+1].per;
            }
        }
    }

    /* Every so often, we reduce the thresholds and PER (different for CCK and OFDM). */
    if (nowMsec - pRc->rssiDownTime >= pRateTable->rssiReduceInterval) {
        for (rate = 0; rate < pRc->rateTableSize; rate++) {
            if (pRc->state[rate].rssiThres > pRateTable->info[rate].rssiAckValidMin) {
                pRc->state[rate].rssiThres -= 1;
            }
            pRc->state[rate].per = 15*pRc->state[rate].per/16;
        }

        pRc->rssiDownTime = nowMsec;
    }

#ifdef DEBUG_PKT_LOG
    log.rc              = *pRc;
    log.rc.misc[0]      = (A_UINT8)pSib->numTxPending;
    log.rc.misc[1]      = pArDev->turboPrimeInfo.currentBoostState;
    log.rc.misc[2]      = pArDev->bssDescr->athAdvCapElement.info.useTurboPrime;
    log.rc.misc[3]      = pRc->rateTableSize;
    log.logType         = LOG_TYPE_RATE_UPDATE;
    log.u.t.rate        = (A_UINT8)txRate;
    log.u.t.xRetry      = (A_UINT8)Xretries;
    log.u.t.retries     = (A_UINT8)retries;
    log.u.t.frameLen    = (A_UINT16)frameLen;
    log.u.t.rssiAck     = rssiAck;
    log.u.t.defAnt      = (A_UINT8)0;
    log.u.t.sibAnt      = (A_UINT8)pSib->antTx;
    logEventRateCtrl(&log);
#endif

}

/*
 * rcGetBestCckRate - given a current tx rate index and tx descriptor,
 * find the previous CCK rate from the working rate table.  Only
 * meaningful in 11g mode.  Returns the best CCK rate if found
 * else original rate.
 */
A_UINT16
rcGetBestCckRate(AR_DESC *pDesc, A_UINT16 curRateIndex)
{
    struct TxRateCtrl_s *pRc;
    A_UINT8             prevIndex;
    AR_CONN_INFO        *pSib       = pDesc->pArConnInfo;
    const RATE_TABLE    *pRateTable = pDesc->pOpBss->pRateTable;

    if (!pSib) {
        return curRateIndex;
    }

    pRc = &pSib->txRateCtrl;

    prevIndex = (A_UINT8)curRateIndex;
    do {
        if (pRateTable->info[prevIndex].phy == WLAN_PHY_CCK) {
            return (A_UINT16)prevIndex;
        }
    } while (rcGetPrevValidTxRate(pRc, prevIndex, &prevIndex));

    return curRateIndex;
}
