/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/arReceive.c#3 $
 *
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 * Implementation of receive (RX) logic for the Target driver.
 */

#include "wlantype.h"
#include "queue.h"
#include "arDev.h"
#include "arReceive.h"
#include "targWdc.h"
#include "halApi.h"

/*******************************************************************************
*
* An explanation of the WDC Target Receive path:
* ==============================================
*
* The major data structures for WDC Target Receive include:
*
*   AR_DESC Descriptor -        A descriptor is used to control the details of
*                               a each Receive operation.  It is through the
*                               descriptor that the driver communicates with
*                               hardware.
*
*   Receive Buffer -            This is an array of bytes, large enough to
*                               hold at least one 802.11 frame.
*
*
*   WDC RX Message -            A message that the Target constructs in order
*                               to inform the Host about Received data.  The
*                               message includes WDC header information,
*                               RECEIVE_INFO (see below), and the received
*                               data itself.
*
*   RECEIVE_INFO -              A structure that contains information about
*                               a particular frame that was received over
*                               the WLAN.  This information is extracted from
*                               the Descriptor, and it's passed to the Host.
*
*   Hardware Descriptor Queue - A queue of Descriptors that is handed to the
*   (rxHwQueue)                 hardware.  A descriptor is put into this
*                               queue in order to make a buffer available for
*                               receive.  It is removed from the queue once
*                               the buffer is filled.  A message is then
*                               generated, and the message is added to the
*                               message queue.
*
*   Message Queue -             A queue of Messages that each contain received
*   (rxMsgQ)                    data.  Messages wait here until the message can
*                               be sent over the WDC Transport to the Host.
*
*
* There are two threads involved in the WDC Target Receive implementation:
*
*   RxCompleteLoop   -          This thread waits until the WLAN driver
*                               notifies it (via arNotifyRxComplete) that
*                               data has been received.  It then removes the
*                               completed descriptor(s) from the Hardware
*                               Descriptor Queue, builds an appropriate
*                               WDC RX Message, and enqueues the message on
*                               the Message Queue.
*
*   MsgDispatchLoop -           This threads waits until a message is added
*                               to the Message Queue.  Then it dequeues the
*                               message, and sends it over the WDC Transport.
*
* Use of multiple threads costs a little bit in terms of context switching;
* but it simplifies the logic, and avoids complex queueing and synchronization
* that might otherwise be needed.
*
* The WLAN driver may use interrupts (as it currently does), or it may use
* polling.  As long as the driver calls arNotifyRxComplete, received data
* will get passed to the Host.
*
* Note that there is very little synchronization with the Host.  The Target
* receives data whenever Target Receive buffers are available.  The Target
* Sends data whenever the Host is able to accept data.  (If there are no
* Host-side receive buffers available, the Transport blocks until buffers
* become available.)
*
* In order to avoid data copies, the WDC Messages are pre-allocated with
* sufficient room for receive buffers.   So a Descriptor handed off to
* hardware contains a pointer into the middle of a WDC Message.  Additionally,
* space is reserved in the Message for RECEIVE_INFO, which is filled in from
* the descriptor when the Receive has completed.  In theory, it's possible
* to make use of more buffers than descriptors; but in practice, the size
* of a descriptor is small compared to a receive buffer, so we allocate
* exactly the same number of messages, buffers, and descriptors.  The
* descriptors are placed in memory just before the corresponding message
* (and buffer).
*
* Receive data never passes through the CPU.  We therefore avoid coherency
* operations as well as buffer copies.  [The Receive buffer is always
* DMA-coherent, but it is not made CPU-coherent.]
*
* A side-effect of the one-for-one association of descriptors with Receive
* buffers is that there's no need for an explicit Descriptor Free Queue:
* Either the descriptor is on the Hardware queue (waiting for WLAN data to
* fill in a buffer), or the descriptor is waiting for data that was just
* read to be sent to the Host.
*
* The rxHwQueue is accessed directly by hardware.  In order to avoid
* RX Overrun errors (RXORN), we always keep at least one descriptor at
* the end of the rxHwQueue which is "self-linked".  That way, the hardware
* always believes that there's an RX buffer available.
*
* When debugging, here are some notworthy places to set breakpoints:
*
*  arReceiveBuf         - A buffer is made available to hardware.
*
*  arNotifyRxComplete   - The WLAN driver signals that data is received.
*
*  TARG_wdcRxMsgEnqueue - A message is enqueued on the Message Queue
*                         (this is done by the RxCompleteLoop thread).
*
*  TARG_wdcRxMsgDequeue - A message is dequeued from the Message Queue
*                         (this is done by the MsgDispatchLoop thread).
*
*  TARG_wdcRxMsgSend    - The message is leaving the Target and headed to
*                         the WDC Transport.
*
*  arRecycleRxBuf       - The message has successfully been sent to the
*                         Host, and the buffer can now be re-used for
*                         another Receive operation.
*
******************************************************************************/

/*
 * The number of outstanding Receives.
 * This controls the number of receive buffers, receive messages,
 * and number of buffers that can be in the RX Message queue.
 */
int     RX_OUTSTANDING_COUNT;


/* Local functions */
LOCAL A_UINT8 getRxMulticastHashIndex(char *pMacAddress);

#if defined(OVERLAPPING_RX_BUFFERS)
#define OVERLAP_RXMSG_DATA_LENGTH  (1600)
#define OVERLAP_RXMSG_LENGTH       (OVERLAP_RXMSG_DATA_LENGTH + \
                                    WDC_RXMSG_HEADER_LENGTH)
#endif

#if defined(OVERLAPPING_RX_BUFFERS) && defined(FAST_FRAMES_ENABLED)
#error "-DOVERLAPPING_RX_BUFFERS && -DFAST_FRAMES_ENABLED are mutually exclusive."
#endif

/*
 * Change the RX filter
 */
void
TARG_wdcSetRxFilter(
    DEVICE_HANDLE       deviceHandle,
    RX_FILTER_FLAGS     rxFilter,
    A_STATUS            *pStatus
)
{
    AR_DEV_INFO     *pArDev     = (AR_DEV_INFO *)deviceHandle;

    ASSERT(pArDev);

    halSetRxFilter(pArDev, rxFilter);

    if (pStatus) {
        *pStatus = A_OK;
    }
}


/*
 * Initialize the multicast filter.
 *
 * If bReset = TRUE, all filter bits are 0 (all multicast are filtered)
 *      Individual multicast filter bits are enabled via wdcAddRxMulticastFilter
 * If bReset = FALSE, all filter bits are 1 (all multicast are allowed)
 *      This is useful for software multicast filtering.
 *
 */
void
TARG_wdcInitRxMulticastFilter(
    DEVICE_HANDLE       deviceHandle,
    A_BOOL              bReset
)
{
    AR_DEV_INFO     *pArDev     = (AR_DEV_INFO *)deviceHandle;
    A_UINT32        value;

    ASSERT(pArDev);

    if (bReset) {
        value = 0;
    } else {
        value = 0xffffffff;
    }

    halSetMulticastFilter(pArDev, value, value);
}

/*
 * Add the specified MAC address to the list of multicast addresses
 * allowed to be received.
 */
void
TARG_wdcSetRxMulticastFilter(
    DEVICE_HANDLE       deviceHandle,
    WLAN_MACADDR        *pMcastAddr,
    A_STATUS            *pStatus
)
{
    AR_DEV_INFO     *pArDev     = (AR_DEV_INFO *)deviceHandle;
    A_UINT8         hashIdx;

    ASSERT(pArDev);
    ASSERT(pMcastAddr);

    hashIdx = getRxMulticastHashIndex(pMcastAddr->octets);
    halMulticastFilterIndex(pArDev, hashIdx, TRUE);

    if (pStatus) {
        *pStatus = A_OK;
    }
}

/*
 * Remove the specified MAC address from the list of multicast addresses
 * allowed to be received.
 */
void
TARG_wdcClearRxMulticastFilter(
    DEVICE_HANDLE       deviceHandle,
    WLAN_MACADDR        *pMcastAddr,
    A_STATUS            *pStatus
)
{
    AR_DEV_INFO     *pArDev     = (AR_DEV_INFO *)deviceHandle;
    A_UINT8         hashIdx;

    ASSERT(pArDev);
    ASSERT(pMcastAddr);

    hashIdx = getRxMulticastHashIndex(pMcastAddr->octets);
    halMulticastFilterIndex(pArDev, hashIdx, FALSE);

    if (pStatus) {
        *pStatus = A_OK;
    }
}


/******************************************************************************
 * getRxMulticastHashIndex -
 *    Calculate the hash index for multicast address according to ar5211 spec.
 *
 * Input: Mac address
 *
 * RETURNS: hash index
 */
LOCAL A_UINT8
getRxMulticastHashIndex(
    char *pMacAddress
)
{
    char    macAddr[8];
    char    hashIdx;

    ASSERT(pMacAddress);

    macAddr[0] = pMacAddress[0] & 0x3F;
    macAddr[1] = ((pMacAddress[1] & 0x0F)<<2) | ((pMacAddress[0] & 0xC0) >> 6);
    macAddr[2] = ((pMacAddress[2] & 0x03)<<4) | ((pMacAddress[1] & 0xF0) >> 4);
    macAddr[3] = (pMacAddress[2] & 0xFC) >> 2;
    macAddr[4] = pMacAddress[3]& 0x3F;
    macAddr[5] = ((pMacAddress[4] & 0x0F)<<2) | ((pMacAddress[3] & 0xC0) >> 6);
    macAddr[6] = ((pMacAddress[5] & 0x03)<<4) | ((pMacAddress[4] & 0xF0) >> 4);
    macAddr[7] = (pMacAddress[5] & 0xFC) >> 2;

    hashIdx = macAddr[0]^macAddr[1]^macAddr[2]^macAddr[3]^macAddr[4]^macAddr[5]
        ^macAddr[6]^macAddr[7];

    return hashIdx;
}


/*
 * Program the receive hardware with a pointer to the descriptor chain,
 * enable the receiver.
 */
void
arStartReceive(
    AR_DEV_INFO *pArDev
)
{
    AR_DESC       *pDesc;

    /* Determine the start of the RX descriptor chain */
    if ((pDesc = arDescQGetHead(&pArDev->rxHwQueue)) != NULL)  {

        halSetRxDP(pArDev, pDesc->thisPhysPtr);
        
        /* Enable DMA RX engine. */
        halEnableReceive(pArDev);
        
        /* Enable PCU RX engine. */
        halStartPcuReceive(pArDev);

#if !defined(AR_POLL)
        halEnableInterrupts(pArDev, HAL_INT_RX);
#endif
    }

    /* Unlock the RX completion thread */
    A_SEM_UNLOCK(pArDev->rxCompleteLock);
}

/* Stop receive hardware */
void
arStopReceive(
    AR_DEV_INFO *pArDev
)
{

    A_TASK_ASSERT_UNLOCKED();

    /* Lock the RX completion thread */
    A_SEM_LOCK(pArDev->rxCompleteLock,    WAIT_FOREVER);

    A_TASK_LOCK();

    /* TASK_LOCK - Make sure we shut down RX cleanly without a buffer
       recycle interrupting and starting things up again. */

    /* Disable PCU RX engine. */
    halStopPcuReceive(pArDev);

    /* Disable DMA RX engine. */
    halStopDmaReceive(pArDev);

    A_TASK_UNLOCK();

    /* Drain remaining items from the RX queue */
    arRxCompleteHandler(pArDev);
}



/*
 * RETURNS
 *   TRUE  - Filter Packet (do not send it to the host)
 *   FALSE - Do not filter this packet (send it to the host)
 *
 * Note: This sw-filter does not filter any packets currently.
 * All packets are passed to the host.
 */
LOCAL A_BOOL
arRxFilterHandler(
    AR_DEV_INFO  *pArDev,
    AR_DESC      *pDesc,
    RECEIVE_INFO *pReceiveInfo
)
{
    WLAN_MGT_MAC_HEADER *wlanHdr;

    if (pReceiveInfo->status != RX_STATUS_OK)
        return FALSE;

    wlanHdr = ((WLAN_MGT_MAC_HEADER *) ARDESC_TO_RXBUF(pDesc));
    A_DATA_CACHE_INVAL(wlanHdr, sizeof(*wlanHdr));

    /* Record the last Beacon RSSI for ANI */

    if ((wlanHdr->frameControl.fSubtype != SUBT_BEACON) ||
        (wlanHdr->frameControl.fType    != FRAME_MGT)   ||
        A_MACADDR_COMP(&wlanHdr->bssId, &pArDev->bssTable[0].bssId))
        return FALSE;

    pArDev->arAniLastBeaconRssi = pReceiveInfo->rssi;
    return FALSE;
}




/*
 * Handle one or more completed RX descriptors on the hardware queue.
 * When we exit this function, if there are still descriptors on the
 * RX queue, re-enable RX interrupts.
 *
 * RETURNS
 *   TRUE -  There are more non-tail descriptors on the RX queue
 *   FALSE - All descriptors have been processed.
 */
A_BOOL
arRxCompleteHandler(
    AR_DEV_INFO *pArDev
)
{
    AR_DESC                 *pDesc;
    A_STATUS                halStatus;
    RECEIVE_INFO            *pReceiveInfo;
    AR_DESC_RX_STATUS       *pDescStatus;
    WDC_RXMSG               *rxMsg;
    char                    *rxBuf;
    A_UINT32                hwIndex;

    /* Update the flag to indicate that HW is doing something */
    pArDev->phyActivityFlag = TRUE;

    /* xxxxLMWxxxx TBD: MIB SUPPORT */

    for (;;) {
        pDesc = arDescQGetHead(&pArDev->rxHwQueue);

        /*
         * With self-linked queues, pDesc should normally not
         * be NULL, except for certain cases such as init time.
         */
        if (pDesc == NULL) {
            return FALSE;
        }

#if defined(RX_SELF_LINKED_QUEUE)

        /*
         * Don't ever remove the self-linked descriptor
         * at the end of the hardware descriptor list.
         */
        if (pDesc->nextPhysPtr == pDesc->thisPhysPtr) {
            /*
             * We've cleaned up everything on the list.
             * No need to re-enable RX interrupts.
             */
            return FALSE;
        }
#else
        if (pDesc == NULL) {
            return FALSE;
        }
#endif

        halStatus = halProcessRxDesc(pArDev, pDesc);
        if (halStatus == A_EBUSY) {
#if !defined(AR_POLL)
            /*
             * No descriptor to process, since the first descriptor is busy.
             * We need to re-enable RX interrupts, though, so that we know
             * when another descriptor completes.
             */
            halEnableInterrupts(pArDev, HAL_INT_RX);
#endif
            return TRUE;
        }

        pDescStatus = &pDesc->status.rx;

        rxBuf = ARDESC_TO_RXBUF(pDesc);

        // Invalidate the cacheline that contains the start
        // of RX data so that we do not corrupt it when
        // writing the RXMSG Header (i.e. RECEIVE_INFO)


        A_DATA_CACHE_INVAL((void *) rxBuf, 1);

        /* pReceiveInfo points to the right spot in the RX Message Buffer! */
        pReceiveInfo = ARDESC_TO_RECEIVE_INFO(pDesc);

        ASSERT(pReceiveInfo);

        switch (halStatus) {
        case A_EBUSY:
            /* Special cased, above. */
            ASSERT(0);
            break;
        case A_ERROR:
            pReceiveInfo->status = RX_STATUS_ERR;
            break;

        case A_PHY_ERROR:
            pReceiveInfo->status = RX_STATUS_PHY_ERR;
            break;

        case A_CRC_ERROR:
            pReceiveInfo->status = RX_STATUS_CRC_ERR;
            break;

        default:
            /*
             * DECRYPT_CRC trumps DECRYPT_MIC.
             * TODO: clean up rest of code to not need error bitmap
             */
            if (pDescStatus->decryptError & DECRYPTERROR_CRC) {
                pReceiveInfo->status = RX_STATUS_DECRYPT_CRC_ERR;
            } else if (pDescStatus->decryptError & DECRYPTERROR_MIC) {
                pReceiveInfo->status = RX_STATUS_DECRYPT_MIC_ERR;
            } else {
                pReceiveInfo->status = RX_STATUS_OK;
            }
            break;
        }

        pReceiveInfo->timestamp.high = 0;
        pReceiveInfo->timestamp.low  = pDescStatus->timestamp;
        pReceiveInfo->frameLength    = pDescStatus->dataLength;
        pReceiveInfo->rate           = pDescStatus->rate;
        pReceiveInfo->antenna        = pDescStatus->antenna;
        pReceiveInfo->rssi           = pDescStatus->rssi;
        pReceiveInfo->channel        = CURRENT_FREQ(pArDev);
        pReceiveInfo->decryptError   = pDescStatus->decryptError;
        pReceiveInfo->keyCacheMiss   = pDescStatus->keyCacheMiss;

        /*
         * Set the connection ID or the key table index.
         * The host can sort this out based on the destination address.
         */
        hwIndex = pDescStatus->hwIndex;
        /*
         * We assume no default hw index is bigger than
         * the shift for TKIP receive
         */
        if (hwIndex >= 32) {
            hwIndex -= 32;
        }
        pReceiveInfo->connIndex = ((hwIndex >= MAX_SHARED_KEYS) ?
                                   (hwIndex - MAX_SHARED_KEYS) :
                                    hwIndex);
        arDescQPopHead(&pArDev->rxHwQueue);

        rxMsg = WDC_RXBUF_TO_RXMSG(rxBuf);

        rxMsg->rxMsgLength   = pReceiveInfo->frameLength +
                               WDC_RXMSG_HEADER_LENGTH;

        rxMsg->msgOpcode     = WDCMSG_DATA_INDICATION;

        rxMsg->generationNum = pArDev->RxGenerationNum;

        if (arRxFilterHandler(pArDev, pDesc, pReceiveInfo)) {
            arRecycleRxBuf(pArDev, rxBuf);
        } else {
            TARG_wdcRxMsgSend(pArDev->masterHandle->masterMsgHandle, rxMsg);
        }

    } // for (;;)
}

#if !defined(AR_POLL)

/*
 * Initiate RX Completion processing.
 * Called from the WLAN driver (interrupt or poll function)
 * when a data is received.
 */
INLINE void
arNotifyRxComplete(
    AR_DEV_INFO *pArDev
)
{
    A_CNTSEM_POST(pArDev->rxCompleteSem);
}


/*
 * This runs in its own context.   It moves received data from the
 * hardware RX queue to the WDC RX Message Dispatch Queue.
 * When there are no new completed RX descriptors to handle,
 * the thread sleeps.  It is awoken when the WLAN RX interrupt
 * handler (or WLAN SR polling loop) calls arNotifyRxComplete.
 */
void
arRxCompleteLoop(
    AR_DEV_INFO *pArDev
)
{
    A_STATUS status;
    for(;;) {
        A_CNTSEM_WAIT(pArDev->rxCompleteSem);
        A_SEM_LOCK(pArDev->rxCompleteLock, WAIT_FOREVER);

        /* Restore power to Awake state is sleeping */
        status = arMacSetAwakeMode(pArDev, TRUE);
        ASSERT(status == A_OK);

        arRxCompleteHandler(pArDev);

        /* Restore power to former or desired state */
        status = arMacSetAwakeMode(pArDev, FALSE);
        ASSERT(status == A_OK);

        A_SEM_UNLOCK(pArDev->rxCompleteLock);

    }
}
#endif

/* Convert from a RX Buffer to the associated AR_DESC */
#define RXBUF_TO_PDESC(rxBuf) \
        ((AR_DESC *)((A_UINT32)WDC_RXBUF_TO_RXMSG(rxBuf) - sizeof(AR_DESC)))

/*
 * Make a receive buffer & descriptor available to hardware.
 * On entry, pDesc is a pointer to a Receive Descriptor, which
 * points to a pre-allocated Receive Buffer of length bufLength.
 */
static void
arReceiveBuf(
    AR_DEV_INFO         *pArDev,
    AR_DESC             *pDesc,
    A_UINT32            bufLength
)
{
    A_STATUS status = A_OK;
    ASSERT(pDesc != NULL);

    status = arMacSetAwakeMode(pArDev, TRUE);
    ASSERT(status == A_OK);

    /* Initialize RX descriptor */
    halSetupRxDesc(pArDev, pDesc, bufLength);

    /* Push descriptor onto HW RX queue */
    arDescQPushTail(&pArDev->rxHwQueue, pDesc, pDesc);

#if defined(RX_SELF_LINKED_QUEUE)
    /*
     * If we were the first buffer (other than the self-linked tail)
     * to be put on the queue, then re-enable RX interrupts.
     */

#if !defined(AR_POLL)
    if (arDescQGetHead(&pArDev->rxHwQueue)->pNextVirtPtr == pDesc)
        halEnableInterrupts(pArDev, HAL_INT_RX);
#endif
#else
    if (arDescQGetHead(&pArDev->rxHwQueue) == pDesc)
        halSetRxDP(pArDev, pDesc->thisPhysPtr);
    
    if (pArDev->bTargetStarted && (pArDev->bRxEnabled == FALSE)) {
        halStartPcuReceive(pArDev);
    }

    if (pArDev->bRxEnabled == TRUE) {
        halEnableReceive(pArDev);
#if !defined(AR_POLL)        
        halEnableInterrupts(pArDev, HAL_INT_RX);
#endif
    }
#endif
    /* Restore MAC to the old state */
    status = arMacSetAwakeMode(pArDev, FALSE);
    ASSERT(status == A_OK);
}

/*
 * This is an interface provided to the WDC Messaging layer.
 * It is intended to be called after the Messaging Layer has
 * successfully sent a Receive Buffer to the Host.  At that
 * point, the buffer can be used again for another receive.
 */
void
arRecycleRxBuf(
    DEVICE_HANDLE       deviceHandle,
    char                *rxBuf
)
{
    AR_DEV_INFO     *pArDev     = (AR_DEV_INFO *)deviceHandle;
    AR_DESC         *rxDesc     = RXBUF_TO_PDESC(rxBuf);

    ASSERT(rxDesc->pBufferVirtPtr.byte == (A_UINT8 *)rxBuf);

#if defined(OVERLAPPING_RX_BUFFERS)
    arReceiveBuf(pArDev, rxDesc, OVERLAP_RXMSG_DATA_LENGTH);
#else
    arReceiveBuf(pArDev, rxDesc, WDC_RXMSG_DATA_LENGTH);
#endif


}


/*
 * Called once during startup to allocate stuctures, buffers, and
 * threads needed for the Receive path.
 *
 * Initialize the Target Receive path and start receiving:
 *   RX Descriptors
 *   RX Messages (which contain RX buffers)
 *   Initialize the RX Message Queue and start the RX Message Dispatcher
 *   Hand RX buffers to the hardware and
 */
A_STATUS
arRxInit(
    AR_DEV_INFO  *pArDev
)
{
    A_STATUS         status;
    A_UINT32         rxMsgSpaceSz;

    void             *rxMsgSpace = NULL;
    AR_DESC          *rxDesc;
    WDC_RXMSG        *rxMsg;
    char             *rxBuf;

    A_UINT32         physRxMsgSpace;
    A_UINT32         physRxDesc;
    A_UINT32         physRxBuf;
    A_UINT32         physRxMsg;

    int i;

    pArDev->RxGenerationNum = 0; /* Restart RX generation number  */

    /* Initialize HW RX Descriptor Queue.... */
    status = arDescQInit(&pArDev->rxHwQueue);
    if (status != A_OK) {
        goto arRxInitError;
    }

#if defined(RX_SELF_LINKED_QUEUE)
    /* ....and mark the queue as having a self-linked tail descriptor. */
    pArDev->rxHwQueue.isTailSelfLinked = TRUE;
#endif

    /*
     * Allocate RX Messages/Descriptors/Buffers.
     * These are the same number of each of these, so they are
     * allocated and managed together.  The storage is layed out
     * with AR_DESC first followed by WDC_RXMSG and ending with
     * the actual data buffer.  Macros are used to convert from
     * an RX data buffer pointer to the associated AR_DESC and
     * to the associated WDC_RXMSG.
     */

#if defined(OVERLAPPING_RX_BUFFERS)
    rxMsgSpaceSz = RX_OUTSTANDING_COUNT * (OVERLAP_RXMSG_LENGTH + sizeof(AR_DESC));
#else
    rxMsgSpaceSz = RX_OUTSTANDING_COUNT * (WDC_RXMSG_LENGTH     + sizeof(AR_DESC));
#endif

    rxMsgSpace     = A_DRIVER_MALLOC(rxMsgSpaceSz);
    physRxMsgSpace = A_DATA_V2P(rxMsgSpace);
    if (!rxMsgSpace) {
        status = A_NO_MEMORY;
        goto arRxInitError;
    }

    /*
     * Prime the pump with RX buffers to receive data.
     * Once a buffer has been delivered to the Host, it
     * is re-used for more incoming data.
     */

    rxDesc     = (AR_DESC *)rxMsgSpace;
    rxMsg      = (WDC_RXMSG *)((A_UINT32)rxDesc + sizeof(AR_DESC));
    rxBuf      = (char *)WDC_RXMSG_TO_RXBUF(rxMsg);

    physRxDesc = physRxMsgSpace;
    physRxMsg  = (A_UINT32)physRxDesc + sizeof(AR_DESC);
    physRxBuf  = (A_UINT32)WDC_RXMSG_TO_RXBUF(physRxMsg);

#if defined(OVERLAPPING_RX_BUFFERS)
#define NEXT_ITEM(item) ((A_UINT32)(item) + OVERLAP_RXMSG_LENGTH + sizeof(AR_DESC))
#else
#define NEXT_ITEM(item) ((A_UINT32)(item) + WDC_RXMSG_LENGTH     + sizeof(AR_DESC))
#endif
    for (i=0; i<RX_OUTSTANDING_COUNT; i++) {

        rxDesc->thisPhysPtr                 = physRxDesc;
        *(char **)(&rxDesc->pBufferVirtPtr) = rxBuf;
        rxDesc->bufferPhysPtr               = physRxBuf;

#if defined(OVERLAPPING_RX_BUFFERS)
        arReceiveBuf(pArDev, rxDesc, OVERLAP_RXMSG_DATA_LENGTH);
#else
        arReceiveBuf(pArDev, rxDesc, WDC_RXMSG_DATA_LENGTH);
#endif

        rxDesc     =   (AR_DESC *)NEXT_ITEM(rxDesc);
        rxMsg      = (WDC_RXMSG *)NEXT_ITEM(rxMsg);
        rxBuf      =      (char *)NEXT_ITEM(rxBuf);
        physRxDesc =              NEXT_ITEM(physRxDesc);
        physRxMsg  =              NEXT_ITEM(physRxMsg);
        physRxBuf  =              NEXT_ITEM(physRxBuf);
    }

    /* Start the RX Completion thread. */
    A_SEM_INIT(pArDev->rxCompleteLock, 0, CREATE_LOCKED);

#if !defined(AR_POLL)
    A_CNTSEM_INIT(pArDev->rxCompleteSem, 0);
    A_TASK_CREATE(arRxCompleteLoop, pArDev);
#endif

    goto arRxInitExit;

arRxInitError:
    if (rxMsgSpace) {
        A_DRIVER_FREE(rxMsgSpace, rxMsgSpaceSz);
    }

arRxInitExit:
    return status;
}
