/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/arReset.c#3 $
 *
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 */

/*
 * Atheros Target Driver implmentation of Reset and related functions.
 */

#include "arDev.h"
#include "halApi.h"
#include "arReceive.h"
#include "arTransmit.h"
#include "arEvent.h"
#if defined(ECOS)
#include "cyg/hal/plf_intr.h"
#endif

/*
 * Reset a target in the most thorough way possible.
 * State and configuration are lost.
 * Not intended for normal, correct operation.
 */
void
TARG_wdcTargetReset(
    IN  DEVICE_HANDLE       deviceHandle,
    OUT A_STATUS            *pStatus
)
{
    HAL_PLATFORM_RESET();
}


void
TARG_wdcTargetSetChannel(
    IN  DEVICE_HANDLE       deviceHandle,
    IN  CHANNEL_BAND        channelBand,
    IN  CHANNEL_FREQ        channelFrequency,
    IN  PHY_MODS            phyModulations,
    IN  A_UINT32            maxRDPower,
    IN  A_UINT32            cfgCtl,
    IN  A_UINT32            twiceAntennaReduction,
	IN  A_BOOL              keepRCContent,
    OUT A_STATUS            *pStatus
)
{
    AR_DEV_INFO     *pArDev     = (AR_DEV_INFO *)deviceHandle;
    WLAN_MODE       wlanMode;
    A_UINT32        rxfilter;

    /* Derive wlanMode */
    pArDev->curPrimeState = BASE_MODE;
    if (channelBand == CHANNEL_BAND_2GHz) {
        if (phyModulations & PHY_MOD_OFDM) {
            if (phyModulations & PHY_MOD_XR) {
                wlanMode = WLAN_MODE_11g_XR;
            } else {
                if (phyModulations & PHY_MOD_TURBO) {
                    wlanMode = WLAN_MODE_11g_TURBO;
                    pArDev->curPrimeState = TURBO_MODE;
                } else {
                    wlanMode = WLAN_MODE_11g;
                }
            }
        } else {
            wlanMode = WLAN_MODE_11b;
        }
    } else {
        ASSERT(channelBand == CHANNEL_BAND_5GHz);
        if (phyModulations & PHY_MOD_XR) {
            wlanMode = WLAN_MODE_11a_XR;
        } else {
            if (phyModulations & PHY_MOD_TURBO) {
                wlanMode = WLAN_MODE_11a_TURBO;
                pArDev->curPrimeState = TURBO_MODE;
            } else {
                wlanMode = WLAN_MODE_11a;
            }
        }
    }

    pArDev->config.chanDesc.channelBand    = channelBand;
    pArDev->config.chanDesc.channelFreq    = channelFrequency;
    pArDev->config.chanDesc.phyModulations = phyModulations;
    pArDev->config.chanDesc.wlanMode       = wlanMode;
    pArDev->config.chanDesc.maxRDPower = (A_UINT16)maxRDPower;
    pArDev->config.chanDesc.cfgCtl = (A_UINT8)cfgCtl;
    pArDev->config.chanDesc.twiceAntennaReduction = (A_INT8)twiceAntennaReduction;

    /* Disable Global Interrupts while we channel change. */
    halDisableInterrupts(pArDev, HAL_INT_GLOBAL);
    A_UNTIMEOUT(&pArDev->periodicCalTimer);
    A_UNTIMEOUT(&pArDev->autoNoiseImmunityTimer);

    /* Detach and disable BMISS interrupts.  The host will
       rejoin this BSS after the channel change */
    halWriteAssocid(pArDev, 0, 0);

    /* Stop asynchronous events */
    arStopEvents(pArDev);

    /* 
	 * Stop Transmit
     * Turbo Switching is the only one using keepRCContent with TRUE value. In the
     * hacked fix to drain the packets during the Turboswitch, the keepRCContent
     * flag is used to determine whether this drvChangeChannel is invoked for turbo switch
     * or not, any other function using the flag that should check this usage.
     */
    arStopTransmit(pArDev, keepRCContent? DONT_WAIT : DO_WAIT);

    /* Stop Receive */
    rxfilter = pArDev->rxFilterReg;
    halSetRxFilter(pArDev, 0);    
    arStopReceive(pArDev);

    /* Disable DSRs and other threads which may attempt to write
       MAC registers during a reset/asleep */
    A_DISABLE_ALL_INTERRUPTS();
    A_TASK_LOCK();

    *pStatus = halReset(pArDev, CURRENT_SERVICE_TYPE(pArDev), &pArDev->config.chanDesc, TRUE);

    /* Renable DSRs & other threads */
    A_TASK_UNLOCK();
    A_ENABLE_ALL_INTERRUPTS();

    /* Renable Global Interrupts */
    halEnableInterrupts(pArDev, HAL_INT_GLOBAL);

    /* Increase Generation Number */
    pArDev->RxGenerationNum++;

    /* Start Receive */
    pArDev->rxFilterReg = rxfilter;
    halSetRxFilter(pArDev, pArDev->rxFilterReg);
    arStartReceive(pArDev);

    /* Start Transmit */
    arStartTransmit(pArDev);

    /* Start aynchronous events */
    arStartEvents(pArDev);
    A_TIMEOUT(&pArDev->autoNoiseImmunityTimer,
              100 * 2,  // Every 2 seconds
              TRUE);

    if (pArDev->config.calibrationInterval != 0) {
        A_TIMEOUT(&pArDev->periodicCalTimer, 
                  100 * pArDev->config.calibrationInterval, TRUE);
    }
    
    return;
}
