/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/arTransmit.c#3 $
 *
 * Copyright (c) 2000-2004 Atheros Communications, Inc., All Rights Reserved
 *
 * This file contains the implementation for transmit logic for the target driver..
 */

#include "wlantype.h"
#include "queue.h"
#include "arDev.h"
#include "arTransmit.h"
#include "arRateCtrl.h"
#include "halApi.h"
#include "hal.h"
#include "histograms.h"
#include "targWdc.h"

int     TX_OUTSTANDING_COUNT;
#define TXBUF_TO_ARDESC(txbuf)  \
       ((AR_DESC *)((A_UINT32)WDC_TXBUF_TO_TXMSG(txbuf) - sizeof(AR_DESC)))
#define TXMSG_TO_ARDESC(txMsg)  \
       ((AR_DESC *)((A_UINT32)txMsg - sizeof(AR_DESC)))
#define ARDESC_TO_TXMSG(pArDesc) \
       ((WDC_TXMSG *)((A_UINT32) pArDesc + sizeof(AR_DESC)))



#if 0
HISTOGRAM_DEFINE(arTx, 1, 256, 2);
HISTOGRAM_DEFINE(arTxDelta, 1, 16, 0);
HISTOGRAM_DEFINE(arTxTimer, 1, 256, 2);
#endif

#ifdef TRANSMIT_RESPONSE_DEBUG
int TxResponseOutstanding = 0;
#endif

#define swrDump(arDev, pSib) (0)
#define apMovePsToSwRetryQueue(_pSib)
#define apUpdateTimForSib(_pArDev, _pSib)
#define apHandlePowerMgt(_pArDev, a, b, c)           A_ERROR

LOCAL void
sequenceNumberRecover(AR_DEV_INFO *pArDev, AR_DESC *pDesc);

LOCAL void
arTxService(AR_DEV_INFO *pArDev, AR_DESC *pTxDesc);

LOCAL A_STATUS
ar5MacSend(AR_DEV_INFO *pArDev, AR_QUEUE_INFO *pQueue,
	   AR_DESC *pHead, A_BOOL psPolledFrame);

LOCAL void
drainRetryQueue(AR_DEV_INFO *pArDev, AR_QUEUE_INFO *pQueue,
		AR_CONN_INFO *pSib);

LOCAL A_STATUS
frameRetransmit(AR_DEV_INFO *pArDev, AR_QUEUE_INFO *pQueue,
		AR_DESC *pThisTail);

LOCAL A_STATUS
swretryLookAhead(AR_DEV_INFO *pArDev, AR_DESC *pTxDesc);

LOCAL A_BOOL
txIsDone(AR_DEV_INFO *pArDev);

void
resetTxQueue(AR_DEV_INFO *pArDev, AR_QUEUE_INFO *pQueue);

void
arStartTransmit(IN AR_DEV_INFO *pArDev)
{

    TARG_wdcInitTxQueues((DEVICE_HANDLE) pArDev);

    /* Unlock the TX thread */
    A_SEM_UNLOCK(pArDev->txMsgDispatchLock);

    /* UnLock the TX completion thread */
    A_SEM_UNLOCK(pArDev->txCompleteLock);

#if !defined(AR_POLL)
    halEnableInterrupts(pArDev, HAL_INT_TX);
#endif

    pArDev->bTxEnabled = TRUE;

    return;
}

static void 
arResetConnTxInfo(AR_DEV_INFO *pArDev)
{
   int connId;
   
   for(connId = 0; connId < AR_MAX_CONN; connId++)
   {
     AR_CONN_INFO *pConnInfo = &pArDev->connTable[connId];
     if(!IS_FREE_AR_CONN(pConnInfo)) {
         pConnInfo->numTxPending = 0;
         pConnInfo->numFilteredPending = 0;
         pConnInfo->needClearDest = TRUE;
         resetTxQueue(pArDev, &pConnInfo->swRetryQueue);
     }
   }
}


void
arStopTransmit(IN AR_DEV_INFO *pArDev, DRAIN_ENUM waitForDrain)
{

    pArDev->bTxEnabled = FALSE;

    /* Lock the TX completion thread */
    A_SEM_LOCK(pArDev->txCompleteLock,    WAIT_FOREVER);

    /* Lock the TX thread */
    A_SEM_LOCK(pArDev->txMsgDispatchLock, WAIT_FOREVER);

	/* 
	 * We can't drain tx queue naturally during turbo switch as it has to be done
	 * very fast
	 */
    if (waitForDrain == DO_WAIT) {
        arTxCompleteHandler(pArDev);
	}

    TARG_wdcFiniTxQueues((DEVICE_HANDLE) pArDev);
    arResetConnTxInfo(pArDev);
    return;
}

void
arRecycleTxDesc(DEVICE_HANDLE  deviceHandle,
                AR_DESC          *pDesc)
{

    WDC_TXMSG *txMsg = ARDESC_TO_TXMSG(pDesc);
    ASSERT(pDesc);
    ASSERT(pDesc->txCookie == 0);

    pDesc                      = TXMSG_TO_ARDESC(txMsg);

    pDesc->bufferPhysPtr       = A_DATA_V2P(txMsg->txData);
    pDesc->nextPhysPtr         = (A_UINT32) NULL;
    pDesc->pNextVirtPtr        = (A_UINT32) NULL;
    pDesc->thisPhysPtr         = A_DATA_V2P(pDesc);
    pDesc->pBufferVirtPtr.byte = (A_UCHAR *) txMsg->txData;
    pDesc->pTxFirstDesc        = pDesc;
    pDesc->pTxLastDesc         = pDesc;
    pDesc->swretryCount        = 0;  
    pDesc->staleFlag           = 0;
    pDesc->hw.word[0]          = 0;
    pDesc->hw.word[1]          = 0;
    pDesc->hw.word[2]          = 0;
    pDesc->hw.word[3]          = 0;
    pDesc->hw.word[4]          = 0;
    pDesc->hw.word[5]          = 0;

    TARG_wdcDataMsgRecv(txMsg);
    return;

}



static __inline__
void
arRecycleTxDescChain(IN DEVICE_HANDLE  deviceHandle,
                     IN AR_DESC       *pDesc) {
    AR_DESC *pDescTmp;

    do {
        pDescTmp = pDesc->pNextVirtPtr;
        arRecycleTxDesc(deviceHandle, pDesc);
        pDesc = pDescTmp;
    } while (pDesc != NULL);

    return;
}


// Handle TX Completion Interrupts
// WLAN -> TX Interrupt -> USB Credit / WDCMSG_TX_IND

LOCAL A_BOOL
arSendTxComplete(AR_DEV_INFO    *pArDev,
		 TX_COOKIE      txCookie,
		 TRANSMIT_INFO  *transmitInfo) {

    WDC_MSG   *txCompleteMsg;
    MASTER_MSG_HANDLE masterMsgHandle = pArDev->masterHandle->masterMsgHandle;

    txCompleteMsg        = TARG_wdcCtrlMsgCreate(masterMsgHandle,
                                                 WDCMSG_SEND_COMPLETE);
    ASSERT(txCompleteMsg);
    txCompleteMsg->msgId = (A_UINT32)txCookie;

    TARG_wdcMsgAddData(txCompleteMsg,
		       sizeof(TRANSMIT_INFO),
		       (void *) transmitInfo);
    TARG_wdcCtrlMsgSend(masterMsgHandle, txCompleteMsg);

#ifdef TRANSMIT_RESPONSE_DEBUG
    A_TASK_LOCK();
    TxResponseOutstanding--;
    ASSERT(TxResponseOutstanding >= 0);
    A_TASK_UNLOCK();
#endif

    return TRUE;
}

#if !defined(AR_POLL)
INLINE void
arNotifyTxComplete(AR_DEV_INFO *pArDev)
{
	A_CNTSEM_POST(pArDev->txCompleteSem);
}

void
arFlushTxCompleteMsg(AR_DEV_INFO *pArDev)

{
    int              semcount   = 0;

    arTxCompleteHandler(pArDev);

    do {
        A_CNTSEM_PEEK(pArDev->txCompleteSem, &semcount);
        diag_printf("txcc:%d\n", semcount);

        if (semcount) {
            /* Wait for a TX Message to become available. */
            A_CNTSEM_WAIT(pArDev->txCompleteSem);
        }
    } while (semcount);
}

LOCAL void
arTxCompleteLoop(AR_DEV_INFO *pArDev)
{
    A_STATUS status;

    A_SEM_LOCK(pArDev->txCompleteLock, WAIT_FOREVER);
#if !defined(AR_POLL)
    halEnableInterrupts(pArDev, HAL_INT_TX);
#endif
    A_SEM_UNLOCK(pArDev->txCompleteLock);

    for(;;) {
        A_CNTSEM_WAIT(pArDev->txCompleteSem);
        A_SEM_LOCK(pArDev->txCompleteLock, WAIT_FOREVER);

        /* Restore power to Awake state is sleeping */
        status = arMacSetAwakeMode(pArDev, TRUE);
        ASSERT(status == A_OK);

        arTxCompleteHandler(pArDev);
        
#if !defined(AR_POLL)
        halEnableInterrupts(pArDev, HAL_INT_TX);
#endif
        /* Restore power to former or desired state */
        status = arMacSetAwakeMode(pArDev, FALSE);
        ASSERT(status == A_OK);
        A_SEM_UNLOCK(pArDev->txCompleteLock);
    }
}

#endif

// TX Completion Queue

LOCAL A_STATUS
TARG_wdcTxMsgQInit(MASTER_HANDLE masterHandle)
{
    A_STATUS  status = A_OK;
    int       i;

    TAILQ_INIT(&masterHandle->TxMsgQHead);
    TAILQ_INIT(&masterHandle->TxMsgQFree);

    for (i=0; i<TX_OUTSTANDING_COUNT; i++) {
        struct TxMsgQ *msgQItem;

        msgQItem = A_DRIVER_MALLOC(sizeof(*msgQItem));
        if (!msgQItem) {
            ASSERT(0);
            status = A_NO_MEMORY;
            goto msgQInitExit;
        }
        A_MEM_ZERO(msgQItem, sizeof(*msgQItem));

        TAILQ_INSERT_TAIL(&masterHandle->TxMsgQFree, msgQItem, Qlinkage);
    }

    A_CNTSEM_INIT(masterHandle->TxMsgQSem,  0);

 msgQInitExit:
    return status;
}


void
TARG_wdcTxMsgEnqueue(
    AR_DEV_INFO   *pArDev,
    WDC_TXMSG     *txMsg
)
{
    MASTER_HANDLE masterHandle;
    struct TxMsgQ *msgQItem;

    masterHandle = pArDev->masterHandle;

    /* Synchronize with TARG_wdcTxMsgDequeue, trying to remove a Message */
    A_TASK_LOCK();

    /* Get the first msgQItem from the free list.... */
    msgQItem = TAILQ_FIRST(&masterHandle->TxMsgQFree);
    ASSERT(msgQItem);

    /* ...and pop it off the free list */
    TAILQ_REMOVE(&masterHandle->TxMsgQFree, msgQItem, Qlinkage);
    ASSERT(msgQItem);
    ASSERT(msgQItem->txMsg == NULL);

    msgQItem->txMsg = txMsg;

    /* Now put the msgQItem at the end of the Tx Message Dispatch Queue */
    TAILQ_INSERT_TAIL(&masterHandle->TxMsgQHead, msgQItem, Qlinkage);

    A_TASK_UNLOCK();

    /* Notify TX Message Dispatcher that a new message awaits. */
    A_CNTSEM_POST(masterHandle->TxMsgQSem);

}

WDC_TXMSG *
TARG_wdcTxMsgDequeue(
    MASTER_HANDLE masterHandle
)
{
    struct TxMsgQ *msgQItem;
    WDC_TXMSG *txMsg;

    /* Wait for a TX Message to become available. */
    A_CNTSEM_WAIT(masterHandle->TxMsgQSem);

    /* Synchronize with TARG_wdcTxMsgEnqueue, trying to add a new Message */
    A_TASK_LOCK();

    /* Pull the first Message off the TX Message Dispatch Queue */
    msgQItem = TAILQ_FIRST(&masterHandle->TxMsgQHead);
    ASSERT(msgQItem);
    TAILQ_REMOVE(&masterHandle->TxMsgQHead, msgQItem, Qlinkage);

    txMsg = msgQItem->txMsg;
    ASSERT(txMsg);

    msgQItem->txMsg = NULL;
    TAILQ_INSERT_TAIL(&masterHandle->TxMsgQFree, msgQItem, Qlinkage);

    A_TASK_UNLOCK();

    return txMsg;
}


void
TARG_wdcTxMsgFlushqueue(
    AR_DEV_INFO     *pArDev
    )
{
    WDC_TXMSG       *txMsg = NULL;
    int             semcount   = 0;
    int             index;
    MASTER_HANDLE   masterHandle;

    masterHandle = pArDev->masterHandle;
    A_CNTSEM_PEEK(masterHandle->TxMsgQSem, &semcount);
    for (index=0;index<semcount;index++) {
        txMsg = TARG_wdcTxMsgDequeue(masterHandle);
        if (txMsg != NULL) {
            A_DATA_CACHE_INVAL(txMsg,64);
            if (txMsg->msgOpcode == WDCMSG_FLUSH) {
                TARG_wdcTxMsgEnqueue(pArDev,txMsg);
            } else {    
                /* Recycle TxMsg */
                arRecycleTxDesc(pArDev, TXMSG_TO_ARDESC(txMsg));
	    }
        } else {
	    ASSERT(0);
	    break;
	}
    }
}


LOCAL void
TARG_wdcTxFlushHandler(MASTER_HANDLE masterHandle, WDC_TXMSG *txMsg) 
{
    AR_DEV_INFO    *pArDev     = (AR_DEV_INFO *) masterHandle->deviceHandle;
    int            count = 0;

    pArDev->txBarrierInProgress = TRUE;

    arRecycleTxDesc(pArDev, TXMSG_TO_ARDESC(txMsg));

    /*
     * Wait for all previous TX requests to complete.
     * We hold further TX requests that have been queued
     * until the BARRIER has completed.
     */
    
    A_CNTSEM_INIT(pArDev->txWaitDoneSem, 0); // migrate to condvar
    while (!txIsDone(pArDev)) {
        if (count > 4) break;
        A_CNTSEM_TIMED_WAIT(pArDev->txWaitDoneSem,
                            cyg_current_time() + 100);
        count ++;
    }
    
    /* XXX: Note this function assumes the txMsgDispatchLock is acquired */

    A_SEM_UNLOCK(pArDev->txMsgDispatchLock);            
    A_CNTSEM_POST(pArDev->ctrlDispatcherBarrier);
    A_CNTSEM_WAIT(pArDev->dataDispatcherBarrier);
    pArDev->txBarrierInProgress = FALSE;

    A_SEM_LOCK(pArDev->txMsgDispatchLock, WAIT_FOREVER);
    return;
}

void
TARG_wdcTxMsgHandler(MASTER_HANDLE masterHandle, WDC_TXMSG *txMsg)
{
    A_STATUS        status;
    AR_DESC        *pDesc      = NULL;
    AR_QUEUE_INFO  *pArTxQueue = NULL;
    AR_BSS_INFO    *pArBss     = NULL;
    AR_CONN_INFO   *pArConn    = NULL;
    AR_DEV_INFO    *pArDev     = (AR_DEV_INFO *) masterHandle->deviceHandle;

    if (txMsg == NULL)
        return;

    ASSERT(txMsg->msgOpcode == WDCMSG_SEND);

    pDesc      = TXMSG_TO_ARDESC(txMsg);
    pArTxQueue = &pArDev->txHwQueues[txMsg->txQueueId];

    /* Validate Connection */
    if (txMsg->txConnId != WDC_INVALID_CONN_ID) {
        TRANSMIT_INFO    txInfo;

        txInfo.status = TX_ERR;
        pArConn       = &pArDev->connTable[txMsg->txConnId];

        ASSERT(!IS_FREE_AR_CONN(pArConn));
        if (IS_FREE_AR_CONN(pArConn)) {
            /* Recycle TxMsg */
            if (txMsg->responseDesired && txMsg->msgId) {
                arSendTxComplete(pArDev,
                                 (TX_COOKIE) txMsg->msgId,
                                 &txInfo);
                txMsg->msgId = 0;
                arRecycleTxDesc(pArDev, pDesc);
                return;
            }
        }

        /* Use BSS that is bound to the connection */
        pArBss = pArConn->pOpBss;

        /* Validate BSS */

#if 0
        // xxx: Workaround until we have wdcDeleteConnection()
        ASSERT(IS_VALID_AR_BSS(pArBss));
        if (!IS_VALID_AR_BSS(pArBss)) {
            /* Recycle TxMsg */
            if (txMsg->responseDesired && txMsg->msgId) {
                arSendTxComplete(pArDev,
                                 (TX_COOKIE) txMsg->msgId,
                                 &txInfo);
                txMsg->msgId = 0;
                arRecycleTxDesc(pArDev, pDesc);
                return;
            }
            return;
        }
#else
        if (!IS_VALID_AR_BSS(pArBss)) {
            pArConn = NULL;
            pArBss  = NULL;
        }
#endif

    } else {
        /* Connectionless transmit (e.g. probe requests) */
        /* Note pArBSS and pArConn are NULL at this point */
    }

    ASSERT(pDesc);
    ASSERT(txMsg);
    ASSERT(pArTxQueue);

    // AR_DESC - Hardware
    ASSERT(pDesc->bufferPhysPtr == (A_DATA_V2P(txMsg->txData)));
    pDesc->hw.txControl.more         = 0;
    pDesc->hw.txControl.frameLength  = (txMsg->txMsgLength -
                                        WDC_TXMSG_HEADER_LENGTH);
    pDesc->hw.txControl.bufferLength = txMsg->bufferLength;
    pDesc->nextPhysPtr               = (A_UINT32) NULL;

    // AR_DESC - Software
    ASSERT(pDesc->thisPhysPtr         == (A_DATA_V2P(pDesc)));
    ASSERT(pDesc->pBufferVirtPtr.byte == (A_UCHAR *) txMsg->txData);

    pDesc->pNextVirtPtr              = (A_UINT32) NULL;
    pDesc->pTxFirstDesc              = pDesc;
    pDesc->pTxLastDesc               = pDesc;
    pDesc->pArConnInfo               = pArConn;
    pDesc->pOpBss                    = pArBss;

    /* Translate the conn Id in key cache HW index */
    if (txMsg->txConnId != WDC_INVALID_CONN_ID) {
        pDesc->hwIndex = txMsg->txConnId + MAX_SHARED_KEYS;
    } else {
        pArBss = &pArDev->bssTable[0];
        pDesc->hwIndex = pArBss->config.bssDefaultKey;
    }

    if (txMsg->responseDesired)
        pDesc->txCookie              = (TX_COOKIE)txMsg->msgId;
    else
        pDesc->txCookie              = 0;

#ifdef TRANSMIT_RESPONSE_DEBUG
    if (txMsg->responseDesired) {
        A_TASK_LOCK();
        TxResponseOutstanding++;
        ASSERT(TxResponseOutstanding <= TX_OUTSTANDING_COUNT);
        A_TASK_UNLOCK();
    }
#endif
    status = ar5MacSend(pArDev, pArTxQueue, pDesc, FALSE);

}

// USB -> WLAN
void
TARG_wdcTxMsgDispatchLoop(MASTER_HANDLE masterHandle)
{
    AR_DEV_INFO    *pArDev     = (AR_DEV_INFO *) masterHandle->deviceHandle;
    WDC_TXMSG      *txMsg      = NULL;
    A_STATUS        status;
    int             iSemCount;

    A_TASK_SET_HIGH_PRIORITY();

    for (;;) {

        txMsg = TARG_wdcTxMsgDequeue(masterHandle);
        

        A_SEM_LOCK(pArDev->txMsgDispatchLock, WAIT_FOREVER);
        A_DATA_CACHE_INVAL(txMsg,sizeof(txMsg));

        /* Restore power to Awake state is sleeping */
        status = arMacSetAwakeMode(pArDev, TRUE);
        ASSERT(status == A_OK);

        do {
            if (txMsg->msgOpcode == WDCMSG_FLUSH) {
                TARG_wdcTxFlushHandler(masterHandle, txMsg);
            } else {
                if (pArDev->bTargetStarted == FALSE) {
                    arRecycleTxDesc(pArDev, TXMSG_TO_ARDESC(txMsg));
                } else {
                    TARG_wdcTxMsgHandler(masterHandle, txMsg);
                }
            }

            A_CNTSEM_PEEK(masterHandle->TxMsgQSem, &iSemCount);
            if (iSemCount) {
                txMsg = TARG_wdcTxMsgDequeue(masterHandle);
            } else {
                txMsg = NULL;
            }
        } while(txMsg != NULL);

#if !defined(AR_POLL)
        /* Proactively reap any WLAN TX buffers */
        arTxCompleteHandler(pArDev);
#endif
        A_SEM_UNLOCK(pArDev->txMsgDispatchLock);            

        /* Restore power to former or desired state */
        status = arMacSetAwakeMode(pArDev, FALSE);
        ASSERT(status == A_OK);
    }
}


A_STATUS
TARG_wdcTxMsgDispatchInit(MASTER_HANDLE masterHandle)
{
    AR_DEV_INFO *pArDev = (AR_DEV_INFO *) masterHandle->deviceHandle;
    A_STATUS     status = A_OK;

    status = TARG_wdcTxMsgQInit(masterHandle);
    if (status != A_OK) {
        goto TxMsgDispatchInitExit;
    }

	A_SEM_INIT(pArDev->txMsgDispatchLock, 0, CREATE_UNLOCKED);

#if defined(AGGRESSIVE_TASK_STACKSZ)
    // Measured usage on 05/24/2004: 1224
    A_TASK_CREATE_STACKSZ(TARG_wdcTxMsgDispatchLoop, masterHandle, 1536);
#else
    A_TASK_CREATE(TARG_wdcTxMsgDispatchLoop, masterHandle);
#endif

 TxMsgDispatchInitExit:
    return status;
}

/*
 * Allocate buffers to be used by hardware for compression.
 */
LOCAL void
compressionInit(AR_DEV_INFO *pArDev)
{
#if defined(COMPRESSION_ENABLED)
    if (pArDev->hwInfo.compressSupport) {
        pArDev->compBufKbs = COMP_BUF_KBS;
        pArDev->virtCompBuffers =
            (A_UINT32) A_DRIVER_MALLOC(
                            (COMP_BUF_COUNT * COMP_BUF_KBS * 1024) +
                            COMP_BUF_ALIGN_SIZE);

        ASSERT(pArDev->virtCompBuffers != (A_UINT32) NULL);

        pArDev->virtCompBuffers = pArDev->virtCompBuffers + COMP_BUF_ALIGN_SIZE;
        pArDev->virtCompBuffers = pArDev->virtCompBuffers & ~COMP_BUF_ALIGN_MASK;
        pArDev->physCompBuffers = A_DATA_V2P(pArDev->virtCompBuffers);
    }
#else
    return;
#endif /* COMPRESSION_ENABLED */
}


A_STATUS
arTxInit(
         AR_DEV_INFO *pArDev
) {
    A_STATUS       status;
    WDC_TXMSG     *txMsgSpace = NULL;
    WDC_TXMSG     *txMsg;
    A_UINT32       txMsgSpaceSz;
    MASTER_HANDLE  masterHandle = pArDev->masterHandle;
    int            i;

    compressionInit(pArDev);

    pArDev->hwInfo.numTxQueues  = HAL_NUM_TX_QUEUES;

    for (i = 0; i < AR_MAX_QUEUES; i++) {
        status = arDescQInit(&pArDev->txHwQueues[i]);
        if (status != A_OK) {
            goto arTxInitError;
        }
    }

    /* Initialize barriers (supports WDCMSG_FLUSH). */
    pArDev->txBarrierInProgress = FALSE;
    A_CNTSEM_INIT(pArDev->txWaitDoneSem, 0);
    A_CNTSEM_INIT(pArDev->dataDispatcherBarrier, 0);
    A_CNTSEM_INIT(pArDev->ctrlDispatcherBarrier, 0);

   txMsgSpaceSz = TX_OUTSTANDING_COUNT * (WDC_TXMSG_LENGTH + sizeof(AR_DESC));
    txMsgSpace   = (WDC_TXMSG *) A_DRIVER_MALLOC(txMsgSpaceSz);
    if (!txMsgSpace) {
        status = A_NO_MEMORY;
        goto arTxInitError;
    }
    A_MEM_ZERO(txMsgSpace, txMsgSpaceSz);

    txMsg = (WDC_TXMSG *) ((A_UINT32) txMsgSpace + sizeof(AR_DESC));
    for (i = 0; i < TX_OUTSTANDING_COUNT; i++) {
        AR_DESC *pDesc;

        pDesc                      = TXMSG_TO_ARDESC(txMsg);
        pDesc->bufferPhysPtr       = A_DATA_V2P(txMsg->txData);
        pDesc->nextPhysPtr         = (A_UINT32) NULL;
        pDesc->pNextVirtPtr        = (A_UINT32) NULL;
        pDesc->thisPhysPtr         = A_DATA_V2P(pDesc);
        pDesc->pBufferVirtPtr.byte = (A_UCHAR *) txMsg->txData;
        pDesc->pTxFirstDesc        = pDesc;
        pDesc->pTxLastDesc         = pDesc;

        TARG_wdcDataMsgRecv(txMsg);
        txMsg = (WDC_TXMSG *)((A_UINT32) txMsg +
                              WDC_TXMSG_LENGTH +
                              sizeof(AR_DESC));
    }

    status = TARG_wdcTxMsgDispatchInit(masterHandle);
    if (status != A_OK) {
        goto arTxInitError;
    }

    /* Start the TX Completion thread */
	A_SEM_INIT(pArDev->txCompleteLock, 0, CREATE_LOCKED);


#if !defined(AR_POLL)
	A_CNTSEM_INIT(pArDev->txCompleteSem, 0);
    A_TASK_CREATE(arTxCompleteLoop, pArDev);
#endif

    /* Initialize BSS queues */
    pArDev->bssTable[0].pTxQueue    = &pArDev->txHwQueues[TXQ_ID_FOR_DATA];
    pArDev->bssTable[0].pBurstQueue = &pArDev->txHwQueues[TXQ_ID_FOR_GBURST];

    ASSERT(AR_MAX_BSS == 1);

    goto arTxInitExit;

 arTxInitError:

#if defined(ECOS)
    ASSERT(0);
#endif

    if (txMsgSpace) {
	A_DRIVER_FREE(txMsgSpace, txMsgSpaceSz);
    }
 arTxInitExit:
    return status;
}


void
TARG_wdcSetupTxQueue(
    IN  DEVICE_HANDLE       deviceHandle,
    IN  A_UINT32            wdcTxQueueId,
    IN  TXQ_ATTRIBUTES     *pTxQueueInfo)
{
    AR_DEV_INFO        *pArDev = (AR_DEV_INFO *)deviceHandle;
    HAL_TX_QUEUE_INFO  queueParam;
    A_UINT32           halTxQueueNum = HAL_NUM_TX_QUEUES + 1;
    AR_QUEUE_INFO       *pArQueueInfo;

    if (pArDev->pHalInfo->txQueueAllocMask & (1 << pTxQueueInfo->priority))
        return;

    ASSERT(wdcTxQueueId < HAL_NUM_TX_QUEUES);
    if (!(wdcTxQueueId < HAL_NUM_TX_QUEUES)) {
        return;
    }

    AR_HAL_QUEUE_INFO_INIT(&queueParam, pTxQueueInfo);
    queueParam.mode          = TXQ_MODE_NORMAL;
    queueParam.priority      = pTxQueueInfo->priority;
    queueParam.burstTime     = pTxQueueInfo->burstTime;
    queueParam.qFlags        = TXQ_FLAG_TXINT_ENABLE;
#if defined(BURST_ENABLED)
    queueParam.qFlags       |= TXQ_FLAG_TXDESCINT_ENABLE;
#endif

#if defined(COMPRESSION_ENABLED)
    if (pTxQueueInfo->compression) {
        ASSERT(pArDev->physCompBuffers);
        /* 
         * TBD: When we want to support more than one compression buffer,
         * they'll all be pre-allocated at physCompBuffers.  This is the
         * place to choose a buffer for this particular queue.
         *
         * XXX: This code assumes we are using at most two Queues:
         * TXQ_ID_FOR_DATA and TXQ_ID_FOR_GBURST.  
         */

        if (wdcTxQueueId == TXQ_ID_FOR_GBURST)
            queueParam.physCompBuf = ((A_UINT32) pArDev->physCompBuffers +
                                      (COMP_BUF_KBS * 1024));
        else 
            queueParam.physCompBuf = pArDev->physCompBuffers;

        queueParam.qFlags     |= TXQ_FLAG_COMPRESSION_ENABLE;
    }
#endif /* COMPRESSION_ENABLED */

    halTxQueueNum = halSetupTxQueue(pArDev, &queueParam);
    ASSERT(halTxQueueNum < pArDev->hwInfo.numTxQueues);
    if (halTxQueueNum >= pArDev->hwInfo.numTxQueues) {
        return;
    }

    pArQueueInfo = &pArDev->txHwQueues[halTxQueueNum];
    pArQueueInfo->halTxQueueNum = halTxQueueNum;
    pArQueueInfo->wdc.burstTime = queueParam.burstTime;

    //xxxxPGSxxxx - temporary (for now).  Init target side tx queue structure
    A_SEM_INIT(pArQueueInfo->qSem, 0, CREATE_UNLOCKED);

    return;

}

void
TARG_wdcInitTxQueues(IN DEVICE_HANDLE deviceHandle) {
    AR_DEV_INFO       *pArDev = (AR_DEV_INFO *) deviceHandle;
    A_UINT32           uiTmp;

    ASSERT(HAL_NUM_TX_QUEUES == AR_MAX_QUEUES);

    for (uiTmp = 0; uiTmp < HAL_NUM_TX_QUEUES; uiTmp++) {
        if (pArDev->txQueueInitialized & (1 << uiTmp)) {
            TARG_wdcSetupTxQueue(deviceHandle, uiTmp,
                                 &pArDev->txQueueInfo[uiTmp]);
        }
    }
    return;
}

void
TARG_wdcFiniTxQueues(IN DEVICE_HANDLE deviceHandle) {
    AR_DEV_INFO       *pArDev = (AR_DEV_INFO *) deviceHandle;
    A_UINT32           uiTmp;
    A_UINT32           queueNum;

#ifndef DMA_STOP_WAIT_TIME
#define DMA_STOP_WAIT_TIME 4
#endif

    for (uiTmp = 0; uiTmp < HAL_NUM_TX_QUEUES; uiTmp++) {

        if (pArDev->txQueueInitialized & (1 << uiTmp)) {
            queueNum = pArDev->txQueueInfo[uiTmp].priority;
            if (pArDev->pHalInfo->txQueueAllocMask & (1<<queueNum)) {
                halStopTxDma(pArDev, queueNum, DMA_STOP_WAIT_TIME);
                halReleaseTxQueue(pArDev, queueNum);
                resetTxQueue(pArDev, &pArDev->txHwQueues[queueNum]);
            }
        }
    }

    return;
}


/*
 * arQueueTxDescriptor - queue descriptor to transmit queue tail
 *
 * RETURNS: void
 */
static __inline__
void
arQueueTxDescriptorLocked(AR_DEV_INFO *pArDev, AR_QUEUE_INFO *pQueue,
                          AR_DESC *pDesc, AR_DESC *pTail)
{
    ASSERT(pTail->pNextVirtPtr == NULL);
    ASSERT(pTail->nextPhysPtr == 0);

#ifdef SWCOHERENCYWB
	{
		AR_DESC *pVarDesc;

		for (pVarDesc = pDesc; pVarDesc; pVarDesc = pVarDesc->pNextVirtPtr) {
			A_DATA_CACHE_FLUSH(pVarDesc->pBufferVirtPtr, pVarDesc->hw.txControl.bufferLength);
			A_DESC_CACHE_FLUSH(pVarDesc);
		}
	}
#endif

    if (pQueue->pDescQueueTail == NULL) {
        ASSERT(pQueue->pDescQueueHead == NULL);
        pQueue->pDescQueueHead = pDesc;
        pQueue->isDmaStopped   = TRUE;
    } else {
        pQueue->pDescQueueTail->pNextVirtPtr = pDesc;
        pQueue->pDescQueueTail->nextPhysPtr  = pDesc->thisPhysPtr;
    }
    A_PIPEFLUSH(); // sync the write buffer

    pQueue->pDescQueueTail = pTail;

    if (pQueue->isDmaStopped) {
        halSetTxDP(pArDev, pQueue->halTxQueueNum, pDesc->thisPhysPtr);
        pQueue->isDmaStopped = FALSE;
    }

    return;
}

/*
 * arTxService - prepare the frame descriptor for transmit by
 *  setting up the rate, duration etc..
 *  prepare the frame for transmit by putting it in the
 *  right byte order
 */

LOCAL void
arTxService(AR_DEV_INFO *pArDev, AR_DESC *pTxDesc)
{
    halSetupTxDesc(pArDev, pTxDesc, pTxDesc->hwIndex);
    pTxDesc->pTxLastDesc->status.tx.status = AR_NOT_DONE;
}

AR_STATS dummyStats;

/*
 * arProcessTxQueueLocked - interrupt service routines
 *
 *   This routine is called by processInt to handle the transmitted interrupt.
 *   It releases the tx buffers to the buffer pool.  It also releases the
 *   descriptors to the tx queue.
 */


void
arProcessTxQueueLocked(AR_DEV_INFO *pArDev, AR_QUEUE_INFO *pQueue)
{
    AR_STATS               *pDevStats = &pArDev->devStats;
    AR_STATS               *pSibStats;
    AR_DESC                *pFirstDesc, *pTxDesc;
    AR_DESC_TX_STATUS      *pTxStatus;
    WLAN_FRAME_HEADER      *pHdr;
    AR_CONN_INFO           *pSib;
    A_STATUS                status;
    TRANSMIT_INFO           txInfo;

#if 0
    HISTOGRAM_START_COUNTER(arTx);
#endif

    /*
     * iterate over the frames remembering that only the first descriptor
     * contains control information and only the last descriptor has its
     * status updated by hw; also whenever the hw runs out of tx descs, it
     * will expect to use the final descriptor as a springboard to get to a
     * (newly-chained) desc when it is restarted; so software must not free
     * the final descriptor but keep it as a holding descriptor - the sw
     * does so by marking it with the staleFlag;
     * always start the loop from the queue head as the station side wlan
     * layer tx completion code in the loop may recursively call this routine,
     * or the resetQueue routine, or may queue up new frames to this queue
     */
    while ((pFirstDesc = pQueue->pDescQueueHead)) {
        AR_DESC *pFreeDesc = NULL;

#if 0
        HISTOGRAM_START_TIMER(arTxTimer);
#endif

        /* skip the stale/holding descriptor if any */
        if (pFirstDesc->staleFlag) {
            if (!pFirstDesc->pNextVirtPtr) {
                /* no more descriptors to process */

                if (pArDev->txBarrierInProgress) {
                    A_CNTSEM_POST(pArDev->txWaitDoneSem);
                }

                break;
            }
            pFirstDesc = pFirstDesc->pNextVirtPtr;
        }

        if (pFirstDesc->pArConnInfo && 
            (pFirstDesc->pArConnInfo->bInUse == FALSE)) {
            isrPrintf("%s:%d Race avoidance.\n", __FILE__, __LINE__);
            pFirstDesc->pArConnInfo = NULL;
        }

        /* look at the status on the last descriptor only */
        pTxDesc = pFirstDesc->pTxLastDesc;
        ASSERT(pTxDesc->pTxFirstDesc == pFirstDesc);

        pTxStatus = &pTxDesc->status.tx;
    
        if (pTxStatus->status == AR_NOT_DONE) {
            status = halProcessTxDesc(pArDev, pTxDesc);
            if (status == A_EBUSY) {
                break;
            }
            ASSERT(status == A_OK);
        }

        if (swretryLookAhead(pArDev, pTxDesc) != A_OK) {
            pTxStatus->status = AR_NOT_DONE;
            break;
        }

        ASSERT(pTxStatus->status != AR_NOT_DONE);
        pSib = pFirstDesc->pArConnInfo;

        /* make the STA side watchdogs happy */
        /* xxxxPGSxxxx - TODO - move to host */
        pArDev->TransmitBusy    = FALSE;
        pArDev->phyActivityFlag = TRUE;

#if 0
        HISTOGRAM_COUNT(arTx);
#endif

        /*
         * Pass failed unicast frames to swretry.  Group frames
         * do not have a SIB.
         */
        if ((pTxStatus->status != AR_TRANSMIT_OK) && pSib &&
            pArDev->config.swTxRetries != 0)
        {
            ASSERT(!isGrp(&pFirstDesc->pBufferVirtPtr.header->address1));
            if (frameRetransmit(pArDev, pQueue, pTxDesc) == A_OK) {
                 /* swretry has re-organized the transmit queue - loop again */
                AR_FF_QUEUE_DRAIN(pArDev, pQueue);
                continue;
            }
        }

        /*
         * book-keeping; cabQ frameCounting is lacking for now as
         * the queue is populated in the ISR context and would need
         * some thought on doing it in a race free manner
         */
        ASSERT(pQueue->qFrameCount || pQueue->isCabQueue);
        pQueue->qFrameCount--;
        if (pSib) {


#if 0
            /* xxxxPGSxxxx - remove this assert for now. Not sure what to do with the local SIB */
            ASSERT(pSib->numTxPending ||
                  ((pSib == pArDev->localSta) && pQueue->isCabQueue));
#endif
            pSib->numTxPending--;
            apUpdateTimForSib(pArDev, pSib);
        }

        /*
         * make this the holding descriptor and advance the queue head; do this
         * and subsequent freeing of the frames in this order to keep it race
         * free w.r.t. the ISR code working on the cabQ in SWBA - we need to
         * move the head before freeing the frames; also do this before calling
         * upper layer tx completion code on these frames since some of the upper
         * layer tx completion code on the station will recursively call this
         * procedure, or reset this queue, or try to transmit via this queue
         */
        pTxDesc->staleFlag    = 1;
        pTxDesc->pTxFirstDesc = pTxDesc;
        pTxDesc->pTxLastDesc  = pTxDesc;
        if (pTxDesc != pQueue->pDescQueueHead) {
            pFreeDesc = pQueue->pDescQueueHead;
            pQueue->pDescQueueHead = pTxDesc;
        }
        AR_FF_QUEUE_DRAIN(pArDev, pQueue);

        pHdr = pFirstDesc->pBufferVirtPtr.header;

#if defined(NO_CONN_STATS)
        pSibStats = &dummyStats;
#else
        pSibStats = pSib ? &pSib->stats : &dummyStats;
#endif

        if (pTxStatus->status == AR_TRANSMIT_OK) {
            WLAN_MACADDR *pDestAddr;

#if !defined(ECOS_NOTDONE)
            END_ERR_ADD(pArDev->pOSHandle, MIB2_OUT_UCAST, +1);
#else
//	    ECOS_NOTDONE_XXX_C("MIB2_OUT_UCAST");
#endif

            AR_UPDATE_STATS(pDevStats, pSibStats, GoodTransmits,
                                       1 + pFirstDesc->ffFlag);
            AR_UPDATE_STATS(pDevStats, pSibStats, GoodTransmitBytes,
                                       pFirstDesc->hw.txControl.frameLength);

            pDestAddr = &pHdr->address1;
            if ((pArDev->config.serviceType == WLAN_STA_SERVICE) &&
                (pArDev->bssTable[0].config.bssType == INFRASTRUCTURE_BSS))
            {
                pDestAddr = &pHdr->address3;
            }

            if (!isGrp(pDestAddr)) {
                AR_UPDATE_STATS(pDevStats, pSibStats, TxUnicastFrames, 1);

                /* TURBO_PRIME */
#ifdef BUILD_AP
                if (pArDev->config.abolt & ABOLT_TURBO_PRIME &&
                    pSib &&
                    pSib->staState & STATE_ASSOC)
                {
                    if (pSib->athAdvCapElement.info.useTurboPrime) {
                        pArDev->turboPrimeInfo.primeByteCount += pFirstDesc->hw.txControl.frameLength;
                    } else {
                        pArDev->turboPrimeInfo.legacyActivity = A_MS_TICKGET();
                        pArDev->turboPrimeInfo.legacyPresent = TRUE;
                    }
                }
#endif

            } else if (isBcast(pDestAddr)) {
                AR_UPDATE_STATS(pDevStats, pSibStats, TxBroadcastFrames, 1);
            } else {
                AR_UPDATE_STATS(pDevStats, pSibStats, TxMulticastFrames, 1);
            }
        } else {
#if !defined(ECOS_NOTDONE)
            END_ERR_ADD(pArDev->pOSHandle, MIB2_OUT_ERRS, +1);
#else
//	    ECOS_NOTDONE_XXX_C("MIB2_OUT_ERRS");
#endif
            AR_UPDATE_STATS(pDevStats, pSibStats, TxFramesDropped, 1);
        }

        /* WDC tx status */
        switch (pTxStatus->status) {
        case AR_TRANSMIT_OK:
            txInfo.status = TX_OK;
            break;

        case AR_EXCESSIVE_RETRIES:
        case AR_FILTERED:
            txInfo.status = TX_ERR;
            break;

        default:
            /* Should never hit this */
            ASSERT(0);
            txInfo.status = TX_ERR;
        }

        txInfo.timeStamp     = pTxStatus->timestamp;
        txInfo.retryCount    = pTxStatus->retryCount;
        txInfo.xmitRate      = 0;  /* xxxxPGSxxxx - TODO */
        txInfo.ackRssi       = 0;  /* xxxxPGSxxxx - TODO */

        if (pTxDesc->txCookie) {
            A_TASK_UNLOCK();
            arSendTxComplete(pArDev, pTxDesc->txCookie, &txInfo);
            A_TASK_LOCK();
            pTxDesc->txCookie = 0;
        }

        /*
         * Free target tx descriptors (up until holding descriptor)
         *
         * Note: the descriptors are already popped from the tx queue
         *       at this point
         */

        while (pFreeDesc && (pFreeDesc != pTxDesc)) {
            AR_DESC *pTmpDesc = pFreeDesc;
            pFreeDesc = pFreeDesc->pNextVirtPtr;

            if (pTmpDesc->txCookie) {
                A_TASK_UNLOCK();
                arSendTxComplete(pArDev, pTmpDesc->txCookie, &txInfo);
                A_TASK_LOCK();
                pTmpDesc->txCookie = 0;
            }

            ASSERT(pTmpDesc->txCookie == 0);
            arRecycleTxDesc(pArDev, pTmpDesc);
        }

#if 0
        HISTOGRAM_STOP_TIMER(arTxTimer, 0);
#endif
    }

#if 0
    HISTOGRAM_STOP_COUNTER(arTx, 0);
#endif
}

static A_BOOL
txQueueIsEmpty(
    AR_QUEUE_INFO *pQueue
)
{
    AR_DESC *pFirstDesc;

    pFirstDesc = pQueue->pDescQueueHead;
    if (!pFirstDesc) {
        return TRUE;
    }

    if (pFirstDesc->staleFlag) {
        if (!pFirstDesc->pNextVirtPtr) {
            return TRUE;
        }
    }

    return FALSE;
}

/*
 * Returns TRUE if all Transmits have been sent and reaped.
 * FALSE if there is still TX activity in progress.
 */
static A_BOOL
txIsDone(
    AR_DEV_INFO *pArDev
)
{
    A_BOOL isDone = TRUE;

    if (pArDev->config.abolt & ABOLT_BURST) {
        if (!txQueueIsEmpty(pArDev->bssTable[0].pBurstQueue))
            isDone = FALSE;
    }
#ifdef WME
    if (pArDev->config.WmeEnabled) {
        if (!txQueueIsEmpty(&pArDev->baseBss.wmequeues[TXQ_ID_FOR_AC0])) {
            isDone = FALSE;
        }
        if (!txQueueIsEmpty(&pArDev->baseBss.wmequeues[TXQ_ID_FOR_AC1])) {
            isDone = FALSE;
        }
        if (!txQueueIsEmpty(&pArDev->baseBss.wmequeues[TXQ_ID_FOR_AC2])) {
            isDone = FALSE;
        }
        if (!txQueueIsEmpty(&pArDev->baseBss.wmequeues[TXQ_ID_FOR_AC3])) {
            isDone = FALSE;
        }
    } else {
        if (!txQueueIsEmpty(&pArDev->baseBss.txQueue)) {
            isDone = FALSE;
        }
    }
#else
    if (!txQueueIsEmpty(pArDev->bssTable[0].pTxQueue)) {
        isDone = FALSE;
    }
#endif

    return isDone;
}

void
arTxCompleteHandler(AR_DEV_INFO *pArDev)
{
    A_TASK_LOCK();
    if (pArDev->config.abolt & ABOLT_BURST) {
        arProcessTxQueueLocked(pArDev, pArDev->bssTable[0].pBurstQueue);
    }

#ifdef WME
    if (pArDev->config.WmeEnabled) {
        arProcessTxQueueLocked(pArDev, &pArDev->baseBss.wmequeues[TXQ_ID_FOR_AC0]);
        arProcessTxQueueLocked(pArDev, &pArDev->baseBss.wmequeues[TXQ_ID_FOR_AC1]);
        arProcessTxQueueLocked(pArDev, &pArDev->baseBss.wmequeues[TXQ_ID_FOR_AC2]);
        arProcessTxQueueLocked(pArDev, &pArDev->baseBss.wmequeues[TXQ_ID_FOR_AC3]);
    } else {
        arProcessTxQueueLocked(pArDev, &pArDev->baseBss.txQueue);
    }
#else
    arProcessTxQueueLocked(pArDev, pArDev->bssTable[0].pTxQueue);
#endif
    A_TASK_UNLOCK();

}


/*
 * sequenceNumberRecover - called during excessiveRetry
 *  processing; good candidate to be moved (part of the
 *  functionality) to HAL going forward
 *  Note that this routine does lookahead processing
 *  of the remainder of a fragment train after the
 *  frame which suffered the excessive retry.  The state
 *  variable which denotes this is the fragSeqNumValid
 *  counter.  Because of the lookahead, we assume that
 *  the caller ensures that the fragment train is marked
 *  done by the DMA before calling us.
 */

static void
sequenceNumberRecover(AR_DEV_INFO *pArDev, AR_DESC *pDesc)
{
    AR_DESC           *head, *tail;
    WLAN_FRAME_HEADER *pWlanHdr;
    A_UINT16           seqNum = 0;

    head = pDesc;
    tail = pDesc->pTxLastDesc;

    pWlanHdr = head->pBufferVirtPtr.header;

    /*
     *  Capture sequence number :
     *  1)  always if non-fragmented
     *  2)  for fragments, only if not previously captured
     */
    if ((!pWlanHdr->frameControl.moreFrag) &&
        (WLAN_GET_FRAGNUM(pWlanHdr->seqControl) == 0))
    {
        WLAN_SET_SEQNUM(pWlanHdr->seqControl, tail->status.tx.seqNum);
        return;
    }
    if (!head->fragSeqNumValid) {
        seqNum = tail->status.tx.seqNum;
        WLAN_SET_SEQNUM(pWlanHdr->seqControl, seqNum);
        head->fragSeqNumValid = 1;
    }

    /*
     *  For a fragment train, insert the seqNum from the error frame
     *  into every subsequent fragmented frame...but only do it once.
     *  Every subsequent frame will be marked filtered.
     */
    while (pWlanHdr->frameControl.moreFrag) {
        head = tail->pNextVirtPtr;
        ASSERT(head != NULL); // if more frags, then more descs
        tail = head->pTxLastDesc;

        pWlanHdr = head->pBufferVirtPtr.header;

        /* we know the rest of train is done already and it's all filtered */
        A_DESC_CACHE_INVAL(tail);
        ASSERT(halGetTxDescDone(pArDev, tail));

        if (head->fragSeqNumValid == 0) {
            /* Set the seqNum saved from the descriptor above.
             * Frame is still in network order.
             */
            WLAN_SET_SEQNUM(pWlanHdr->seqControl, seqNum);
            head->fragSeqNumValid = 1;
        }
    }
    return;
}


/*******************************************************************************
 * ar5MacSend - main transmit routine handling a (chain of) frame(s) to
 *   be transmitted; flow enters from DS/SME code, powersave code, and
 *   swRetry code
 *   Consumes the frames passed to it
 */
A_STATUS
ar5MacSend(AR_DEV_INFO *pArDev, AR_QUEUE_INFO *pQueue, AR_DESC *pHead, A_BOOL psPolledFrame)
{
    AR_DESC   *pTxDesc, *pLastTail = NULL;
    A_BOOL    burstTrigger = FALSE;
    A_BOOL    isBurstQueue = FALSE;
    A_STATUS  status       = A_OK;

    A_TASK_LOCK();

#if !defined(ECOS_NOTDONE)
    if (HACK_TO_PDEV(pArDev)->NicResetInProgress) {

        freeBuffandDescChain(pArDev->hostHandle, AR_2_ATHEROS(pHead));
        logMsg("ar5MacSend: Failed\n");
        status = A_ERROR;
        goto exit;
    }
#endif

    ASSERT(pHead != NULL);
    ASSERT(pQueue != NULL);

    if ((pQueue == pHead->pOpBss->pBurstQueue) &&
        (pArDev->config.modeCTS != PROT_MODE_NONE))
    {
        INIT_WLAN_INTR_LOCK(intKey);
        ASSERT(pArDev->config.abolt & ABOLT_BURST);
        isBurstQueue = TRUE;

        LOCK_WLAN_INTR(intKey);
        ASSERT(!pQueue->queuingBurst);
        /*
         * This request if going to start a new burst so trigger it
         * after it was queued
         */
        burstTrigger = !pQueue->qBurstCount;
        pQueue->queuingBurst = TRUE;
        UNLOCK_WLAN_INTR(intKey);
    }

    for (pTxDesc = pHead; pTxDesc; pTxDesc = pLastTail ? pLastTail->pNextVirtPtr : pHead) {
        AR_CONN_INFO *pSib = pTxDesc->pArConnInfo;


        /*
         * Ensure only properly cleanedup descriptors the first time around; assert
         * to validate the assumed redundancy in the original code
         */
        ASSERT(pTxDesc->swRetryFlag || (!pTxDesc->pBufferVirtPtr.header->frameControl.retry));
        ASSERT(pTxDesc->swRetryFlag || (!pTxDesc->fragSeqNumValid));

        /* hold onto frames while we're in the swretry process */
        if ((!pTxDesc->swRetryFlag) && pSib && pSib->numFilteredPending) {
            /*
             * group frames should never end up on the swretry process! they
             * succeed the first time around
             */
            ASSERT(!isGrp(&pTxDesc->pBufferVirtPtr.header->address1));
            ASSERT(pArDev->config.swTxRetries != 0);

            /* splice out this frame and put it on the swretry queue */
            spliceFrame_HELPER(&pHead, pLastTail, pTxDesc->pTxLastDesc);
            arDescQPushTail(&pSib->swRetryQueue, pTxDesc, pTxDesc->pTxLastDesc);
            ++pSib->swRetryQueue.qFrameCount;

            continue;
        }

#ifdef BUILD_AP
        /*
         * Handle power management here (for the AP only); need to handle it
         * on a per frame basis as this frame chain could be the swretry queue
         * getting drained and only some type of frames end up on the PS queue
         */
        if ((!psPolledFrame) && (pArDev->config.serviceType == WLAN_AP_SERVICE)
            && (apHandlePowerMgt(pArDev, &pHead, pLastTail, pTxDesc) == A_OK))
        {
            /*
             * the above may have dropped the frame; but since the
             * STA side code is the only one that cares about the
             * error code, and this particular conditonal being AP
             * side code only - it should be OK to return A_OK here
             */
            continue;
        }
#endif

        /*
         * this will determine rate, duration etc. for the frame,
         * setting up the various desc fields like clearDestMask
         * etc; may also apply hw/platform specific policies;
         * converts it to the right byte order
         */
        arTxService(pArDev, pTxDesc);
        A_DESC_CACHE_FLUSH(pTxDesc);

        /* book-keeping */
#ifdef UPSD
        if (!(pTxDesc->txQandMode & TX_IS_UPSD))
#endif
        {
            pQueue->qFrameCount++;
            if (pSib) {
                pSib->numTxPending++;
            }
        }

        pLastTail = pTxDesc->pTxLastDesc;
    }

    if (!pHead) {
        if (isBurstQueue)
        {
            A_BOOL      burstNow;
            INIT_WLAN_INTR_LOCK(intKey);
            /* See if there was burst complete action which happened
             * while we were processing descriptors which is signaled
             * by burstQueing set to FALSE
             */
            LOCK_WLAN_INTR(intKey);
            burstNow =  !pQueue->queuingBurst;
            pQueue->queuingBurst = FALSE;
            if (burstNow && pQueue->qBurstCount > 0) {
                pQueue->qBurstCount -= 1;
            }
            UNLOCK_WLAN_INTR(intKey);

            if (burstNow) {
                /*
                 * The ISR  missed a chance to burst pending descriptors
                 * so we have to start another burst, but the problem could be
                 * that we may pending burst to station which entered PS Mode
                 */
                halStartTxDma(pArDev, pQueue->halTxQueueNum);
            }
        }
        goto exit;
    }
    ASSERT(pLastTail);

#if !defined(ECOS_NOTDONE)
    /* shouldn't ever be here for cabQueue */
    ASSERT(!pQueue->isCabQueue);
#if (defined(UPSD) && defined(BUILD_AP))
    {
        if (pHead->txQandMode & TX_IS_UPSD) {
            queueSwTxDescriptor(pdevInfo, pQueue, pHead, pLastTail, 2);
            goto exit;
        }
    }
#endif
#endif

    /* Add to tx queue -  kick-start the hw if empty queue */
    arQueueTxDescriptorLocked(pArDev, pQueue, pHead, pLastTail);

    /*
     * Note this write is not strictly necessary if hw was
     * behind sw and hadn't hit null yet; alternate strategy
     * would be to defer it until we get a TXEOL interrupt;
     * we decided to avoid the interrupt cost
     */
    if (isBurstQueue)
    {
        A_BOOL      burstNow;
        INIT_WLAN_INTR_LOCK(intKey);

        LOCK_WLAN_INTR(intKey);
        burstNow =  !pQueue->queuingBurst;
        pQueue->queuingBurst = FALSE;
        if (burstNow && pQueue->qBurstCount > 0) {
            pQueue->qBurstCount -= 1;
        }
        UNLOCK_WLAN_INTR(intKey);

        if (burstTrigger || burstNow) {
            /*
             * Start the Burst if our queueing started a burst queue
             * or if the ISR has missed triggering the Burst since
             * we are here manipulating the queue
             */
            halStartTxDma(pArDev, pQueue->halTxQueueNum);
        }
    } else {
        halStartTxDma(pArDev, pQueue->halTxQueueNum);
    }
    
 exit:
    A_TASK_UNLOCK();
    
    return(status);
}


/*
 * drainRetryQueue - Drain a software retry queue by moving queued
 *   frames to the transmit queue.  Called from wlanFrameRetransmit when
 *   either we've sucessfully software-retried a frame or when we've given
 *   up trying to software-retry a frame.  If called while software retry
 *   is still in progress, wlanDrainRetryQueue does nothing.
 *
 * RETURNS: void
 */
static void
drainRetryQueue(AR_DEV_INFO *pArDev, AR_QUEUE_INFO *pQueue, AR_CONN_INFO *pSib)
{
    AR_DESC *pTxDesc;

    if ((!pSib->numFilteredPending) && pSib->swRetryQueue.qFrameCount) {
        pTxDesc = pSib->swRetryQueue.pDescQueueHead;
        pSib->swRetryQueue.pDescQueueHead =
            pSib->swRetryQueue.pDescQueueTail = NULL;
        pSib->swRetryQueue.qFrameCount = 0;
        ar5MacSend(pArDev, pQueue, pTxDesc, FALSE);
    }
}


/*
 *
 * frameRetransmit - entry point for swretried frames re-joining the
 *   tx flow; called by the tx completion code; operates on one frame/
 *   frag-train at a time - called when frame is still on the tx queue
 *   with this frame being at the head of the queue (behind any holding
 *   descriptor that may be)
 *   Incoming frame is in network order
 *
 * RETURNS:
 *   A_OK if the frame gets retransmitted and/or tx queue gets reorganized
 *   A_ERROR if frame hit swretry limit and won't be retransmitted; the
 *       transmit queue should be left undisturbed in this case
 */
static A_STATUS
frameRetransmit(AR_DEV_INFO *pArDev, AR_QUEUE_INFO *pQueue, AR_DESC *pThisTail)
{
    AR_DESC           *pTxDesc    = pThisTail->pTxFirstDesc;
    AR_DESC_TX_STATUS *pTxStatus  = &pThisTail->status.tx;
    AR_CONN_INFO      *pSib       = pTxDesc->pArConnInfo;

    ASSERT(pTxStatus->status != AR_TRANSMIT_OK);

    /* safeguard against issues which were seen in Crete gen */
    ASSERT((pTxStatus->status == AR_EXCESSIVE_RETRIES) || (pTxStatus->retryCount == 0));

    /* swretry related asserts */
    ASSERT((pTxStatus->status == AR_EXCESSIVE_RETRIES) || pSib->numFilteredPending
            || swrDump(pArDev, pSib));
    ASSERT((pTxStatus->status == AR_EXCESSIVE_RETRIES) || (pTxDesc->swretryCount == 0)
            || swrDump(pArDev, pSib));
    ASSERT((pTxStatus->status == AR_EXCESSIVE_RETRIES)
            || (pTxDesc->pBufferVirtPtr.header->frameControl.retry == 0)
            || swrDump(pArDev, pSib));

    if (pTxStatus->status == AR_EXCESSIVE_RETRIES) {
#ifdef VPRINT
        txPrintf("sw %d\n", pQueue->halTxQueueNum);
#endif

        /* this frame was attempted by hw - there can't be any filters
         * pending; there can be nothing on the swRetryQueue at this
         * time; atleast this frame is still been counted towards
         * transmits pending
         */
        if  (!(pArDev->config.abolt & ABOLT_BURST) ||
            (pQueue != pTxDesc->pOpBss->pBurstQueue))
        {
            ASSERT((!pSib->numFilteredPending) || swrDump(pArDev, pSib));
        }
        ASSERT((!pSib->swRetryQueue.qFrameCount) || swrDump(pArDev, pSib));
        ASSERT(pSib->numTxPending);

        /* next frame queued up will need to have the clearDestMask
         * bit set; capture the num of transmits pending to stop new
         * frames from getting queued ahead of the retransmits; all frames
         * currently on the powersave queue have to move to the swretry
         * queue as the retransmits need to be ahead of them also
         */
        pSib->needClearDest = TRUE;
        pSib->numFilteredPending = pSib->numTxPending;
        apMovePsToSwRetryQueue(pSib);

        ++pTxDesc->swretryCount;

        if ((pTxDesc->swretryCount > pArDev->config.swTxRetries) || (!pArDev->bTxEnabled)) {
            /*
             * this is going to be discarded by the processTxQueue code;
             * account for the filteredPending count; we could ideally
             * cause all following fragments of this one to be discarded
             * too.. but the code isn't organized well for that case - so
             * live with this unoptimal behavior for now!
             */
            --pSib->numFilteredPending;
            AR_UPDATE_STATS(&pArDev->devStats, &pSib->stats,
                            swRetryMaxRetriesExceeded, 1);
            drainRetryQueue(pArDev, pQueue, pSib);

            return A_ERROR;
        }
        /* total of all frames ever retried on this connection */
        AR_UPDATE_STATS(&pArDev->devStats, &pSib->stats, swRetryTotalCnt, 1);

        /* Capture some transient stuff on the first swretry */
        if (pTxDesc->swretryCount == 1) {
            sequenceNumberRecover(pArDev, pTxDesc);
        }
        pTxDesc->pBufferVirtPtr.header->frameControl.retry = 1;
    }

    /* this should never be called on the cabQ */
    ASSERT(!pQueue->isCabQueue);

    /*
     * the frame failed! swretryLookAhead has already ensured that
     * either this frame/frag-train is the last frame on the transmit
     * queue, or the frame following this frame/frag-train is marked
     * done; we don't need the holding descriptor in either case;
     * also this frame is at the head of the tx queue with possibly
     * only the holding descriptor ahead of it
     */
    ASSERT(pQueue->pDescQueueHead);
    if (pQueue->pDescQueueHead->staleFlag) {
        /* use the following order to keep it race-free against ISR
         * code operations of the kind that happen on the cabQ
         */
        AR_DESC *pVarDesc = pQueue->pDescQueueHead;
        pQueue->pDescQueueHead = pVarDesc->pNextVirtPtr;

        pVarDesc->pNextVirtPtr = NULL;
        pVarDesc->nextPhysPtr = 0;
        arRecycleTxDesc(pArDev, pVarDesc);
    }

    /* need to submit entire fragtrain in one shot */
    ASSERT(pQueue->pDescQueueHead == pTxDesc);
    while (pTxDesc) {
        pTxDesc->swRetryFlag = 1;
        pThisTail = pTxDesc->pTxLastDesc;

        while(pTxDesc != pThisTail) {
            /*
             * We don't care to change the last descriptor's next phy ptr
             * since we reset thisTail later
             */
            pTxDesc = pTxDesc->pNextVirtPtr;
        }
        /* temporary book-keeping */
        ASSERT(pQueue->qFrameCount);
        pQueue->qFrameCount--;
        ASSERT(pSib->numTxPending);
        pSib->numTxPending--;
        pSib->numFilteredPending--;

        if (!pTxDesc->pTxFirstDesc->pBufferVirtPtr.header->frameControl.moreFrag) {
            break;
        }
        pTxDesc = pThisTail->pNextVirtPtr;
        ASSERT(halGetTxDescDone(pArDev, pTxDesc->pTxLastDesc));
    }

    /* take these frames off the main queue */
    pTxDesc = pQueue->pDescQueueHead;
    pQueue->pDescQueueHead = pThisTail->pNextVirtPtr;
    if (!pQueue->pDescQueueHead) {
        pQueue->pDescQueueTail = NULL;
    }
    pThisTail->pNextVirtPtr = NULL;
    pThisTail->nextPhysPtr = 0;

    ar5MacSend(pArDev, pQueue, pTxDesc, FALSE);

    drainRetryQueue(pArDev, pQueue, pSib);

    return A_OK;
}

/*
 * swretryLookAhead - helper used in context of processTxQueue to
 *  discern our ability to swretry in the current state of the
 *  transmit queue
 *
 * RETURNS: A_OK for green light
 *          A_ERROR for red light
 */

static A_STATUS
swretryLookAhead(AR_DEV_INFO *pArDev, AR_DESC *pTxDesc)
{
    AR_CONN_INFO *pSib = pTxDesc->pTxFirstDesc->pArConnInfo;

    /* won't swretry successful frames, on disabled, or without a sib */
    if ((pTxDesc->status.tx.status == AR_TRANSMIT_OK) ||
        (pArDev->config.swTxRetries == 0) || (!pSib))
    {
        /* frame should be tried by hw and successfully completed only
         * if there were no filters pending
         */
        return A_OK;
    }

    /* an errored fragment train needs to be fully baked
     * for us to handle it
     */
    while (pTxDesc->pTxFirstDesc->pBufferVirtPtr.header->frameControl.moreFrag) {
        pTxDesc = pTxDesc->pNextVirtPtr->pTxLastDesc;
        A_DESC_CACHE_INVAL(pTxDesc);
        if (!halGetTxDescDone(pArDev, pTxDesc)) {
            return A_ERROR;
        }
    }

    /* we can handle if this is the last frame on the tx queue */
    if (!pTxDesc->pNextVirtPtr) {
        return A_OK;
    }

    /* if we're not the last descriptor in the transmit queue
     * swretry can be attempted until the next descriptor is done
     */
    pTxDesc = pTxDesc->pNextVirtPtr->pTxLastDesc;
    A_DESC_CACHE_INVAL(pTxDesc);
    if (halGetTxDescDone(pArDev, pTxDesc)) {
        return A_OK;
    }
    return A_ERROR;
}

void
resetTxQueue(AR_DEV_INFO *pArDev, AR_QUEUE_INFO *pQueue)
{
    AR_DESC  *pFirstDesc;
    AR_STATS *pConnStats;

    ASSERT(pQueue);

#if 0
    A_SEM_LOCK(pQueue->qSem, WAIT_FOREVER);
#else
    A_TASK_LOCK();
#endif

    pFirstDesc = pQueue->pDescQueueHead;

    while (pFirstDesc) {
        ASSERT(pFirstDesc->pNextVirtPtr != pFirstDesc);

#if !defined(ECOS_NOTDONE)
        END_ERR_ADD(pVportBss->vportCommon.pOSHandle, MIB2_OUT_ERRS, +1);
#endif


#if defined(NO_CONN_STATS)
        pConnStats = &dummyStats;
#else
        pConnStats = pFirstDesc->pArConnInfo ? 
                     &pFirstDesc->pArConnInfo->stats : &dummyStats;
#endif

        AR_UPDATE_STATS(&pArDev->devStats, pConnStats, TxFramesDropped, 1);

        if (pFirstDesc->txCookie != 0) {
            TRANSMIT_INFO txInfo;

            txInfo.status     = TX_ERR;
            txInfo.timeStamp  = 0;  /* xxx */
            txInfo.retryCount = 0;  /* xxx */
            txInfo.xmitRate   = 0;  /* xxxxPGSxxxx - TODO */
            txInfo.ackRssi    = 0;  /* xxxxPGSxxxx - TODO */

            A_TASK_UNLOCK();
            arSendTxComplete(pArDev, pFirstDesc->txCookie, &txInfo);
            A_TASK_LOCK();
            pFirstDesc->txCookie = 0;
        }

        pFirstDesc = pFirstDesc->pTxLastDesc->pNextVirtPtr;
    }

    if ((pFirstDesc = pQueue->pDescQueueHead)) {
        pQueue->pDescQueueHead   = NULL;
        pQueue->pDescQueueTail   = NULL;
        pQueue->qFrameCount      = 0;
        pQueue->qBurstCount      = 0;
        pQueue->pBurstHeadDesc   = NULL;
        pQueue->pBurstTailDesc   = NULL;
        pQueue->queuingBurst     = FALSE;
        pQueue->pendBFrameCount  = 0;
        pQueue->burstCTSDur      = 0;

        arRecycleTxDescChain(pArDev, pFirstDesc);
    }

#if 0
    A_SEM_UNLOCK(pQueue->qSem);
#else
    A_TASK_UNLOCK();
#endif
}

