/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/targDispatch.c#3 $
 *
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 */

#include "wlantype.h"
#include "wdcApi.h"
#include "wdcMsg.h"
#include "queue.h"
#include "arMsgApi.h"
#include "arDev.h"
#include "arMsgApi.h"
#include "targWdc.h"
#include "masterInfo.h"
#include "arReceive.h"
#include "arTransmit.h"
#include "arEvent.h"
#include "halApi.h"
/* Build-time version linking between Host WLAN driver and
  * Target Firmware.
  */
#include "../../host/include/wlanvers.h"

/*
 * WDC Control Message Dispatch
 *
 * There is a Control Message Dispatch thread that takes WDC Messages
 * off of a Control Message Dispatch Queue, and then calls the appropriate
 * WDC function.  Messages are put onto the end of the queue by the
 * WDC Message layer (when it receives a message from the master).
 * If the Dispatcher thread gets through all available Control Messages,
 * it is suspended until a new message arrives.  The number of elements
 * in the Dispatch Queue is limited to CTRL_MSG_INCOMING_COUNT.
 */

/* Initialize the WDC Control Message Dispatch Queue. */
A_STATUS
TARG_wdcCtrlMsgQInit(MASTER_HANDLE masterHandle)
{
    int         i;
    A_STATUS    status;

    TAILQ_INIT(&masterHandle->CtrlMsgQHead);

    TAILQ_INIT(&masterHandle->CtrlMsgQFree);
    for (i=0; i<CTRL_MSG_INCOMING_COUNT; i++) {
        struct CtrlMsgQ *msgQItem;

        msgQItem = A_DRIVER_MALLOC(sizeof(*msgQItem));
        if (!msgQItem) {
            ASSERT(0);
            status = A_NO_MEMORY;
            goto msgQInitExit;
        }
        A_MEM_ZERO(msgQItem, sizeof(*msgQItem));

        TAILQ_INSERT_TAIL(&masterHandle->CtrlMsgQFree, msgQItem, Qlinkage);
    }

    A_CNTSEM_INIT(masterHandle->CtrlMsgQSem, 0);

    status = A_OK;

msgQInitExit:
    return status;
}

/*
 * Enqueue a WDC message to be handled by the WDC Message Dispatcher.
 * Called when a message is received.
 */
void
TARG_wdcCtrlMsgEnqueue(
    MASTER_HANDLE masterHandle,
    WDC_MSG *wdcMsg
)
{
    struct CtrlMsgQ *msgQItem;

    A_TASK_LOCK();
    /* Get the first msgQItem from the free list */
    msgQItem = TAILQ_FIRST(&masterHandle->CtrlMsgQFree);
    ASSERT(msgQItem);

    /* ...and pop it off the free list */
    TAILQ_REMOVE(&masterHandle->CtrlMsgQFree, msgQItem, Qlinkage);
    ASSERT(msgQItem);
    ASSERT(msgQItem->wdcMsg == NULL);

    msgQItem->wdcMsg = wdcMsg;

    /* Now put the msgQItem at the end of the Control Message Dispatch Queue */
    TAILQ_INSERT_TAIL(&masterHandle->CtrlMsgQHead, msgQItem, Qlinkage);
    A_TASK_UNLOCK();

    /* Notify Control Message Dispatcher that a new message awaits. */
    A_CNTSEM_POST(masterHandle->CtrlMsgQSem);
}

/*
 * Dequeue a WDC message to be handled by the WDC Message Dispatcher.
 * Called by the message dispatcher.
 * If no message is available, suspend until one is available.
 */
WDC_MSG *
TARG_wdcCtrlMsgDequeue(
    MASTER_HANDLE masterHandle
)
{
    struct CtrlMsgQ *msgQItem;
    WDC_MSG *wdcMsg;

    /* Wait for a Control Message to become available. */
    A_CNTSEM_WAIT(masterHandle->CtrlMsgQSem);

    /* Synchronize with interrupt-level code, trying to add a new Message */
    A_TASK_LOCK();

    /* Pull the first Message off the Control Message Dispatch Queue */
    msgQItem = TAILQ_FIRST(&masterHandle->CtrlMsgQHead);
    ASSERT(msgQItem);
    TAILQ_REMOVE(&masterHandle->CtrlMsgQHead, msgQItem, Qlinkage);

    wdcMsg = msgQItem->wdcMsg;
    ASSERT(wdcMsg);

    msgQItem->wdcMsg = NULL;
    TAILQ_INSERT_TAIL(&masterHandle->CtrlMsgQFree, msgQItem, Qlinkage);
    A_TASK_UNLOCK();

    return wdcMsg;
}

/*
 * The number of bytes that can hold "most" incoming WDC Control Messages.
 * The Control Message Dispatcher cache-invalidates this many bytes of
 * the message before processing the message. If the message's length
 * indicates that it's longer than the amount invalidated, the Dispatcher
 * then invalidates more.
 */
#define WDC_CTRL_MSG_LENGTH 64

/*
 * A quick lookup table for per-MessageType information
 * that helps during dispatch.
 */
struct {
    A_BOOL       responseImplied;         /* This message type always demands
                                             a Target Response.  Used for any
                                             "GET" messages from the Host. */
#if defined(DEBUG)
    char         *opcodeName;             /* For debug messages */
#endif
} msgTypeInfo[WDCMSG_MAX];

static void
TARG_wdcCtrlMsgTypeInit(void)
{
    msgTypeInfo[WDCMSG_TARGET_GET_CAPABILITY].responseImplied       = TRUE;
    msgTypeInfo[WDCMSG_TARGET_GET_STATUS].responseImplied           = TRUE;
    msgTypeInfo[WDCMSG_GET_CONNECTION_STATS].responseImplied        = TRUE;
    msgTypeInfo[WDCMSG_FLUSH].responseImplied                       = TRUE;

#if defined(DEBUG)
    msgTypeInfo[WDCMSG_BIND].opcodeName =
        "WDCMSG_BIND";

    msgTypeInfo[WDCMSG_UNBIND].opcodeName =
        "WDCMSG_UNBIND";

    msgTypeInfo[WDCMSG_TARGET_GET_CAPABILITY].opcodeName =
        "WDCMSG_TARGET_GET_CAPABILITY";

    msgTypeInfo[WDCMSG_TARGET_SET_CONFIGURATION].opcodeName =
        "WDCMSG_TARGET_SET_CONFIGURATION";

    msgTypeInfo[WDCMSG_TARGET_GET_STATUS].opcodeName =
        "WDCMSG_TARGET_GET_STATUS";

    msgTypeInfo[WDCMSG_TARGET_START].opcodeName =
        "WDCMSG_TARGET_START";

    msgTypeInfo[WDCMSG_TARGET_STOP].opcodeName =
        "WDCMSG_TARGET_STOP";

    msgTypeInfo[WDCMSG_TARGET_SET_CHANNEL].opcodeName =
        "WDCMSG_TARGET_SET_CHANNEL";

    msgTypeInfo[WDCMSG_TARGET_RESET].opcodeName =
        "WDCMSG_TARGET_RESET";

    msgTypeInfo[WDCMSG_TARGET_ENABLE].opcodeName =
        "WDCMSG_TARGET_ENABLE";

    msgTypeInfo[WDCMSG_TARGET_DISABLE].opcodeName =
        "WDCMSG_TARGET_DISABLE";

    msgTypeInfo[WDCMSG_STA_JOIN_BSS].opcodeName =
        "WDCMSG_STA_JOIN_BSS";

    msgTypeInfo[WDCMSG_STA_ASSOC_BSS].opcodeName =
        "WDCMSG_STA_ASSOC_BSS";

    msgTypeInfo[WDCMSG_STA_START_BSS].opcodeName =
        "WDCMSG_STA_START_BSS";

    msgTypeInfo[WDCMSG_STA_DETACH_BSS].opcodeName =
        "WDCMSG_STA_DETACH_BSS";
    
    msgTypeInfo[WDCMSG_UPDATE_BSS_ATTRIBUTE].opcodeName =
        "WDCMSG_UPDATE_BSS_ATTRIBUTE";

    msgTypeInfo[WDCMSG_UPDATE_BSS_IE].opcodeName =
        "WDCMSG_UPDATE_BSS_IE";

    msgTypeInfo[WDCMSG_CREATE_CONNECTION].opcodeName =
       "WDCMSG_CREATE_CONNECTION";

    msgTypeInfo[WDCMSG_UPDATE_CONNECTION_ATTRIBUTE].opcodeName =
        "WDCMSG_UPDATE_CONNECTION_ATTRIBUTE";

    msgTypeInfo[WDCMSG_SET_CONNECTION_KEY].opcodeName =
        "WDCMSG_SET_CONNECTION_KEY";

    msgTypeInfo[WDCMSG_GET_CONNECTION_STATS].opcodeName =
        "WDCMSG_GET_CONNECTION_STATS";

    msgTypeInfo[WDCMSG_DELETE_CONNECTION].opcodeName =
        "WDCMSG_DELETE_CONNECTION";

    msgTypeInfo[WDCMSG_SETUP_TX_QUEUE].opcodeName =
        "WDCMSG_SETUP_TX_QUEUE";

    msgTypeInfo[WDCMSG_UPDATE_TX_QUEUE_ATTRIBUTE].opcodeName =
        "WDCMSG_UPDATE_TX_QUEUE_ATTRIBUTE";

    msgTypeInfo[WDCMSG_SET_RX_FILTER].opcodeName =
        "WDCMSG_SET_RX_FILTER";

    msgTypeInfo[WDCMSG_SET_RX_MULTICAST_FILTER].opcodeName =
        "WDCMSG_SET_RX_MULTICAST_FILTER";

    msgTypeInfo[WDCMSG_INIT_RX_MULTICAST_FILTER].opcodeName =
        "WDCMSG_INIT_RX_MULTICAST_FILTER";

    msgTypeInfo[WDCMSG_CLEAR_RX_MULTICAST_FILTER].opcodeName =
        "WDCMSG_CLEAR_RX_MULTICAST_FILTER";

    msgTypeInfo[WDCMSG_SEND].opcodeName =
        "WDCMSG_SEND";

    msgTypeInfo[WDCMSG_RESTORE_INFO].opcodeName =
        "WDCMSG_RESTORE_INFO";

    msgTypeInfo[WDCMSG_FLUSH].opcodeName =
        "WDCMSG_FLUSH";

    msgTypeInfo[WDCMSG_TARGET_STATS_UPDATE].opcodeName =
        "WDCMSG_TARGET_STATS_UPDATE";

    msgTypeInfo[WDCMSG_CONNECTION_STATS_UPDATE].opcodeName =
        "WDCMSG_CONNECTION_STATS_UPDATE";

    msgTypeInfo[WDCMSG_BEACON_MISS_INTERRUPT].opcodeName =
        "WDCMSG_BEACON_MISS_INTERRUPT";

    msgTypeInfo[WDCMSG_STORE_INFO].opcodeName =
        "WDCMSG_STORE_INFO";

    msgTypeInfo[WDCMSG_RETRIEVE_INFO].opcodeName =
        "WDCMSG_RETRIEVE_INFO";

    msgTypeInfo[WDCMSG_TARGET_BEACON_UPDATE].opcodeName =
        "WDCMSG_TARGET_BEACON_UPDATE";

    msgTypeInfo[WDCMSG_ADHOC_MERGE_UPDATE].opcodeName =
        "WDCMSG_ADHOC_MERGE_UPDATE";

    msgTypeInfo[WDCMSG_DEVICE_AVAILABLE].opcodeName =
        "WDCMSG_DEVICE_AVAILABLE";

    msgTypeInfo[WDCMSG_DEVICE_UNAVAILABLE].opcodeName =
        "WDCMSG_DEVICE_UNAVAILABLE";

    msgTypeInfo[WDCMSG_DEBUG_MESSAGE_NOTIFY].opcodeName =
        "WDCMSG_DEBUG_MESSAGE_NOTIFY";

    msgTypeInfo[WDCMSG_TARGET_SET_PWR_MODE].opcodeName =
        "WDCMSG_TARGET_SET_PWR_MODE";

#endif /* DEBUG*/
}



/*
 * Target message dispatcher.
 *
 * When a message is received from the host, this function is called
 * to decode it and dispatch to the appropriate message handler.
 */
void
TARG_wdcCtrlMsgDispatch(
    MASTER_HANDLE     masterHandle,
    WDC_MSG           *wdcMsg)
{
    A_UINT32          msgLength;
    WDC_MSG_OPCODE    msgOpcode;
    char              *msgBuffer;
    A_STATUS          status = A_OK;
    DEVICE_HANDLE     deviceHandle    = masterHandle->deviceHandle;
    MASTER_MSG_HANDLE masterMsgHandle = masterHandle->masterMsgHandle;
    WDC_MSG_CURSOR    cursor = 0;
    A_UINT32          responseDesired;
    WDC_MSG           *responseMsg;
    AR_DEV_INFO       *pArDev         = (AR_DEV_INFO *) masterHandle->deviceHandle;

    /* Invalidate enough to read the message length */
    A_DATA_CACHE_INVAL(wdcMsg, WDC_CTRL_MSG_LENGTH);
    msgLength  = wdcMsg->msgLength;
    if (wdcMsg->msgLength > WDC_CTRL_MSG_LENGTH) {
        /* It's a long incoming message, so we need to invalidate more */
        A_DATA_CACHE_INVAL(wdcMsg, wdcMsg->msgLength);
    }
    msgOpcode  = wdcMsg->msgOpcode;
    msgBuffer  = (char *)wdcMsg->msgBuffer;
    // diag_printf("control msg recv : length = %d opcode = %d \n",msgLength,msgOpcode);

    /* Validate message */
    ASSERT(msgLength <= WDC_MSG_LENGTH_MAX);
    ASSERT(msgOpcode > WDCMSG_INVAL);
    ASSERT(msgOpcode < WDCMSG_MAX);

    responseDesired = wdcMsg->r.responseDesired;
    if (responseDesired ||
        msgTypeInfo[msgOpcode].responseImplied) {
        /* Create response message */
        responseMsg = TARG_wdcCtrlMsgCreate(masterMsgHandle, msgOpcode);
        if (!responseMsg) {
            status = A_NO_MEMORY;
            ASSERT(0);
            return;
        }
    } else {
        responseMsg = NULL;
    }

#ifdef DEBUG
    uiPrintf("%s: RECV %s %d bytes msgId=0x%x %s\n",
                __FUNCTION__,
                msgTypeInfo[msgOpcode].opcodeName ?
                    msgTypeInfo[msgOpcode].opcodeName : "UNKNOWN_OPCODE",
                msgLength,
                wdcMsg->msgId,
                wdcMsg->r.responseDesired ? "RESP" : "NORESP");

    WDCMSG_PRINT(wdcMsg);

#endif

    switch(msgOpcode) {

    case WDCMSG_HOST_AVAILABLE: {
        A_UINT32 ath_sw_ver_major;
         A_UINT32 ath_sw_ver_minor;
         A_UINT32 ath_sw_ver_patch;
         A_UINT32 ath_sw_ver_build;

         TARG_wdcMsgReadParam(wdcMsg, &cursor, &ath_sw_ver_major);
         TARG_wdcMsgReadParam(wdcMsg, &cursor, &ath_sw_ver_minor);
         TARG_wdcMsgReadParam(wdcMsg, &cursor, &ath_sw_ver_patch);
         TARG_wdcMsgReadParam(wdcMsg, &cursor, &ath_sw_ver_build);

         if ((ath_sw_ver_major != ATH_SW_VER_MAJOR) ||
             (ath_sw_ver_minor != ATH_SW_VER_MINOR) ||
             (ath_sw_ver_patch != ATH_SW_VER_PATCH) ||
             (ath_sw_ver_build != ATH_SW_VER_BUILD)) {
             /* If there is a version mis-match, reset the target
              * and re-download new firmware. */
             hal_ar5523_reset();
         }

        TARG_wdcHostAvailable(masterHandle, deviceHandle, &status);
        break;
    }

    case WDCMSG_BIND: {
        A_UINT32 hostApiVersion;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, &hostApiVersion);
        TARG_wdcBind(masterHandle, deviceHandle, hostApiVersion, &status);

        break;
    }

    case WDCMSG_UNBIND: {
        TARG_wdcUnbind(masterHandle, deviceHandle);
        break;
    }

    case WDCMSG_TARGET_GET_CAPABILITY: {
        TARGET_CAP_ID       capId;
        A_UINT32            capSz;
        A_STATUS            status;

        ASSERT(responseMsg);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&capId);

        /* Determine if this is fixed- or variable-sized capability data */
        TARG_wdcTargetGetCapability(deviceHandle,
                                    capId,
                                    &capSz,
                                    NULL,
                                    &status);
        ASSERT(status == A_OK);

        TARG_wdcMsgAddParam(responseMsg, capSz);

        if (capSz) {
            /* Variable-sized capability data */
            TARGET_CAP_DATA     pCapData;
            A_UINT32            maxLength;
            A_UINT32            newCapSz;

            /*
             * We'll fetch the variable-size capability directly into
             * the response message buffer.
             */
            TARG_wdcMsgPrepData(responseMsg, &pCapData, &maxLength);
            ASSERT(capSz <= maxLength);

            TARG_wdcTargetGetCapability(deviceHandle,
                                        capId,
                                        &newCapSz,
                                        pCapData,
                                        &status);
            ASSERT(newCapSz == capSz);
            TARG_wdcMsgCommitData(responseMsg, newCapSz);
        } else {
            /* Fixed-sized capability data */
            TARGET_CAP_DATA     capData;

            TARG_wdcTargetGetCapability(deviceHandle,
                                        capId,
                                        &capSz,
                                        &capData,
                                        &status);
            ASSERT(capSz == 0);
            TARG_wdcMsgAddParam(responseMsg, (A_UINT32)capData);
        }

        break;
        }

    case WDCMSG_TARGET_SET_CONFIGURATION: {
        TARGET_CONFIG_ID    cfgId;
        A_UINT32            cfgSz;
        A_STATUS            status;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&cfgId);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&cfgSz);

        /* Determine if this is fixed- or variable-sized configuration data */
        if (cfgSz) {
            /* Variable-sized configuration data */
            TARGET_CONFIG_VAL     *pCfgData;

            /*
             * We'll set the variable-size configuration directly from
             * the message buffer.
             */
            TARG_wdcMsgSeekData(wdcMsg, &cursor, (void **)&pCfgData);

            TARG_wdcTargetSetConfiguration(deviceHandle,
                                           cfgId,
                                           cfgSz,
                                           pCfgData,
                                           &status);
        } else {
            /* Fixed-sized configuration data */
            TARGET_CONFIG_VAL     cfgData;

            TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&cfgData);

            TARG_wdcTargetSetConfiguration(deviceHandle,
                                           cfgId,
                                           0,
                                           cfgData,
                                           &status);
        }

        break;
        }

    case WDCMSG_TARGET_GET_STATUS: {
        TARGET_STATUS_ID    statusId;
        A_UINT32            statusSz;
        A_STATUS            status;

        ASSERT(responseMsg);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&statusId);

        /* Determine if this is fixed- or variable-sized Status data */
        TARG_wdcTargetGetStatus(deviceHandle,
                                statusId,
                                &statusSz,
                                NULL,
                                &status);
        ASSERT(status == A_OK);

        TARG_wdcMsgAddParam(responseMsg, statusSz);

        if (statusSz) {
            /* Variable-sized status data */
            TARGET_STATUS_DATA  pStatusData;
            A_UINT32            maxLength;

            /*
             * We'll fetch the variable-size status directly into
             * the response message buffer.
             */
            TARG_wdcMsgPrepData(responseMsg, &pStatusData, &maxLength);
            ASSERT(statusSz <= maxLength);

            TARG_wdcTargetGetStatus(deviceHandle,
                                    statusId,
                                    &statusSz,
                                    pStatusData,
                                    &status);
            TARG_wdcMsgCommitData(responseMsg, statusSz);
        } else {
            /* Fixed-sized Status data */
            TARGET_STATUS_DATA     statusData;

            TARG_wdcTargetGetStatus(deviceHandle,
                                    statusId,
                                    &statusSz,
                                    &statusData,
                                    &status);
            ASSERT(statusSz == 0);
            TARG_wdcMsgAddParam(responseMsg, (A_UINT32)statusData);
        }

        break;
    }
    
    case WDCMSG_TARGET_GET_STATS: {
        arNotifyDeviceStats(masterHandle);
        break;
    }

    case WDCMSG_TARGET_START: {
        TARG_wdcTargetStart(deviceHandle, &status);
        {
            static int iStartCount;
            struct mallinfo mem_info;
            mem_info = mallinfo();
            iStartCount++;
            isrPrintf("Target started: Memory used: %d Free: %d [%d]\n\n",
                      mem_info.uordblks,
                      mem_info.fordblks, iStartCount);
        }
        if (responseMsg) {
            TARG_wdcMsgAddParam(responseMsg, 
                halGetRandomSeed((AR_DEV_INFO *)deviceHandle));
        }
        break;
    }

    case WDCMSG_TARGET_STOP: {
        TARG_wdcTargetStop(deviceHandle, &status);
        break;
    }

    case WDCMSG_TARGET_SET_CHANNEL: {
        CHANNEL_BAND        channelBand;
        CHANNEL_FREQ        channelFrequency;
        PHY_MODS            phyModulations;
        A_UINT32            maxRDPower;
        A_UINT32            cfgCtl;
        A_UINT32            twiceAntennaReduction;
        A_UINT32            param;
		A_BOOL              keepRCContent;

        ASSERT(sizeof(CHANNEL_BAND) <= sizeof(A_UINT32));
        ASSERT(sizeof(CHANNEL_FREQ) <= sizeof(A_UINT32));
        ASSERT(sizeof(PHY_MODS) <= sizeof(A_UINT32));

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        channelBand = (CHANNEL_BAND) param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        channelFrequency = (CHANNEL_FREQ) param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        phyModulations = (PHY_MODS) param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        maxRDPower = (A_UINT32) param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        cfgCtl = (A_UINT32) param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        twiceAntennaReduction = (A_UINT32) param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        keepRCContent = (A_BOOL) param;

        TARG_wdcTargetSetChannel(deviceHandle, channelBand, channelFrequency,
                                 phyModulations, maxRDPower,cfgCtl,
                                 twiceAntennaReduction, keepRCContent, &status);

        uiPrintf("%s: SET_CHANNEL chan %x freq: %x phymod: %x\n",
                 __FUNCTION__,
                 channelBand,
                 channelFrequency,
                 phyModulations);
        break;
    }

    case WDCMSG_TARGET_RESET: {
	    TARG_wdcTargetReset(deviceHandle, &status);
	    break;
    }

    case WDCMSG_TARGET_ENABLE:
        TARG_wdcTargetEnable(deviceHandle, &status);
        break;

    case WDCMSG_TARGET_DISABLE:
        TARG_wdcTargetDisable(deviceHandle, &status);
        break;

    case WDCMSG_STA_JOIN_BSS: {
        A_UINT32            wdcBssId;
        WLAN_MACADDR       *pBssId;
        WDC_BSS_ATTRIBUTES *pBssConfig;
        A_UINT32            sz;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&wdcBssId);

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&sz);
        ASSERT(sz == sizeof(*pBssId));
        TARG_wdcMsgSeekData(wdcMsg, &cursor, (void **) &pBssId);
        TARG_wdcMsgConsumeData(wdcMsg, &cursor, sizeof(*pBssId));
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&sz);
        ASSERT(sz == sizeof(*pBssConfig));
        TARG_wdcMsgSeekData(wdcMsg, &cursor, (void **) &pBssConfig);

        TARG_wdcStaJoinBss(deviceHandle, wdcBssId, pBssId, pBssConfig);
        break;
    }
    case WDCMSG_STA_ASSOC_BSS: {
        A_UINT32            wdcBssId;
        A_UINT32            assocId;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&wdcBssId);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&assocId);

        TARG_wdcStaAssocBss(deviceHandle, wdcBssId, assocId);
        break;
    }
    case WDCMSG_STA_START_BSS: {
        A_UINT32            wdcBssId;
        A_UINT32            sz;
        void                *pBuffer;
        WDC_BUFFER_DESC     beaconBufferDesc;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&wdcBssId);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&sz);

        if (sz) {
            TARG_wdcMsgSeekData(wdcMsg, &cursor, (void **) &pBuffer);

            A_MEM_SET(&beaconBufferDesc, 0, sizeof(beaconBufferDesc));
            beaconBufferDesc.pBuffer        = pBuffer;
            beaconBufferDesc.bufferLength   = sz;
            beaconBufferDesc.frameLength    = sz;

            TARG_wdcStartBss(deviceHandle, wdcBssId, &beaconBufferDesc);
        } else {
            TARG_wdcStartBss(deviceHandle, wdcBssId, NULL);
        }
        break;
    }
    case WDCMSG_STA_DETACH_BSS: {
        A_UINT32            wdcBssId;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&wdcBssId);
        TARG_wdcDetachBss(deviceHandle, wdcBssId);
        break;
    }
    case WDCMSG_UPDATE_BSS_ATTRIBUTE: {
        A_UINT32            wdcBssId;
        WDC_BSS_ATTRIBUTE   attribute;
        A_UINT32            cfgSz;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&wdcBssId);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&attribute);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&cfgSz);

        /* Determine if this is fixed- or variable-sized configuration data */
        if (cfgSz) {
            /* Variable-sized configuration data */
            TARGET_CONFIG_VAL     *pCfgData;

            /*
             * We'll set the variable-size configuration directly from
             * the message buffer.
             */
            TARG_wdcMsgSeekData(wdcMsg, &cursor, (void **)&pCfgData);

            TARG_wdcUpdateBssAttribute(deviceHandle,
                                       wdcBssId,
                                       attribute,
                                       cfgSz,
                                       pCfgData);
        } else {
            /* Fixed-sized configuration data */
            TARGET_CONFIG_VAL     cfgData;

            TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&cfgData);

            TARG_wdcUpdateBssAttribute(deviceHandle,
                                       wdcBssId,
                                       attribute,
                                       0,
                                       cfgData);
        }

        break;
        }

    case WDCMSG_UPDATE_BSS_IE:
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;

    case WDCMSG_CREATE_CONNECTION: {
        WLAN_MACADDR       *pMacAddr;
        A_UINT32            wdcConnId;
        A_UINT32            wdcBssId;
        CONN_ATTRIBUTES     *pOptions;
        A_UINT32            sz;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&sz);
        ASSERT(sz == sizeof(*pMacAddr));
        TARG_wdcMsgSeekData(wdcMsg, &cursor, (void **) &pMacAddr);
        TARG_wdcMsgConsumeData(wdcMsg, &cursor, sizeof(*pMacAddr));

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&wdcConnId);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&wdcBssId);

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&sz);
        ASSERT(sz == sizeof(*pOptions));
        TARG_wdcMsgSeekData(wdcMsg, &cursor, (void **) &pOptions);


        TARG_wdcCreateConnection(deviceHandle, pMacAddr, wdcConnId,
                                 wdcBssId, pOptions);
        break;
     }

    case WDCMSG_UPDATE_CONNECTION_ATTRIBUTE: {
        A_UINT32            wdcConnId;
        CONN_ATTRIBUTE      attribute;
        A_UINT32            cfgSz;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&wdcConnId);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&attribute);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&cfgSz);

        /* Determine if this is fixed- or variable-sized configuration data */
        if (cfgSz) {
            /* Variable-sized configuration data */
            TARGET_CONFIG_VAL     *pCfgData;

            /*
             * We'll set the variable-size configuration directly from
             * the message buffer.
             */
            TARG_wdcMsgSeekData(wdcMsg, &cursor, (void **)&pCfgData);

            TARG_wdcUpdateConnectionAttribute(deviceHandle,
                                              wdcConnId,
                                              attribute,
                                              cfgSz,
                                              pCfgData);
        } else {
            /* Fixed-sized configuration data */
            TARGET_CONFIG_VAL     cfgData;

            TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&cfgData);

            TARG_wdcUpdateConnectionAttribute(deviceHandle,
                                              wdcConnId,
                                              attribute,
                                              0,
                                              cfgData);
        }

        break;
        }

    case WDCMSG_SET_CONNECTION_KEY:
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;

    case WDCMSG_GET_CONNECTION_STATS:
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;

    case WDCMSG_DELETE_CONNECTION: {
        A_UINT32            wdcConnId;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&wdcConnId);
        TARG_wdcDeleteConnection(deviceHandle, wdcConnId);
        break;
    }
    case WDCMSG_SETUP_TX_QUEUE:
    {

        A_UINT32            wdcTxQueueId;
        TXQ_ATTRIBUTES     *pTxQueueInfo;
    	A_UINT32            param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        wdcTxQueueId = param;

        ASSERT(wdcTxQueueId  <  AR_MAX_QUEUES);

        pTxQueueInfo = &pArDev->txQueueInfo[wdcTxQueueId];

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        pTxQueueInfo->priority = param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        pTxQueueInfo->aifs = param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        pTxQueueInfo->logCwMin = param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        pTxQueueInfo->logCwMax = param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        pTxQueueInfo->burstTime = param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        pTxQueueInfo->compression = param;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
        pTxQueueInfo->qosMode = param;

        pArDev->txQueueInitialized |= (1 << wdcTxQueueId);
        TARG_wdcSetupTxQueue(deviceHandle, wdcTxQueueId, pTxQueueInfo);
	    break;
    }
    case WDCMSG_UPDATE_TX_QUEUE_ATTRIBUTE:
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;

    case WDCMSG_SET_RX_FILTER: {
        RX_FILTER_FLAGS     rxFilter;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&rxFilter);
        TARG_wdcSetRxFilter(deviceHandle, rxFilter, &status);
        break;
    }

    case WDCMSG_SET_RX_MULTICAST_FILTER:
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;

    case WDCMSG_INIT_RX_MULTICAST_FILTER: {
        A_UINT32 bReset;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, &bReset);
        TARG_wdcInitRxMulticastFilter(deviceHandle, bReset ? TRUE : FALSE);
        break;
    }

    case WDCMSG_CLEAR_RX_MULTICAST_FILTER:
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;

    case WDCMSG_SEND:
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;

    case WDCMSG_RESTORE_INFO:
        /* Received a message that we're not ready to handle yet */
        ASSERT(0);
        break;

    case WDCMSG_FLUSH: {
        A_UINT32   disableEnableCounter;

        /*
         * All previous Control Messages must have been handled already --
         * they're all handled serially.
         *
         * Synchronize with the Data Message Dispatcher.
         */
        
        A_CNTSEM_POST(pArDev->dataDispatcherBarrier);
        A_CNTSEM_WAIT(pArDev->ctrlDispatcherBarrier);

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&disableEnableCounter);
        ASSERT(responseMsg);
        TARG_wdcMsgAddParam(responseMsg, disableEnableCounter);

        break;
    }

    case WDCMSG_TARGET_SET_PWR_MODE: {
        TARGET_POWER_MODE wdcPsMode;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&wdcPsMode);
        TARG_wdcTargetSetPsMode(deviceHandle, wdcPsMode, &status);
        ASSERT(status == A_OK);
        break;
    }

    case WDCMSG_TARGET_SET_LED_STATE: {
        A_UINT32            ledState;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&ledState);
        TARG_wdcSetLedState(deviceHandle, ledState);
        break;
    }
    case WDCMSG_BEACON_MISS_ACK: {
        A_UINT32            status;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&status);
        TARG_wdcBmissAck(deviceHandle, (A_BOOL)status);
        break;
    }

    case WDCMSG_TARGET_SET_LED_STEADY_MODE: {
    	A_UINT32    ledNum, mode;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&ledNum);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&mode);
        TARG_wdcSetLedSteadyMode(deviceHandle, ledNum, mode);
        break;
    }

    case WDCMSG_TARGET_SET_LED_BLINK_MODE: {
    	A_UINT32    ledNum, ledMode, blinkRate, slowMode;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&ledNum);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&ledMode);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&blinkRate);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&slowMode);
        TARG_wdcSetLedBlinkMode(deviceHandle, ledNum, ledMode, blinkRate, slowMode);
        break;
    }
    
    case WDCMSG_TARGET_APPLY_MKK_IRREV_UPDATE: {
        A_UINT32 enOddU1, undoUpdate;
         
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&enOddU1);
        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&undoUpdate);
        halApplyMKKIrreversibleUpdate(pArDev, enOddU1, undoUpdate);
        halFillCapabilityInfo(pArDev);

        break;
     }
    case WDCMSG_TARGET_MKK_LOCK_EEPROM: {
		A_UINT32    param;
		A_BOOL      readOnly;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
		readOnly = (A_BOOL) param;
		halMkkLockEeprom(pArDev, readOnly);
        break;
     }
    case WDCMSG_TARGET_RAW_EEPROM_READ: {
		A_UINT32    param;
		A_UINT16    offset,data;

        TARG_wdcMsgReadParam(wdcMsg, &cursor, (A_UINT32 *)&param);
		offset = (A_UINT16) param;
		halRawEepromRead(pArDev, offset, &data);
        TARG_wdcMsgAddParam(responseMsg, (A_UINT32)data);
        break;
     }
     
    default:
        /* Unknown message */
        ASSERT(0);
    }

    if (responseMsg) {
        REFLECT_MSG_ID(wdcMsg, responseMsg);
        REFLECT_MSG_DEBUG(wdcMsg, responseMsg);

        responseMsg->r.targetStatus = status;

#ifdef DEBUG
        uiPrintf("%s: RESP %s %d bytes msgId=0x%x Status: 0x%x\n",
                    __FUNCTION__,
                    msgTypeInfo[responseMsg->msgOpcode].opcodeName ?
                        msgTypeInfo[responseMsg->msgOpcode].opcodeName :
                        "UKNOWN_OPCODE",
                    responseMsg->msgLength,
                    responseMsg->msgId,
                    responseMsg->r.targetStatus);

        WDCMSG_PRINT(responseMsg);
#endif

        /* Finally, send the response message to the master. */
        TARG_wdcCtrlMsgSend(masterMsgHandle, responseMsg);
    } else {
#ifdef DEBUG
	uiPrintf("%s:     %s Status: 0x%x\n",
		 __FUNCTION__,
		 (msgTypeInfo[msgOpcode].opcodeName ?
		  msgTypeInfo[msgOpcode].opcodeName : "UKNOWN_OPCODE"),
		 status);
#endif
    }
}


/*
 * Target Control Message dispatch loop.
 *
 * This is the main loop of the Control Message Dispatch Thread.
 */
void
TARG_wdcCtrlMsgDispatchLoop(MASTER_HANDLE masterHandle)
{
    WDC_MSG *wdcMsg;

    A_TASK_SET_HIGH_PRIORITY();

    /*
     * Loop forever, removing messages from the Control Message
     * Dispatch Queue and dispatching them.
     */
    for(;;) {
        /* Pull the next message from the Control Message Dispatch Queue */
        wdcMsg = TARG_wdcCtrlMsgDequeue(masterHandle);

        /* Process the request. */
        TARG_wdcCtrlMsgDispatch(masterHandle, wdcMsg);

        /*
         * We're done with this message buffer.
         * Recycle it so that it can be used to receive the
         * next Control Message.
         */
        TARG_wdcCtrlMsgRecv(wdcMsg);
    }
}


/*
 * Initialize the Control Message Dispatcher.
 */
A_STATUS
TARG_wdcCtrlMsgDispatchInit(MASTER_HANDLE masterHandle)
{
    A_STATUS      status = A_OK;

    /* Setup information about message types that facilitates dispatch. */
    TARG_wdcCtrlMsgTypeInit();

    /* Initialize the Message Queue. */
    status = TARG_wdcCtrlMsgQInit(masterHandle);
    if (status != A_OK) {
        goto initExit;
    }

    /* Then, start the dispatcher loop */
#if defined(AGGRESSIVE_TASK_STACKSZ)
    // Measured usage on 05/24/2004: 2184
    A_TASK_CREATE_STACKSZ(TARG_wdcCtrlMsgDispatchLoop, masterHandle, 3072);
#else
    A_TASK_CREATE(TARG_wdcCtrlMsgDispatchLoop, masterHandle);
#endif


initExit:

    return status;
}
