/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/targMsg.c#3 $
 *
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 */

/* Target-Side Messaging */

#include "wlantype.h"
#include "wdcApi.h"
#include "wdcMsg.h"
#include "queue.h"
#include "arMsgApi.h"
#include "arDev.h"
#include "arMsgApi.h"
#include "targWdc.h"
#include "masterInfo.h"
#include "arReceive.h"
#include "arTransmit.h"
#include "athusbdrv.h"
#include "targTransport.h"
#include "targMsg.h"


/* Which pipe to use for control messages */
#define USB_PIPE_NUM_CONTROL  0

/* Which pipe to use for data (RX) messages */
#define USB_PIPE_NUM_DATA     1

struct master_msg_info theMasterMsgStruct;
MASTER_MSG_HANDLE theMasterMsgHandle = &theMasterMsgStruct;

/* Forward references */
static void TARG_wdcMsgRecvInd(MASTER_MSG_HANDLE masterMsgHandle,
    A_UINT8 pipeNum, A_UINT8 *wdcMsg, A_UINT32 bytesReceived);

static void TARG_wdcMsgSendCnf(MASTER_MSG_HANDLE masterMsgHandle,
    A_UINT8 pipeNum, A_UINT8 *pBuffer, A_UINT32 bytesSend);

static void _OutgoingCtrlMsgFreeListInit(MASTER_MSG_HANDLE masterMsgHandle);

static WDC_MSG * _OutgoingCtrlMsgAlloc(MASTER_MSG_HANDLE masterMsgHandle);

static void _OutgoingCtrlMsgFree(MASTER_MSG_HANDLE masterMsgHandle, WDC_MSG *wdcMsg);

void TARG_wdcCtrlMsgRecv(WDC_MSG *wdcMsg);

/* USB API defines this as the way to determine message sizes per pipe */
extern A_UINT32  sendPipePktSize[];
extern A_UINT32  recvPipePktSize[];


MASTER_MSG_HANDLE
TARG_wdcMsgInit(MASTER_HANDLE masterHandle)
{
    DRV_HANDLE   drvHandle;
    A_STATUS     status;
    A_UINT8      sendPipes;
    A_UINT8      recvPipes;
    A_UINT32     ctrlPipeRecvSz;
    A_UINT32     ctrlPipeSendSz;
    int          i;

    sendPipePktSize[USB_PIPE_NUM_DATA] = WDC_DATA_MSG_LENGTH_MAX;
    recvPipePktSize[USB_PIPE_NUM_DATA] = WDC_DATA_MSG_LENGTH_MAX;

    status = athUsbDrvInit(0,
                           theMasterMsgHandle,
                           (RECV_INDICATION_HANDLER)TARG_wdcMsgRecvInd,
                           (SEND_CONFIRM_HANDLER)TARG_wdcMsgSendCnf,
                           &drvHandle,
                           &sendPipes,
                           &recvPipes);
    if (status != A_OK) {
        ASSERT(0); /* TBD */
    }

    ctrlPipeRecvSz = recvPipePktSize[USB_PIPE_NUM_CONTROL];
    ctrlPipeSendSz = sendPipePktSize[USB_PIPE_NUM_CONTROL];

    theMasterMsgHandle->masterHandle       = masterHandle;
    theMasterMsgHandle->drvHandle          = drvHandle;
    theMasterMsgHandle->sendPipes          = sendPipes;
    theMasterMsgHandle->recvPipes          = recvPipes;
    theMasterMsgHandle->ctrlPipeRecvSz     = ctrlPipeRecvSz;
    theMasterMsgHandle->ctrlPipeSendSz     = ctrlPipeSendSz;

    /*
     * Give the messaging layer buffers to receive Control Messages from the Host.
     */
    for (i=0; i<CTRL_MSG_INCOMING_COUNT; i++) {
        theMasterMsgHandle->ctrlPipeRecvBuf[i] =
            (WDC_MSG *)A_DRIVER_MALLOC(ctrlPipeRecvSz);
        if (theMasterMsgHandle->ctrlPipeRecvBuf[i] == NULL) {
            ASSERT(0); /* TBD */
        }

        TARG_wdcCtrlMsgRecv(theMasterMsgHandle->ctrlPipeRecvBuf[i]);
    }

    /*
     * Create the free list for outgoing Control Messages to the Host.
     */
    _OutgoingCtrlMsgFreeListInit(theMasterMsgHandle);
    for (i=0; i<CTRL_MSG_OUTGOING_COUNT; i++) {
        WDC_MSG *wdcMsg;

        wdcMsg = (WDC_MSG *)A_DRIVER_MALLOC(theMasterMsgHandle->ctrlPipeRecvSz);
        ASSERT(wdcMsg);
        _OutgoingCtrlMsgFree(theMasterMsgHandle, wdcMsg);
    }

    return theMasterMsgHandle;
}

void
TARGET_wdcMsgFini(
    MASTER_MSG_HANDLE masterMsgHandle
)
{
    A_STATUS    status;
    int         i;

    status = athUsbDrvShutdown(masterMsgHandle->drvHandle);

    if (status != A_OK) {
        ASSERT(0); /* TBD */
    }

    for (i=0; i<CTRL_MSG_INCOMING_COUNT; i++) {
        A_DRIVER_FREE(theMasterMsgHandle->ctrlPipeRecvBuf[i],
                      theMasterMsgHandle->ctrlPipeRecvSz);
        theMasterMsgHandle->ctrlPipeRecvBuf[i] = NULL;
    }

    for (i=0; i<CTRL_MSG_OUTGOING_COUNT; i++) {
        WDC_MSG *wdcMsg;

        wdcMsg = _OutgoingCtrlMsgAlloc(theMasterMsgHandle);
        A_DRIVER_FREE(wdcMsg, theMasterMsgHandle->ctrlPipeRecvSz);
    }

    masterMsgHandle->drvHandle = NULL;
}
    

void
TARG_wdcMsgRecvInd(
    MASTER_MSG_HANDLE masterMsgHandle,
    A_UINT8           pipeNum,
    A_UINT8          *pMsg,
    A_UINT32          bytesReceived)
{
    if (bytesReceived == 0) {
        /* Incomplete receive.  Re-receive the packet. */
        /* xxxxLMWxxxx TODO: Prevent endless retry */
        if (pipeNum == USB_PIPE_NUM_CONTROL) {
            TARG_wdcCtrlMsgRecv((WDC_MSG *) pMsg);
        } else {
            ASSERT(pipeNum == USB_PIPE_NUM_DATA);
            TARG_wdcDataMsgRecv((WDC_TXMSG *) pMsg);
        }
    } else {
        if (pipeNum == USB_PIPE_NUM_CONTROL) {
            TARG_wdcCtrlMsgEnqueue(masterMsgHandle->masterHandle, 
                                   (WDC_MSG *) pMsg);
        } else {

            ASSERT(pipeNum == USB_PIPE_NUM_DATA);

            // Invalidate the WLAN_FRAME_HEADER in addition
            // to the WDC_TXMSG_HEADER as the hal layer
            // will look into the WLAN_FRAME_HEADER.

            A_DATA_CACHE_INVAL((void *) pMsg, 
                               WDC_TXMSG_HEADER_LENGTH +
                               sizeof(WLAN_FRAME_HEADER));

            { 

#if defined(AR_POLL)
                /* To improve latency in attempts to qualify for WiFi Logo:
                 * 
                 * This code should only work under polling as this callback
                 * will occur at thread level (as opposed to DSR level) which
                 * will allow us to sleep, if needed, in the TX path.  This
                 * code also assumes that the appropriate TX lock is held so
                 * as to avoid race conditions w/ channel change & tx flushing. 
                 *
                 * If the current pending TX Queue is empty and the
                 * current incoming TX message is not a WDCMSG_FLUSH,
                 * do not queue the TX message onto the TX queue.  Rather
                 * directly add it to the hardware MAC queue.
                 *
                 */
                
                AR_DEV_INFO *pArDev = masterMsgHandle->masterHandle->deviceHandle;
                int semcount        = 0;

                A_CNTSEM_PEEK(masterMsgHandle->masterHandle->TxMsgQSem, 
                              &semcount);

                if (((WDC_TXMSG *)pMsg)->msgOpcode == WDCMSG_FLUSH) {
                    pArDev->txBarrierInProgress = TRUE;
                }

                if ((semcount == 0) &&
                    (pArDev->txBarrierInProgress != TRUE) &&
                    (pArDev->bTxEnabled == TRUE) &&
                    A_SEM_TRYLOCK(pArDev->txMsgDispatchLock)) {

                    extern void TARG_wdcTxMsgHandler(MASTER_HANDLE, 
                                                     WDC_TXMSG *);
                    TARG_wdcTxMsgHandler(masterMsgHandle->masterHandle, 
                                         (WDC_TXMSG *) pMsg);
                    A_SEM_UNLOCK(pArDev->txMsgDispatchLock);
                } else 
#endif
                {
                    TARG_wdcTxMsgEnqueue(masterMsgHandle->masterHandle->
                                         deviceHandle, (WDC_TXMSG *) pMsg);
                }
            }
        }
    }
}
/* 
 * Send Confirm -- called to confirm that the Host has received a message
 * that we sent earlier.
 */
void
TARG_wdcMsgSendCnf(
    MASTER_MSG_HANDLE masterMsgHandle,
    A_UINT8           pipeNum,
    A_UINT8           *pBuffer,
    A_UINT32          bytesSent)
{

#ifndef DO_ZERO_TERM
    if (bytesSent != sendPipePktSize[pipeNum]) {

        /* Incomplete send.  Re-send the packet. */
        /* xxxxLMWxxxx TODO: Prevent endless retry, stats */

        athUsbDrvSend(masterMsgHandle->drvHandle,
                      pipeNum,
                      pBuffer);
        return;
    } 
#endif
    if (pipeNum == USB_PIPE_NUM_CONTROL) {
        /*
         * Control Message Buffers are allocated as needed, and
         * are destroyed after the message has been sent to the Host.
         */
        TARG_wdcCtrlMsgDestroy(masterMsgHandle, (WDC_MSG *)pBuffer);
    } else { /* USB_PIPE_NUM_DATA  */
        
        /* 
         * Data Message Buffers are allocated during initialization, and
         * are re-used after the RX data has been sent to the Host.
         */
        ASSERT(pipeNum == USB_PIPE_NUM_DATA);
        arRecycleRxBuf(masterMsgHandle->masterHandle->deviceHandle,
                       WDC_RXMSG_TO_RXBUF(pBuffer));

    }
   
}

A_STATUS
TARG_wdcCtrlMsgSend(
    MASTER_MSG_HANDLE masterMsgHandle,
    WDC_MSG *wdcMsg)
{
    A_STATUS status;
    
#ifdef DO_ZERO_TERM
     // diag_printf("control msg send : length = %d opcode = %d \n",wdcMsg->msgLength, wdcMsg->msgOpcode);
    status = athUsbDrvSend(masterMsgHandle->drvHandle,
                           USB_PIPE_NUM_CONTROL,
                           (A_UINT8 *)wdcMsg,
                           wdcMsg->msgLength);
#else
    status = athUsbDrvSend(masterMsgHandle->drvHandle,
                           USB_PIPE_NUM_CONTROL,
                           (A_UINT8 *)wdcMsg);
#endif

    ASSERT(status == A_OK);
    return status;
}

/* Send a RX data message over the DATA pipe to the Host. */
A_STATUS
TARG_wdcRxMsgSend(
    MASTER_MSG_HANDLE masterMsgHandle,
    WDC_RXMSG         *rxMsg)
{
    A_STATUS status;

#if !defined(DO_ZERO_TERM)
    status = athUsbDrvSend(masterMsgHandle->drvHandle,
                           USB_PIPE_NUM_DATA,
                           (A_UINT8 *)rxMsg);
#else
    status = athUsbDrvSend(masterMsgHandle->drvHandle,
                           USB_PIPE_NUM_DATA,
                           (A_UINT8 *)rxMsg,
                           rxMsg->rxMsgLength);
#endif

    ASSERT(status == A_OK);
    return status;
}

#define WDCMSG_FREELINK(wdcMsg) ((WDC_MSG **)(wdcMsg))

static void
_OutgoingCtrlMsgFreeListInit(MASTER_MSG_HANDLE masterMsgHandle)
{
    masterMsgHandle->OutgoingCtrlMsgFreeList = NULL;
    A_CNTSEM_INIT(masterMsgHandle->OutgoingCtrlMsgSem, 0);
}

static WDC_MSG *
_OutgoingCtrlMsgAlloc(MASTER_MSG_HANDLE masterMsgHandle)
{
    WDC_MSG *wdcMsg;

    A_CNTSEM_WAIT(masterMsgHandle->OutgoingCtrlMsgSem);

    A_TASK_LOCK();
    wdcMsg = masterMsgHandle->OutgoingCtrlMsgFreeList;
    masterMsgHandle->OutgoingCtrlMsgFreeList =
        *WDCMSG_FREELINK(masterMsgHandle->OutgoingCtrlMsgFreeList);
    A_TASK_UNLOCK();

    return wdcMsg;
}

static void
_OutgoingCtrlMsgFree(
    MASTER_MSG_HANDLE masterMsgHandle,
    WDC_MSG *wdcMsg
)
{
    A_TASK_LOCK();
    *WDCMSG_FREELINK(wdcMsg) = masterMsgHandle->OutgoingCtrlMsgFreeList;
    masterMsgHandle->OutgoingCtrlMsgFreeList = wdcMsg;
    A_TASK_UNLOCK();

    A_CNTSEM_POST(masterMsgHandle->OutgoingCtrlMsgSem);
}
#undef WDCMSG_FREELINK

WDC_MSG *
TARG_wdcCtrlMsgCreate(MASTER_MSG_HANDLE masterMsgHandle, WDC_MSG_OPCODE msgOpcode)
{
    WDC_MSG *wdcMsg;

    ASSERT(msgOpcode > WDCMSG_INVAL);
    ASSERT(msgOpcode < WDCMSG_MAX);

    wdcMsg = _OutgoingCtrlMsgAlloc(masterMsgHandle);
    ASSERT(wdcMsg);
    wdcMsg->msgLength = WDCMSG_HEADER_LENGTH;
    wdcMsg->msgOpcode = msgOpcode;

    return wdcMsg;
}

void
TARG_wdcCtrlMsgDestroy(MASTER_MSG_HANDLE masterMsgHandle, WDC_MSG *wdcMsg)
{
    ASSERT(wdcMsg);
    wdcMsg->msgOpcode = WDCMSG_INVAL; /* sanity */
    _OutgoingCtrlMsgFree(masterMsgHandle, wdcMsg);
}

void
TARG_wdcCtrlMsgRecv(WDC_MSG *wdcMsg)
{
    athUsbDrvRecv(theMasterMsgHandle->drvHandle,
                  USB_PIPE_NUM_CONTROL,
                  (A_UINT8 *)wdcMsg);
}


void
TARG_wdcDataMsgRecv(WDC_TXMSG *wdcTxMsg) 
{
    athUsbDrvRecv(theMasterMsgHandle->drvHandle,
                  USB_PIPE_NUM_DATA,
                  (A_UINT8 *)wdcTxMsg);

}

/************************************************************
 * Functions to place data into a message buffer.
 *
 * For fixed-sized data, use TARG_wdcMsgAddParam.
 * For variable-sized data, there are two choices:
 * 1) Use TARG_wdcMsgAddData, which copies data from
 *    a user-supplied buffer into the message buffer
 *    and updates your position in the buffer.
 *    OR
 * 2) Use TARG_wdcMsgPrepData, manually copy the data,
 *    and then TARG_wdcMsgCommitData.  Prep returns a
 *    pointer into the message buffer, but it doesn't
 *    copy any data and it doesn't update your position.
 *    Commit updates your position.
 */

void
TARG_wdcMsgAddParam(WDC_MSG *wdcMsg, A_UINT32 param)
{
    A_UINT32 *pParam;

    pParam = (A_UINT32 *)((A_UINT32)wdcMsg + wdcMsg->msgLength);

    /* Store data */
    *pParam = param;

    /* Adjust message length */
    wdcMsg->msgLength += sizeof(A_UINT32);
}

void
TARG_wdcMsgAddData(WDC_MSG *wdcMsg, A_UINT32 length, void *data)
{
    char       *pData;
    A_UINT32   paddedLength;

    /* Round length up 4-bytes */
    paddedLength = (length + 3) & ~0x3;

    ASSERT(paddedLength + sizeof(A_UINT32) + wdcMsg->msgLength <= WDC_MSG_LENGTH_MAX);
    pData = (char *)((A_UINT32)wdcMsg + wdcMsg->msgLength);

    /* Save length of this data item */
    *((A_UINT32 *)pData) = length;
    pData += sizeof(A_UINT32);

    /* Save data itself */
    A_DRIVER_BCOPY(data, pData, length);

    /* Adjust total message length to account for size, data, and padding. */
    wdcMsg->msgLength += sizeof(A_UINT32) + paddedLength;
}

/*
 * Prepare a WDC Message for data, which will be copied into the
 * buffer by the caller.  Caller (and receiver) are responsible
 * for handling any endianness issues.  The data must subsequently
 * be commited via TARG_wdcMsgCommitData.
 *
 * Returns:
 *   pData     - the start address where caller's data should be copied
 *   maxLength - the maximum length of data that can be accommodated
 */
void
TARG_wdcMsgPrepData(WDC_MSG   *wdcMsg,
                    void *    *pData,
                    A_UINT32  *pMaxLength)
{
    char *msgData;

    msgData = (char *)((A_UINT32)wdcMsg + wdcMsg->msgLength);

    if (pData) {
        *pData = msgData;
    }

    if (pMaxLength) {
        *pMaxLength = WDC_MSG_LENGTH_MAX - wdcMsg->msgLength;
    }
}

void
TARG_wdcMsgCommitData(WDC_MSG   *wdcMsg,
                      A_UINT32  length)
{
    A_UINT32   paddedLength;

    /* Round length up 4-bytes */
    paddedLength = (length + 3) & ~0x3;

    ASSERT(paddedLength + wdcMsg->msgLength <= WDC_MSG_LENGTH_MAX);

    /* Adjust total message length to account for size, data, and padding. */
    wdcMsg->msgLength += paddedLength;
}



/************************************************************
 * Functions to extract data from a message buffer.
 *
 * For fixed-sized data, use TARG_wdcMsgReadParam.
 * For variable-sized data, there are two choices:
 * 1) Use TARG_wdcMsgReadData, which copies data from
 *    the message buffer into a user-supplied buffer
 *    and updates your "position" in the buffer.
 *    OR
 * 2) Use TARG_wdcMsgSeekData, manually copy the data,
 *    and then TARG_wdcMsgConsumeData.  Seek returns a
 *    pointer into the message buffer, but it doesn't
 *    copy any data and it doesn't update your position.
 *    Consume updates your position.
 */


/*
 * Read a single parameter from a message.
 */
void
TARG_wdcMsgReadParam(WDC_MSG        *wdcMsg,
                     WDC_MSG_CURSOR *pCursor,
                     A_UINT32       *pParam)
{
    A_UINT32       param;
    WDC_MSG_CURSOR newCursor;

    ASSERT(pParam);

    if (pCursor) {
        newCursor = *pCursor;
    } else {
        newCursor = 0;
    }

    ASSERT(newCursor <= wdcMsg->msgLength - WDCMSG_HEADER_LENGTH);

    param = *(A_UINT32 *)((A_UINT32)wdcMsg + WDCMSG_HEADER_LENGTH + newCursor);

    if (pCursor) {
        newCursor += sizeof(A_UINT32);
        *pCursor = newCursor;
    }

    *pParam = param;
}

/*
 * Read variable length data from a message.
 */
void
TARG_wdcMsgReadData(WDC_MSG        *wdcMsg,
                    WDC_MSG_CURSOR *pCursor,
                    A_UINT32       *pLength,
                    void           *pData)
{
    A_UINT32       length;
    A_UINT32       paddedLength;
    WDC_MSG_CURSOR newCursor;
    char           *msgData;

    ASSERT(pData);

    if (pCursor) {
        newCursor = *pCursor;
    } else {
        newCursor = 0;
    }

    TARG_wdcMsgReadParam(wdcMsg, &newCursor, &length);
    ASSERT(length > 0);

    /* Round length up 4-bytes */
    paddedLength = (length + 3) & ~0x3;

    msgData = (char *)((A_UINT32)wdcMsg + WDCMSG_HEADER_LENGTH + newCursor);
    A_DRIVER_BCOPY(msgData, (char *)pData, length);

    if (pCursor) {
        newCursor += paddedLength;
        *pCursor = newCursor;
    }

    if (pLength) {
        *pLength = length;
    }
}

/*
 * Seek to start of variable-sized data, which will be copied from the
 * buffer by the caller.  Caller and sender are responsible for handling
 * any endianness issues.  The data must subsequently be consumed via
 * TARG_wdcMsgConsumeData.  The cursor is modified by ConsumeData, not
 * by SeekData.
 *
 * Returns:
 *   ppData    - a pointer into the message buffer where data begins
 */
void
TARG_wdcMsgSeekData(WDC_MSG        *wdcMsg,
                    WDC_MSG_CURSOR *pCursor,
                    void           **ppData)
{
    A_UINT32       cursor;
    char           *msgData;

    ASSERT(pCursor);

    cursor = *pCursor;

    msgData = (char *)((A_UINT32)wdcMsg + WDCMSG_HEADER_LENGTH + cursor);

    if (ppData) {
        *ppData = msgData;
    }
}

/*
 * Consume variable-sized data -- in other words, move the cursor --
 * without copying any data.
 */
void
TARG_wdcMsgConsumeData(WDC_MSG        *wdcMsg,
                       WDC_MSG_CURSOR *pCursor,
                       A_UINT32       length)
{
    A_UINT32       paddedLength;
    WDC_MSG_CURSOR newCursor;

    ASSERT(pCursor);

    newCursor = *pCursor;

    /* Round length up 4-bytes */
    paddedLength = (length + 3) & ~0x3;

    newCursor += paddedLength;

    *pCursor = newCursor;
}
