/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/src/USB/wlan/target/src/targPlatform.c#3 $
 *
 * Copyright (c) 2004 Atheros Communications, Inc., All Rights Reserved
 *
 */

#include "wlantype.h"
#include "queue.h"
#include "wdcApi.h"
#include "wdcMsg.h"
#include "arDev.h"
#include "arMsgApi.h"
#include "targWdc.h"
#include "targTransport.h"
#include "targPlatform.h"
#include "halApi.h"
#include "hal.h"
#include <pkgconf/hal.h>
#include <cyg/hal/plf_misc.h>
#include <cyg/hal/plf_cpusl.h>

/* XXX: Resume Workaround */
#define AR5523_RESUME_WAR

typedef struct _PLATFORM_INTR_CTXT {
    A_INTERRUPT_OBJECT      intrObj;
    A_HANDLE                intHandle;
}PLATFORM_INTR_CTXT;

typedef enum {
    BOARD_STATE_NORMAL,
    BOARD_STATE_SUSPEND
}BOARD_STATE;

typedef struct _BOARD_PARAM {
    A_UINT32                cpuClk;             /* CPU Clk Source */
    A_UINT32                uartEnabled;        /* UART Status */
    A_UINT32                gpioEnabled;        /* GPIO Status */
    A_UINT32                clkGating;          /* Clk Gating  */
    /* Other Board Parameters to be added */
}BOARD_PARAM;
              
typedef struct _AR5523_PLATFORM_HANDLE {
    MASTER_HANDLE           masterHandle;       /* Back ptr to context sent by caller */
    MASTER_MSG_HANDLE       masterMsgHandle;    /* Ptr to Master Msg Handle */
    PLATFORM_INTR_CTXT      lineStCtxt;         /* Line state Intr Ctxt */
    BOARD_STATE             boardState;         /* Board State */
    BOARD_PARAM             boardAttrib;        /* Board Attributes */
    /* Context for other Miscellaneous 
    ** interrupts to be added */
#if defined(AR5523_RESUME_WAR)
    volatile A_UINT32       iUglyWarCnt;        /* Bug WAR - Event Counter */
#endif
}AR5523_PLATFORM_HANDLE;


#if defined(DEBUG)
#define plfPrintf diag_printf
#else
#define plfPrintf emptyfn
#endif
     
static void 
TARG_boardParamInit(BOARD_PARAM *pBoardAttrib)
{
    pBoardAttrib->cpuClk = hal_ar5523_get_cpu_clk();
    pBoardAttrib->gpioEnabled = 0;
    pBoardAttrib->uartEnabled = 0;
    pBoardAttrib->clkGating   = 1;
}

static void 
TARG_BoardSleepEvent (
    AR5523_PLATFORM_HANDLE  *plfHandle
    )
{
    A_STATUS    status = A_OK;

    if (plfHandle->boardAttrib.clkGating) {
        hal_ar5523_clk_gating_disable(TRUE);
    }

    /* Change the CPU clk to 1 Mhz Internal Clock 
     *
     * Place USB in suspend prior to setting the cpu clk to
     * 1MHZ.  XXX: In doing so, we cannot rely on the
     * 30mhz CPU clock which is derived from the USB clock.
     * To use 30mhz, invert suspend/set_cpu clk.
     */

#if defined(ARC_SUSPEND)
    hal_ar5523_set_usb_suspend(TRUE);
#endif

    plfHandle->boardAttrib.cpuClk = hal_ar5523_get_cpu_clk();
    hal_ar5523_set_cpu_clk(AR5523_CPU_CLK_INTOSC);

    status = arDevSetPowerMode(plfHandle->masterHandle->deviceHandle, 
                               AR_DEVICE_SLEEP_MODE);
    
    if (status != A_OK) {

        /* Change the CPU clk src to Normal */
        hal_ar5523_set_cpu_clk(plfHandle->boardAttrib.cpuClk);

#if defined(ARC_SUSPEND)
        hal_ar5523_set_usb_suspend(FALSE);
#endif
        
        if (plfHandle->boardAttrib.clkGating) {
            hal_ar5523_clk_gating_disable(FALSE);
        }
        
        plfPrintf("Device Sleep Failed\n");
        ASSERT(FALSE);
   
    } else {

        if (plfHandle->boardAttrib.gpioEnabled) {
            /* No GPIO changes for Now */
        }
        if (plfHandle->boardAttrib.uartEnabled) {
            hal_ar5523_uart_clk_enable(FALSE);
        }
        
        if (plfHandle->boardAttrib.clkGating) {
            hal_ar5523_clk_gating_disable(FALSE);
        }

    }
}

static void 
TARG_BoardResumeEvent (
    AR5523_PLATFORM_HANDLE  *plfHandle
    )
{
    A_STATUS    status = A_OK;

    if (plfHandle->boardAttrib.clkGating) {
        hal_ar5523_clk_gating_disable(TRUE);
    }
    /* Call the Platform HAL to resume */
    status = arDevSetPowerMode(plfHandle->masterHandle->deviceHandle, 
                               AR_DEVICE_NORMAL_MODE);

    if (status != A_OK) {
        plfPrintf ("Device awake Failed\n");
        ASSERT(FALSE);
    } else {

        /* Change the CPU clk src to Normal */
        hal_ar5523_set_cpu_clk(plfHandle->boardAttrib.cpuClk);

#if defined(ARC_SUSPEND)
        hal_ar5523_set_usb_suspend(FALSE);
#endif

        if (plfHandle->boardAttrib.uartEnabled) {
            hal_ar5523_uart_clk_enable(TRUE);
        }       
        if (plfHandle->boardAttrib.gpioEnabled) {
            /* No GPIO changes for Now */
        }
    
        if (plfHandle->boardAttrib.clkGating) {
            hal_ar5523_clk_gating_disable(FALSE);
        }
    }
}


static void 
TARG_lineChangeDsr(A_UINT32 intvector, A_UINT32 intcount, A_UINT32 intrHandle)
{
    AR5523_PLATFORM_HANDLE *plfHandle;

    plfHandle = (AR5523_PLATFORM_HANDLE *)intrHandle;

    /* 
     * No need to clear the interrupt reading the register 
     * while decoding the interrupt clears it 
     */


    /* 
     * XXX: AR5523_RESUME_WAR.
     * 
     * This is extra ugly. 
     *
     * Recommended implementation:
     *
     * USB 2.0 Transceiver Macrocell Document, Page 51
     *
     * Note: Handling glitches on the USB during suspend
     * When a device is suspended, a J State is on the bus and the SIE
     * should be looking for a K (resume) or a SE0 (reset).  In either case,
     * a glitch that looks like a K or an SE0 will cause the macrocell
     * oscillator to start up.  The macrocell must hold off any CLK
     * transitions until CLK is "usable". Once CLK is running 
     * (typically several ms later), the SIE must check LineState.
     * If a J state is asserted, then the SIE returns to the suspend state,
     * if a K state is asserted, then the SIE starts the resume process, 
     * and if an SE0 is asserted then the SIE starts the reset process.
     *
     * Problem (or what I believe it to be anyway) :
     *
     * The utmi_linestate information resides under the USB_RX_FIFO 
     * (AR5523_GPIO + 0x80C) register in GPIO address space.  
     * This register is inaccessible when USB is in low power.
     * To read this register we must first un-suspend USB.
     * 
     * Simply put, in order to determine if we should resume, we must
     * resume.
     *
     * Workaround: 
     *
     * In suspend, Attempt to resume on any linechange interrupt -- 
     * even though we may not be sure what the line change is.
     *
     * If the resume sequence times out (i.e. we fail to see 
     * an USB INIT event) after RESUME_COUNT poll attempts AND
     * the USB chip reports we're in suspend, put the platform 
     * back to sleep.  
     *
     * Workaround part deaux: 
     *
     * Because we are unable to differentiate between a glitch 
     * in linestate and an actual change (nor are we able to
     * reset the last known line state from Predator's perspective) --
     * we run the risk of going to sleep and never receiving a 
     * LINECHANGE interrupt until it's too late.  This happens
     * on NEC host controllers during Windows device disable/enable.
     * For this reason, we maintain state in the device of whether
     * or not it is 'safe' to go into boardSleep.  (yuck)  We have
     * no other workaround for this problem, as during boardsleep
     * the only state we have at our disposal is the LINECHANGE interrupt
     * bit.
     *
     * Requirements: 
     *
     * USB-IF states that the maximum current draw for
     * a high-powered device should be < 2.5mA within 10ms of a suspend.
     * As such, the resume count is set aggressively.
     *
     */

#define AR5523_WAR_RESUME_COUNT 150000

    if (plfHandle->boardState == BOARD_STATE_SUSPEND) {
        A_BOOL bSuspend = FALSE;
        plfHandle->iUglyWarCnt = 0;

        A_DISABLE_ALL_INTERRUPTS();

        /* Resume event for the platform */
        TARG_BoardResumeEvent(plfHandle);


        /* Poll until we emerge from suspend */
        while (((volatile BOARD_STATE) 
                plfHandle->boardState != BOARD_STATE_NORMAL)) {

            TARG_txportPoll(plfHandle->masterMsgHandle);

            bSuspend = TARG_txportGetLineState(plfHandle->masterMsgHandle);

            if (bSuspend &&
                (++plfHandle->iUglyWarCnt > AR5523_WAR_RESUME_COUNT) &&
                (plfHandle->masterHandle->deviceHandle->bUSBPowerSave == TRUE))
                break;
        }


        if (bSuspend &&
            (++plfHandle->iUglyWarCnt > AR5523_WAR_RESUME_COUNT) &&
            (plfHandle->masterHandle->deviceHandle->bUSBPowerSave == TRUE)) {

            /* USB Suspend Event */
            TARG_BoardSleepEvent(plfHandle);
            
            /* Change the state of the board */
            plfHandle->boardState = BOARD_STATE_SUSPEND;

            A_ENABLE_INTERRUPT(CYGNUM_HAL_INT_LINE_CHANGE_VEC);

        } else {

            /* Enable the txport */

            TARG_txportEnable(plfHandle->masterMsgHandle);
            
            A_ENABLE_INTERRUPT(CYGNUM_HAL_INT_INTCLK_VEC);
        }

        A_ENABLE_ALL_INTERRUPTS();

    } else {
        /* If we're not in SUSPEND mode and we received this interrupt 
         * reset it.  We shouldn't have to do this, but we enabled the
         * interrupt in the first place. 
         */
#if 0
        A_ENABLE_INTERRUPT(CYGNUM_HAL_INT_LINE_CHANGE_VEC);
#endif
    }
}


static A_UINT32 
TARG_lineChangeIsr(A_UINT32 intvector, A_UINT32 intrHandle)
{
    AR5523_PLATFORM_HANDLE *plfHandle;

    plfHandle = (AR5523_PLATFORM_HANDLE *)intrHandle;
    /* Mask the Interrupted vector */
    A_DISABLE_INTERRUPT(intvector);                   

    /* Clean up needed:  Call DSR from ISR here
     * Since the handling this line change is time critical. 
     * and the upper levels may have the scheduler locked
     * which would delay the processing */

    TARG_lineChangeDsr(intvector, 0, intrHandle);

    return  (CYG_ISR_HANDLED);
}

static void 
TARG_InstallLineStateChgIsr (AR5523_PLATFORM_HANDLE *plfHandle)
{
    ASSERT(plfHandle);

     /* Create the Interrupt Obj */
    A_CREATE_INTERRUPT (
        CYGNUM_HAL_INT_LINE_CHANGE_VEC,     /* Vector to attach to       */
        (A_UINT32)plfHandle,                /* Data pointer              */
        TARG_lineChangeIsr,                 /* Interrupt Service Routine */
        TARG_lineChangeDsr,                 /* Deferred Service Routine  */
        &plfHandle->lineStCtxt.intHandle,   /* returned handle           */
        &plfHandle->lineStCtxt.intrObj      /* interrupt                 */
        );
 
    /* Connect to the Interupt Object */
    if (plfHandle->lineStCtxt.intHandle) {
        A_ATTACH_INTERRUPT(plfHandle->lineStCtxt.intHandle);
    }
}



void 
TARG_platformTxportEvent(
    TARG_PLATFORM_EVENT_HANDLE  appHandle,
    TXPORT_PLATFORM_EVENT       txportEvent
    )
{
    AR5523_PLATFORM_HANDLE  *plfHandle;

    plfHandle = (AR5523_PLATFORM_HANDLE  *)appHandle;
    ASSERT(plfHandle);

    switch (txportEvent) {
        case TXPORT_SLEEP_EVENT :
            plfPrintf ("USB Sleep\n\n");

            /* XXX: Only enter suspend during Windows hibernation */
            if ((plfHandle->boardState == BOARD_STATE_NORMAL) &&
                (plfHandle->masterHandle->deviceHandle->bUSBPowerSave == TRUE))
            {

                /* Disable the txport */
                TARG_txportDisable(plfHandle->masterMsgHandle);

                A_DISABLE_INTERRUPT(CYGNUM_HAL_INT_INTCLK_VEC);

                /* USB Suspend Event */
                TARG_BoardSleepEvent(plfHandle);

                /* Change the state of the board */
                plfHandle->boardState = BOARD_STATE_SUSPEND;

                /* Enable the interrupt for the line state change event */
                A_ENABLE_INTERRUPT(CYGNUM_HAL_INT_LINE_CHANGE_VEC);
            } 
            break;

        case TXPORT_WAKE_EVENT : 
            if (plfHandle->boardState == BOARD_STATE_NORMAL) {
                plfPrintf("USB Wake\n");
            }
            plfHandle->iUglyWarCnt = 0;
            break;

        case TXPORT_RESET_EVENT : 
            if (plfHandle->boardState == BOARD_STATE_NORMAL) {
                plfPrintf("USB Reset\n");
            }
            plfHandle->iUglyWarCnt = 0;
            break;

        case TXPORT_INIT_EVENT : 
            if (plfHandle->boardState == BOARD_STATE_NORMAL) {
                plfPrintf("USB Init\n");
            } else {
                plfHandle->boardState = BOARD_STATE_NORMAL;
            }
            break;

        default:
            ASSERT (0);
    }
}


PLATFORM_HANDLE * 
TARG_platformInit(MASTER_HANDLE     masterHandle,     
                  MASTER_MSG_HANDLE masterMsgHandle
    )
{
    AR5523_PLATFORM_HANDLE *plfHandle;

    plfHandle = (AR5523_PLATFORM_HANDLE *)A_DRIVER_MALLOC(sizeof(AR5523_PLATFORM_HANDLE));
    if (!plfHandle) {
        ASSERT(0); /* To debug */
        return NULL;
    }
    A_MEM_ZERO( plfHandle, sizeof(AR5523_PLATFORM_HANDLE));

    plfHandle->masterHandle     = masterHandle;
    plfHandle->masterMsgHandle  = masterMsgHandle;
    plfHandle->boardState       = BOARD_STATE_NORMAL;
    
    /* Initialise the Borad Params */
    TARG_boardParamInit(&plfHandle->boardAttrib);

    /* Initialize the Miscellaneous Interrupt, the Interrupts 
     * are still controlled by the individual misc masks */
    A_ENABLE_INTERRUPT(CYGNUM_HAL_INT_MISC_VEC);

    /* Install the ISR for the Line change event */
    TARG_InstallLineStateChgIsr(plfHandle);

    /* Register with the USB driver for USB events */
    TARG_txportRegisterPSEvent (plfHandle->masterMsgHandle,
                                TARG_platformTxportEvent, 
                                plfHandle);

    /* Register with the WLAN driver for WLAN ps events */
    /* Create the Interrupt Obj */
    
    return ((PLATFORM_HANDLE)plfHandle);
}

