/*
 * $Id: //depot/sw/branches/1.3_USB_LINUX_port/wpa_supplicant-0.2.5/driver_athusb.c#3 $
 * 
 * Copyright (c) 2004, Atheros Communications Inc.
 *
 * WPA Supplicant - driver interaction with Atheros WLAN USB Station driver
 * 
 */

/*
 * Refer driver.h for a detailed description on the wpa_driver_ops structure
 * and the function definitions
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <errno.h>

#include "wireless_copy.h"
#include "common.h"
#include "driver.h"
#include "driver_wext.h"
#include "eloop.h"
#include "wpa_supplicant.h"
#include "linuxioctl.h"

/*
 * athusb_setvalue - Sets a private wireless parameter contained
 *                   in the private subioctl of the driver
 */
static int athusb_setvalue(const char *ifname, int Command, int Value)
{
	struct iwreq		wrq;
    int                 sd;
    int                 retval = 0;

	if ((sd = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
        perror("socket");
        return (-1);
    }

    memset(&wrq, 0, sizeof(struct iwreq));
	wrq.u.mode = Command;
    strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
    memcpy (wrq.u.name+sizeof(unsigned int), &Value, sizeof(unsigned int));
    retval = ioctl(sd, ATHUSB_IOCTL_SET_VALUE, &wrq);
    close(sd);

	return ((retval < 0) ?  -1 : 0);
}

/*
 * athusb_setpriv - Sets a private wireless parameter contained
 *                  in the private ioctl of the driver
 */
static int athusb_setpriv(const char *ifname, int Command, void *pData, int Length)
{
	struct iwreq		wrq;
    int                 sd;
    int                 retval = 0;

	if ((sd = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
        perror("socket");
        return (-1);
    }

    memset(&wrq, 0, sizeof(struct iwreq));
	wrq.u.mode = Command;
    strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
	if (Length < IFNAMSIZ) {
		memcpy(wrq.u.name, pData, Length);
	} else {
		wrq.u.data.pointer = pData;
		wrq.u.data.length = Length;
	}

    retval = ioctl(sd, Command, &wrq);
    close(sd);

	return ((retval < 0) ?  -1 : 0);
}


static int wpa_driver_athusb_set_wpa(const char *ifname, int enabled)
{
	return athusb_setvalue(ifname, ATHUSB_SUBIOCTL_WPA, enabled);
}


static int wpa_driver_athusb_set_key(const char *ifname, wpa_alg alg,
				   unsigned char *addr, int key_idx,
				   int set_tx, u8 *seq, size_t seq_len,
				   u8 *key, size_t key_len)
{
	int ret = 0;
    struct athusb_wpakey WpaKey;
    ENCRYPTION_ALGORITHM encAlg = EncryptionDisabled;

	char *alg_name;

	switch (alg) {
	case WPA_ALG_NONE:
		alg_name = "none";
        encAlg = EncryptionDisabled;
		break;
	case WPA_ALG_WEP:
		alg_name = "WEP";
        encAlg = EncryptionWEP;
		break;
	case WPA_ALG_TKIP:
		alg_name = "TKIP";
        encAlg = EncryptionTKIP;
		break;
	case WPA_ALG_CCMP:
		alg_name = "CCMP";
        encAlg = EncryptionAESCCM;
		break;
	default:
		return -1;
	}

	wpa_printf(MSG_DEBUG, "%s: alg=%s key_idx=%d set_tx=%d seq_len=%d "
		   "key_len=%d", __FUNCTION__, alg_name, key_idx, set_tx,
		   seq_len, key_len);

    /* Fill-in the WPA Key structure and pass it on to the driver */
    WpaKey.wpaAlgorithm = encAlg;
    memcpy(WpaKey.Address, addr, ETH_ALEN);
    WpaKey.KeyIndex = key_idx;
    WpaKey.DefaultTx = set_tx;
    memcpy(WpaKey.Sequence, seq, seq_len);
    WpaKey.SeqLength = seq_len;
    memcpy(WpaKey.KeyData, key, key_len);
    WpaKey.KeyLength = key_len; 
    
    /* WPA Key for TKIP is arranged differently in the supplicant and
     * driver. Rearrange before sending it to the driver */
    if (alg == WPA_ALG_TKIP &&  key_len == 32) {
        u8 KeyTemp[8];
        memcpy(KeyTemp, WpaKey.KeyData + 16, 8);
        memcpy(WpaKey.KeyData + 16, WpaKey.KeyData + 24, 8);
        memcpy(WpaKey.KeyData + 24, KeyTemp, 8);
    }

    ret = athusb_setpriv(ifname, ATHUSB_IOCTL_SETKEY, &WpaKey,
            sizeof(struct athusb_wpakey));

	return ret;
}

static int wpa_driver_athusb_set_countermeasures(const char *ifname,
						 int enabled)
{
    /* Countermeasures are taken care by the driver, and it doesnt
     * have a means to control it yet. Just ignore the command */
	wpa_printf(MSG_DEBUG, "%s: enabled=%d", __FUNCTION__, enabled);
	return (0);
}


static int wpa_driver_athusb_set_drop_unencrypted(const char *ifname,
						  int enabled)
{
	wpa_printf(MSG_DEBUG, "%s: enabled=%d", __FUNCTION__, enabled);
	return athusb_setvalue(ifname, ATHUSB_SUBIOCTL_PRIVACY, enabled);
}


static int
wpa_driver_athusb_deauthenticate(const char *ifname, u8 *addr, int reason_code)
{
	struct athusb_mlme MlmeCommand;

	wpa_printf(MSG_DEBUG, "%s", __FUNCTION__);
	MlmeCommand.mlmeRequest = MLME_REQUEST_DEAUTH;
	MlmeCommand.mlmeReason = reason_code;
	return athusb_setpriv(ifname, ATHUSB_IOCTL_SETMLME, &MlmeCommand, sizeof(struct athusb_mlme));
}

static int
wpa_driver_athusb_disassociate(const char *ifname, u8 *addr, int reason_code)
{
	struct athusb_mlme MlmeCommand;

	wpa_printf(MSG_DEBUG, "%s", __FUNCTION__);
	MlmeCommand.mlmeRequest = MLME_REQUEST_DISASSOC;
	MlmeCommand.mlmeReason = reason_code;
	return athusb_setpriv(ifname, ATHUSB_IOCTL_SETMLME, &MlmeCommand, sizeof(struct athusb_mlme));
}

static int wpa_driver_athusb_associate(const char *ifname, const char *bssid,
				       const char *ssid, size_t ssid_len,
				       int freq,
				       const char *wpa_ie, size_t wpa_ie_len,
				       wpa_cipher pairwise_suite,
				       wpa_cipher group_suite,
				       wpa_key_mgmt key_mgmt_suite)
{
	int ret = 0;
    IEEE80211_AUTHENTICATION_MODE authMode;
    ENCRYPTION_ALGORITHM encAlg = EncryptionDisabled;

	wpa_printf(MSG_DEBUG, "%s", __FUNCTION__);

    switch (key_mgmt_suite) {
    case KEY_MGMT_PSK :         /* WPA-PSK */
        authMode = AuthModeWPAPSK;
        break;
    case KEY_MGMT_802_1X:       /* WPA */
        authMode = AuthModeWPA;
        break;
    case KEY_MGMT_NONE:         /* 802.1X or None ??? */
        authMode = AuthModeAutoSwitch;
        break;
    default:
        authMode = AuthModeOpen;
        break;
    }
	if (athusb_setvalue(ifname, ATHUSB_SUBIOCTL_AUTHTYPE, authMode) < 0)
        ret = -1;

    switch (group_suite) {
        case CIPHER_NONE :
            encAlg = EncryptionDisabled;
            break;
        case CIPHER_WEP40:
        case CIPHER_WEP104:
            encAlg = EncryptionWEP;
            break;
        case CIPHER_CCMP:
            encAlg = EncryptionAESCCM;
            break;
        case CIPHER_TKIP:
            encAlg = EncryptionTKIP;
            break;
    }
	if (athusb_setvalue(ifname, ATHUSB_SUBIOCTL_ENCRYPTION, encAlg) < 0)
        ret = -1;

	if (wpa_driver_wext_set_ssid(ifname, ssid, ssid_len) < 0)
		ret = -1;
	if (wpa_driver_wext_set_bssid(ifname, bssid) < 0)
		ret = -1;

	return ret;
}

static int wpa_driver_athusb_set_auth_alg(const char *ifname, int auth_alg)
{
    IEEE80211_AUTHENTICATION_MODE authMode;

	    if(auth_alg & AUTH_ALG_LEAP){
		/* Disable LEAP in Driver for Supplicant to take over */
		athusb_setvalue(ifname,ATHUSB_SUBIOCTL_LEAPDRIVER,0);
		return athusb_setvalue(ifname,ATHUSB_SUBIOCTL_LEAPENABLED,1);
	}else if (auth_alg & AUTH_ALG_SHARED_KEY) {
        authMode = AuthModeShared;
    } else {
        authMode = AuthModeOpen;
    }
	
	    
	    athusb_setvalue(ifname,ATHUSB_SUBIOCTL_LEAPENABLED,0);
    /* What we set here doesnt matter really; the driver sets a differnt
     * authentication algorithm when WPA is selected */
	return athusb_setvalue(ifname, ATHUSB_SUBIOCTL_AUTHTYPE, auth_alg);
}


struct wpa_driver_ops wpa_driver_athusb_ops = {
	.get_bssid              = wpa_driver_wext_get_bssid,
	.get_ssid               = wpa_driver_wext_get_ssid,
	.set_wpa                = wpa_driver_athusb_set_wpa,
	.set_key                = wpa_driver_athusb_set_key,
	.events_init            = wpa_driver_wext_events_init,
	.events_deinit          = wpa_driver_wext_events_deinit,
	.set_countermeasures    = wpa_driver_athusb_set_countermeasures,
	.set_drop_unencrypted   = wpa_driver_athusb_set_drop_unencrypted,
	.scan                   = wpa_driver_wext_scan,
	.get_scan_results       = wpa_driver_wext_get_scan_results,
	.deauthenticate         = wpa_driver_athusb_deauthenticate,
	.disassociate           = wpa_driver_athusb_disassociate,
	.associate              = wpa_driver_athusb_associate,
	.set_auth_alg           = wpa_driver_athusb_set_auth_alg,
};
